############################################################
### Preparing and checking data collected on Gorilla     ###
###                                                      ###
### by Lisa Scholten, l.scholten@tudelft.nl              ### 
############################################################

# ---- FIND, FETCH AND MERGE FILES     ----
# -----------------------------------------
#? Using code from Gorilla support packages and R support
#Optional install of the 'matdir' package below. This is because the later, dcast() function, is not in the core R installation. 
#install.packages('matdir') # library(matdir) - outdated package, not available for r v.3.6.1

library(plyr)
library(dplyr)
library(tidyr)
library(reshape2)
library(lubridate)
library(purrr)

# Set working directory
# setwd("WRITE PATH")
setwd("WRITE PATH")

# List all CSV files in working directory 
# Wrapper function that accounts for file size 
# From https://stackoverflow.com/questions/39530734/list-files-taking-account-of-file-size-in-r
list_files  <- function(path = ".", pattern = NULL, min_size = 1, 
                        all.files = FALSE, full.names = FALSE, recursive = FALSE,
                        ignore.case = FALSE, include.dirs = FALSE, no..= FALSE) {
  
  pre <- list.files(path, pattern, all.files, full.names, recursive, ignore.case, 
                    include.dirs, no..)
  
  purrr::discard(pre, ~file.size(.)< min_size)
  
}

# Take all files > 1 kB (i.e. only nodes containing data)
# If taking all data, data set merges below will not work!
files <- list_files(pattern = "*.csv", min_size = 1010) # 1000 B = 1kB

# Extract node id and type
node.id <- NA
type <- NA

for (x in 1:length(files)) {
  node.id[x] <- gsub('.{4}$', '', strsplit(as.character(files[x]), "-")[[1]][3])
  type[x]    <- ifelse(grepl("questionnaire",files[x]),"q", "t")
  }

# Create lookup table
lookup <- data.frame(e.seq = NA, type = type, node = node.id, branch = NA, dom = NA,
                     stringsAsFactors = F)
lookup <- unique(lookup); row.names(lookup) <- 1:nrow(lookup) # cleanup doubles

## ---- Define node order in experiment sequence ----
e.seq <-c(# questionnaires
  "71i9" = 1, "mem2" = 2, "wrfg" = 3, "eylt" = 7, "b8fd" = 8,
  "umw3" = 9, "797b" = 10, "rhwm" = 11, "gdbs" = 12, "byqx" = 13,
  "ogqy" = 14,
  # tasks
  "p8gu" = 4, # cognitive tests
  # cond = 0, i.e. tree stimuli
  "5i45" = 5, "msd6" = 51, "3nzn" = 52, # ekf
  "43ro" = 5, "smg4" = 61, "h62i" = 62, # floodh
  "g4bl" = 5, "2o95" = 71, "ex6r" = 72, # floodr
  "eitm" = 5, "lwfm" = 81, "bc9r" = 82, # bath
  # cond = 1, i.e. icon stimuli
  "kwum" = 6, "7n6r" = 151, "n7sy" = 152, # ekf
  "9pma" = 6, "b5do" = 161, "awan" = 162, # floodh
  "76ml" = 6, "coex" = 171, "fctj" = 172, # floodr
  "ohyf" = 6, "8xra" = 181, "3ar1" = 182) # bath

e.seq.2 <-c(# questionnaires
  "71i9" = 1, "mem2" = 2, "wrfg" = 3, "eylt" = 5, "b8fd" = 6, "bo25" = 6,
  "wxgm" = 6, "78v2" = 6, "t7op" = 6, "3q6n" = 6, "r2il" = 6,
  "rga1" = 6,
  "umw3" = 8, "797b" = 11, "rhwm" = 12, "gdbs" = 13, "byqx" = 14,
  "ogqy" = 15,
  # tasks
  "cbv6" = 4, "p8gu" = 10, # cognitive tests (GLT, CRT and BNT)
  # cond = 0, i.e. tree stimuli
  "5i45" = 5, "msd6" = 51, "3nzn" = 52, # ekf
  "43ro" = 5, "smg4" = 61, "h62i" = 62, # floodh
  "g4bl" = 5, "2o95" = 71, "ex6r" = 72, # floodr
  "eitm" = 5, "lwfm" = 81, "bc9r" = 82, # bath
  # cond = 1, i.e. icon stimuli
  "kwum" = 7, "7n6r" = 151, "n7sy" = 152, # ekf
  "9pma" = 7, "b5do" = 161, "awan" = 162, # floodh
  "76ml" = 7, "coex" = 171, "fctj" = 172, # floodr
  "ohyf" = 7, "8xra" = 181, "3ar1" = 182) # bath



## ---- Specify whether on main (2), control (0) or treatment (1) branch ----
e.branch <- c(# questionnaires
  "71i9" = 2, "mem2" = 2, "wrfg" = 2, "eylt" = 2, "b8fd" = 2,
  "umw3" = 2, "797b" = 2, "rhwm" = 2, "gdbs" = 2, "byqx" = 2,
  "ogqy" = 2,
  # tasks
  "p8gu" = 2, # cognitive tests
  # cond = 0, i.e. tree stimuli
  "5i45" = 0, "msd6" = 0, "3nzn" = 0, # ekf
  "43ro" = 0, "smg4" = 0, "h62i" = 0, # floodh
  "g4bl" = 0, "2o95" = 0, "ex6r" = 0, # floodr
  "eitm" = 0, "lwfm" = 0, "bc9r" = 0, # bath
  # cond = 1, i.e. icon stimuli
  "kwum" = 1, "7n6r" = 1, "n7sy" = 1, # ekf
  "9pma" = 1, "b5do" = 1, "awan" = 1, # floodh
  "76ml" = 1, "coex" = 1, "fctj" = 1, # floodr
  "ohyf" = 1, "8xra" = 1, "3ar1" = 1) # bath

e.dom <- c(# questionnaires ----
  "71i9" = "all", "mem2" = "all", "wrfg" = "all", "eylt" = "all", "b8fd" = "all",
  "umw3" = "all", "797b" = "all", "rhwm" = "all", "gdbs" = "all", "byqx" = "all",
  "ogqy" = "all",
  # tasks
  "p8gu" = "all", # cognitive tests
  # cond = 0, i.e. tree stimuli
  "5i45" = "ekf", "msd6" = "ekf", "3nzn" = "ekf", # ekf
  "43ro" = "floodh", "smg4" = "floodh", "h62i" = "floodh", # floodh
  "g4bl" = "floodr", "2o95" = "floodr", "ex6r" = "floodr", # floodr
  "eitm" = "bath", "lwfm" = "bath", "bc9r" = "bath", # bath
  # cond = 1, i.e. icon stimuli
  "kwum" = "ekf", "7n6r" = "ekf", "n7sy" = "ekf", # ekf
  "9pma" = "floodh", "b5do" = "floodh", "awan" = "floodh", # floodh
  "76ml" = "floodr", "coex" = "floodr", "fctj" = "floodr", # floodr
  "ohyf" = "bath", "8xra" = "bath", "3ar1" = "bath") # bath

for (x in 1: nrow(lookup)) {
  lookup[x,]$e.seq  <- e.seq[which(lookup[x,]$node == names(e.seq))][[1]]
  lookup[x,]$branch <- e.branch[which(lookup[x,]$node == names(e.branch))][[1]]
  lookup[x,]$dom    <- e.dom[which(lookup[x,]$node == names(e.dom))][[1]]
} ; rm(e.seq, e.branch, e.dom)


## ---- Bind all csv's in "files" by row, retaining all columns ----
d <- do.call("rbind.fill", lapply(files, read.csv, header = TRUE, fill = TRUE, stringsAsFactors = FALSE))

# faster alternative, but has issues if files from different experiment versions
# d <- do.call("bind_rows", lapply(files, read.csv, header = TRUE, fill = TRUE, stringsAsFactors = FALSE)) 


## ---- Remove entries that are not needed ----

# COLUMN-WISE OPERATIONS
# Meanings see https://gorilla.sc/support/reference/faq/metrics#columns
rm.cols <- c(
  # Experiment columns
  "UTC.Timestamp", "Local.Timestamp", "Local.Timezone",
  "Repeat.Key", "Participant.Starting.Group","Participant.Private.ID",
  "Participant.Device.Type", "Participant.Device", "Participant.OS",
  "Participant.Browser", "Participant.Monitor.Size", "Participant.Viewport.Size",
  "Schedule.ID","Checkpoint", "branch.b5u7", "checkpoint.2upf", "checkpoint.bypm",
  "reject.7die", "checkpoint.19o9", "checkpoint.b1g4","checkpoint.2sb9",
  "checkpoint.4ojy", "checkpoint.rj24", "checkpoint.ffgp", "checkpoint.6skl",
  "checkpoint.41h1", "checkpoint.3hh5", 
  
  # Task columns
  "Task.Version","Dishonest", "X.Coordinate", "Y.Coordinate", "Spreadsheet.Name", 
  "Zone.Name", "Zone.Type", "Attempt", "Incorrect", "Timed.Out", 
  # Additional data
  "Progress", "Best", "Worst", 
  "IconA", "IconB", "Blocktext","IconA2", "IconB2", "x.A",
  "px.A", "y.A", "py.A", "x.B", "px.B", "y.B", "py.B",
  "type", "xA", "pxA", "yA", "pyA", "xB", "pxB", "yB",
  "pyB", "pyB2", "yB2", "pxB2", "xB2", "pyA2", "yA2", 
  "pxA2", "xA2", "GLTpic", "GLTunits", "pic2", "pic3", "CRTquestion", "CRTunit")

d <- d[,-which(colnames(d) %in% rm.cols)]

# remove further unnecessary columns
d <- subset(d, select = -c(gamble1, gamble2, Q1c, Q3c, Q4c, correct,  
                           GLTquestion, GLTanswer, CRTanswer, checkpoint.c5bo))

# Keep
# EXPERIMENT COLUMNS
# [1] "?..Event.Index"                  "UTC.Date"                        "Local.Date"                      "Experiment.ID"                  
# [5] "Experiment.Version"              "Tree.Node.Key" (new:"Node")      "Participant.Public.ID"          "Participant.Status"             
# [9] "Participant.Completion.Code"     "Participant.External.Session.ID" "Task.Name"                       "branch.53gw"                    
# [13] "randomiser.36ox"                 "branch.k1cj"                     "checkpoint.2sb9"                 "branch.auin"                    
# [17] "branch.rhot"                     "branch.uma7"                     "branch.8rhg"                   "randomise_blocks"
# "randomise_trials",                           

# QUESTIONNAIRE COLUMNS
# "Question.Key"                    "Response"                        
# TASK COLUMNS
# [21] "Spreadsheet.Row"                 "Trial.Number"                    "Screen.Number"                  
# [25] "Screen.Name"                     "Reaction.Time"                   "Correct"  

# ADDITIONAL USER-DEFINED COLUMNS FROM SPREADSHEETS
# [65] "display"                 "gamble"           "manipulated              "correct"                        


# Rename cols or col content
names(d)[names(d) == "branch.53gw"] <- "consent"
names(d)[names(d) == "branch.8rhg"] <- "SewerConn"
names(d)[names(d) == "randomiser.36ox"] <- "e.cond" # experimental condition
names(d)[names(d) == "branch.k1cj"] <- "0-1stRank" # 1st ranked | control
names(d)[names(d) == "branch.rhot"] <- "0-2ndRank"
names(d)[names(d) == "branch.auin"] <- "1-1stRank" # 1st ranked | treatment
names(d)[names(d) == "branch.uma7"] <- "1-2ndRank"
names(d)[names(d) == "Tree.Node.Key"] <- "Node"
names(d)[names(d) == "Participant.Public.ID"] <- "prolific.id"
names(d)[names(d) == "Correct"] <- "q.correct"
names(d)[names(d) == "correct"] <- "t.correct"

# Retain only node ids
d$Node <- sapply(strsplit(as.character(d$Node), "-"), `[`, 2)


## ROW-WISE OPERATIONS
# Across nodes
d <- d[d$?..Event.Index != "END OF FILE",] # "END OF FILE" rows
d <- d[-which(d$Question.Key == "BEGIN QUESTIONNAIRE"),] # rows
d <- d[-which(d$Trial.Number == "BEGIN TASK"),] # rows

d[d == ""] <- NA # overwrite empty fields with NA

# # If merging across versions, some alignment is needed
# d[which(d$Task.Name == "Intro and consent"),]$Task.Name <- "Intro and consent - Prolific" 
# d[which(d$Task.Name == "End"),]$Task.Name <- "End-Prolific" 

# d <- d[-which(d$Question.Key =="BEGIN QUESTIONNAIRE"),]
# d <- d[-which(d$Trial.Number == "BEGIN TASK"),]
# DO BEFORE deletions (otherwise timing calculation might be wrong) check dplyr, tidyr etc
# Cast response time from END QUESTIONNAIRE into new column "Response.Time" and remove line
# tody <- as.numeric(d[d$Question.Key == "END QUESTIONNAIRE",]$Response)


# 71i9 - Intro and consent: d[d$Node == "71i9", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]
d <- d[-which(d$Question.Key == "Consent-quantised"),]

# "797b" - Experience with wastewater services: 
# d[d$Node == "797b", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]
d <- d[-which(d$Question.Key == "ExpCost-quantised" |
                d$Question.Key == "expEkf-quantised" |
                d$Question.Key == "ExpFloodh-quantised" |
                d$Question.Key == "expFloodR-quantised" |
                d$Question.Key == "attentionchckyes-quantised" |
                d$Question.Key == "expBathing-quantised" |
                d$Question.Key == "expProf-quantised"),]


# The Fatigue - feeling checkpoints
# Scaled as 1 ++++ 2 +++ 3 ++ 4 +  5 0 6 - 7 -- 8 --- 9 ----

# "wrfg" - Fatigue 1: 
# d[d$Node == "wrfg",][1:5,]
d <- d[-which(d$Question.Key == "PleasantUnpleasant-quantised" |
                d$Question.Key == "ArousedSleepy-quantised"),]
d[which(d$Node == "wrfg" & d$Task.Name == "Fatigue"),]$Task.Name <- "Affect1" # rename so they are kept separate through timepoints



# "b8fd" - Fatigue 2
# d[d$Node == "b8fd", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]
# also applies to "3q6n", "78v2","bo25", "t7op", # "bo25"
d[which(d$Node == "b8fd" & d$Task.Name == "Fatigue"|
          d$Node == "bo25" & d$Task.Name == "Fatigue"|
          d$Node == "wxgm"& d$Task.Name == "Fatigue"|
          d$Node == "78v2" & d$Task.Name == "Fatigue"|
          d$Node == "t7op" & d$Task.Name == "Fatigue"|
          d$Node == "3q6n" & d$Task.Name == "Fatigue"|
          d$Node == "r2il" & d$Task.Name == "Fatigue"|
          d$Node == "rga1" & d$Task.Name == "Fatigue"|
          d$Node == "bo25" & d$Task.Name == "Fatigue"|
          d$Node == "5pa1" & d$Task.Name == "Fatigue"),]$Task.Name <- "Affect2"




# "umw3" - Fatigue 3 :d[d$Node == "umw3", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]
d[which(d$Node == "umw3" & d$Task.Name == "Fatigue"),]$Task.Name <- "Affect3"

# "eylt" - Outcome ranking
# d[d$Node == "eylt", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]
d <- d[-which(d$Question.Key == "AttentionOutcomes-quantised"),]
# DO+++ Translate ranks and domains e.g. column = domain, cell entry = rank

# "gdbs" - Retrospective CBD and confirm consent
# d[d$Node == "gdbs", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]
d <- d[-which(d$Question.Key == "Detection1-quantised"|
                d$Question.Key == "Detection1-quantised"|
                d$Question.Key == "Detection2-quantised"|
                d$Question.Key == "Detection3-quantised"|
                d$Question.Key == "Detect4-quantised"|
                d$Question.Key == "Detect5-quantised"|
                d$Question.Key == "ConfirmConsent-quantised"),]

# "mem2" - Sewersystem screen
# d[d$Node == "mem2", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]
d <- d[-which(d$Question.Key == "sewerservice-quantised"),]

# "rhwm" - Demographics
# [d$Node == "rhwm", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]
d <- d[-which(d$Question.Key == "gender-quantised"|
                d$Question.Key == "birthyear-mixed"|
                d$Question.Key == "response-1-quantised"),]


# TASKS

# Merge gambles into 1 column

nrow(d[which(d$Task.Name == "Gambles practice" & d$Screen.Name == "Screen 1"),]) > 0

# "9pma" - Gambles practice
# d[d$Node == "9pma",][1:5,]
d <- d[-which(d$Task.Name == "Gambles practice" & d$Screen.Name == "Screen 1"),]
d <- d[-which(d$Task.Name == "Gambles practice" & d$Screen.Name == "Screen 2"),]
d <- d[-which(d$Task.Name == "Gambles practice" & d$Screen.Name == "Screen 3"),]
d <- d[-which(d$Task.Name == "Gambles practice" & d$Screen.Name == "Screen 4"),]
d <- d[-which(d$Task.Name == "Gambles practice" & d$Screen.Name == "end"),]
# also applies to "eitm", "kwum", "ohyf", "5i45"

# "7n6r"  - Risk preference elicitation
# d[d$Node == "7n6r",][1:5,]
d <- d[-which(d$Task.Name == "Risk preference elicitation" & d$Screen.Name == "intro"),]

# also applies to "8xra", "b5do", "msd6", "smg4"

# "3ar1" - Risk preference elicitation - validation
# d[d$Node == "3ar1",][1:5,]
d <- d[-which(d$Task.Name == "Risk preference elicitation - validation" & d$Screen.Name == "intro"),]
# also applies to "awan", "bc9r", lwfm", "n7sy", "3nzn", "h62i"

# # "p8gu" - Cognitive tests - FOR MERGING WITH TRIAL DATA ONLY
# # d[d$Node == "p8gu",][1:10,]
# d <- d[-which(d$Task.Name == "Cognitive tests" & d$Screen.Name == "intro"),]
# d <- d[-which(d$Task.Name == "Cognitive tests" & d$Screen.Name == "BNT-finish"),]
# d <- d[-which(d$Task.Name == "Cognitive tests" & d$Screen.Name == "GLTintro"),]
# d <- d[-which(d$Task.Name == "Cognitive tests" & d$Screen.Name == "end"),]

# "cbv6" - Graph literacy task
# d[d$Node == "cbv6",][1:5,]
d <- d[-which(d$Task.Name == "GLT" & d$Screen.Name == "intro"),]
d <- d[-which(d$Task.Name == "GLT" & d$Screen.Name == "end"),]

# "byqx" - End
# d[d$Node == "byqx", !map_lgl(d, ~ all(is.na(.))), drop = FALSE]

d <- d[-which(d$Screen.Name == "Confirm A" & is.na(d$Response)),] 
d <- d[-which(d$Screen.Name == "Confirm B" & is.na(d$Response)),] 

# ---- Convert data into useable format ---- 

# ## Convert character string to date object 
# d$UTC.Date <- parse_date_time2(d[1,]$UTC.Date, orders = c("%d/%m/%Y %H:%M:%S"))
# d$Local.Date <- parse_date_time2(d[1,]$Local.Date, orders = c("%d/%m/%Y %H:%M:%S"))


# Coalesce values spread across columns of different names
d$FirstRank <- coalesce(as.character(d$"0-1stRank"), as.character(d$"1-1stRank"))
d$SecondRank <- coalesce(as.character(d$"0-2ndRank"), as.character(d$"1-2ndRank"))
d <- d[,-which(names(d) %in% c("0-1stRank", "1-1stRank", "0-2ndRank", "1-2ndRank"))]

# Unite columns into one
d <- unite(d, "Version",
      Experiment.ID:Experiment.Version, 
      na.rm = TRUE, remove = TRUE)

# Rename and swap for intuition
d[which(d$Question.Key == "END QUESTIONNAIRE"),]$Question.Key <- "timeontask"
d[which(d$Trial.Number == "END TASK"),]$Trial.Number <- "timeontask"

# Swap Reponse and Question.Key for ranking
d$help <- NA
d$help <- ifelse(d$Question.Key  %in% c("rank-1","rank-2","rank-3","rank-4","rank-5"), "h", 0) # 0 is vital
d <- transform(d, Question.Key = ifelse(help == "h", Response, Question.Key),
          Response = ifelse(help == "h", Question.Key, Response))
d$help <- NULL

# Sort out some issues?

# Screen.Name
d[which(d$Screen.Name == "CRT" & d$Spreadsheet.Row == 2),]$Screen.Name <- "CRT1"
d[which(d$Screen.Name == "CRT" & d$Spreadsheet.Row == 3),]$Screen.Name <- "CRT2"
d[which(d$Screen.Name == "CRT" & d$Spreadsheet.Row == 4),]$Screen.Name <- "CRT3"

d[which(d$Screen.Name == "GLT" & d$Spreadsheet.Row == 2),]$Screen.Name <- "GLT1"
d[which(d$Screen.Name == "GLT" & d$Spreadsheet.Row == 3),]$Screen.Name <- "GLT2"
# d[which(d$Screen.Name == "GLT" & d$Spreadsheet.Row == 7),]$Screen.Name <- "GLT1"  
# d[which(d$Screen.Name == "GLT" & d$Spreadsheet.Row == 8),]$Screen.Name <- "GLT2"

# unique(d[d$Screen.Name == "BNT-Q1",]$Response)
# unique(d[d$Screen.Name == "BNT-Q2a",]$Response)
# unique(d[d$Screen.Name == "BNT-Q2b",]$Response)
# unique(d[d$Screen.Name == "BNT-Q3",]$Response)





# ---- Transpose data from long to wide ---- 
# Transform separately by questionnaire data and by task data

# Left join filling empty "type" fields in d by matching with lookup table
# deprecated: for (i in 1:nrow(d)){ d[i,]$type <- lookup[lookup$node == d[i,]$Node,]$type[1] }
names(d)[names(d) == "Node"] <- "node"
tody <- left_join(x = d, y = lookup[,2:3], by = "node")
d$type <- tody$type 
rm(tody)

# Any duplicated entries?
table(duplicated(d))

# Now, separate into questionnaire (q) and task (t) data
d.q <- d[d$type == "q",]
d.t <- d[d$type == "t",]

# Remove not necessary stuff from d.q

# Add number to indicate choice series (block), to make sure this does not get lost when casting
d.t$series <- NA
# d.t[which(d.t$Task.Name == "Risk preference elicitation"|
#             d.t$Task.Name == "Risk preference elicitation - validation"),]$series <- 
#   ifelse(is.na(d.t[which(d.t$Task.Name == "Risk preference elicitation"|
#                             d.t$Task.Name == "Risk preference elicitation - validation"),]$Spreadsheet.Row), NA,
#                      ifelse(d.t[which(d.t$Task.Name == "Risk preference elicitation"|
#                                         d.t$Task.Name == "Risk preference elicitation - validation"),]$Spreadsheet.Row < 21, 1, 2 ))

d.t[which(d.t$Task.Name == "Risk preference elicitation"|
            d.t$Task.Name == "Risk preference elicitation - validation"),]$series <- 
  ifelse(is.na(d.t[which(d.t$Task.Name == "Risk preference elicitation"|
                           d.t$Task.Name == "Risk preference elicitation - validation"),]$randomise_blocks), NA,
         ifelse(d.t[which(d.t$Task.Name == "Risk preference elicitation"|
                            d.t$Task.Name == "Risk preference elicitation - validation"),]$randomise_blocks == 1, 1, 2 ))



# Custom function to enable long-to-wide conversion of string data so that it picks first string
select.first<-function(x){ first(x) }

  
# Transform the data from long to wide. 
# Questionnaire data
d.q <- dcast(d.q, prolific.id + Participant.External.Session.ID + Participant.Status # Id vars(columns to keep)
            ~ Question.Key + Task.Name,                        # Measured variables (column to make several columns from)
            value.var = "Response",                            # Value variables (column to populate the new columns above)
            fun = select.first)                                # Aggregation function

# Task data 
dt2 <- d.t
d.t <- dcast(d.t, prolific.id + Participant.External.Session.ID + Participant.Status + e.cond                                                   
             ~  Task.Name + gamble + Screen.Name + series,                    
             value.var= c("Response"),             
             fun = select.first)  

d.tt <- dcast(dt2, prolific.id + Participant.External.Session.ID + Participant.Status + e.cond                                                   
              ~  Task.Name + gamble + Screen.Name + series,                    
              value.var= c("Reaction.Time"),             
              fun = select.first)

rm(dt2)

# ----  Tidy up column names etc ----

# Questionnaires
# --------------
# Coalesce if necessary
# d.q$sewerservice <- coalesce(as.character(d.q$"sewerservice_Experience with wastewater services"), 
#                              as.character(d.q$"sewerservice_Pre-screening"))
#                    
# d.q$"sewerservice_Experience with wastewater services" <- d.q$"sewerservice_Pre-screening" <- NULL

# Rename  
  
oldname <- c("ArousedSleepy_Affect1", #1
             "ArousedSleepy_Affect2", 
             "ArousedSleepy_Affect3",
             "PleasantUnpleasant_Affect1", 
             "PleasantUnpleasant_Affect2", #5
             "PleasantUnpleasant_Affect3",
             "response-1_Demographics", 
             "response-1-text_Demographics",
             "Detection1_Debriefing and confirm consent",                                                 
             "Detection2_Debriefing and confirm consent",                                                 
             "Detection3_Debriefing and confirm consent",
             "timeontask_Affect1", 
             "timeontask_Affect2",#10
             "timeontask_Affect3",
             "timeontask_Debriefing and confirm consent",                                                 
             "timeontask_Demographics",                                                                   
             "timeontask_End-Prolific",                                                                   
             "timeontask_Experience with wastewater services", #15                                           
             "timeontask_Intro and consent - Prolific",                                                   
             "timeontask_Outcome ranking"                                                                
             #"timeontask_Pre-screening"
             )

newname <- c("arousal1", #1
             "arousal2", 
             "arousal3",
             "valence1",
             "valence2", #5
             "valence3",
             "uk", 
             "other.country",
             "Detect1_Debriefing and confirm consent",                                                 
             "Detect2_Debriefing and confirm consent",                                                 
             "Detect3_Debriefing and confirm consent",
             "tot.affect1", 
             "tot.affect2", #10
             "tot.affect3",
             "tot.debriefing",                                                 
             "tot.demographics",                                                                   
             "tot.end",                                                                   
             "tot.experience", #15                                           
             "tot.intro",                                       
             "tot.outcomes"                                                                
             #"tot.prescreening"
             )

d.q <- d.q %>% rename_at(vars(oldname), ~ newname)

# These require some brute force !!! RECHECK POSITION WITH COMPLETE DATA
colnames(d.q)[c(4,10,33,36,37)]
colnames(d.q)[c(4,10,33,36,37)] <- c("cost", "bath", "ekf", "floodr", "floodh")

colnames(d.q) <- sapply(colnames(d.q), function(x) strsplit(x, "_")[[1]][1], USE.NAMES=FALSE)
d.q$"NA" <- NULL 

# Rename all columns by lowercase
colnames(d.q) <- sapply(colnames(d.q),tolower)

# Cleanup
rm(oldname, newname)

# Tasks
# --------------
## Response data
## **************

# Remove columns with all values == NA
not_all_na <- function(x) any(!is.na(x))
d.t  <- d.t %>% select_if(not_all_na)


# Collect cognitive test data
## CRT data
d.t$CT_1_CRT <- as.character(d.t$"Cognitive numeric_NA_CRT1_NA")
d.t$CT_2_CRT <- as.character(d.t$"Cognitive numeric_NA_CRT2_NA")
d.t$CT_3_CRT <- as.character(d.t$"Cognitive numeric_NA_CRT3_NA")

# BNT data
d.t$CT_1_BNT  <- as.character(d.t$"Cognitive numeric_NA_BNT-Q1_NA")
d.t$CT_2a_BNT <- as.character(d.t$"Cognitive numeric_NA_BNT-Q2a_NA")
d.t$CT_2b_BNT <- as.character(d.t$"Cognitive numeric_NA_BNT-Q2b_NA")
d.t$CT_3_BNT <- as.character(d.t$"Cognitive numeric_NA_BNT-Q3_NA")

## GLT data
d.t$CT_1_GLT <- as.character(d.t$"GLT_NA_GLT1_NA")
d.t$CT_2_GLT <- as.character(d.t$"GLT_NA_GLT2_NA")


# Risk preference data
d.t$G_1_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_1_Confirm A_1"),
                              as.character(d.t$"Risk preference elicitation_1_Confirm B_1"))
d.t$G_2_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_2_Confirm A_1"),
                              as.character(d.t$"Risk preference elicitation_2_Confirm B_1"))
d.t$G_5_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_5_Confirm A_1"),
                              as.character(d.t$"Risk preference elicitation_5_Confirm B_1"))
d.t$G_6_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_6_Confirm A_1"),
                              as.character(d.t$"Risk preference elicitation_6_Confirm B_1"))
d.t$G_9_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_9_Confirm A_1"),
                              as.character(d.t$"Risk preference elicitation_9_Confirm B_1"))
d.t$G_10_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_10_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_10_Confirm B_1"))
d.t$G_11_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_11_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_11_Confirm B_1"))
d.t$G_12_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_12_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_12_Confirm B_1"))
d.t$G_13_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_13_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_13_Confirm B_1"))
d.t$G_14_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_14_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_14_Confirm B_1"))
d.t$G_15_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_15_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_15_Confirm B_1"))
d.t$G_17_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_17_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_17_Confirm B_1"))
d.t$G_20_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_20_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_20_Confirm B_1"))
d.t$G_21_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_21_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_21_Confirm B_1"))
d.t$G_22_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_22_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_22_Confirm B_1"))
d.t$G_23_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_23_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_23_Confirm B_1"))
d.t$G_24_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_24_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_24_Confirm B_1"))
d.t$G_25_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_25_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_25_Confirm B_1"))
d.t$G_26_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_26_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_26_Confirm B_1"))
d.t$G_27_Confirm_1 <- coalesce(as.character(d.t$"Risk preference elicitation_27_Confirm A_1"),
                               as.character(d.t$"Risk preference elicitation_27_Confirm B_1"))

d.t$G_1_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_1_Confirm A_2"),
                              as.character(d.t$"Risk preference elicitation_1_Confirm B_2"))
d.t$G_2_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_2_Confirm A_2"),
                              as.character(d.t$"Risk preference elicitation_2_Confirm B_2"))
d.t$G_5_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_5_Confirm A_2"),
                              as.character(d.t$"Risk preference elicitation_5_Confirm B_2"))
d.t$G_6_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_6_Confirm A_2"),
                              as.character(d.t$"Risk preference elicitation_6_Confirm B_2"))
d.t$G_9_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_9_Confirm A_2"),
                              as.character(d.t$"Risk preference elicitation_9_Confirm B_2"))
d.t$G_10_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_10_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_10_Confirm B_2"))
d.t$G_11_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_11_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_11_Confirm B_2"))
d.t$G_12_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_12_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_12_Confirm B_2"))
d.t$G_13_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_13_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_13_Confirm B_2"))
d.t$G_14_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_14_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_14_Confirm B_2"))
d.t$G_15_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_15_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_15_Confirm B_2"))
d.t$G_17_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_17_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_17_Confirm B_2"))
d.t$G_20_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_20_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_20_Confirm B_2"))
d.t$G_21_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_21_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_21_Confirm B_2"))
d.t$G_22_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_22_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_22_Confirm B_2"))
d.t$G_23_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_23_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_23_Confirm B_2"))
d.t$G_24_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_24_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_24_Confirm B_2"))
d.t$G_25_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_25_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_25_Confirm B_2"))
d.t$G_26_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_26_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_26_Confirm B_2"))
d.t$G_27_Confirm_2 <- coalesce(as.character(d.t$"Risk preference elicitation_27_Confirm A_2"),
                               as.character(d.t$"Risk preference elicitation_27_Confirm B_2"))

# Rename
oldname <- c("Gambles practice_NA_Q1_NA",
             "Gambles practice_NA_Q3_NA",
             "Gambles practice_NA_Q4_NA",
             "Risk preference elicitation_1_Choice_1", 
             "Risk preference elicitation_2_Choice_1",  
             "Risk preference elicitation_5_Choice_1",             
             "Risk preference elicitation_6_Choice_1",
             "Risk preference elicitation_9_Choice_1",           
             "Risk preference elicitation_10_Choice_1",
             "Risk preference elicitation_11_Choice_1",
             "Risk preference elicitation_12_Choice_1",
             "Risk preference elicitation_13_Choice_1",
             "Risk preference elicitation_14_Choice_1",
             "Risk preference elicitation_15_Choice_1",
             "Risk preference elicitation_17_Choice_1",
             "Risk preference elicitation_20_Choice_1",
             "Risk preference elicitation_21_Choice_1",            
             "Risk preference elicitation_22_Choice_1",            
             "Risk preference elicitation_23_Choice_1",
             "Risk preference elicitation_24_Choice_1",            
             "Risk preference elicitation_25_Choice_1",
             "Risk preference elicitation_26_Choice_1",
             "Risk preference elicitation_27_Choice_1",
             "Risk preference elicitation_1_Choice_2", 
             "Risk preference elicitation_2_Choice_2",  
             "Risk preference elicitation_5_Choice_2",             
             "Risk preference elicitation_6_Choice_2",
             "Risk preference elicitation_9_Choice_2",           
             "Risk preference elicitation_10_Choice_2",
             "Risk preference elicitation_11_Choice_2",
             "Risk preference elicitation_12_Choice_2",
             "Risk preference elicitation_13_Choice_2",
             "Risk preference elicitation_14_Choice_2",
             "Risk preference elicitation_15_Choice_2",
             "Risk preference elicitation_17_Choice_2",
             "Risk preference elicitation_20_Choice_2",
             "Risk preference elicitation_21_Choice_2",            
             "Risk preference elicitation_22_Choice_2",            
             "Risk preference elicitation_23_Choice_2",
             "Risk preference elicitation_24_Choice_2",            
             "Risk preference elicitation_25_Choice_2",
             "Risk preference elicitation_26_Choice_2",
             "Risk preference elicitation_27_Choice_2",
             "Risk preference elicitation - validation_1_Choice_1", 
             "Risk preference elicitation - validation_2_Choice_1",  
             "Risk preference elicitation - validation_5_Choice_1",             
             "Risk preference elicitation - validation_6_Choice_1",
             "Risk preference elicitation - validation_9_Choice_1",           
             "Risk preference elicitation - validation_10_Choice_1",
             "Risk preference elicitation - validation_11_Choice_1",
             "Risk preference elicitation - validation_12_Choice_1",
             "Risk preference elicitation - validation_13_Choice_1",
             "Risk preference elicitation - validation_14_Choice_1",
             "Risk preference elicitation - validation_15_Choice_1",
             "Risk preference elicitation - validation_17_Choice_1",
             "Risk preference elicitation - validation_20_Choice_1",
             "Risk preference elicitation - validation_21_Choice_1",            
             "Risk preference elicitation - validation_22_Choice_1",            
             "Risk preference elicitation - validation_23_Choice_1",
             "Risk preference elicitation - validation_24_Choice_1",            
             "Risk preference elicitation - validation_25_Choice_1",
             "Risk preference elicitation - validation_26_Choice_1",
             "Risk preference elicitation - validation_27_Choice_1",
             "Risk preference elicitation - validation_1_Choice_2", 
             "Risk preference elicitation - validation_2_Choice_2",  
             "Risk preference elicitation - validation_5_Choice_2",             
             "Risk preference elicitation - validation_6_Choice_2",
             "Risk preference elicitation - validation_9_Choice_2",           
             "Risk preference elicitation - validation_10_Choice_2",
             "Risk preference elicitation - validation_11_Choice_2",
             "Risk preference elicitation - validation_12_Choice_2",
             "Risk preference elicitation - validation_13_Choice_2",
             "Risk preference elicitation - validation_14_Choice_2",
             "Risk preference elicitation - validation_15_Choice_2",
             "Risk preference elicitation - validation_17_Choice_2",
             "Risk preference elicitation - validation_20_Choice_2",
             "Risk preference elicitation - validation_21_Choice_2",            
             "Risk preference elicitation - validation_22_Choice_2",            
             "Risk preference elicitation - validation_23_Choice_2",
             "Risk preference elicitation - validation_24_Choice_2",            
             "Risk preference elicitation - validation_25_Choice_2",
             "Risk preference elicitation - validation_26_Choice_2",
             "Risk preference elicitation - validation_27_Choice_2",
             "G_1_Confirm_1","G_2_Confirm_1",
             "G_5_Confirm_1","G_6_Confirm_1",
             "G_9_Confirm_1","G_10_Confirm_1",
             "G_11_Confirm_1","G_12_Confirm_1",
             "G_13_Confirm_1","G_14_Confirm_1",
             "G_15_Confirm_1","G_17_Confirm_1",
             "G_20_Confirm_1","G_21_Confirm_1",
             "G_22_Confirm_1","G_23_Confirm_1",
             "G_24_Confirm_1","G_25_Confirm_1",
             "G_26_Confirm_1","G_27_Confirm_1",
             "G_1_Confirm_2","G_2_Confirm_2",
             "G_5_Confirm_2","G_6_Confirm_2",
             "G_9_Confirm_2","G_10_Confirm_2",
             "G_11_Confirm_2","G_12_Confirm_2",
             "G_13_Confirm_2","G_14_Confirm_2",
             "G_15_Confirm_2","G_17_Confirm_2",
             "G_20_Confirm_2","G_21_Confirm_2",
             "G_22_Confirm_2","G_23_Confirm_2",
             "G_24_Confirm_2","G_25_Confirm_2",
             "G_26_Confirm_2","G_27_Confirm_2")

newname <- c("practice.1", "practice.2", "practice.3",
             "rpe_1_1", "rpe_2_1",  
             "rpe_5_1", "rpe_6_1",
             "rpe_9_1",           
             "rpe_10_1",
             "rpe_11_1",
             "rpe_12_1",
             "rpe_13_1",
             "rpe_14_1",
             "rpe_15_1",
             "rpe_17_1",
             "rpe_20_1",
             "rpe_21_1",            
             "rpe_22_1",            
             "rpe_23_1",
             "rpe_24_1",            
             "rpe_25_1",
             "rpe_26_1",
             "rpe_27_1",
             "rpe_1_2", 
             "rpe_2_2",  
             "rpe_5_2",             
             "rpe_6_2",
             "rpe_9_2",           
             "rpe_10_2",
             "rpe_11_2",
             "rpe_12_2",
             "rpe_13_2",
             "rpe_14_2",
             "rpe_15_2",
             "rpe_17_2",
             "rpe_20_2",
             "rpe_21_2",            
             "rpe_22_2",            
             "rpe_23_2",
             "rpe_24_2",            
             "rpe_25_2",
             "rpe_26_2",
             "rpe_27_2",
             "rpv_1_1", 
             "rpv_2_1",  
             "rpv_5_1",             
             "rpv_6_1",
             "rpv_9_1",           
             "rpv_10_1",
             "rpv_11_1",
             "rpv_12_1",
             "rpv_13_1",
             "rpv_14_1",
             "rpv_15_1",
             "rpv_17_1",
             "rpv_20_1",
             "rpv_21_1",            
             "rpv_22_1",            
             "rpv_23_1",
             "rpv_24_1",            
             "rpv_25_1",
             "rpv_26_1",
             "rpv_27_1",
             "rpv_1_2", 
             "rpv_2_2",  
             "rpv_5_2",             
             "rpv_6_2",
             "rpv_9_2",           
             "rpv_10_2",
             "rpv_11_2",
             "rpv_12_2",
             "rpv_13_2",
             "rpv_14_2",
             "rpv_15_2",
             "rpv_17_2",
             "rpv_20_2",
             "rpv_21_2",            
             "rpv_22_2",            
             "rpv_23_2",
             "rpv_24_2",            
             "rpv_25_2",
             "rpv_26_2",
             "rpv_27_2",
             "rpe.why_1_1","rpe.why_2_1",
             "rpe.why_5_1","rpe.why_6_1",
             "rpe.why_9_1","rpe.why_10_1",
             "rpe.why_11_1","rpe.why_12_1",
             "rpe.why_13_1","rpe.why_14_1",
             "rpe.why_15_1","rpe.why_17_1",
             "rpe.why_20_1","rpe.why_21_1",
             "rpe.why_22_1","rpe.why_23_1",
             "rpe.why_24_1","rpe.why_25_1",
             "rpe.why_26_1","rpe.why_27_1",
             "rpe.why_1_2","rpe.why_2_2",
             "rpe.why_5_2","rpe.why_6_2",
             "rpe.why_9_2","rpe.why_10_2",
             "rpe.why_11_2","rpe.why_12_2",
             "rpe.why_13_2","rpe.why_14_2",
             "rpe.why_15_2","rpe.why_17_2",
             "rpe.why_20_2","rpe.why_21_2",
             "rpe.why_22_2","rpe.why_23_2",
             "rpe.why_24_2","rpe.why_25_2",
             "rpe.why_26_2","rpe.why_27_2")

d.t <- d.t %>% rename_at(vars(oldname), ~ newname)

# Rename all columns by lowercase
colnames(d.t) <- sapply(colnames(d.t),tolower)

# Remove all columns with some values = NA left from coalesce
not_any_na <- function(x) all(!is.na(x))
d.t <- bind_cols(select_if(d.t[,c(1:179)], not_any_na), d.t[,c(180:length(d.t))]) # position where CT and rpe.why start

# Force null on replaced ones
d.t$"cognitive numeric_na_bnt-q1_na"  <- NULL
d.t$"cognitive numeric_na_crt1_na"    <- NULL
d.t$"cognitive numeric_na_crt2_na"    <- NULL
d.t$"cognitive numeric_na_crt3_na"    <- NULL
d.t$"glt_na_glt1_na"                  <- NULL
d.t$"glt_na_glt2_na"                  <- NULL

## EXTRA CHECKS ON TASK DATA WHEN DATA IS IN
# - Does above work or are now other columns missing?
# - Did anybody provide "Other reason" when confirming?



## Task time data
## **************

# Remove columns with all values == NA
not_all_na <- function(x) any(!is.na(x))
d.tt  <- d.tt %>% select_if(not_all_na)

## CRT data
d.tt$CT_1_CRT <- as.character(d.tt$"Cognitive numeric_NA_CRT1_NA")
d.tt$CT_2_CRT <- as.character(d.tt$"Cognitive numeric_NA_CRT2_NA")
d.tt$CT_3_CRT <- as.character(d.tt$"Cognitive numeric_NA_CRT3_NA")

# BNT data
d.tt$CT_1_BNT  <- as.character(d.tt$"Cognitive numeric_NA_BNT-Q1_NA")
d.tt$CT_2a_BNT <- as.character(d.tt$"Cognitive numeric_NA_BNT-Q2a_NA")
d.tt$CT_2b_BNT <- as.character(d.tt$"Cognitive numeric_NA_BNT-Q2b_NA")
d.tt$CT_3_BNT <- as.character(d.tt$"Cognitive numeric_NA_BNT-Q3_NA")

## GLT data
d.tt$CT_1_GLT <- as.character(d.tt$"GLT_NA_GLT1_NA")
d.tt$CT_2_GLT <- as.character(d.tt$"GLT_NA_GLT2_NA")


# Risk preference data
d.tt$G_1_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_1_Confirm A_1"),
                               as.character(d.tt$"Risk preference elicitation_1_Confirm B_1"))
d.tt$G_2_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_2_Confirm A_1"),
                               as.character(d.tt$"Risk preference elicitation_2_Confirm B_1"))
d.tt$G_5_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_5_Confirm A_1"),
                               as.character(d.tt$"Risk preference elicitation_5_Confirm B_1"))
d.tt$G_6_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_6_Confirm A_1"),
                               as.character(d.tt$"Risk preference elicitation_6_Confirm B_1"))
d.tt$G_9_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_9_Confirm A_1"),
                               as.character(d.tt$"Risk preference elicitation_9_Confirm B_1"))
d.tt$G_10_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_10_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_10_Confirm B_1"))
d.tt$G_11_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_11_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_11_Confirm B_1"))
d.tt$G_12_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_12_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_12_Confirm B_1"))
d.tt$G_13_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_13_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_13_Confirm B_1"))
d.tt$G_14_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_14_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_14_Confirm B_1"))
d.tt$G_15_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_15_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_15_Confirm B_1"))
d.tt$G_17_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_17_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_17_Confirm B_1"))
d.tt$G_20_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_20_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_20_Confirm B_1"))
d.tt$G_21_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_21_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_21_Confirm B_1"))
d.tt$G_22_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_22_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_22_Confirm B_1"))
d.tt$G_23_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_23_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_23_Confirm B_1"))
d.tt$G_24_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_24_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_24_Confirm B_1"))
d.tt$G_25_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_25_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_25_Confirm B_1"))
d.tt$G_26_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_26_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_26_Confirm B_1"))
d.tt$G_27_Confirm_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_27_Confirm A_1"),
                                as.character(d.tt$"Risk preference elicitation_27_Confirm B_1"))

d.tt$G_1_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_1_Confirm A_2"),
                               as.character(d.tt$"Risk preference elicitation_1_Confirm B_2"))
d.tt$G_2_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_2_Confirm A_2"),
                               as.character(d.tt$"Risk preference elicitation_2_Confirm B_2"))
d.tt$G_5_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_5_Confirm A_2"),
                               as.character(d.tt$"Risk preference elicitation_5_Confirm B_2"))
d.tt$G_6_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_6_Confirm A_2"),
                               as.character(d.tt$"Risk preference elicitation_6_Confirm B_2"))
d.tt$G_9_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_9_Confirm A_2"),
                               as.character(d.tt$"Risk preference elicitation_9_Confirm B_2"))
d.tt$G_10_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_10_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_10_Confirm B_2"))
d.tt$G_11_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_11_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_11_Confirm B_2"))
d.tt$G_12_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_12_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_12_Confirm B_2"))
d.tt$G_13_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_13_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_13_Confirm B_2"))
d.tt$G_14_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_14_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_14_Confirm B_2"))
d.tt$G_15_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_15_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_15_Confirm B_2"))
d.tt$G_17_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_17_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_17_Confirm B_2"))
d.tt$G_20_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_20_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_20_Confirm B_2"))
d.tt$G_21_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_21_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_21_Confirm B_2"))
d.tt$G_22_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_22_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_22_Confirm B_2"))
d.tt$G_23_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_23_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_23_Confirm B_2"))
d.tt$G_24_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_24_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_24_Confirm B_2"))
d.tt$G_25_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_25_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_25_Confirm B_2"))
d.tt$G_26_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_26_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_26_Confirm B_2"))
d.tt$G_27_Confirm_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_27_Confirm A_2"),
                                as.character(d.tt$"Risk preference elicitation_27_Confirm B_2"))

d.tt$G_1_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_1_Mask A_1"),
                            as.character(d.tt$"Risk preference elicitation_1_Mask B_1"))
d.tt$G_2_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_2_Mask A_1"),
                            as.character(d.tt$"Risk preference elicitation_2_Mask B_1"))
d.tt$G_5_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_5_Mask A_1"),
                            as.character(d.tt$"Risk preference elicitation_5_Mask B_1"))
d.tt$G_6_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_6_Mask A_1"),
                            as.character(d.tt$"Risk preference elicitation_6_Mask B_1"))
d.tt$G_9_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_9_Mask A_1"),
                            as.character(d.tt$"Risk preference elicitation_9_Mask B_1"))
d.tt$G_10_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_10_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_10_Mask B_1"))
d.tt$G_11_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_11_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_11_Mask B_1"))
d.tt$G_12_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_12_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_12_Mask B_1"))
d.tt$G_13_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_13_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_13_Mask B_1"))
d.tt$G_14_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_14_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_14_Mask B_1"))
d.tt$G_15_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_15_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_15_Mask B_1"))
d.tt$G_17_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_17_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_17_Mask B_1"))
d.tt$G_20_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_20_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_20_Mask B_1"))
d.tt$G_21_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_21_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_21_Mask B_1"))
d.tt$G_22_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_22_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_22_Mask B_1"))
d.tt$G_23_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_23_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_23_Mask B_1"))
d.tt$G_24_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_24_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_24_Mask B_1"))
d.tt$G_25_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_25_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_25_Mask B_1"))
d.tt$G_26_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_26_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_26_Mask B_1"))
d.tt$G_27_Mask_1 <- coalesce(as.character(d.tt$"Risk preference elicitation_27_Mask A_1"),
                             as.character(d.tt$"Risk preference elicitation_27_Mask B_1"))

d.tt$G_1_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_1_Mask A_2"),
                            as.character(d.tt$"Risk preference elicitation_1_Mask B_2"))
d.tt$G_2_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_2_Mask A_2"),
                            as.character(d.tt$"Risk preference elicitation_2_Mask B_2"))
d.tt$G_5_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_5_Mask A_2"),
                            as.character(d.tt$"Risk preference elicitation_5_Mask B_2"))
d.tt$G_6_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_6_Mask A_2"),
                            as.character(d.tt$"Risk preference elicitation_6_Mask B_2"))
d.tt$G_9_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_9_Mask A_2"),
                            as.character(d.tt$"Risk preference elicitation_9_Mask B_2"))
d.tt$G_10_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_10_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_10_Mask B_2"))
d.tt$G_11_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_11_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_11_Mask B_2"))
d.tt$G_12_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_12_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_12_Mask B_2"))
d.tt$G_13_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_13_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_13_Mask B_2"))
d.tt$G_14_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_14_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_14_Mask B_2"))
d.tt$G_15_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_15_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_15_Mask B_2"))
d.tt$G_17_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_17_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_17_Mask B_2"))
d.tt$G_20_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_20_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_20_Mask B_2"))
d.tt$G_21_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_21_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_21_Mask B_2"))
d.tt$G_22_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_22_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_22_Mask B_2"))
d.tt$G_23_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_23_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_23_Mask B_2"))
d.tt$G_24_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_24_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_24_Mask B_2"))
d.tt$G_25_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_25_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_25_Mask B_2"))
d.tt$G_26_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_26_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_26_Mask B_2"))
d.tt$G_27_Mask_2 <- coalesce(as.character(d.tt$"Risk preference elicitation_27_Mask A_2"),
                             as.character(d.tt$"Risk preference elicitation_27_Mask B_2"))

# Rename
oldname <- c("Gambles practice_NA_Q1_NA",
             "Gambles practice_NA_Q3_NA", 
             "Gambles practice_NA_Q4_NA",
             "Risk preference elicitation_1_Choice_1", 
             "Risk preference elicitation_2_Choice_1",  
             "Risk preference elicitation_5_Choice_1",             
             "Risk preference elicitation_6_Choice_1",
             "Risk preference elicitation_9_Choice_1",           
             "Risk preference elicitation_10_Choice_1",
             "Risk preference elicitation_11_Choice_1",
             "Risk preference elicitation_12_Choice_1",
             "Risk preference elicitation_13_Choice_1",
             "Risk preference elicitation_14_Choice_1",
             "Risk preference elicitation_15_Choice_1",
             "Risk preference elicitation_17_Choice_1",
             "Risk preference elicitation_20_Choice_1",
             "Risk preference elicitation_21_Choice_1",            
             "Risk preference elicitation_22_Choice_1",            
             "Risk preference elicitation_23_Choice_1",
             "Risk preference elicitation_24_Choice_1",            
             "Risk preference elicitation_25_Choice_1",
             "Risk preference elicitation_26_Choice_1",
             "Risk preference elicitation_27_Choice_1",
             "Risk preference elicitation_1_Choice_2", 
             "Risk preference elicitation_2_Choice_2",  
             "Risk preference elicitation_5_Choice_2",             
             "Risk preference elicitation_6_Choice_2",
             "Risk preference elicitation_9_Choice_2",           
             "Risk preference elicitation_10_Choice_2",
             "Risk preference elicitation_11_Choice_2",
             "Risk preference elicitation_12_Choice_2",
             "Risk preference elicitation_13_Choice_2",
             "Risk preference elicitation_14_Choice_2",
             "Risk preference elicitation_15_Choice_2",
             "Risk preference elicitation_17_Choice_2",
             "Risk preference elicitation_20_Choice_2",
             "Risk preference elicitation_21_Choice_2",            
             "Risk preference elicitation_22_Choice_2",            
             "Risk preference elicitation_23_Choice_2",
             "Risk preference elicitation_24_Choice_2",            
             "Risk preference elicitation_25_Choice_2",
             "Risk preference elicitation_26_Choice_2",
             "Risk preference elicitation_27_Choice_2",
             "Risk preference elicitation - validation_1_Choice_1", 
             "Risk preference elicitation - validation_2_Choice_1",  
             "Risk preference elicitation - validation_5_Choice_1",             
             "Risk preference elicitation - validation_6_Choice_1",
             "Risk preference elicitation - validation_9_Choice_1",           
             "Risk preference elicitation - validation_10_Choice_1",
             "Risk preference elicitation - validation_11_Choice_1",
             "Risk preference elicitation - validation_12_Choice_1",
             "Risk preference elicitation - validation_13_Choice_1",
             "Risk preference elicitation - validation_14_Choice_1",
             "Risk preference elicitation - validation_15_Choice_1",
             "Risk preference elicitation - validation_17_Choice_1",
             "Risk preference elicitation - validation_20_Choice_1",
             "Risk preference elicitation - validation_21_Choice_1",            
             "Risk preference elicitation - validation_22_Choice_1",            
             "Risk preference elicitation - validation_23_Choice_1",
             "Risk preference elicitation - validation_24_Choice_1",            
             "Risk preference elicitation - validation_25_Choice_1",
             "Risk preference elicitation - validation_26_Choice_1",
             "Risk preference elicitation - validation_27_Choice_1",
             "Risk preference elicitation - validation_1_Choice_2", 
             "Risk preference elicitation - validation_2_Choice_2",  
             "Risk preference elicitation - validation_5_Choice_2",             
             "Risk preference elicitation - validation_6_Choice_2",
             "Risk preference elicitation - validation_9_Choice_2",           
             "Risk preference elicitation - validation_10_Choice_2",
             "Risk preference elicitation - validation_11_Choice_2",
             "Risk preference elicitation - validation_12_Choice_2",
             "Risk preference elicitation - validation_13_Choice_2",
             "Risk preference elicitation - validation_14_Choice_2",
             "Risk preference elicitation - validation_15_Choice_2",
             "Risk preference elicitation - validation_17_Choice_2",
             "Risk preference elicitation - validation_20_Choice_2",
             "Risk preference elicitation - validation_21_Choice_2",            
             "Risk preference elicitation - validation_22_Choice_2",            
             "Risk preference elicitation - validation_23_Choice_2",
             "Risk preference elicitation - validation_24_Choice_2",            
             "Risk preference elicitation - validation_25_Choice_2",
             "Risk preference elicitation - validation_26_Choice_2",
             "Risk preference elicitation - validation_27_Choice_2",
             "Risk preference elicitation_1_Fixation_1", 
             "Risk preference elicitation_2_Fixation_1",  
             "Risk preference elicitation_5_Fixation_1",             
             "Risk preference elicitation_6_Fixation_1",
             "Risk preference elicitation_9_Fixation_1",           
             "Risk preference elicitation_10_Fixation_1",
             "Risk preference elicitation_11_Fixation_1",
             "Risk preference elicitation_12_Fixation_1",
             "Risk preference elicitation_13_Fixation_1",
             "Risk preference elicitation_14_Fixation_1",
             "Risk preference elicitation_15_Fixation_1",
             "Risk preference elicitation_17_Fixation_1",
             "Risk preference elicitation_20_Fixation_1",
             "Risk preference elicitation_21_Fixation_1",            
             "Risk preference elicitation_22_Fixation_1",            
             "Risk preference elicitation_23_Fixation_1",
             "Risk preference elicitation_24_Fixation_1",            
             "Risk preference elicitation_25_Fixation_1",
             "Risk preference elicitation_26_Fixation_1",
             "Risk preference elicitation_27_Fixation_1",
             "Risk preference elicitation_1_Fixation_2", 
             "Risk preference elicitation_2_Fixation_2",  
             "Risk preference elicitation_5_Fixation_2",             
             "Risk preference elicitation_6_Fixation_2",
             "Risk preference elicitation_9_Fixation_2",           
             "Risk preference elicitation_10_Fixation_2",
             "Risk preference elicitation_11_Fixation_2",
             "Risk preference elicitation_12_Fixation_2",
             "Risk preference elicitation_13_Fixation_2",
             "Risk preference elicitation_14_Fixation_2",
             "Risk preference elicitation_15_Fixation_2",
             "Risk preference elicitation_17_Fixation_2",
             "Risk preference elicitation_20_Fixation_2",
             "Risk preference elicitation_21_Fixation_2",            
             "Risk preference elicitation_22_Fixation_2",            
             "Risk preference elicitation_23_Fixation_2",
             "Risk preference elicitation_24_Fixation_2",            
             "Risk preference elicitation_25_Fixation_2",
             "Risk preference elicitation_26_Fixation_2",
             "Risk preference elicitation_27_Fixation_2",
             "Risk preference elicitation - validation_1_Fixation_1", 
             "Risk preference elicitation - validation_2_Fixation_1",  
             "Risk preference elicitation - validation_5_Fixation_1",             
             "Risk preference elicitation - validation_6_Fixation_1",
             "Risk preference elicitation - validation_9_Fixation_1",           
             "Risk preference elicitation - validation_10_Fixation_1",
             "Risk preference elicitation - validation_11_Fixation_1",
             "Risk preference elicitation - validation_12_Fixation_1",
             "Risk preference elicitation - validation_13_Fixation_1",
             "Risk preference elicitation - validation_14_Fixation_1",
             "Risk preference elicitation - validation_15_Fixation_1",
             "Risk preference elicitation - validation_17_Fixation_1",
             "Risk preference elicitation - validation_20_Fixation_1",
             "Risk preference elicitation - validation_21_Fixation_1",            
             "Risk preference elicitation - validation_22_Fixation_1",            
             "Risk preference elicitation - validation_23_Fixation_1",
             "Risk preference elicitation - validation_24_Fixation_1",            
             "Risk preference elicitation - validation_25_Fixation_1",
             "Risk preference elicitation - validation_26_Fixation_1",
             "Risk preference elicitation - validation_27_Fixation_1",
             "Risk preference elicitation - validation_1_Fixation_2", 
             "Risk preference elicitation - validation_2_Fixation_2",  
             "Risk preference elicitation - validation_5_Fixation_2",             
             "Risk preference elicitation - validation_6_Fixation_2",
             "Risk preference elicitation - validation_9_Fixation_2",           
             "Risk preference elicitation - validation_10_Fixation_2",
             "Risk preference elicitation - validation_11_Fixation_2",
             "Risk preference elicitation - validation_12_Fixation_2",
             "Risk preference elicitation - validation_13_Fixation_2",
             "Risk preference elicitation - validation_14_Fixation_2",
             "Risk preference elicitation - validation_15_Fixation_2",
             "Risk preference elicitation - validation_17_Fixation_2",
             "Risk preference elicitation - validation_20_Fixation_2",
             "Risk preference elicitation - validation_21_Fixation_2",            
             "Risk preference elicitation - validation_22_Fixation_2",            
             "Risk preference elicitation - validation_23_Fixation_2",
             "Risk preference elicitation - validation_24_Fixation_2",            
             "Risk preference elicitation - validation_25_Fixation_2",
             "Risk preference elicitation - validation_26_Fixation_2",
             "Risk preference elicitation - validation_27_Fixation_2",
             "Risk preference elicitation_1_Done_1", 
             "Risk preference elicitation_2_Done_1",  
             "Risk preference elicitation_5_Done_1",             
             "Risk preference elicitation_6_Done_1",
             "Risk preference elicitation_9_Done_1",           
             "Risk preference elicitation_10_Done_1",
             "Risk preference elicitation_11_Done_1",
             "Risk preference elicitation_12_Done_1",
             "Risk preference elicitation_13_Done_1",
             "Risk preference elicitation_14_Done_1",
             "Risk preference elicitation_15_Done_1",
             "Risk preference elicitation_17_Done_1",
             "Risk preference elicitation_20_Done_1",
             "Risk preference elicitation_21_Done_1",            
             "Risk preference elicitation_22_Done_1",            
             "Risk preference elicitation_23_Done_1",
             "Risk preference elicitation_24_Done_1",            
             "Risk preference elicitation_25_Done_1",
             "Risk preference elicitation_26_Done_1",
             "Risk preference elicitation_27_Done_1",
             "Risk preference elicitation_1_Done_2", 
             "Risk preference elicitation_2_Done_2",  
             "Risk preference elicitation_5_Done_2",             
             "Risk preference elicitation_6_Done_2",
             "Risk preference elicitation_9_Done_2",           
             "Risk preference elicitation_10_Done_2",
             "Risk preference elicitation_11_Done_2",
             "Risk preference elicitation_12_Done_2",
             "Risk preference elicitation_13_Done_2",
             "Risk preference elicitation_14_Done_2",
             "Risk preference elicitation_15_Done_2",
             "Risk preference elicitation_17_Done_2",
             "Risk preference elicitation_20_Done_2",
             "Risk preference elicitation_21_Done_2",            
             "Risk preference elicitation_22_Done_2",            
             "Risk preference elicitation_23_Done_2",
             "Risk preference elicitation_24_Done_2",            
             "Risk preference elicitation_25_Done_2",
             "Risk preference elicitation_26_Done_2",
             "Risk preference elicitation_27_Done_2",
             "Risk preference elicitation - validation_1_Done_1", 
             "Risk preference elicitation - validation_2_Done_1",  
             "Risk preference elicitation - validation_5_Done_1",             
             "Risk preference elicitation - validation_6_Done_1",
             "Risk preference elicitation - validation_9_Done_1",           
             "Risk preference elicitation - validation_10_Done_1",
             "Risk preference elicitation - validation_11_Done_1",
             "Risk preference elicitation - validation_12_Done_1",
             "Risk preference elicitation - validation_13_Done_1",
             "Risk preference elicitation - validation_14_Done_1",
             "Risk preference elicitation - validation_15_Done_1",
             "Risk preference elicitation - validation_17_Done_1",
             "Risk preference elicitation - validation_20_Done_1",
             "Risk preference elicitation - validation_21_Done_1",            
             "Risk preference elicitation - validation_22_Done_1",            
             "Risk preference elicitation - validation_23_Done_1",
             "Risk preference elicitation - validation_24_Done_1",            
             "Risk preference elicitation - validation_25_Done_1",
             "Risk preference elicitation - validation_26_Done_1",
             "Risk preference elicitation - validation_27_Done_1",
             "Risk preference elicitation - validation_1_Done_2", 
             "Risk preference elicitation - validation_2_Done_2",  
             "Risk preference elicitation - validation_5_Done_2",             
             "Risk preference elicitation - validation_6_Done_2",
             "Risk preference elicitation - validation_9_Done_2",           
             "Risk preference elicitation - validation_10_Done_2",
             "Risk preference elicitation - validation_11_Done_2",
             "Risk preference elicitation - validation_12_Done_2",
             "Risk preference elicitation - validation_13_Done_2",
             "Risk preference elicitation - validation_14_Done_2",
             "Risk preference elicitation - validation_15_Done_2",
             "Risk preference elicitation - validation_17_Done_2",
             "Risk preference elicitation - validation_20_Done_2",
             "Risk preference elicitation - validation_21_Done_2",            
             "Risk preference elicitation - validation_22_Done_2",            
             "Risk preference elicitation - validation_23_Done_2",
             "Risk preference elicitation - validation_24_Done_2",            
             "Risk preference elicitation - validation_25_Done_2",
             "Risk preference elicitation - validation_26_Done_2",
             "Risk preference elicitation - validation_27_Done_2",
             "G_1_Confirm_1","G_2_Confirm_1",
             "G_5_Confirm_1","G_6_Confirm_1",
             "G_9_Confirm_1","G_10_Confirm_1",
             "G_11_Confirm_1","G_12_Confirm_1",
             "G_13_Confirm_1","G_14_Confirm_1",
             "G_15_Confirm_1","G_17_Confirm_1",
             "G_20_Confirm_1","G_21_Confirm_1",
             "G_22_Confirm_1","G_23_Confirm_1",
             "G_24_Confirm_1","G_25_Confirm_1",
             "G_26_Confirm_1","G_27_Confirm_1",
             "G_1_Confirm_2","G_2_Confirm_2",
             "G_5_Confirm_2","G_6_Confirm_2",
             "G_9_Confirm_2","G_10_Confirm_2",
             "G_11_Confirm_2","G_12_Confirm_2",
             "G_13_Confirm_2","G_14_Confirm_2",
             "G_15_Confirm_2","G_17_Confirm_2",
             "G_20_Confirm_2","G_21_Confirm_2",
             "G_22_Confirm_2","G_23_Confirm_2",
             "G_24_Confirm_2","G_25_Confirm_2",
             "G_26_Confirm_2","G_27_Confirm_2",
             "G_1_Mask_1","G_2_Mask_1",
             "G_5_Mask_1","G_6_Mask_1",
             "G_9_Mask_1","G_10_Mask_1",
             "G_11_Mask_1","G_12_Mask_1",
             "G_13_Mask_1","G_14_Mask_1",
             "G_15_Mask_1","G_17_Mask_1",
             "G_20_Mask_1","G_21_Mask_1",
             "G_22_Mask_1","G_23_Mask_1",
             "G_24_Mask_1","G_25_Mask_1",
             "G_26_Mask_1","G_27_Mask_1",
             "G_1_Mask_2","G_2_Mask_2",
             "G_5_Mask_2","G_6_Mask_2",
             "G_9_Mask_2","G_10_Mask_2",
             "G_11_Mask_2","G_12_Mask_2",
             "G_13_Mask_2","G_14_Mask_2",
             "G_15_Mask_2","G_17_Mask_2",
             "G_20_Mask_2","G_21_Mask_2",
             "G_22_Mask_2","G_23_Mask_2",
             "G_24_Mask_2","G_25_Mask_2",
             "G_26_Mask_2","G_27_Mask_2",
             "Gambles practice_NA_NA_NA",
             "Risk preference elicitation_NA_NA_NA", 
             "Risk preference elicitation - validation_NA_NA_NA",
             "Risk preference elicitation_NA_block_1",
             "Risk preference elicitation_NA_block_2",
             #"Risk preference elicitation_NA_end_2",
             "Risk preference elicitation - validation_NA_block_1",
             "Risk preference elicitation - validation_NA_block_2"
             #"Risk preference elicitation - validation_NA_end_2"
             )

newname <- c("practice.1",  "practice.2",  "practice.3", 
             "rpe_1_1","rpe_2_1","rpe_5_1","rpe_6_1",
             "rpe_9_1","rpe_10_1","rpe_11_1","rpe_12_1",
             "rpe_13_1","rpe_14_1","rpe_15_1","rpe_17_1",
             "rpe_20_1","rpe_21_1","rpe_22_1","rpe_23_1",
             "rpe_24_1","rpe_25_1","rpe_26_1","rpe_27_1",
             "rpe_1_2","rpe_2_2","rpe_5_2","rpe_6_2",
             "rpe_9_2","rpe_10_2","rpe_11_2","rpe_12_2",
             "rpe_13_2","rpe_14_2","rpe_15_2","rpe_17_2",
             "rpe_20_2","rpe_21_2","rpe_22_2","rpe_23_2",
             "rpe_24_2","rpe_25_2","rpe_26_2","rpe_27_2",
             "rpv_1_1","rpv_2_1","rpv_5_1","rpv_6_1",
             "rpv_9_1","rpv_10_1","rpv_11_1","rpv_12_1",
             "rpv_13_1","rpv_14_1","rpv_15_1","rpv_17_1",
             "rpv_20_1","rpv_21_1","rpv_22_1","rpv_23_1",
             "rpv_24_1","rpv_25_1","rpv_26_1","rpv_27_1",
             "rpv_1_2","rpv_2_2","rpv_5_2","rpv_6_2",
             "rpv_9_2","rpv_10_2","rpv_11_2","rpv_12_2",
             "rpv_13_2","rpv_14_2","rpv_15_2","rpv_17_2",
             "rpv_20_2","rpv_21_2","rpv_22_2","rpv_23_2",
             "rpv_24_2","rpv_25_2","rpv_26_2","rpv_27_2",
             "rpe_1_Fix_1","rpe_2_Fix_1","rpe_5_Fix_1","rpe_6_Fix_1",
             "rpe_9_Fix_1","rpe_10_Fix_1","rpe_11_Fix_1","rpe_12_Fix_1",
             "rpe_13_Fix_1","rpe_14_Fix_1","rpe_15_Fix_1","rpe_17_Fix_1",
             "rpe_20_Fix_1","rpe_21_Fix_1","rpe_22_Fix_1","rpe_23_Fix_1",
             "rpe_24_Fix_1","rpe_25_Fix_1","rpe_26_Fix_1","rpe_27_Fix_1",
             "rpe_1_Fix_2","rpe_2_Fix_2","rpe_5_Fix_2","rpe_6_Fix_2",
             "rpe_9_Fix_2","rpe_10_Fix_2","rpe_11_Fix_2","rpe_12_Fix_2",
             "rpe_13_Fix_2","rpe_14_Fix_2","rpe_15_Fix_2","rpe_17_Fix_2",
             "rpe_20_Fix_2","rpe_21_Fix_2","rpe_22_Fix_2","rpe_23_Fix_2",
             "rpe_24_Fix_2","rpe_25_Fix_2","rpe_26_Fix_2","rpe_27_Fix_2",
             "rpv_1_Fix_1","rpv_2_Fix_1","rpv_5_Fix_1","rpv_6_Fix_1",
             "rpv_9_Fix_1","rpv_10_Fix_1","rpv_11_Fix_1","rpv_12_Fix_1",
             "rpv_13_Fix_1","rpv_14_Fix_1","rpv_15_Fix_1","rpv_17_Fix_1",
             "rpv_20_Fix_1","rpv_21_Fix_1","rpv_22_Fix_1","rpv_23_Fix_1",
             "rpv_24_Fix_1","rpv_25_Fix_1","rpv_26_Fix_1","rpv_27_Fix_1",
             "rpv_1_Fix_2","rpv_2_Fix_2","rpv_5_Fix_2","rpv_6_Fix_2",
             "rpv_9_Fix_2","rpv_10_Fix_2","rpv_11_Fix_2","rpv_12_Fix_2",
             "rpv_13_Fix_2","rpv_14_Fix_2","rpv_15_Fix_2","rpv_17_Fix_2",
             "rpv_20_Fix_2","rpv_21_Fix_2","rpv_22_Fix_2","rpv_23_Fix_2",
             "rpv_24_Fix_2","rpv_25_Fix_2","rpv_26_Fix_2","rpv_27_Fix_2",
             "rpe_1_End_1","rpe_2_End_1","rpe_5_End_1","rpe_6_End_1",
             "rpe_9_End_1","rpe_10_End_1","rpe_11_End_1","rpe_12_End_1",
             "rpe_13_End_1","rpe_14_End_1","rpe_15_End_1","rpe_17_End_1",
             "rpe_20_End_1","rpe_21_End_1","rpe_22_End_1","rpe_23_End_1",
             "rpe_24_End_1","rpe_25_End_1","rpe_26_End_1","rpe_27_End_1",
             "rpe_1_End_2","rpe_2_End_2","rpe_5_End_2","rpe_6_End_2",
             "rpe_9_End_2","rpe_10_End_2","rpe_11_End_2","rpe_12_End_2",
             "rpe_13_End_2","rpe_14_End_2","rpe_15_End_2","rpe_17_End_2",
             "rpe_20_End_2","rpe_21_End_2","rpe_22_End_2","rpe_23_End_2",
             "rpe_24_End_2","rpe_25_End_2","rpe_26_End_2","rpe_27_End_2",
             "rpv_1_End_1","rpv_2_End_1","rpv_5_End_1","rpv_6_End_1",
             "rpv_9_End_1","rpv_10_End_1","rpv_11_End_1","rpv_12_End_1",
             "rpv_13_End_1","rpv_14_End_1","rpv_15_End_1","rpv_17_End_1",
             "rpv_20_End_1","rpv_21_End_1","rpv_22_End_1","rpv_23_End_1",
             "rpv_24_End_1","rpv_25_End_1","rpv_26_End_1","rpv_27_End_1",
             "rpv_1_End_2","rpv_2_End_2","rpv_5_End_2","rpv_6_End_2",
             "rpv_9_End_2","rpv_10_End_2","rpv_11_End_2","rpv_12_End_2",
             "rpv_13_End_2","rpv_14_End_2","rpv_15_End_2","rpv_17_End_2",
             "rpv_20_End_2","rpv_21_End_2","rpv_22_End_2","rpv_23_End_2",
             "rpv_24_End_2","rpv_25_End_2","rpv_26_End_2","rpv_27_End_2",
             "rpe.why_1_1","rpe.why_2_1","rpe.why_5_1","rpe.why_6_1",
             "rpe.why_9_1","rpe.why_10_1", "rpe.why_11_1","rpe.why_12_1",
             "rpe.why_13_1","rpe.why_14_1", "rpe.why_15_1","rpe.why_17_1",
             "rpe.why_20_1","rpe.why_21_1", "rpe.why_22_1","rpe.why_23_1",
             "rpe.why_24_1","rpe.why_25_1", "rpe.why_26_1","rpe.why_27_1",
             "rpe.why_1_2","rpe.why_2_2", "rpe.why_5_2","rpe.why_6_2",
             "rpe.why_9_2","rpe.why_10_2", "rpe.why_11_2","rpe.why_12_2",
             "rpe.why_13_2","rpe.why_14_2", "rpe.why_15_2","rpe.why_17_2",
             "rpe.why_20_2","rpe.why_21_2", "rpe.why_22_2","rpe.why_23_2",
             "rpe.why_24_2","rpe.why_25_2", "rpe.why_26_2","rpe.why_27_2",
             "rpe.why_1_Mask_1","rpe.why_2_Mask_1","rpe.why_5_Mask_1","rpe.why_6_Mask_1",
             "rpe.why_9_Mask_1","rpe.why_10_Mask_1", "rpe.why_11_Mask_1","rpe.why_12_Mask_1",
             "rpe.why_13_Mask_1","rpe.why_14_Mask_1", "rpe.why_15_Mask_1","rpe.why_17_Mask_1",
             "rpe.why_20_Mask_1","rpe.why_21_Mask_1", "rpe.why_22_Mask_1","rpe.why_23_Mask_1",
             "rpe.why_24_Mask_1","rpe.why_25_Mask_1", "rpe.why_26_Mask_1","rpe.why_27_Mask_1",
             "rpe.why_1_Mask_2","rpe.why_2_Mask_2", "rpe.why_5_Mask_2","rpe.why_6_Mask_2",
             "rpe.why_9_Mask_2","rpe.why_10_Mask_2", "rpe.why_11_Mask_2","rpe.why_12_Mask_2",
             "rpe.why_13_Mask_2","rpe.why_14_Mask_2", "rpe.why_15_Mask_2","rpe.why_17_Mask_2",
             "rpe.why_20_Mask_2","rpe.why_21_Mask_2", "rpe.why_22_Mask_2","rpe.why_23_Mask_2",
             "rpe.why_24_Mask_2","rpe.why_25_Mask_2", "rpe.why_26_Mask_2","rpe.why_27_Mask_2",
             "GP.tot", "rpe.tot", 
             "rpv.tot", "rpe_block_1",
             "rpe_block_2", 
             "rpv_block_1","rpv_block_2")

d.tt <- d.tt %>% rename_at(vars(oldname), ~ newname)

# Rename all columns by lowercase
colnames(d.tt) <- sapply(colnames(d.tt),tolower)

# Remove all columns with some values = NA left from coalesce (but keep partial answers on "rpe.why")
not_any_na <- function(x) all(!is.na(x))
d.tt <- bind_cols(select_if(d.tt[,c(1:429)], not_any_na), d.tt[,c(430:length(d.tt))])

# Force null on replaced ones
d.tt$"cognitive numeric_na_bnt-finish_na" <- NULL
d.tt$"cognitive numeric_na_intro_na"   <- NULL
d.tt$"cognitive numeric_na_end_na"     <- NULL
d.tt$"cognitive numeric_na_na_na"      <- NULL
d.tt$"cognitive numeric_na_bnt-q1_na"  <- NULL
d.tt$"cognitive numeric_na_crt1_na"    <- NULL
d.tt$"cognitive numeric_na_crt2_na"    <- NULL
d.tt$"cognitive numeric_na_crt3_na"    <- NULL
d.tt$"glt_na_glt1_na"                  <- NULL
d.tt$"glt_na_glt2_na"                  <- NULL
d.tt$"glt_na_na_na"                    <- NULL
d.tt$"risk preference elicitation_na_end_na" <- NULL
d.tt$"risk preference elicitation - validation_na_end_na" <- NULL

# ---- Anonymize ---- 
colnames(d.q)[2]  <- "participant.id"
colnames(d.t)[2]  <- "participant.id"
colnames(d.tt)[2] <- "participant.id"

d.q$prolific.id <- d.t$prolific.id <- d.tt$prolific.id <- "anonymized"

# ---- Export the whole thing ---- 

out.path <- "D:/Projects/LSE/Courses/4_PB451E_Dissertation/Analysis/Prolific/"

write.csv(d.q, file = paste0(out.path, "questionnaires-", Sys.Date(), ".csv"), row.names = FALSE)
write.csv(d.t, file = paste0(out.path, "tasks-", Sys.Date(), ".csv"), row.names = FALSE)
write.csv(d.tt, file = paste0(out.path, "tot-", Sys.Date(), ".csv"), row.names = FALSE)

# cleanup
rm(list = ls()); gc()

