function [solution,error,convergence] = SSCornering_Newton_Iterations( varargin )
%SSCORNERING_NEWTON_ITERATIONS - Solves the non-linear equilibrium equations of
%   a steady-state cornering vehicle model using Newton's method.
%   Adapted damped Newton iterations are performed to find the solution of
%   the non-linear steady state equations, according to the theory
%   described on p. 238-243 of Scientific Computing - Michael T. Heath.
%
%   Syntax:
%       [solution,error,convergence] = ...
%                                SSCORNERING_NEWTON_ITERATIONS(V, rho, plotbool)
%
%   Description:
%       SSCORNERING_NEWTON_ITERATIONS Applies damped Newton iteration to find
%       the solution to the equilibrium equations described by a predefined
%       objective function. This objective function contains the
%       equilibrium equations, which in case of a steady-state solution
%       should all equate to zero. The solution for which this is true is
%       found using the indicated iterative method.
%    
%   Inputs:
%       V - Vehicle velocity
%       rho - Corner radius
%       plotbool - Plot boolean
%
%   Outputs:
%       solution - struct containing the details of the final solution
%       error - the error in the steady-state equations of the final
%       solution.
%       convergence - boolean indicating whether the solution has converged
%       or not.
%
%   Other m-files required: SSCornering_Bicycle, SSCornering_TwoTrack,...
%                           SSCornering_TwoTrackBus
%   Subfunctions: none
%   MAT-files required: none
%
%   See also: 

%   Author: Camiel Beckers
%   email: c.j.j.beckers@tue.nl
%   Website: https://research.tue.nl/en/persons/camiel-jj-beckers
%   ORCID: https://orcid.org/0000-0002-3383-1092
%   DOI of this dataset: http://doi.org/10.4121/uuid:e6560568-4203-43c5-9fdb-a91bc5e2cee4
%
%   Date: 23-Jul-2018; Last revision: 26-Feb-2019

%   Copyright (c) 2020, C.J.J.Beckers
% 
%   This program is free software: you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation, either version 3 of the License, or
%   (at your option) any later version.
% 
%   This program is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
% 
%   You should have received a copy of the GNU General Public License
%   along with this program.  If not, see <https://www.gnu.org/licenses/>.

%% Parse input data and prescribe default values
V_default    = 22.5/3.6; %[m/s] vehicle velocity
rho_default  = 10;       %[m] corner radius
pbool_default= true;     %plotboolean

p = inputParser;
addOptional(p, 'V' , V_default   ,@(x)isnumeric(x)&&isscalar(x)&&sign(x)==1);
addOptional(p,'rho',rho_default  ,@(x)isnumeric(x)&&isscalar(x)&&sign(x)==1);
addOptional(p, 'pb',pbool_default,@(x)isa(x,'logical'));
parse(p,varargin{:})
clear rho_default V_default pbool_default

V        = p.Results.V;
rho      = p.Results.rho;
plotbool = p.Results.pb;

%Vehicle parameters (for calculation of initial guess, not critical for final solution)
vehicle.a       = 3.5;                %[m] distance CoG-front axle
vehicle.l       = 6;                  %[m] vehicle wheelbase
vehicle.b       = vehicle.l-vehicle.a;%[m] distance CoG-rear axle
vehicle.r_e     = 0.5;                %[m] free tyre radius

%% Generate folder to save figures that are generated
if ~exist('Figures_generated', 'dir')
       mkdir('Figures_generated')
end

%% Set parameters
                     
% Choose objective function:
f  = @(x,pb) SSCornering_EoM_TwoTrackBus(x,V,rho,pb);   %Advanced twotrack model with bus parameters

%Determine initial guess based on linear bicycle model at very low speeds:
x(1,1) = atan( vehicle.l/sqrt(rho^2-vehicle.b^2)); %delta [rad]
x(2,1) = atan(-vehicle.b/sqrt(rho^2-vehicle.b^2)); %beta [rad]
x(3,1) = V/vehicle.r_e; %OmegaL [rad/s]
x(4,1) = V/vehicle.r_e; %OmegaR [rad/s]
h = [0.0001; 0.0001; 0.01 ; 0.01  ]; %finite difference perturbation in [rad],[rad],[rad/s],[rad/s]
                                 %used to determine the Jacobian of the
                                 %objective function              
% Numerical parameters                   
gammastart= 0.03;  %Starting value of alpha for initial f(x)
                   %alpha is the step size of the Netwon iterations
p         = 2;     %stepsize scales with |f(x)|^-p
maxiter   = 300;   %Maximum number of Newton iterations
tol       = 1e-3;  %Maximum acceptable error in objective function

%% Determine Jacobian
%By central difference method
%Scalar equivalent: f'(x) = (f(x+h)-f(x-h))/(2*h)    h << 1 

%First determine number of output states of objective function:
nOutput = size(f(x,0),1); 

%Determine Jacobian (as a function of x):
J=@(x) (f(repmat(x,size(x'))+diag(h),0)-f(repmat(x,size(x'))-diag(h),0))...
                              ./...
                    (2*repmat(h',nOutput,1));
              
%% Perform Newton-Raphson iteration

if plotbool %plot only if requested by user
    figh = figure;
    
    h = semilogy(0,norm(f(x,0)),'*:');hold on; %plot first point
    plot([0 maxiter],[tol tol],'r');           %tollerance limit
    
    %Set axis/labels
    axis([0 maxiter tol*1e-1 norm(f(x,0))*1e1]);
    xlabel('iteration number');
    ylabel('|f_{obj}(x)|');
    title('Convergence plot');
end

% Initialize variables 
convergence = false;          %boolean indicating convergence
i_save  = 0;                  %iteration number
x_save  = x;                  %state vector
fx_save = f(x,0);             %objective function evaluation
fn_save = norm(fx_save);      %norm of objective function evaluation

% Determine step size gain:
gamma = gammastart*fn_save^p;     %step size gain
for i = 1:maxiter
      
    fx = f(x,0);    %evaluate objective function
    Jx = J(x);      %evaluate Jacobian
    
    % Newton's method (linear)
    s = Jx\-fx;     %Compute Newton step
     
    %determine step size as function of the norm of the objective function
    %The closer the current state is to the solution, the larger the step
    %size will be. Taking a small step size at the beginning improves
    %convergence. 
    alpha = min( 1 , gamma * (norm(fx))^(-p) );
     
    x = x + alpha*s; %Update solution
    
    %Store data
    i_save  = [i_save  i ];
    x_save  = [x_save  x ];
    fx_save = [fx_save fx];
    fn_save = [fn_save norm(fx)];
    
    % Update data of convergence plot:
    if plotbool
        h.XData = i_save;
        h.YData = fn_save;
        drawnow
    end
    
    % If required tollerance has been reached, stop iterating
    if norm(fx_save(:,end)) < tol
        convergence = true;
        break
    end
end

if plotbool %if requested by used: plot the results
    close(figh); %Close convergence plot
    
    figure; %Plot degrees of freedom
    subplot(4,1,1);
    plot(i_save,rad2deg(x_save(1,:)),'*:');
    xlim([0 i_save(end)])
    ylabel('\delta [deg]');
    subplot(4,1,2);
    plot(i_save,rad2deg(x_save(2,:)),'*:')
    xlim([0 i_save(end)])
    ylabel('\beta [deg]');
    subplot(4,1,3);
    plot(i_save,x_save(3,:)*60/(2*pi),'*:');
    xlim([0 i_save(end)])
    ylabel('\omega_L [RPM]');
    subplot(4,1,4);
    plot(i_save,x_save(4,:)*60/(2*pi),'*:');
    xlim([0 i_save(end)])
    ylabel('\omega_R [RPM]');
    xlabel('iteration number');
    
    figure; %Plot objective function
    hold on;
    plot(i_save(2:end),abs(fx_save(1,2:end)),'*-k');
    plot(i_save(2:end),abs(fx_save(2,2:end)),'*--k');
    plot(i_save(2:end),abs(fx_save(3,2:end)),'*:k');
    plot(i_save(2:end),abs(fx_save(4,2:end)),'*-.k');
    legend({'$\Sigma F_x - m a_x = 0$','$\Sigma F_y - m a_y = 0$','$\Sigma M_z = 0$','$T_L-T_R = 0$'},'Interpreter','latex') 
    %set(gca, 'YScale', 'log')
    xlabel('Iteration number');
    ylabel('Error [N]/[Nm]');
    saveas(gcf,'./Figures_generated/NewtonIterations_ObjFun','epsc')
    saveas(gcf,'./Figures_generated/NewtonIterations_ObjFun','fig')
end

%% Display results
%Display final solution in command window
[error,solution] = f(x,plotbool);

%Comand window output:
if plotbool == true
disp(['Cornering Resistance Power: ' num2str(solution.PcRes/1e3,3) ' [kW]']);
disp(['Cornering Scrub Losses    : ' num2str(solution.Pslip/1e3,3) ' [kW]']);
disp(['Combined Cornering Losses : ' num2str(solution.Pdriv/1e3,3) ' [kW]']);
end

end
