function [error,s] = SSCornering_EoM_TwoTrackBus( states, V, rho, plotbool )
%SSCORNERING_EOM_TWOTRACKBUS - Contains the equations of motion of a
%non-linear twotrack model, under steady state cornering assumptions.
%
%   Syntax:
%       error = SSCORNERING_EOM_TWOTRACKBUS(states, V, rho, plotbool)
%       [error,s] = SSCORNERING_EOM_TWOTRACKBUS(states, V, rho, plotbool)
%
%   Description:
%       SSCORNERING_EOM_TWOTRACKBUS() - Gives the solution 's' of the
%       equations of motion of a twotrack model under steady state
%       corndering conditions. The equations are not in closed form, but
%       are a function of four input states, which may or may not result in
%       a steady-state solution (see: Outputs). Therefore, they are to be
%       solved iteratively, by minimizing the errors in the equations of
%       motion. These are expressed by the output 'errors'.
%    
%   Inputs:
%       states - double with 4 rows containging estimates of the four
%       input states. These are:    - Average steering angle [rad]
%                                   - Vehicle side slip angle [rad]
%                                   - Angular axle velocity left [rad/s]
%                                   - Angular axle velocity right [rad/s]
%       If the input contains multiple columns of 4 rows each, the entire
%       calculation is repeated for each of the columns.
%       V - double indicating the forward velocity of the vehicle in [m/s].
%       rho - double indicating the corner radius in [m].
%       plotbool - boolean indicating whether or not a plot should be made
%       of the resulting solution. plotbool = TRUE will result in a plot.
%
%   Outputs:
%       error - double array of size 4, containing the errors in the
%       equations of motion in respectively [N], [N], [Nm], and [Nm]
%       for the given inputs. These are used to indicate whether the input
%       states actually result in a steady-state solution or not. 
%       s - struct containing the solution data
%
%   Other m-files required: none
%   Subfunctions: MFy
%   MAT-files required: none
%
%   See also: struct

%   Author: Camiel Beckers
%   email: c.j.j.beckers@tue.nl
%   Website: https://research.tue.nl/en/persons/camiel-jj-beckers
%   ORCID: https://orcid.org/0000-0002-3383-1092
%   DOI of this dataset: http://doi.org/10.4121/uuid:e6560568-4203-43c5-9fdb-a91bc5e2cee4
%
%   Date: 23-Jul-2018; Last revision: 18-Mar-2019

%   Copyright (c) 2020, C.J.J.Beckers
% 
%   This program is free software: you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation, either version 3 of the License, or
%   (at your option) any later version.
% 
%   This program is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
% 
%   You should have received a copy of the GNU General Public License
%   along with this program.  If not, see <https://www.gnu.org/licenses/>.

%% Set parameters

%------------------------------------------------------%
%Custom numbers for example vehicle in TRD paper
m       = 15000;         
a       = 3.5; 
b       = 2.5;
l       = a+b;
h       = 1.5;
rsd     = 0.25;
s1      = 1;
s2      = 1;
s3      = 1-0.32;
%------------------------------------------------------%

%Define steering relation: Ackermann steering:
SteeringRelation1 = @(delta) atan(l/(l/tan(delta)-s1)); %Left steer angle [rad]
SteeringRelation2 = @(delta) atan(l/(l/tan(delta)+s1)); %Right steer angle [rad]

%Enviromental parameters
g    = 9.81;   %[m/s^2] gravitational acceleration

%front tyre Magic Formula parameters
p.tyre1.R0   =       0.548;    % Free tyre radius [m]
p.tyre1.Fz0  =   50000;        % tyre nominal force [N]
p.tyre1.kz   = 1190000;        % Tyre vertical stiffness [N/m]
p.tyre1.pCx1 =       1.6;      % Shape factor Cfx for longitudinal force         
p.tyre1.pDx1 =       0.75;     % Longitudinal friction Mux at Fznom         
p.tyre1.pDx2 =      -0.05;     % Variation of friction Mux with load         
p.tyre1.pKx1 =      15;        % Longitudinal slip stiffness Kfx/Fz at Fznom 
p.tyre1.pCy1 =       1.3;      % Shape factor Cfy for lateral forces
p.tyre1.pDy1 =       0.7;      % Lateral friction coefficient at Fz0
p.tyre1.pDy2 =      -0.1;      % Lateral friction coefficient dependency on Fz
p.tyre1.pEy1 =       0.15126;  % Lateral curvature Efy at Fznom
p.tyre1.pEy2 =       0.0;      % Lateral curvature dependency on Fz
p.tyre1.pKy1 =       7.702;    % peak cornering stiffness (Cfa=pKy1*Fz0)
p.tyre1.pKy2 =       2.13;     % vertical force at which peak Cfa occurs (Fz=pKy2*Fz0)

%rear tyre Magic Formula parameters
p.tyre2.R0   =     0.548;    % Free tyre radius [m]
p.tyre2.Fz0  = 50000;        % tyre nominal force [N]
p.tyre2.kz   = 1130000;      % Tyre vertical stiffness [N/m]
p.tyre2.pCx1 =     1.6;      % Shape factor Cfx for longitudinal force         
p.tyre2.pDx1 =     0.75;     % Longitudinal friction Mux at Fznom         
p.tyre2.pDx2 =    -0.05;     % Variation of friction Mux with load         
p.tyre2.pKx1 =    15;        % Longitudinal slip stiffness Kfx/Fz at Fznom  
p.tyre2.pCy1 =     1.3;      % Shape factor Cfy for lateral forces
p.tyre2.pDy1 =     0.7;      % Lateral friction coefficient at Fz0
p.tyre2.pDy2 =    -0.1;      % Lateral friction coefficient dependency on Fz
p.tyre2.pEy1 =    -0.26327;  % Lateral curvature Efy at Fznom
p.tyre2.pEy2 =     0.0;      % Lateral curvature dependency on Fz
p.tyre2.pKy1 =     6.484;    % peak cornering stiffness (Cfa=pKy1*Fz0)
p.tyre2.pKy2 =     1.9557;   % vertical force at which peak Cfa occurs (Fz=pKy2*Fz0)

%% Input check
%Check if the input is in the correct format
validateattributes(states,{'numeric'},{'nrows',4})

%Initialize output
ncols = size(states,2); %number of columns in input
error = zeros(4,ncols); %declare output

for icol = 1:ncols %repeat entire calculation for each of the input columns
    %state variables
    delta  = states(1,icol);           %[rad] average steering angle 
    beta   = states(2,icol);           %[rad] vehicle side slip angle
    OmegaL = states(3,icol);           %[rad/s] angular velocity rear left axle
    OmegaR = states(4,icol);           %[rad/s] angular velocity rear right axle
    
    %Further steering angles:
    delta1 = SteeringRelation1(delta); %[rad] steering angle front right
    delta2 = SteeringRelation2(delta); %[rad] steering angle front right
    delta3 = 0;                        %[rad] steering angle left-rear axle
    delta4 = 0;                        %[rad] steering angle right-rear axle    

    %% Non-linear kinematics of Two Track model
    %Velocities in vehicle frame:
    vx = V*cos(beta);        %[m/s] longitudinal vehicle velocity
    vy =-V*sin(beta);        %[m/s] lateral vehicle velocity
    r = V/rho;              %[rad/s] vehicle yaw velocity
    
    %Wheel positions in vehicle frame w.r.t. GoG  (vector = [x;y;z]):
    r1 = [ a; s1;0]; %front left
    r2 = [ a;-s1;0]; %front right
    r3 = [-b; s2;0]; %rear left (outer)
    r4 = [-b; s3;0]; %rear left (inner)
    r5 = [-b;-s3;0]; %rear right (inner)
    r6 = [-b;-s2;0]; %rear right (outer)

    %Rotation matrices from vehicle frame to front wheel frames:
    A1  = [ cos(delta1), sin(delta1), 0
           -sin(delta1), cos(delta1), 0
               0      ,     0       , 1 ]; %front left wheel
    A2  = [ cos(delta2), sin(delta2), 0
           -sin(delta2), cos(delta2), 0
               0      ,     0       , 1 ]; %front right wheel
    A34 = [ cos(delta3), sin(delta3), 0
           -sin(delta3), cos(delta3), 0
               0      ,     0       , 1 ]; %rear left wheels
    A56 = [ cos(delta4), sin(delta4), 0
           -sin(delta4), cos(delta4), 0
               0      ,     0       , 1 ]; %rear right wheels
    
    %Wheel velocities in wheel frame (vector = [x;y;z]):
    V1 = A1*([vx;vy;0]+cross([0;0;r],r1)); %front left
    V2 = A2*([vx;vy;0]+cross([0;0;r],r2)); %front right
    V3 = A34*([vx;vy;0]+cross([0;0;r],r3)); %rear left
    V4 = A34*([vx;vy;0]+cross([0;0;r],r4)); %rear left (inner)
    V5 = A56*([vx;vy;0]+cross([0;0;r],r5)); %rear right(inner)
    V6 = A56*([vx;vy;0]+cross([0;0;r],r6)); %rear right

    %Side slip angles for each of the wheels:
    alpha1 = atan(-V1(2)/abs(V1(1))); %front left
    alpha2 = atan(-V2(2)/abs(V2(1))); %front right
    alpha3 = atan(-V3(2)/abs(V3(1))); %rear left
    alpha4 = atan(-V4(2)/abs(V4(1))); %rear left (inner)
    alpha5 = atan(-V5(2)/abs(V5(1))); %rear right(inner)
    alpha6 = atan(-V6(2)/abs(V6(1))); %rear right

    %Calculation of the vertical force per tyre (incl. load transfer):
    ay       = (V^2/rho)*cos(beta); %[m/s^2] lateral acceleration 
    ax       = (V^2/rho)*sin(beta); %[m/s^2] longitudinal acceleration 
    Mroll       =  m*ay*h; %[Nm] moment acting on CoG in lateral direction
    Mpitch      = -m*ax*h; %[Nm] moment acting on CoG in longitudinal direction
    deltaFzlong =         Mpitch/(  l  );  %[N] Longitudinal load transfer per axle
    deltaFz12   =    rsd *Mroll/( 2*s1);  %[N] Lateral Load transfer front axle
    deltaFz3456 = (1-rsd)*Mroll/(s2+s3);  %[N] Lateral Load transfer rear axle
    
    %Vertical tyre forces: 
    Fz1 = 0.5 *m*g*b/l - deltaFz12     + deltaFzlong/2;%[N] vertical force front left wheel
    Fz2 = 0.5 *m*g*b/l + deltaFz12     + deltaFzlong/2;%[N] vertical force front right wheel
    Fz3 = 0.25*m*g*a/l - deltaFz3456/2 - deltaFzlong/4;%[N] vertical force rear left wheel
    Fz4 = 0.25*m*g*a/l - deltaFz3456/2 - deltaFzlong/4;%[N] vertical force rear left wheel (inner)
    Fz5 = 0.25*m*g*a/l + deltaFz3456/2 - deltaFzlong/4;%[N] vertical force rear right wheel(inner)
    Fz6 = 0.25*m*g*a/l + deltaFz3456/2 - deltaFzlong/4;%[N] vertical force rear right wheel
    %Note: assumption is made that load transfer at the rear (deltaFz3456) is
    %distributed equally between the inner and the outer wheels.
    
    % Calculate loaded tyre radius of rear tyres:
    Re3 = p.tyre2.R0 - Fz3/p.tyre2.kz; %rear left wheel
    Re4 = p.tyre2.R0 - Fz4/p.tyre2.kz; %rear left wheel (inner)
    Re5 = p.tyre2.R0 - Fz5/p.tyre2.kz; %rear right wheel (inner)
    Re6 = p.tyre2.R0 - Fz6/p.tyre2.kz; %rear right wheel

    %Longitudinal slip speed between wheel and road surface:
    Vsx3 = -(V3(1)-Re3*OmegaL); %Slip speed [m/s]
    Vsx4 = -(V4(1)-Re4*OmegaL); %Slip speed [m/s]
    Vsx5 = -(V5(1)-Re5*OmegaR); %Slip speed [m/s]
    Vsx6 = -(V6(1)-Re6*OmegaR); %Slip speed [m/s]
    
    %Longitudinal slip ratios:
    kappa1 = 0;    %No longitudinal slip at front wheels
    kappa2 = 0;
    kappa3 = Vsx3/(abs(V3(1))+eps);
    kappa4 = Vsx4/(abs(V4(1))+eps);
    kappa5 = Vsx5/(abs(V5(1))+eps);
    kappa6 = Vsx6/(abs(V6(1))+eps);

    %% Nonlinear Dynamics of Two Track model

    %Lateral (Fy) and Longitudinal (Fx) tyre forces [N] in wheel frame:
    %Based on the Magic Formula
    [Fx1,Fy1] = MFy(kappa1,alpha1,Fz1,p.tyre1,0); %front left
    [Fx2,Fy2] = MFy(kappa2,alpha2,Fz2,p.tyre1,0); %front right
    [Fx3,Fy3] = MFy(kappa3,alpha3,Fz3,p.tyre2,0); %rear left
    [Fx4,Fy4] = MFy(kappa4,alpha4,Fz4,p.tyre2,0); %rear left (inner)
    [Fx5,Fy5] = MFy(kappa5,alpha5,Fz5,p.tyre2,plotbool); %rear right(inner)
    [Fx6,Fy6] = MFy(kappa6,alpha6,Fz6,p.tyre2,0); %rear right

    %Moments on each rear axle:
    ML = Fx3*Re3+Fx4*Re4; %[Nm] moment on left axle
    MR = Fx6*Re6+Fx5*Re5; %[Nm] moment on left axle 
    
    %Wheel forces in vehicle frame (vector = [x;y;z]):
    F1 = A1\[Fx1;Fy1;Fz1]; %front left
    F2 = A2\[Fx2;Fy2;Fz2]; %front right
    F3 = A34\[Fx3;Fy3;Fz3]; %rear left
    F4 = A34\[Fx4;Fy4;Fz4]; %rear left (inner)
    F5 = A56\[Fx5;Fy5;Fz5]; %rear right(inner)
    F6 = A56\[Fx6;Fy6;Fz6]; %rear right

    %% Force/moment equilibrium equations
    % forces/moments acting at the center of gravity in the vehicle frame
    FxCoG = F1(1) + F2(1) + F3(1) + F6(1) + F4(1) + F5(1); %longitudinal
    FyCoG = F1(2) + F2(2) + F3(2) + F6(2) + F4(2) + F5(2); %lateral
    FzCoG = F1(3) + F2(3) + F3(3) + F6(3) + F4(3) + F5(3); %vertical
    MzCoG = cross(r1,F1) + cross(r2,F2) +...
            cross(r3,F3) + cross(r6,F6) +...
            cross(r4,F4) + cross(r5,F5)     ; %moment around z-axis
    
    % Convert to tangential/normal axis system
    SumFt = FxCoG*cos(beta)-FyCoG*sin(beta); %tangential
    SumFn = FxCoG*sin(beta)+FyCoG*cos(beta); %normal
    
    % Calculate errors from the three Equations of Motion
    error(1,icol) = SumFt;             %Tangential force equilibrium CoG
    error(2,icol) = SumFn - m*V^2/rho; %Normal force equilibrium CoG
    error(3,icol) = MzCoG(3);           %Moment equilibrium z-axis CoG
    error(4,icol) = ML-MR;             %Moment equilibrium on rear differential
    
%     %ALTERNATIVE: also possible but numerically slower:
%     error(1,icol) = FxCG - m*ax;     %Longitudinal force equilibrium CoG
%     error(2,icol) = FyCG - m*ay;     %lateral force equilibrium CoG
end

%% Store data:
s.delta1 = delta1;      %Steering angle left [rad]
s.delta2 = delta2;      %Steering angle right [rad]
s.beta   = beta;        %Vehicle side slip angle [rad]
s.alpha1 = alpha1;     %Side slip angle of front left wheel [rad]
s.alpha2 = alpha2;     %Side slip angle of front right wheel [rad]
s.alpha3 = alpha3;     %Side slip angle of rear left wheel [rad]
s.alpha4 = alpha4;     %Side slip angle of rear left (inner) wheel [rad]
s.alpha5 = alpha5;     %Side slip angle of rear right (inner) wheel [rad]
s.alpha6 = alpha6;     %Side slip angle of rear right wheel [rad]
s.kappa1 = kappa1;     %long. slip ratio of front left wheel [rad]
s.kappa2 = kappa2;     %long. slip ratio of front right wheel [rad]
s.kappa3 = kappa3;     %long. slip ratio of rear left wheel [rad]
s.kappa4 = kappa4;     %long. slip ratio of rear left (inner) wheel [rad]
s.kappa5 = kappa5;     %long. slip ratio of rear right (inner) wheel [rad]
s.kappa6 = kappa6;     %long. slip ratio of rear right wheel [rad]
s.r_diff = OmegaR/OmegaL;%Velocity ratio between right and left rear axle [-]
s.Fx1    = Fx1;       
s.Fy1    = Fy1;
s.Fz1    = Fz1;
s.Fx2    = Fx2;
s.Fy2    = Fy2;
s.Fz2    = Fz2;
s.Fx3    = Fx3;
s.Fy3    = Fy3;
s.Fz3    = Fz3;
s.Fx4    = Fx4;
s.Fy4    = Fy4;
s.Fz4    = Fz4;
s.Fx5    = Fx5;
s.Fy5    = Fy5;
s.Fz5    = Fz5;
s.Fx6    = Fx6;
s.Fy6    = Fy6;
s.Fz6    = Fz6;
s.an     = SumFn/m;     %normal acceleration [m/s^2]
s.at     = SumFt/m;     %tangential acceleration [m/s^2]
s.ax     = FxCoG/m;      %longitudinal acceleration [m/^2]
s.ay     = FyCoG/m;      %lateral acceleration [m/^2]

s.PcRes  = Fx3*V3(1)+ Fx6*V6(1)+ Fx4*V4(1)+ Fx5*V5(1);%Cornering resistance power [J/s]
s.Pslip  = Fx3*Vsx3 + Fx6*Vsx6 + Fx4*Vsx4 + Fx5*Vsx5; %Lost power in long. slip [J/s]
s.Pdriv  = ML*OmegaL+MR*OmegaR; %Total Required driving power [J/s] (sum of above 2)

%% Plot results:
if plotbool == true %plot only if requested
    
        
    %% Vertical force plot
    figure; hold on;
    hbar1 = bar([1 2],[Fz1,Fz2]);
    hbar2 = bar([3:6],[Fz3,Fz4,Fz5,Fz6]);
    set(hbar1,'FaceColor',[0 0 0]);
    set(hbar2,'FaceColor',[0.7 0.7 0.7]);
    xlim([0.5 6.5]);
    xlabel('Tire number');
    ylabel('Vertical tire force [N]');
    legend('Front','Rear','location','north');
    saveas(gcf,'./Figures_generated/VerticalTireForces','epsc')
    saveas(gcf,'./Figures_generated/VerticalTireForces','fig')
    
    % Vertical force warning
    if any([Fz1,Fz2,Fz3,Fz4,Fz5,Fz6] < 0)
        warning('In the displayed solution, one or more of the calculated vertical tire forces is negative. The results are not physically correct!');
    s.WARNING = 'In the displayed solution, one or more of the calculated vertical tire forces is negative. The results are not physically correct!';
    end
    
    %% Solution Plot
    %Set custom colors
    gray  = [0.6 0.6 0.6];
    green = [0   175 80 ]/255;
    red   = [0.9 0   0  ];
    
    %Set tire size in plot
    tirDiam = 2*0.548;  %tire diameter [m]
    tirWidth= 0.318;    %tire width [m]
    
    %Set scale for vectors
    ForScale = 1/13000;  %force scale [m/N]
    VelScale = 1/6;     %velocity scale [m/(ms^-2)]
    
    %load image of CoF
    CoGimg = imread('CoG.png');
    
    figure; hold on; grid off; axis equal;
    set(gca,'Visible','off')
    set(gca,'XDir','rev','YDir','rev') %axis system:  
                                                    % <----o
                                                    %  x   |
                                                    %      |
                                                    %      v y
    %Plot path of circle
    th = (1.5*pi-beta-(b+0.5)/rho):2*pi/100:(1.5*pi-beta+(a+1.5)/rho);
    xunit = rho * cos(th) + rho*sin(beta);
    yunit = rho * sin(th) + rho*cos(beta);
    plot(xunit, yunit,'k--');
    
    %Plot vehicle lines from CoG to wheels
    plot([0 r1(1) r1(1)],[0 0 r1(2)],'LineWidth',4,'color',gray);
    plot([0 r2(1) r2(1)],[0 0 r2(2)],'LineWidth',4,'color',gray);
    plot([0 r3(1) r3(1)],[0 0 r3(2)],'LineWidth',4,'color',gray);
    plot([0 r4(1) r4(1)],[0 0 r4(2)],'LineWidth',4,'color',gray);
    plot([0 r5(1) r5(1)],[0 0 r5(2)],'LineWidth',4,'color',gray);
    plot([0 r6(1) r6(1)],[0 0 r6(2)],'LineWidth',4,'color',gray);

    %Plot points at wheel locations
    plot(r1(1),r1(2),'.k','Markersize',20);
    plot(r2(1),r2(2),'.k','Markersize',20);
    plot(r3(1),r3(2),'.k','Markersize',20);
    plot(r4(1),r4(2),'.k','Markersize',20);
    plot(r5(1),r5(2),'.k','Markersize',20);
    plot(r6(1),r6(2),'.k','Markersize',20);
    image([-0.1 0.1],[0.1 -0.1],CoGimg); %plot image at CoG
    
    %Determine coordinates of wheel points
    w1 = [ tirDiam/2; tirWidth/2;0];
    w2 = [-tirDiam/2; tirWidth/2;0];
    w3 = [-tirDiam/2;-tirWidth/2;0];
    w4 = [ tirDiam/2;-tirWidth/2;0];
    
    %Determine wheel coordinates for each of the four wheels:
    w11L = A1\w1+r1; %front left
    w21L = A1\w2+r1;
    w31L = A1\w3+r1;
    w41L = A1\w4+r1;
    
    w11R = A2\w1+r2; %front right
    w21R = A2\w2+r2;
    w31R = A2\w3+r2;
    w41R = A2\w4+r2;
    
    w12L = A34\w1+r3; %rear left (outer)
    w22L = A34\w2+r3;
    w32L = A34\w3+r3;
    w42L = A34\w4+r3;
    
    w12R = A56\w1+r6; %rear right (outer)
    w22R = A56\w2+r6;
    w32R = A56\w3+r6;
    w42R = A56\w4+r6;
    
    w13L = A34\w1+r4; %rear left (inner)
    w23L = A34\w2+r4;
    w33L = A34\w3+r4;
    w43L = A34\w4+r4;
    
    w13R = A56\w1+r5; %rear right (inner)
    w23R = A56\w2+r5;
    w33R = A56\w3+r5;
    w43R = A56\w4+r5;
    
    %Plot wheels at each of the wheel locations
    %W1L = rectangle('Position',[w31L(1) w31L(2) tirDiam tirWidth],'Curvature',1,'LineWidth',2,'Edgecolor',gray);
    plot([w11L(1) w21L(1) w31L(1) w41L(1) w11L(1)],[w11L(2) w21L(2) w31L(2) w41L(2) w11L(2)],'LineWidth',2,'color',gray);
    plot([w11R(1) w21R(1) w31R(1) w41R(1) w11R(1)],[w11R(2) w21R(2) w31R(2) w41R(2) w11R(2)],'LineWidth',2,'color',gray);
    plot([w12L(1) w22L(1) w32L(1) w42L(1) w12L(1)],[w12L(2) w22L(2) w32L(2) w42L(2) w12L(2)],'LineWidth',2,'color',gray);
    plot([w12R(1) w22R(1) w32R(1) w42R(1) w12R(1)],[w12R(2) w22R(2) w32R(2) w42R(2) w12R(2)],'LineWidth',2,'color',gray);
    plot([w13L(1) w23L(1) w33L(1) w43L(1) w13L(1)],[w13L(2) w23L(2) w33L(2) w43L(2) w13L(2)],'LineWidth',2,'color',gray);
    plot([w13R(1) w23R(1) w33R(1) w43R(1) w13R(1)],[w13R(2) w23R(2) w33R(2) w43R(2) w13R(2)],'LineWidth',2,'color',gray);
    
    %Determine velocity vectors in vehicle frame
    Vplt   = VelScale* V*[cos(beta);-sin(beta);0];
    V1Lplt = VelScale* (A1\V1);
    V1Rplt = VelScale* (A2\V2);
    V2Lplt = VelScale* (A34\V3);
    V2Rplt = VelScale* (A56\V6);
    V3Lplt = VelScale* (A34\V4);
    V3Rplt = VelScale* (A56\V5);
    
    %Plot velicity vectors
    quiver(0,0,Vplt(1),Vplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',green)
    quiver(r1(1),r1(2),V1Lplt(1),V1Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',green)
    quiver(r2(1),r2(2),V1Rplt(1),V1Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',green)
    quiver(r3(1),r3(2),V2Lplt(1),V2Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',green)
    quiver(r6(1),r6(2),V2Rplt(1),V2Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',green)
    quiver(r4(1),r4(2),V3Lplt(1),V3Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',green)
    quiver(r5(1),r5(2),V3Rplt(1),V3Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',green)
    
    %Write slip angle values next to each of the vectors
    text(  0   ,  0   ,['\beta = '       num2str(rad2deg(beta   ),2) ' deg.    '],'HorizontalAlignment','right');
    text(r1(1),r1(2),['\alpha_{1L} = ' num2str(rad2deg(alpha1),2) ' deg.    '],'HorizontalAlignment','right');
    text(r2(1),r2(2),['\alpha_{1R} = ' num2str(rad2deg(alpha2),2) ' deg.    '],'HorizontalAlignment','right');
    text(r3(1),r3(2),['\alpha_{2L} = ' num2str(rad2deg(alpha3),2) ' deg.    '],'HorizontalAlignment','right');
    text(r6(1),r6(2),['\alpha_{2R} = ' num2str(rad2deg(alpha6),2) ' deg.    '],'HorizontalAlignment','right');
    text(r4(1),r4(2),['\alpha_{3L} = ' num2str(rad2deg(alpha4),2) ' deg.    '],'HorizontalAlignment','right');
    text(r5(1),r5(2),['\alpha_{3R} = ' num2str(rad2deg(alpha5),2) ' deg.    '],'HorizontalAlignment','right');
    
    %Determine decomposed force vectors in vehicle frame:
    Fx1Lplt = ForScale* (A1\[Fx1; 0  ;0]); %Front left 
    Fy1Lplt = ForScale* (A1\[ 0  ;Fy1;0]);
    Fx1Rplt = ForScale* (A2\[Fx2; 0  ;0]); %Front left 
    Fy1Rplt = ForScale* (A2\[ 0  ;Fy2;0]);
    
    Fx2Lplt = ForScale* (A34\[Fx3; 0  ;0]); %Rear left 
    Fy2Lplt = ForScale* (A34\[ 0  ;Fy3;0]);
    Fx2Rplt = ForScale* (A56\[Fx6; 0  ;0]); %Rear right
    Fy2Rplt = ForScale* (A56\[ 0  ;Fy6;0]);
    
    Fx3Lplt = ForScale* (A34\[Fx4; 0  ;0]); %Rear left (inner)
    Fy3Lplt = ForScale* (A34\[ 0  ;Fy4;0]);
    Fx3Rplt = ForScale* (A56\[Fx5; 0  ;0]); %Rear right (inner)
    Fy3Rplt = ForScale* (A56\[ 0  ;Fy5;0]);
    
    %Plot force vectors
    quiver(r1(1),r1(2),Fx1Lplt(1),Fx1Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r1(1),r1(2),Fy1Lplt(1),Fy1Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r2(1),r2(2),Fx1Rplt(1),Fx1Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r2(1),r2(2),Fy1Rplt(1),Fy1Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    
    quiver(r3(1),r3(2),Fx2Lplt(1),Fx2Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r3(1),r3(2),Fy2Lplt(1),Fy2Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r6(1),r6(2),Fx2Rplt(1),Fx2Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r6(1),r6(2),Fy2Rplt(1),Fy2Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    
    quiver(r4(1),r4(2),Fx3Lplt(1),Fx3Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r4(1),r4(2),Fy3Lplt(1),Fy3Lplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r5(1),r5(2),Fx3Rplt(1),Fx3Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    quiver(r5(1),r5(2),Fy3Rplt(1),Fy3Rplt(2),0,'LineWidth',4,'MaxHeadSize',4,'color',red)
    
    %Write force values next to each of the vectors:
%     %text(r1L(1)           ,r1L(2)+Fx1Lplt(2),[' Fx_{1L} = ' num2str(Fx1L,2) ' N'],'HorizontalAlignment','left','color',red);
%     text(r1L(1)+Fy1Lplt(1),r1L(2)+Fy1Lplt(2),[' Fy_{1L} = ' num2str(Fy1L,2) ' N'],'HorizontalAlignment','left','color',red);
%     %text(r1R(1)           ,r1R(2)+Fx1Rplt(2),[' Fx_{1R} = ' num2str(Fx1R,2) ' N'],'HorizontalAlignment','left','color',red);
%     text(r1R(1)+Fy1Rplt(1),r1R(2)+Fy1Rplt(2),[' Fy_{1R} = ' num2str(Fy1R,2) ' N'],'HorizontalAlignment','left','color',red);
%     
%     text(r2L(1)           ,r2L(2)+Fx2Lplt(2),[' Fx_{2L} = ' num2str(Fx2L,2) ' N'],'HorizontalAlignment','left','color',red);
%     text(r2L(1)+Fy2Lplt(1),r2L(2)+Fy2Lplt(2),[' Fy_{2L} = ' num2str(Fy2L,2) ' N'],'HorizontalAlignment','left','color',red);
%     text(r2R(1)           ,r2R(2)+Fx2Rplt(2),[' Fx_{2R} = ' num2str(Fx2R,2) ' N'],'HorizontalAlignment','left','color',red);
%     text(r2R(1)+Fy2Rplt(1),r2R(2)+Fy2Rplt(2),[' Fy_{2R} = ' num2str(Fy2R,2) ' N'],'HorizontalAlignment','left','color',red);
%     
%     text(r3L(1)           ,r3L(2)+Fx3Lplt(2),[' Fx_{3L} = ' num2str(Fx3L,2) ' N'],'HorizontalAlignment','left','color',red);
%     text(r3L(1)+Fy3Lplt(1),r3L(2)+Fy3Lplt(2),[' Fy_{3L} = ' num2str(Fy3L,2) ' N'],'HorizontalAlignment','left','color',red);
%     text(r3R(1)           ,r3R(2)+Fx3Rplt(2),[' Fx_{3R} = ' num2str(Fx3R,2) ' N'],'HorizontalAlignment','left','color',red);
%     text(r3R(1)+Fy3Rplt(1),r3R(2)+Fy3Rplt(2),[' Fy_{3R} = ' num2str(Fy3R,2) ' N'],'HorizontalAlignment','left','color',red);

    %Save figure
    saveas(gcf,['./Figures_generated/solution_V=' num2str(round(V*3.6)) '_rho=' num2str(round(rho))],'epsc')
    saveas(gcf,['./Figures_generated/solution_V=' num2str(round(V*3.6)) '_rho=' num2str(round(rho))],'fig')
end

end

function [Fxmon,Fy]=MFy(kappa,alpha,Fz,par,plotbool)
%MFY - Contains a simplified magic formula formulation that only includes
%simplified expressions for the longitudinal and lateral force.
%
%   Note: the function is able to produce results for Fz<0, which is
%   physically incorrect. However, this is nessesary to ensure smoothness
%   of the resulting forces, as function of parameter variations and thus
%   to ensure convergence of the numerical solver. A manual check of the
%   final result should be performed to ensure positive values of Fz for
%   the final solution of the model.
%
%   Syntax:
%       Fx = MFY(kappa,alpha,Fz,par,plotbool)
%       [Fx,Fy] = MFY(kappa,alpha,Fz,par,plotbool)
%
%   Description:
%       MFY() - Gives the longitudinal and lateral force of a tire, defined
%       by the parameters in par, under the conditions described by the
%       longitudinal slip kappa, the slide slip angle alpha, and vertical
%       force Fz.
%       Instaed of the true longitudinal force, the output Fxmon is given,
%       which is monotonic. This is achieved by increasing Fx linearly with
%       kappa, but only for the range of kappa outside the region of
%       interest.
%       Note that combined slip is not considered and that thus Fx =
%       f(kappa) and Fy = f(alpha).
%    
%   Inputs:
%       kappa - longitudinal slip factor [-]
%       alpha - side slip angle in [rad]
%       Fz - normal force in [N]
%       par - struct containing the tire parameters:
%               par.Fz0  : tyre nominal force [N]
%               par.pCx1 : Shape factor Cfx for longitudinal force         
%               par.pDx1 : Longitudinal friction Mux at Fznom         
%               par.pDx2 : Variation of friction Mux with load         
%               par.pKx1 : Longitudinal slip stiffness Kfx/Fz at Fznom
%               par.pCy1 : Shape factor Cfy for lateral forces
%               par.pDy1 : Lateral friction coefficient at Fz0
%               par.pDy2 : Lateral friction coefficient dependency on Fz
%               par.pEy1 : Lateral curvature Efy at Fznom
%               par.pEy2 : Lateral curvature dependency on Fz
%               par.pKy1 : peak cornering stiffness (Cfa=pKy1*Fz0)
%               par.pKy2 : vertical force at which peak Cfa occurs (Fz=pKy2*Fz0)
%
%   Outputs: 
%       Fx - longitudinal force in [N]
%       Fy - lateral force in [N]
%       
%   Other m-files required: none
%   Subfunctions: none
%   MAT-files required: none
%
%   See also: struct
dfz = (Fz-par.Fz0)/par.Fz0;

%% Lateral force calculation
C   = par.pCy1;
muy = par.pDy1 + par.pDy2*dfz;
D   = Fz.*muy;
E   = min(1,par.pEy1 + par.pEy2*dfz);
Cfa = par.pKy1*par.Fz0*sin(2*atan(Fz/(par.Fz0*par.pKy2)));
B   = Cfa./(C.*D + eps);

Fy  = D.*sin(C.*atan((1-E).*B.*alpha + E.*atan(B.*alpha))); %lateral force [N]

%% Longitudinal force calculation
Cx  = par.pCx1;
mux = par.pDx1 + par.pDx2.*dfz;
Dx  = Fz.*mux;
Kx  = par.pKx1.*Fz;
Bx  = Kx./(Cx*Dx+eps);

Fx  = @(kappa) Dx.*sin(Cx*atan(Bx.*kappa-(Bx.*kappa-atan(Bx.*kappa)))); %longitudinal force [N]

%Make Fx a monotonic function of kappa:
k_c= 0.6;   %[-] critical value for kappa, after which Fx is changed.
e  = 0.2;   %[-] smootness is applied in a band of 2e around k_c
k  = 50000;  % proportional gain after k_c

func_lin = @(kappa) 1/pi.*((kappa+k_c).*(atan(-(kappa+k_c)/e)+pi/2)+...
                           (kappa-k_c).*(atan( (kappa-k_c)/e)+pi/2));
%This function is a linear in kappa for kappa<k_c and kappa>k_c and
%(almost) zero in the range -k_c<kappa<k_c. The function is continuous and
%the smoothness can be adapted using the parameter e.

Fxmon = Fx(kappa) + k*func_lin(kappa);

%% Plot force curves if requested
if plotbool == true
    %Lateral force
    alphaPlot = -0.5:0.01:0.5;
    [~,FyPlot] = MFy(0,alphaPlot,Fz,par,0);
    
    figure; hold on; grid on;
    plot(rad2deg(alphaPlot),FyPlot)
    plot(rad2deg(alpha),Fy,'*r')
    xlabel('side slip angle [deg]');
    ylabel('Lateral force [N]');
    legend('Lateral force curve','final solution');
    
    %Longitudinal force
    kappaPlot = -1:0.001:1;
    [FxPlot,~] = MFy(kappaPlot,0,Fz,par,0);
    
    figure; hold on; grid on;
    plot(kappaPlot,Fx(kappaPlot),':k');
    plot(kappaPlot,FxPlot)
    plot(kappa,Fxmon,'*r')
    xlabel('Longitudinal slip ratio [-]');
    ylabel('Longitudinal tire force [N]');
    legend('Original Magic Formula','Adapted monotonic','Final solution','location','northwest');
    
    saveas(gcf,'./Figures_generated/MF_LongForce','epsc')
    saveas(gcf,'./Figures_generated/MF_LongForce','fig')
end

end