function [P_out,xi_out,phi_out,Phi_out,solution,xi_config] = PowerTransmissionComputation_v3(config,etas,phi,Phi,xi,P_p,P_p1,P_p2,P_inst)
%%% Description
%
% This function calculates the power flow along each power path which
% connects a series of components defining the powertrain. A
% series/parallel partial-hybrid (SPPH) architecture is assumed, based on 
% which eight other powertrain configurations can be derived for specific 
% values of the control parameters. The powertrain architecture is:
%
% .----.  Pf  .----.  Pgt .----.  Ps1 .----. Pp1
% |  F |----->| GT |----->| GB |----->| P1 |----->
% '----'      '----'      '----'      '----'
%                            | 
%                            | Pgb
%                            V
%                         .----.
%                         |EM1 |
%                         '----'
%                            | 
%                            | Pe1
%                            V
%             .----. Pbat .----. Pe2  .----. Ps2  .----. Pp2
%             |BAT |----->|PMAD|----->|EM2 |----->| P2 |----->
%             '----'      '----'      '----'      '----'
%
% The arrows indicate the powerpaths (10 in total), whose arrowhead
% indicates the direction defined as positive in this script. All power
% flows can have negative values (i.e. flow in the other direction) except
% Pf and Pgt. There are two energy sources (fuel and battery), and seven
% powertrain components (gas turbine, gearbox, primary propulsor,
% electro-generator, power management and distribution system,
% electromotor, and secondary propulsor), each of which have an associated
% eneryg conversion efficiency "eta". Note that actually the EM and EG can 
% both act as "motor" and as "generator", depending on the direction of
% power flow.
%
% The amount of input parameters required to define the behaviour of the
% powertrain depends on the architecture selected. For the configuration
% depicted above, three degrees of freedom (DOFs) have to be specified,
% which is equal to the number of nodes plus one. For most configurations,
% one or two of these parameters will be fixed, and thus in most cases only
% one or two input parameters can be specified. Specifying more
% over-constrains the problem. The nine possible architectures that can be 
% generated are:
%
% config            phi   Phi     N of DOF    Possible input variables
% ------------------------------------------------------------------------
% 1. conventional   0     0       1           Pp,Pp1,xi
% 2. turboelectric  0     1       1           Pp,Pp2,xi
% 3. serial         any   1       2           Pp,Pp2,xi,phi
% 4. parallel       any   0       2           Pp,Pp1,xi,phi
% 5. PTE            0     any     2           Pp,Pp1,Pp2,xi,Phi
% 6. SPPH           any   any     3           Pp,Pp1,Pp2,xi,phi,Phi
% 7. e-1            1     0       1           Pp,Pp1,xi
% 8. e-2            1     1       1           Pp,Pp2,xi
% 9. dual-e         1     any     2           Pp,Pp1,Pp2,xi,Phi
%
% Where:
%   - xi is the throttle setting of the gas turbine (in configurations
%       which have a gas turbine; P_gt/P_inst) or of the primary 
%       electromotor (for configurations 7 and 9) or secondary electromotor
%       (configuration 8)
%       configurations 7-9, which are fully electric; P_em/P_inst).
%   - phi is the supplied power ratio, P_bat/(P_bat + P_f)
%   - Phi is the shaft power ratio, P_s2/(Ps_2 + P_s1)
%
% The system has 10 unknowns (the 10 power paths), and 7 equations: one per
% component of the powertrain. At each component, the power balance is
% established as:
%
%   Summation(powers_out) = eta_component * Summation(powers_in)
%
% The remaining three equations necessary to obtain a single solution are
% generated by specifying the three DOF of the system, e.g. by fixing a
% value for phi, Phi and P_p1. The equations are grouped in matrices as:
%
%   A*x = b
%
% Where:
%   - A is the coefficient matrix of size 10 x 10, where the top seven rows
%       correspond to the equations of the nodes and the bottom three rows
%       correspond to the "boundary conditions" or "operating conditions"
%   - x is a column vector containing the ten power paths
%   - b (the RHS) is a column vector of ten elements. The top seven are
%       equal to 0 (power balance at the nodes), and the bottom three
%       depend on the boundary/operating conditions selected.
%
% However, the position and sign of the constants in the matrix A (ones and
% etas) depends on the direction in which the powers flow. A total of 9
% different possibilites exist, depending on whether the propulsors are
% generating thrust (P > 0) or extracting energy from the flow (P < 0), and
% on wether the batteries are charging or discharging. Grouping the power
% paths which can change sign into four groups (I-IV) and establishing that
% at a given node not all powers can simultaneously flow in or out of the
% node, we can define the 9 combinations as shown in the table below. In
% this table, a "+" sign means the powers flow as shown in the figure
% above, while a "-" sign means they flow in opposite direction. Since not
% all combinations are compatible with the different powertrain
% architectures (and some are redundant), the possible matches between
% power-flow-direction combinations and powertrain architectures are
% indicated with a cross.
%
% I:   Ps1, Pp1
% II:  Pgb, Pe1
% III: Pbat
% IV:  Pe2,Ps2,Pp2
%
% Comb.  I   II  III IV  Co. TE. Se. Pa. PTE SPPH e-1 e-2 dual-e 
% ---------------------------------------------------------------
%  1     +   +   +   +   x   x   x   x   x   x        x
%  2     +   +   -   +           x   x       x
%  3     +   +   -   -           x           x        x
%  4     +   -   +   +               x       x    x       x
%  5     +   -   +   -                   x   x            x
%  6     +   -   -   -                       x            x
%  7     -   +   +   +                   x   x            x
%  8     -   +   -   +               x       x    x       x
%  9     -   +   -   -                       x            x
%
%
%
% Input:
%    - config: string determining powertrain layout. The parentheses
%       indicate whether the powertrain contains primary, secondary, or 
%       both propulsors.
%           1. conventional (1)                                
%           2. turboelectric (1,2)                                     
%           3. serial (1,2)                                         
%           4. parallel (1)                                         
%           5. PTE [partial turboelectric] (1,2)  
%           6. SPPH [series-parallel partial hybrid] (1,2)
%           7. e-1 [full-electric, primary propulsor only] (1)               
%           8. e-2 [full-electric, secondary propulsor only] (2)               
%           9. dual-e [full-electric, both propulsors] (1+2)       
%
%   - etas: structure containing the efficiencies of all components.
%       Efficiencies are supposed independent of operating condition (e.g.
%       an electro-motor-generator has the same P_out/P_in (<= 1) value if
%       it is acting as a motor or as a generator). Note that battery power
%       refers to the actual power that the battery has to absorb/produce,
%       neglecting battery efficiency. Efficiency values should be in the
%       interval (0,1). Values of 0 or 1 may lead to singularities in the
%       solution and must be used with caution!
%           .p1: propulsive efficiency of primary propulsors
%           .p2: propulsive efficiency of secondary propulsors
%           .eg: conversion efficiency of electro-generator
%           .em: conversion efficiency of electro-motor
%           .gb: transmission efficiency of gearbox
%           .gt: conversion (thermal) efficiency of gas turbine
%           .pm: transmission efficiency of PMAD system
%
%   - Operating conditions (DOF's). IMPORTANT RULES: 
%       A) one, two or three of these variables must be specified,
%          depending on the configuration (see "N of DOF" in the first
%          table above). 
%       B) The ones that are eligible for each configuration are given in
%          the last column of the first table above
%       C) All other DOF's must be specifed as "[]" or "NaN"
%       D) The three powers, Pp, Pp1, and Pp2, cannot be specified
%          simultaneously
%       E) for this same reason, Pp and Phi (or Phi, Pp1 and Pp2) cannot
%          be specified simultaneously
%       F) If throttle is specified, the maximum installed power (P_inst)
%          of the GT/EM must be specified.
%       
%           - phi: supplied power ratio [-]
%           - Phi: shaft power ratio [-]
%           - xi: throttle [-]
%           - P_p: total propulsive power P = T*V [W]
%           - P_p1: propulsive power produced by primary propulsors [W]
%           - P_p2: propulsive power produced by secondary propulsors [W]
%
%   - P_inst: maximum power of the gas turbine (for configurations 1-6) or
%       the electromotor (for configurations 7-9) in the given flight
%       condition (so not at sea level!) [W] (see notes below)
%
% Output:
%   - P_out: structure containing the power output of each path. If path
%       is not used or input was incorrectly specified, a NaN value is
%       assigned. [W]
%           .f: fuel power
%           .gt: gas turbine power
%           .e1: primary electric machine (generator) electrical power
%           .e2: secondary electric machine (motor) electrical power
%           .s1: shaft power, primary propulsors
%           .s2: shaft power, secondary propulsors
%           .p1: useful/propulsive power, primary propulsors
%           .p2: useful/propulsive power, secondary propulsors
%           .bat: battery power, negative if charging
%           .gb: power between gearbox and electric generator
%           .p: total useful/propulsive power, = p1 + p2
%   - phi_out: supplied power ratio [-] (equal to input, if used)
%   - Phi_out: shaft power ratio [-] (equal to input, if used)
%   - xi_out: throttle setting [-] (equal to input, if used)
%   - solution: 1D array of 9 elements, each one corresponding to one 
%       combination of power-flow-directions (see second table above). 
%       Contains "1" at the position(s) for which a feasible solution
%       was obtained, and 0 everywhere else (i.e. sum(solutions) = the
%       number of feasible solutions obtained).
%   - xi_config: tells us if xi_out refers to the GT (= 1), the primary
%       EM (= 2), or the secondary EM (= 3).
%
% Notes:
%   - For subscripts: upper case letters refer to variables associated to
%       components, lower case letters refer to variables associated to
%       power paths.
%   - It is not recommended to use Pu as input (without P_p1 or P_p2) for 
%       configurations with multiple propulsor systems, since this may lead 
%       to two possible solutions (depending on the remaining input 
%       parameters). Use P_p1 and/or P_p2 instead.
%   - For architectures such as the conventional or TE ones, where the GB
%       and PMAD nodes are not needed, the effiency value of these
%       components can be set to 1 so that they have no effect on the
%       system.
%   - IMPORTANT: for architectures where phi can have any value (series,
%       parallel, and SPPH), if phi = 1, throttle cannot refer to the GT
%       since it produces zero power and leads to a singular matrix.
%       Therefore, if phi = 1 is selected as input for these architectures,
%       then xi is assumed to refer to electromotor throttle instead, and
%       P_inst also refers to the corresponding EM. (EM1 for parallel, or
%       EM2 for serial/SPPH).
%   - For more info, see: de Vries, M. Brown & Vos, (2018)
%
%
%%% Reynard de Vries
%%% TU Delft
%%% Date created: 25-01-18
%%% Last modified: 09-03-22


%% Assign input

% Component efficiencies
e_GT = etas.GT;     % Gas turbine
e_GB = etas.GB;     % Gearbox
e_P1 = etas.P1;     % Primary propulsors
e_EM1 = etas.EM1;   % Primary powertrain electrical machine (normally a generator)
e_PM = etas.PM;     % PMAD (power management and distribution)
e_EM2 = etas.EM2;   % Secondary powertrain electrical machine (normally a motor)
e_P2 = etas.P2;     % Secondary propulsors

% Convert empty inputs to NaN
if isempty(phi); phi=NaN; end
if isempty(Phi); Phi=NaN; end
if isempty(xi); xi=NaN; end
if isempty(P_p); P_p=NaN; end
if isempty(P_p1); P_p1=NaN; end
if isempty(P_p2); P_p2=NaN; end
if isempty(P_inst); P_inst=NaN; end


%% Check if input is consistent with configuration selected

% Make sure that:
% Pp, Pp1 and Pp2 not have been specified simultaneously
if sum(~isnan([P_p P_p1 P_p2])) == 3
    error('The primary, secondary and total useful power cannot be specified simultaneously.')

% Phi, Pp1 and Pp2 (or Pp), have not been specified simultaneously
elseif (sum(~isnan([Phi P_p1 P_p2])) == 3) || ...
        sum(~isnan([Phi P_p P_p1])) == 3 || ...
        sum(~isnan([Phi P_p P_p2])) == 3
    error('The primary and secondary propulsive powers cannot be specified simultaneously with Phi.')
end

% Check/set input parameters for each configuration
switch config
    case 'conventional'
        
        % Enforce OCs
        phi = 0;
        Phi = 0;
        
        % Select input OCs
        OC = [xi P_p P_p1];
        
        % Number of DOFs which have to be specified (for analysis)
        DOFs = 1;
        
        % Does engine throttle refer to the GT (1), EM1 (2), or EM2 (3)?
        xi_config = 1;
        
        % Only evaluate combinations of power flow which are applicable to 
        % the current powertrain configuration, in order to save time
        % and avoid multiple incorrect solutions.
        combis = 1;        
        
    case 'turboelectric'
        phi = 0;
        Phi = 1;
        OC = [xi P_p P_p2];
        DOFs = 1;
        xi_config = 1;
        combis = 1;        
        
    case 'serial'
        Phi = 1;
        OC = [phi xi P_p P_p2];
        DOFs = 2;
        combis = [1 2 3];
        
        % If phi = 1, use throttle to refer to EM2 throttle instead of
        % standard GT (avoid indeterminate)
        if  phi == 1
            xi_config = 3;
        else
            xi_config = 1;
        end       
        
    case 'parallel'
        Phi = 0;
        OC = [phi xi P_p P_p1];
        DOFs = 2;
        combis = [1 2 4 8];
        
       % If phi = 1, use throttle to refer to EM1 throttle instead of
        % standard GT (avoid indeterminate)
        if  phi == 1
            xi_config = 2;
        else
            xi_config = 1;
        end
        
    case 'PTE'
        phi = 0;
        OC = [Phi xi P_p P_p1 P_p2];
        DOFs = 2;
        xi_config = 1;
        combis = [1 5 7];
        
    case 'SPPH'
        OC = [phi Phi xi P_p P_p1 P_p2];
        DOFs = 3;
        combis = [1 2 3 4 5 6 7 8 9];
               
        % If phi = 1, use throttle to refer to EM2 throttle instead of
        % standard GT (avoid indeterminate)
        if  phi == 1
            xi_config = 3;
        else
            xi_config = 1;
        end
        
        
    case 'e-1'
        phi = 1;
        Phi = 0;
        OC = [xi P_p P_p1];
        DOFs = 1;
        xi_config = 2;
        combis = [4 8];
        
    case 'e-2'
        phi = 1;
        Phi = 1;
        OC = [xi P_p P_p2];
        DOFs = 1;
        xi_config = 3;
        combis = [1 3];
        
    case 'dual-e'
        phi = 1;
        OC = [xi Phi P_p P_p1 P_p2];
        DOFs = 2;
        xi_config = 2;
        combis = [4 5 6 7 8 9];
        
    otherwise
        error('Wrong powertrain configuration specified.')
end

% Get name of this script for error message
FN = dbstack();
fn_err = FN(1).file;

% Check whether the correct number of DOFs have been specified
if sum(~isnan(OC)) ~= DOFs
    error(['Exactly ' num2str(DOFs) ' operating condition(s) '...
        'has/have to be specified for analyzing a ' ...
        config ' configuration. See the ''' fn_err ...
        ''' function for more information.'])
end

% Verify that if throttle is specified, the maximum GT (or EM) power in
% this flight condition is also specified
if ~isnan(xi) && isnan(P_inst)
    error(['If throttle is specified as input parameter, the maximum '...
           'available power of the gas turbine (or electromotor) must '...
           ' be specified.'])
end

% If all works, gather OCs (keep this order!)
OCs = [phi Phi xi P_p P_p1 P_p2];

% Is an exensive parameter specified? If the specified power requirements
% equal zero, then all powers = 0 is a solution of the system.
if isnan(P_p) || P_p == 0; P_pSwitch = 0; else; P_pSwitch = 1; end
if isnan(P_p1) || P_p1 == 0; P_p1Switch = 0; else; P_p1Switch = 1; end
if isnan(P_p2) || P_p2 == 0; P_p2Switch = 0; else; P_p2Switch = 1; end
if isnan(xi) || xi == 0
    xiSwitch = 0; 
else
    if isnan(P_inst) || P_inst==0
        xiSwitch = 0; 
    else
        xiSwitch = 1;
    end
end
switches = [P_pSwitch P_p1Switch P_p2Switch xiSwitch];
if sum(switches) == 0
    zeroIsASolution = 1;
else
    zeroIsASolution = 0;
end


%% Construct system matrices

% Merge node equations in top part of coefficient matrix (7 x 10). Do this
% for the 9 possible sign combinations. For each assumed power flow
% direction, a 7x10 matrix is created along the third dimension of A_top.
% After solving, the sign of the results must be checked to see which
% assumption was correct. In almost all cases, only one solution will be
% correct. However (hypothesis: to be proven), for a determined combination
% of input parameters, there might be two possible solutions. It is almost
% impossible to exactly obtain this value, but nonetheless, all possible
% flows should be considered just in case. 
A_top = NaN(7,10,9);

% Case 1: Nominal. Both props are thrusting, battery is discharging
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,1) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   1       1       0       0       0       0       0       0       ;   % gb node
     0      0       0       -e_P1   0       0       0       0       1       0       ;   % p1 node
     0      0       -e_EM1  0       1       0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -e_PM   -e_PM   1       0       0       0       ;   % pmad node
     0      0       0       0       0       0       -e_EM2  1       0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -e_P2   0       1       ];  % p2 node        

% Case 2: Both props thrusting, battery charging
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,2) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   1       1       0       0       0       0       0       0       ;   % gb node
     0      0       0       -e_P1   0       0       0       0       1       0       ;   % p1 node
     0      0       -e_EM1  0       1       0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -e_PM   -1      1       0       0       0       ;   % pmad node
     0      0       0       0       0       0       -e_EM2  1       0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -e_P2   0       1       ];  % p2 node  

% Case 3: Primary prop thrusting, secondary windmilling and battery charging
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,3) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   1       1       0       0       0       0       0       0       ;   % gb node
     0      0       0       -e_P1   0       0       0       0       1       0       ;   % p1 node
     0      0       -e_EM1  0       1       0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -e_PM   -1      +e_PM   0       0       0       ;   % pmad node
     0      0       0       0       0       0       -1      +e_EM2  0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -1      0       +e_P2   ];  % p2 node 
 
% Case 4: Props thrusting, battery delivers power to both
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,4) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   +e_GB   1       0       0       0       0       0       0       ;   % gb node
     0      0       0       -e_P1   0       0       0       0       1       0       ;   % p1 node
     0      0       -1       0      +e_EM1  0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -1      -e_PM   1       0       0       0       ;   % pmad node
     0      0       0       0       0       0       -e_EM2  1       0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -e_P2   0       1       ];  % p2 node   
 
% Case 5: Primary prop thrusting, powered by battery and secondary prop
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,5) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   +e_GB   1       0       0       0       0       0       0       ;   % gb node
     0      0       0       -e_P1   0       0       0       0       1       0       ;   % p1 node
     0      0       -1      0       +e_EM1  0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -1      -e_PM   e_PM    0       0       0       ;   % pmad node
     0      0       0       0       0       0       -1      e_EM2   0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -1      0       e_P2    ];  % p2 node    

% Case 6: Primary prop thrusting, sec. is charging bat and powering primary
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,6) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   +e_GB   1       0       0       0       0       0       0       ;   % gb node
     0      0       0       -e_P1   0       0       0       0       1       0       ;   % p1 node
     0      0       -1      0       e_EM1   0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -1      -1      e_PM    0       0       0       ;   % pmad node
     0      0       0       0       0       0       -1      e_EM2   0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -1      0       e_P2    ];  % p2 node   
% Case 6(OLD)
%{
A_top(:,:,6) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   -e_GB   1       0       0       0       0       0       0       ;   % gb node
     0      0       0       -e_P1   0       0       0       0       1       0       ;   % p1 node
     0      0       1       0       e_EM1   0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -1      -1      e_PM    0       0       0       ;   % pmad node
     0      0       0       0       0       0       -1      e_EM2   0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -1      0       e_P2    ];  % p2 node   
 %} 
 
% Case 7: Primary windmilling, secondary thrusting and bat discharging
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,7) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   1       e_GB    0       0       0       0       0       0       ;   % gb node
     0      0       0       -1      0       0       0       0       e_P1    0       ;   % p1 node
     0      0       -e_EM1  0       1       0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -e_PM   -e_PM   1       0       0       0       ;   % pmad node
     0      0       0       0       0       0       -e_EM2  1       0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -e_P2   0       1       ];  % p2 node      

% Case 8: Primary windmilling, secondary thrusting and bat charging
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,8) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   1       e_GB    0       0       0       0       0       0       ;   % gb node
     0      0       0       -1      0       0       0       0       e_P1    0       ;   % p1 node
     0      0       -e_EM1  0       1       0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -e_PM   -1      1       0       0       0       ;   % pmad node
     0      0       0       0       0       0       -e_EM2  1       0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -e_P2   0       1       ];  % p2 node    

% Case 9: Both props windmilling, charging batteries
%    Pf     Pgt     Pgb     Ps1     Pe1     Pbat    Pe2     Ps2     Pp1     Pp2
A_top(:,:,9) = ...
    [-e_GT  1       0       0       0       0       0       0       0       0       ;   % gt node
     0      -e_GB   1       e_GB    0       0       0       0       0       0       ;   % gb node
     0      0       0       -1      0       0       0       0       e_P1    0       ;   % p1 node
     0      0       -e_EM1  0       1       0       0       0       0       0       ;   % EM1 node
     0      0       0       0       -e_PM   -1      e_PM    0       0       0       ;   % pmad node
     0      0       0       0       0       0       -1      e_EM2   0       0       ;   % EM2 node
     0      0       0       0       0       0       0       -1      0       e_P2    ];  % p2 node 

% Top part of RHS
b_top = [0 0 0 0 0 0 0]';

% Select three OC equations depending on the OCs specified.
A_bot = NaN(3,10);
b_bot = NaN(3,1);
eqN = 1;
for k = 1:length(OCs)
    if ~isnan(OCs(k))
        switch k
            case 1  % phi is specified
                A_bot(eqN,:) = ...
                    [phi    0       0       0       0       (phi-1) 0       0       0       0];
                b_bot(eqN) = 0;
                
            case 2  % Phi is specified
                A_bot(eqN,:) = ...
                    [0      0       0       Phi     0       0       0       (Phi-1) 0       0];
                b_bot(eqN) = 0;
                
            case 3  % xi is specified
                if xi_config == 1
                A_bot(eqN,:) = ...
                    [0      1       0       0       0       0       0       0       0       0];
                elseif xi_config == 2 % Note: nominal operation of a e-1 or dual-e architecture
                A_bot(eqN,:) = ...    % has a negative e1 path flow!
                    [0      0       0       0       -1      0       0       0       0       0];
                elseif xi_config == 3
                A_bot(eqN,:) = ...
                    [0      0       0       0       0       0       1       0       0       0];
                end
                b_bot(eqN) = xi*P_inst;
            case 4  % P_p is specified
                A_bot(eqN,:) = ...
                    [0      0       0       0       0       0       0       0       1       1];
                b_bot(eqN) = P_p;
            
            case 5  % P_p1 is specified
                A_bot(eqN,:) = ...
                    [0      0       0       0       0       0       0       0       1       0];
                b_bot(eqN) = P_p1;
            
            case 6  % P_p2 is specified
                A_bot(eqN,:) = ...
                    [0      0       0       0       0       0       0       0       0       1];
                b_bot(eqN) = P_p2;
        end
        eqN = eqN + 1;
    end
end


%% Solve matrices and verify assumptions

% Merge RHS
b = [b_top; b_bot];

% Loop over different combinations (note: Option a) break loop when a
% feasible solution has been found. Faster. Option b) always evaluate all
% nine possibilities in case multiple solutions exist)
P = NaN(9,10);
solution = zeros(9,1);
for k = combis
    
    % Merge coefficient matrix
    A = [A_top(:,:,k); A_bot];

    % Invert matrix to obtain power paths, save each solution in a row
    P(k,:) = A\b;

    % Check if solution is consistent with assumptions
    switch k
        case 1 % Pgt            Ps1            Pe1            Pbat           Pe2
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)>=0) && (P(k,6)>=0) && (P(k,7)>=0) 
                solution(k) = 1;
            end
        case 2
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)>=0) && (P(k,6)< 0) && (P(k,7)>=0) 
                solution(k) = 1;
            end
        case 3
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)>=0) && (P(k,6)< 0) && (P(k,7)< 0) 
                solution(k) = 1;
            end
        case 4
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)< 0) && (P(k,6)>=0) && (P(k,7)>=0) 
                solution(k) = 1;
            end
        case 5
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)< 0) && (P(k,6)>=0) && (P(k,7)< 0) 
                solution(k) = 1;
            end
        case 6
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)< 0) && (P(k,6)< 0) && (P(k,7)< 0) 
                solution(k) = 1;
            end
        case 7
            if (P(k,2)>=0) && (P(k,4)< 0) && (P(k,5)>=0) && (P(k,6)>=0) && (P(k,7)>=0) 
                solution(k) = 1;
            end
        case 8
            if (P(k,2)>=0) && (P(k,4)< 0) && (P(k,5)>=0) && (P(k,6)< 0) && (P(k,7)>=0) 
                solution(k) = 1;
            end
        case 9
            if (P(k,2)>=0) && (P(k,4)< 0) && (P(k,5)>=0) && (P(k,6)< 0) && (P(k,7)< 0) 
                solution(k) = 1;
            end
            
        % Old version, when "<0" was used instead of "<=0" to avoid 
        % multiple solutions in case a result is zero
        %{
        case 1 % Pgt            Ps1            Pe1            Pbat           Pe2
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)>=0) && (P(k,6)>=0) && (P(k,7)>=0)
                solution(k) = 1;
            end
        case 2
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)>=0) && (P(k,6)< 0) && (P(k,7)>=0)
                solution(k) = 1;
            end
        case 3
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)>=0) && (P(k,6)< 0) && (P(k,7)< 0)
                solution(k) = 1;
            end
        case 4
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)< 0) && (P(k,6)>=0) && (P(k,7)>=0)
                solution(k) = 1;
            end
        case 5
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)< 0) && (P(k,6)>=0) && (P(k,7)< 0)
                solution(k) = 1;
            end
        case 6
            if (P(k,2)>=0) && (P(k,4)>=0) && (P(k,5)< 0) && (P(k,6)< 0) && (P(k,7)< 0)
                solution(k) = 1;
            end
        case 7
            if (P(k,2)>=0) && (P(k,4)< 0) && (P(k,5)>=0) && (P(k,6)>=0) && (P(k,7)>=0)
                solution(k) = 1;
            end
        case 8
            if (P(k,2)>=0) && (P(k,4)< 0) && (P(k,5)>=0) && (P(k,6)< 0) && (P(k,7)>=0)
                solution(k) = 1;
            end
        case 9
            if (P(k,2)>=0) && (P(k,4)< 0) && (P(k,5)>=0) && (P(k,6)< 0) && (P(k,7)< 0)
                solution(k) = 1;
            end
            %}
    end            
end

% Warning if multiple/no solutions have been found. If multiple solutions
% are found, select first. Only give warning of multiple solutions if the
% different solutions are not the same (i.e. when all powers are zero)
if sum(solution)>1
    toggle = 0;
    solIdx = 0;
    while toggle == 0
        solIdx = solIdx + 1;
        if solution(solIdx)==1; toggle = 1; end
    end
    if ~all(P(solIdx,:)==0)
        disp(['    > Warning: multiple solutions found (' ...
            num2str(sum(solution)) ...
            '). Selecting first one of the following possibilities = ' ...
            num2str(solution')])
    end
elseif sum(solution)==0
    if zeroIsASolution == 1
        solIdx = 0;
    else
        disp('    > Warning: no solutions found')
        solIdx = NaN;
    end
else
    [~,solIdx] = max(solution);
end


%% Organize output

% Assign output variables
if isnan(solIdx)
    P_out.f  = NaN;
    P_out.gt = NaN;
    P_out.gb = NaN;
    P_out.s1 = NaN;
    P_out.e1 = NaN;
    P_out.bat= NaN;
    P_out.e2 = NaN;
    P_out.s2 = NaN;
    P_out.p1 = NaN;
    P_out.p2 = NaN;
    P_out.p  = NaN;
elseif solIdx == 0
    P_out.f  = 0;
    P_out.gt = 0;
    P_out.gb = 0;
    P_out.s1 = 0;
    P_out.e1 = 0;
    P_out.bat= 0;
    P_out.e2 = 0;
    P_out.s2 = 0;
    P_out.p1 = 0;
    P_out.p2 = 0;
    P_out.p  = 0;
else   
    P_out.f  = P(solIdx,1);
    P_out.gt = P(solIdx,2);
    P_out.gb = P(solIdx,3);
    P_out.s1 = P(solIdx,4);
    P_out.e1 = P(solIdx,5);
    P_out.bat= P(solIdx,6);
    P_out.e2 = P(solIdx,7);
    P_out.s2 = P(solIdx,8);
    P_out.p1 = P(solIdx,9);
    P_out.p2 = P(solIdx,10);
    P_out.p  = P(solIdx,10) + P(solIdx,9);
    
    % Set variables that are not applicable to powertrain config to NaN
    switch config
        case 'conventional'; P_out.gb = NaN; P_out.e1 = NaN; P_out.bat = NaN;
                             P_out.e2 = NaN; P_out.s2 = NaN; P_out.p2 = NaN;
        case 'turboelectric';P_out.s1 = NaN; P_out.p1 = NaN; P_out.bat = NaN;
        case 'serial';       P_out.s1 = NaN; P_out.p1 = NaN; 
        case 'parallel';     P_out.s2 = NaN; P_out.p2 = NaN; P_out.e2 = NaN;
        case 'PTE';          P_out.bat = NaN;
        case 'SPPH'
        case 'e-1';          P_out.f = NaN; P_out.gt = NaN; P_out.e2 = NaN;
                             P_out.s2 = NaN; P_out.p2 = NaN;
        case 'e-2';          P_out.f = NaN; P_out.gt = NaN; P_out.gb = NaN;
                             P_out.s1 = NaN; P_out.p1 = NaN; P_out.e1 = NaN;
        case 'dual-e';       P_out.f = NaN; P_out.gt = NaN;
    end
end

% Compute operating conditions which had not been specified
if ~isnan(P_inst)
    if xi_config == 1
        xi_out = P_out.gt/P_inst;
    elseif xi_config == 2
        xi_out = -P_out.e1/P_inst;
    elseif xi_config == 3
        xi_out = P_out.e2/P_inst;
    end
else
    xi_out = NaN;
end
if isnan(phi)
    phi_out = P_out.bat/(P_out.bat+P_out.f);
else
    phi_out = phi;
end
if isnan(Phi)
    Phi_out = P_out.s2/(P_out.s2+P_out.s1);
else
    Phi_out = Phi;
end


