function [fig] = PlotPowertrain(P_path,X_in,X_query,figNum,formSpec,description)
%%% Description
%  
% This function takes the three power-structures as generated by
% SizePowertrain.m, and generates a figure showing the powertrain
% components and power flows. Components and paths with NaN values are
% assumed not to exist in the powertrain. The resulting powertrain
% architecture should look as specified in the inpute file (e.g.
% conventional, serial, parallel,...). See
% PowerTransmissionComputation_v3.m for more details.
%
% The input powers can be specified as point values, in which case X_in
% and X_query are ignored, or as arrays, in which case the function
% interpolates the value of P at the specified X-coordinate. The powers
% plotted in the figure have the same units (or the inverse) as the input.
%
% Input:
%   - P_path: structure containing the values of power along the ten paths
%       of the powertrain: f, gt, gb, p1, s1, e1, bat, e2, s2, p2:
%
%     .----.  Pf  .----.  Pgt .----.  Ps1 .----. Pp1
%     |  F |----->| GT |----->| GB |----->| P1 |----->
%     '----'      '----'      '----'      '----'
%                                | 
%                                | Pgb
%                                V
%                             .----.
%                             |EM1 |
%                             '----'
%                                | 
%                                | Pe1
%                                V
%                 .----. Pbat .----. Pe2  .----. Ps2  .----. Pp2
%                 |BAT |----->|PMAD|----->|EM2 |----->| P2 |----->
%                 '----'      '----'      '----'      '----'
%
%   - X_in: 1D array of length equal to the fields in P_path. If
%       only one value (length=1) is specified, this input is ignored.
%   - X_query: X-coordinate at which the P-structures are interpolated, if
%       the input powers have length > 1.
%   - figNum: number of the figure generated. Leave empty or specify as NaN
%       to generate a generic new figure
%   - formSpec: format spec of the numbers incuded in the plots
%   - description: string used in figure title and name.
%
% Output:
%   - fig: figure handle
%
%%% Reynard de Vries
%%% TU Delft
%%% Date created: 08-02-18
%%% Last modified: 12-05-18


%% Settings

% Font size
fontsize = 8;

% Box sizes
A = 0.2;    % Component width
B = 0.1;    % Component height
C = 0.15;   % Propulsor width
D = 0.3;    % Propulsor height

% Box separation
DX = 0.4;   % Grid width
DY = 0.3;   % Grid height

% Colors
colLoss = [1 0 0];
colOff = [0.95 0.95 0.95];
col0 = [0.9 0.9 0.9];
col1 = [0.8 1 1];
col2 = [1 0.8 0.8];


%% Compute component powers and power losses

% Compute losses and sizing condition of each component
%
% The gas turbine is sized in terms of the shaft power it provides, since
% that is the convention, and the power cannot flow in opposite direction.
% For the other components (GB, EG, PM, EM, P1, P2), the "sizing" power is
% assumed to be the summation of powers flowing into the element, which
% will always be greater than the summation of powers flowing out of the
% element. The difference between the ones going in and out is the loss of
% that component, which will have to be accounted for with e.g. cooling.
for i = 1:length(P_path.gt)
    P_comp.GT(i) = P_path.gt(i);
    P_comp.EM1(i) = max([abs(P_path.gb(i)) abs(P_path.e1(i))]);
    P_comp.EM2(i) = max([abs(P_path.e2(i)) abs(P_path.s2(i))]);
    P_comp.P1(i) = max([abs(P_path.p1(i)) abs(P_path.s1(i))]);
    P_comp.P2(i) = max([abs(P_path.p2(i)) abs(P_path.s2(i))]);
    
    % For the GB and PMAD nodes, take all powers flowing into the node as
    % positive. Then add the ones with positive values.
    P1_GB = P_path.gt(i); P2_GB = -P_path.s1(i); P3_GB = -P_path.gb(i);
    P1_PM = P_path.e1(i); P2_PM = P_path.bat(i); P3_PM = -P_path.e2(i);
    P_GB = [P1_GB P2_GB P3_GB];
    P_PM = [P1_PM P2_PM P3_PM];
    if sum(isnan(P_GB)) == 3
        P_comp.GB(i) = NaN;
    else
        P_comp.GB(i) = sum(P_GB(P_GB>0));
    end
    if sum(isnan(P_PM)) == 3
        P_comp.PM(i) = NaN;
    else
        P_comp.PM(i) = sum(P_PM(P_PM>0));
    end
    
    % Compute losses as difference between inflow and outflow at each component
    P_loss.GT(i) = abs(abs(P_path.gt(i))-abs(P_path.f(i)));
    P_loss.EM1(i) = abs(abs(P_path.gb(i))-abs(P_path.e1(i)));
    P_loss.EM2(i) = abs(abs(P_path.e2(i))-abs(P_path.s2(i)));
    P_loss.P1(i) = abs(abs(P_path.p1(i))-abs(P_path.s1(i)));
    P_loss.P2(i) = abs(abs(P_path.p2(i))-abs(P_path.s2(i)));
    if sum(isnan(P_GB)) == 3
        P_loss.GB(i) = NaN;
    else
        P_loss.GB(i) = sum(P_GB(P_GB>0)) + sum(P_GB(P_GB<0));
    end
    if sum(isnan(P_PM)) == 3
        P_loss.PM(i) = NaN;
    else
        P_loss.PM(i) = sum(P_PM(P_PM>0)) + sum(P_PM(P_PM<0));
    end
end


%% Obtain values to be plotted

% Fieldnames
namesp = fieldnames(P_path);
namesc = fieldnames(P_comp);
np = size(namesp,1);
nc = size(namesc,1);

% Interpolate to WS value if array and query point are provided, and
% correct to corresponding operating conditions. Ignore GTM.
if (length(P_path.(namesp{1})) > 1) && (~isnan(X_query) || ~isempty(X_query))
    
    % Loop over all paths
    for i = 1:np
        path.(namesp{i}) = interp1(X_in,P_path.(namesp{i}),X_query,'linear');
    end
    
    % Loop over all components
    for i = 1:nc
        comp.(namesc{i}) = interp1(X_in,P_comp.(namesc{i}),X_query,'linear');
        loss.(namesc{i}) = interp1(X_in,P_loss.(namesc{i}),X_query,'linear');
    end
    
% If no query point is provided, return error    
elseif (length(P_path.(namesp{1})) > 1) && isnan(X_query) && isempty(X_query)
    error(['A WS value must be specified when providing an array of '...
     'input powers, to evaluate the powers at the correct X-coordinate.'])
    
% If only one value has been provided, plot that one
elseif (length(P_path.(namesp{1})) == 1) 
    
    % Loop over all paths
    for i = 1:np
        path.(namesp{i}) = P_path.(namesp{i});
    end
    
    % Loop over all components
    for i = 1:nc
        comp.(namesc{i}) = P_comp.(namesc{i});
        loss.(namesc{i}) = P_loss.(namesc{i});
    end
end


%% Generate figure, settings

% Precalculate box coordinates w.r.t center
boxA = [-A/2 A/2 A/2 -A/2 -A/2];
boxB = [B/2 B/2 -B/2 -B/2 B/2];
boxC = [-C/2 C/2 C/2 -C/2 -C/2];
boxD = [D/2 D/2 -D/2 -D/2 D/2];

% Use predefined figure number, if specified, and clear
if ~isempty(figNum) && figNum > 0
    if ishandle(figNum)
        fig = figure(figNum);
        clf(fig);
    else 
        fig = figure(figNum);
    end
else
    fig = figure;
end

% Set figure characteristics
fig.Color = [1 1 1];
fig.Units = 'centimeters';
fig.PaperSize = [29.7 21.0];
fig.PaperPosition = [0 0 29.7 21.0];
fig.Name = ['Powertrain map: ' description];

% Generate axes
hold on;
grid on; box on;
ax = gca;
axis equal
axis([0 2 0 1.2]);
ax.Position = [0 0 1 1];


%% Create component boxes

% Fuel box
if ~isnan(path.f)
    fill(1*DX+boxA,3*DY+boxB,col0,'edgecolor','k');
    text(1*DX,3*DY,'\bf{Fuel}','HorizontalAlignment','center','VerticalAlignment','middle')
else
    fill(1*DX+boxA,3*DY+boxB,colOff,'edgecolor','none');
end


% GT box
if ~isnan(comp.GT)
    fill(2*DX+boxA,3*DY+boxB,col1,'edgecolor','k');
    text(2*DX,3*DY+B/3,'\bf{Gas turbine}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(2*DX,3*DY,num2str(comp.GT,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    text(2*DX,3*DY-B/3,num2str(loss.GT,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle','Color',colLoss)
else
    fill(2*DX+boxA,3*DY+boxB,colOff,'edgecolor','none');
end

% GB box
if ~isnan(comp.GB)
    fill(3*DX+boxA,3*DY+boxB,col1,'edgecolor','k');
    text(3*DX,3*DY+B/3,'\bf{Gearbox}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(3*DX,3*DY,num2str(comp.GB,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    text(3*DX,3*DY-B/3,num2str(loss.GB,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle','Color',colLoss)
else
    fill(3*DX+boxA,3*DY+boxB,colOff,'edgecolor','none');
end

% EM1 box
if ~isnan(comp.EM1)
    fill(3*DX+boxA,2*DY+boxB,col1,'edgecolor','k');
    if path.e1 < 0
        text(3*DX,2*DY+B/3,'\bf{EM1 (motor)}','HorizontalAlignment','center','VerticalAlignment','middle')
    else
        text(3*DX,2*DY+B/3,'\bf{EM1 (generator)}','HorizontalAlignment','center','VerticalAlignment','middle')
    end
    text(3*DX,2*DY,num2str(comp.EM1,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    text(3*DX,2*DY-B/3,num2str(loss.EM1,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle','Color',colLoss)
else
    fill(3*DX+boxA,2*DY+boxB,colOff,'edgecolor','none');
end

% PMAD box
if ~isnan(comp.PM)
    fill(2*DX+boxA,1.5*DY+boxB,col0,'edgecolor','k');
    text(2*DX,1.5*DY+B/3,'\bf{PMAD}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(2*DX,1.5*DY,num2str(comp.PM,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    text(2*DX,1.5*DY-B/3,num2str(loss.PM,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle','Color',colLoss)
else
    fill(2*DX+boxA,1.5*DY+boxB,colOff,'edgecolor','none');
end

% Battery box
if ~isnan(path.bat)
    fill(1*DX+boxA,1.5*DY+boxB,col0,'edgecolor','k');
    text(1*DX,1.5*DY,'\bf{Battery}','HorizontalAlignment','center','VerticalAlignment','middle')
else
    fill(1*DX+boxA,1.5*DY+boxB,colOff,'edgecolor','none');
end

% EM2 box
if ~isnan(comp.EM2)
    fill(3*DX+boxA,1*DY+boxB,col2,'edgecolor','k');
    if path.e2 < 0
        text(3*DX,1*DY+B/3,'\bf{EM2 (generator)}','HorizontalAlignment','center','VerticalAlignment','middle')
    else
        text(3*DX,1*DY+B/3,'\bf{EM2 (motor)}','HorizontalAlignment','center','VerticalAlignment','middle')
    end
    text(3*DX,1*DY,num2str(comp.EM2,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    text(3*DX,1*DY-B/3,num2str(loss.EM2,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle','Color',colLoss)
else
    fill(3*DX+boxA,1*DY+boxB,colOff,'edgecolor','none');
end

% Propulsor 1
if ~isnan(comp.P1)
    fill(4*DX+boxC,3*DY+boxD,col1,'edgecolor','k');
    text(4*DX,3*DY+B/3,'\bf{Propulsor 1}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(4*DX,3*DY,num2str(comp.P1,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    text(4*DX,3*DY-B/3,num2str(loss.P1,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle','Color',colLoss)
else
    fill(4*DX+boxC,3*DY+boxD,colOff,'edgecolor','none');
end

% Propulsor 2
if ~isnan(comp.P2)
    fill(4*DX+boxC,1*DY+boxD,col2,'edgecolor','k');
    text(4*DX,1*DY+B/3,'\bf{Propulsor 2}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(4*DX,1*DY,num2str(comp.P2,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    text(4*DX,1*DY-B/3,num2str(loss.P2,formSpec),'HorizontalAlignment','center','VerticalAlignment','middle','Color',colLoss)
else
    fill(4*DX+boxC,1*DY+boxD,colOff,'edgecolor','none');
end


%% Create power flow arrows

% Fuel power path
if isnan(path.f)
    quiver(1*DX+A/2,3*DY,DX-A,0,0,'Color',colOff,'ShowArrowHead','off')
else
    text(1.5*DX,3*DY+B/3,'\it{P_{\rm{f}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(1.5*DX,3*DY-B/3,num2str(abs(path.f),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.f < 0    
        quiver(2*DX-A/2,3*DY,-DX+A,0,0,'Color','k','MaxHeadSize',0.5)
    else
        quiver(1*DX+A/2,3*DY,DX-A,0,0,'Color','k','MaxHeadSize',0.5)
    end
end

% GT power path
if isnan(path.gt)
    quiver(2*DX+A/2,3*DY,DX-A,0,0,'Color',colOff,'ShowArrowHead','off')
else
    text(2.5*DX,3*DY+B/3,'\it{P_{\rm{gt}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(2.5*DX,3*DY-B/3,num2str(abs(path.gt),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.gt < 0    
        quiver(3*DX-A/2,3*DY,-DX+A,0,0,'Color','k','MaxHeadSize',0.5)
    else
        quiver(2*DX+A/2,3*DY,DX-A,0,0,'Color','k','MaxHeadSize',0.5)
    end
end

% Shaft 1 power path
if isnan(path.s1)
    quiver(3*DX+A/2,3*DY,DX-A/2-C/2,0,0,'Color',colOff,'ShowArrowHead','off')
else
    text(3.5*DX,3*DY+B/3,'\it{P_{\rm{s1}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(3.5*DX,3*DY-B/3,num2str(abs(path.s1),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.s1 < 0    
        quiver(4*DX-C/2,3*DY,-DX+A/2+C/2,0,0,'Color','k','MaxHeadSize',0.5)
    else
        quiver(3*DX+A/2,3*DY,DX-A/2-C/2,0,0,'Color','k','MaxHeadSize',0.5)
    end
end

% Prop 1 power path
if isnan(path.p1)
    quiver(4*DX+C/2,3*DY,DX-C,0,0,'Color',colOff,'ShowArrowHead','off')
else
    text(4.5*DX,3*DY+B/3,'\it{P_{\rm{p1}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(4.5*DX,3*DY-B/3,num2str(abs(path.p1),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.p1 < 0    
        quiver(5*DX-C/2,3*DY,-DX+C,0,0,'Color','k','MaxHeadSize',0.5)
    else
        quiver(4*DX+C/2,3*DY,DX-C,0,0,'Color','k','MaxHeadSize',0.5)
    end
end

% GB power path
if isnan(path.gb)
    quiver(3*DX,3*DY-B/2,0,-DY+B,0,'Color',colOff,'ShowArrowHead','off')
else
    text(3*DX-A/4,2.5*DY,'\it{P_{\rm{gb}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(3*DX+A/3,2.5*DY,num2str(abs(path.gb),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.gb < 0    
        quiver(3*DX,2*DY+B/2,0,DY-B,0,'Color','k','MaxHeadSize',0.5)
    else
        quiver(3*DX,3*DY-B/2,0,-DY+B,0,'Color','k','MaxHeadSize',0.5)
    end
end

% E1 power path
if isnan(path.e1)
    quiver(3*DX-A/2,2*DY,-DX+A/2,0,0,'Color',colOff,'ShowArrowHead','off')
    quiver(2*DX,2*DY,0,-0.5*DY+B/2,0,'Color',colOff,'ShowArrowHead','off')
else
    text(2.5*DX,2*DY+B/3,'\it{P_{\rm{e1}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(2.5*DX,2*DY-B/3,num2str(abs(path.e1),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.e1 < 0    
        quiver(2*DX,2*DY,DX-A/2,0,0,'Color','k','MaxHeadSize',0.5/1.5)
        quiver(2*DX,1.5*DY+B/2,0,0.5*DY-B/2,0,'Color','k','ShowArrowHead','off')
    else
        quiver(3*DX-A/2,2*DY,-DX+A/2,0,0,'Color','k','ShowArrowHead','off')
        quiver(2*DX,2*DY,0,-0.5*DY+B/2,0,'Color','k','MaxHeadSize',0.5*2)
    end
end

% Battery power path
if isnan(path.bat)
    quiver(1*DX+A/2,1.5*DY,DX-A,0,0,'Color',colOff,'ShowArrowHead','off')
else
    text(1.5*DX,1.5*DY+B/3,'\it{P_{\rm{bat}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(1.5*DX,1.5*DY-B/3,num2str(abs(path.bat),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.bat < 0    
        quiver(2*DX-A/2,1.5*DY,-DX+A,0,0,'Color','k','MaxHeadSize',0.5)
    else
        quiver(1*DX+A/2,1.5*DY,DX-A,0,0,'Color','k','MaxHeadSize',0.5)
    end
end

% E2 power path
if isnan(path.e2)
    quiver(2*DX,1*DY,DX-A/2,0,0,'Color',colOff,'ShowArrowHead','off')
    quiver(2*DX,1.5*DY-B/2,0,-0.5*DY+B/2,0,'Color',colOff,'ShowArrowHead','off')
else
    text(2.5*DX,1*DY+B/3,'\it{P_{\rm{e2}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(2.5*DX,1*DY-B/3,num2str(abs(path.e2),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.e2 < 0    
        quiver(3*DX-A/2,1*DY,-DX+A/2,0,0,'Color','k','ShowArrowHead','off')
        quiver(2*DX,1*DY,0,0.5*DY-B/2,0,'Color','k','MaxHeadSize',0.5*2)
    else
        quiver(2*DX,1*DY,DX-A/2,0,0,'Color','k','MaxHeadSize',0.5/1.5)
        quiver(2*DX,1.5*DY-B/2,0,-0.5*DY+B/2,0,'Color','k','ShowArrowHead','off')
    end
end

% Shaft 2 power path
if isnan(path.s2)
    quiver(3*DX+A/2,1*DY,DX-A/2-C/2,0,0,'Color',colOff,'ShowArrowHead','off')
else
    text(3.5*DX,1*DY+B/3,'\it{P_{\rm{s2}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(3.5*DX,1*DY-B/3,num2str(abs(path.s2),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.s2 < 0    
        quiver(4*DX-C/2,1*DY,-DX+A/2+C/2,0,0,'Color','k','MaxHeadSize',0.5)
    else
        quiver(3*DX+A/2,1*DY,DX-A/2-C/2,0,0,'Color','k','MaxHeadSize',0.5)
    end
end

% Prop 2 power path
if isnan(path.p2)
    quiver(4*DX+C/2,1*DY,DX-C,0,0,'Color',colOff,'ShowArrowHead','off')
else
    text(4.5*DX,1*DY+B/3,'\it{P_{\rm{p2}}}','HorizontalAlignment','center','VerticalAlignment','middle')
    text(4.5*DX,1*DY-B/3,num2str(abs(path.p2),formSpec),'HorizontalAlignment','center','VerticalAlignment','middle')
    if path.p2 < 0    
        quiver(5*DX-C/2,1*DY,-DX+C,0,0,'Color','k','MaxHeadSize',0.5)
    else
        quiver(4*DX+C/2,1*DY,DX-C,0,0,'Color','k','MaxHeadSize',0.5)
    end
end


%% General figure settings

% Add legend
string1 = {'\bf{Legend}\rm',...
           '- Blue boxes are primary powertrain components',...
           '- Red boxes are secondary powertrain components',...
           '- Black numbers in component boxes indicate sizing power',...
           '- Red numbers in component boxes indicate power losses',...
           '- Power paths are presented as absolute value; arrows indicate direction',...
           };
text(0.5*DX,0.5*DY,string1)

% Set fontsize
set(findall(fig,'-property','FontSize'),'FontSize',fontsize)

% Hide axes
ax.Visible = 'off';

% Add title
text(1,1.1,['\bf{' description '}'],'FontSize',fontsize+3,'HorizontalAlignment','center');





