%% MATLAB code for performing the data analysis of the phantom evaluation
% Authors: Jette Bloemberg (j.bloemberg@tudelft.nl), Bruce Hoppener
% Last update: 26/07/2023

% Running the analysis file S3_SlipPneumaticOvipositorNeedle.m requires the
% supplementary raw data set of the phantom evaluation. 


%% Initialize
close all
clear all
clc


%% Condition S2-I05 

data_dir = 'C:\path\your\data\folder\S4_Data\stroke experiments\2mm'; % Replace 'C:\path\your\data\folder\'with the folder that houses the S4_Data folder
data_files = dir(fullfile(data_dir, '*.xlsx'));

%variables
d_theoretical = 2; %mm
interval = 0.5; %seconds

%Calculate the time of one cycle
cycleTime = (interval*10)*7;

%Open a figure to plot the data in
figure

%Make totalSlip variable
totalSlip = zeros(1,1);

for i = 1:length(data_files)
    data = xlsread(fullfile(data_dir,data_files(i).name));
    time{i} = data(:,1);
    pressure{i} = data(:,3);
    distance{i} = data(:,4);
    
    %Smooth the data with a Savitzky-Golay filter, order 3, frame length 11.
    distance_smooth{i} = sgolayfilt(distance{i},3,11);
    
    %Find the peaks and the locations of the peaks
    [pks_d{i},locsp_d{i}] = findpeaks(distance_smooth{i},'MinPeakProminence',0.5);
    
    %Find the locations of the first and the last peak
    loc_peak1 = locsp_d{i}(1)-cycleTime;
    loc_peak31 = locsp_d{i}(29)+cycleTime;
    
    %add them to the new locations vector
    locsp{i}(1) = loc_peak1;
    locsp{i}(2:30) = locsp_d{i};
    locsp{i}(31) = loc_peak31;
    
    pks{i}(1) = distance_smooth{i}(loc_peak1);%Find the first peak
    pks{i}(2:30) = pks_d{i}; %add the found peaks to the new peaks vectors
    pks{i}(31) = distance_smooth{i}(loc_peak31);%Find the last peak
    
    %Find the valleys and the location of the valleys
    [vall{i},locsv{i}] = findpeaks(-distance_smooth{i},'MinPeakProminence',0.5);
    vall{i} = abs(vall{i}); % As this is negative, use the absolute value
    
    %Calculate the slip ratio of the peaks and the valleys
        for k = 2:length(pks{i})
            d_measuredp = pks{i}(k) - pks{i}(k-1); %Measured distance (in mm)
            SRp(i,k-1) = 1 - d_measuredp/d_theoretical;
        end
    
    %Plot all data points of the Slip Ratio
    p1_s = plot(1:length(SRp(i,:)),SRp(i,:),'color','#0072B2');
    set(gca,'fontname','times','fontsize',14)  % Set font to times
    hold on
    
end

%make a matrix for the mean
meanSR2 = zeros(1,30);

for j = 1:30
    meanSR2(:,j) = mean(SRp(:,j));
    S(:,j) = std(SRp(:,j));
end

%Plot the area inbetween the standard deviation
inBetween = [meanSR2+S, fliplr(meanSR2-S)];
x2 = [1:length(meanSR2), fliplr(1:length(meanSR2))];
fill(x2, inBetween,[0.204,0.204,0.204],'FaceAlpha',.2,'LineStyle','none');
hold on

%plot the mean slip ratio
p1_m = plot(1:length(meanSR2),meanSR2,'LineWidth',3,'color','#0072B2');

%add labels and a title
xlabel('Number of actuation cycles')
ylabel('Slip ratio')
title('S2-I05')
ylim([0 1.4]);
legend([p1_s p1_m],{'Single measurement','Mean'},'Location','southeast');

%Add all the slip ratio data to a new vector, for the statistical analysis
Slips2p05i05 = reshape(SRp.',1,[]);


%% Condition S4-I05
close all

data_dir = 'C:\path\your\data\folder\S4_Data\stroke experiments\4mm'; % Replace 'C:\path\your\data\folder\'with the folder that houses the S4_Data folder
data_files = dir(fullfile(data_dir, '*.xlsx'));

%variables
d_theoretical = 4; %mm
interval = 0.5; %seconds

%Calculate the time of one cycle
cycleTime = (interval*10)*7;

%Open a figure to plot the data in
figure

for i = 1:length(data_files)
    data = xlsread(fullfile(data_dir,data_files(i).name));
    time4{i} = data(:,1);
    pressure4{i} = data(:,3);
    distance4{i} = data(:,4);
    
    %Smooth the data with a Savitzky-Golay filter, order 3, frame length 11.
    distance_smooth{i} = sgolayfilt(distance4{i},3,11);
    
    %Find the peaks and the locations of the peaks
    [pks_d{i},locsp_d{i}] = findpeaks(distance_smooth{i},'MinPeakProminence',0.5);
    
    %Find the locations of the first and the last peak
    loc_peak1 = locsp_d{i}(1)-cycleTime;
    loc_peak31 = locsp_d{i}(29)+cycleTime;
    
    %add them to the new locations vector
    locsp{i}(1) = loc_peak1;
    locsp{i}(2:30) = locsp_d{i};
    locsp{i}(31) = loc_peak31;
    
    pks{i}(1) = distance_smooth{i}(loc_peak1);%Find the first peak
    pks{i}(2:30) = pks_d{i}; %add the found peaks to the new peaks vectors
    pks{i}(31) = distance_smooth{i}(loc_peak31);%Find the last peak
    
    %Find the valleys and the location of the valleys
    [vall{i},locsv{i}] = findpeaks(-distance_smooth{i},'MinPeakProminence',0.5);
    vall{i} = abs(vall{i}); % As this is negative, use the absolute value
    
    %Calculate the slip ratio of the peaks and the valleys
        for k = 2:length(pks{i})
            d_measuredp = pks{i}(k) - pks{i}(k-1); %Measured distance (in mm)
            SRp(i,k-1) = 1 - d_measuredp/d_theoretical;        
        end

    p2_s = plot(1:length(SRp(i,:)),SRp(i,:),'color','#D55E00');
    set(gca,'fontname','times','fontsize',14)  % Set font to times
    hold on
    
end

%make a matrix for the mean
meanSR4 = zeros(1,1);

for j = 1:30
    %Calculate the mean Slip Ratio of all the data points
    meanSR4(:,j) = mean(SRp(:,j));
    S(:,j) = std(SRp(:,j));
end

%Plot the area inbetween the standard deviation
inBetween = [meanSR4+S, fliplr(meanSR4-S)];
x2 = [1:length(meanSR4), fliplr(1:length(meanSR4))];
fill(x2, inBetween, [0.204,0.204,0.204],'FaceAlpha',.2,'LineStyle','none');
hold on

%plot the mean slip ratio
p2_m = plot(1:length(meanSR4),meanSR4,'LineWidth',3,'color','#D55E00');

%add labels and a title
xlabel('Number of actuation cycles')
ylabel('Slip ratio')
title('S4-I05')
ylim([0 1.4]);
legend([p2_s p2_m],{'Single measurement','Mean'},'Location','southeast');

%Add all the slip ratio data to a new vector, for the statistical analysis
Slips4p05i05 = reshape(SRp.',1,[]);


%% Condition S4-I03

data_dir = 'C:\path\your\data\folder\S4_Data\speed experiments\i03'; % Replace 'C:\path\your\data\folder\'with the folder that houses the S4_Data folder
data_files = dir(fullfile(data_dir, '*.xlsx'));

%variables
d_theoretical = 4; %mm
interval = 0.3; %seconds

%Calculate the time of one cycle
cycleTime = (interval*10)*7;

%Open a figure to plot the data in
figure

for i = 1:length(data_files)
    data = xlsread(fullfile(data_dir,data_files(i).name));
    time03{i} = data(:,1);
    pressure{i} = data(:,3);
    distance03{i} = data(:,4);
    
        %Smooth the data with a Savitzky-Golay filter, order 3, frame length 11.
    distance_smooth{i} = sgolayfilt(distance03{i},3,11);
    
    %Find the peaks and the locations of the peaks
    [pks_d{i},locsp_d{i}] = findpeaks(distance_smooth{i},'MinPeakProminence',0.5,'NPeaks',29);
    
    %Find the locations of the first and the last peak
    loc_peak1 = locsp_d{i}(1)-cycleTime;
    loc_peak31 = locsp_d{i}(29)+cycleTime;
    
    %add them to the new locations vector
    locsp{i}(1) = loc_peak1;
    locsp{i}(2:30) = locsp_d{i};
    locsp{i}(31) = loc_peak31;
    
    pks{i}(1) = distance_smooth{i}(loc_peak1);%Find the first peak
    pks{i}(2:30) = pks_d{i}; %add the found peaks to the new peaks vectors
    pks{i}(31) = distance_smooth{i}(loc_peak31);%Find the last peak
    
    %Find the valleys and the location of the valleys
    [vall{i},locsv{i}] = findpeaks(-distance_smooth{i},'MinPeakProminence',0.5);
    vall{i} = abs(vall{i}); % As this is negative, use the absolute value
    
    %Calculate the slip ratio of the peaks and the valleys
       for k = 2:length(pks{i})
           d_measuredp = pks{i}(k) - pks{i}(k-1); %Measured distance (in mm)
           SRp(i,k-1) = 1 - d_measuredp/d_theoretical;
       end    
        
    %Plot all data points of the Slip Ratio
    p3_s = plot(1:length(SRp(i,:)),SRp(i,:),'color','#CC79A7');
    set(gca,'fontname','times','fontsize',14)  % Set font to times
    hold on
    
end

%make a matrix for the mean
meanSR03 = zeros(1,1);

for j = 1:30
    %Calculate the mean Slip Ratio of all the data points
    meanSR03(:,j) = mean(SRp(:,j));
    S(:,j) = std(SRp(:,j));
end

%Plot the area inbetween the standard deviation
inBetween = [meanSR03+S, fliplr(meanSR03-S)];
x2 = [1:length(meanSR03), fliplr(1:length(meanSR03))];
fill(x2, inBetween, [0.204 0.204 0.204],'FaceAlpha',.2,'LineStyle','none');
hold on

%plot the mean slip ratio
p3_m = plot(1:length(meanSR03),meanSR03,'LineWidth',3,'color','#CC79A7');

%add labels and a title
xlabel('Number of actuation cycles')
ylabel('Slip ratio')
title('S4-I03')
ylim([0 1.4]);
legend([p3_s p3_m],{'Single measurement','Mean'},'Location','southeast');

%Add all the slip ratio data to a new vector, for the statistical analysis
Slips4p05i03 = reshape(SRp.',1,[]);


%% Condition S4-I01

data_dir = 'C:\path\your\data\folder\S4_Data\speed experiments\i01'; % Replace 'C:\path\your\data\folder\'with the folder that houses the S4_Data folder
data_files = dir(fullfile(data_dir, '*.xlsx'));

%variables
d_theoretical = 4; %mm
interval = 0.1; %seconds

%Calculate the time of one cycle
cycleTime = (interval*10)*7;

%Open a figure to plot the data in
figure

for i = 1:length(data_files)
    data = xlsread(fullfile(data_dir,data_files(i).name));
    time01{i} = data(:,1);
    pressure{i} = data(:,3);
    distance01{i} = data(:,4);
    
    %Smooth the data with a Savitzky-Golay filter, order 4, frame length 11.
    distance_smooth{i} = sgolayfilt(distance01{i},4,11);
    
    %Find the peaks and the locations of the peaks
    [pks_d{i},locsp_d{i}] = findpeaks(distance_smooth{i},'MinPeakProminence',0.5);
    
    %Find the locations of the first and the last peak
    loc_peak1 = locsp_d{i}(1)-cycleTime;
    loc_peak31 = locsp_d{i}(29)+cycleTime;
    
    %add them to the new locations vector
    locsp{i}(1) = loc_peak1;
    locsp{i}(2:30) = locsp_d{i};
    locsp{i}(31) = loc_peak31;
    
    pks{i}(1) = distance_smooth{i}(loc_peak1);%Find the first peak
    pks{i}(2:30) = pks_d{i}; %add the found peaks to the new peaks vectors
    pks{i}(31) = distance_smooth{i}(loc_peak31);%Find the last peak
    
    %Find the valleys and the location of the valleys
    [vall{i},locsv{i}] = findpeaks(-distance_smooth{i},'MinPeakProminence',0.5);
    vall{i} = abs(vall{i}); % As this is negative, use the absolute value

    %Calculate the slip ratio of the peaks and the valleys
        for k = 2:length(pks{i})
            d_measuredp = pks{i}(k) - pks{i}(k-1); %Measured distance (in mm)
            SRp(i,k-1) = 1 - d_measuredp/d_theoretical;
        end
        
    %Plot all data points of the Slip Ratio
    p4_s = plot(1:length(SRp(i,:)),SRp(i,:),'color','#009E73');
    set(gca,'fontname','times','fontsize',14)  % Set font to times
    hold on
    
end

%make a matrix for the mean
meanSR01 = zeros(1,1);

for j = 1:30
    %Calculate the mean Slip Ratio of all the data points
    meanSR01(:,j) = mean(SRp(:,j));
    S(:,j) = std(SRp(:,j));
end

%Plot the area inbetween the standard deviation
inBetween = [meanSR01+S, fliplr(meanSR01-S)];
x2 = [1:length(meanSR01), fliplr(1:length(meanSR01))];
fill(x2, inBetween, [0 0.158 0.115],'FaceAlpha',.2,'LineStyle','none');
hold on

%plot the mean slip ratio
p4_m = plot(1:length(meanSR01),meanSR01,'LineWidth',3,'color','#009E73');

%add labels and a title
xlabel('Number of actuation cycles')
ylabel('Slip ratio')
title('S4-I01')
ylim([0 1.4]);
legend([p4_s p4_m],{'Single measurement','Mean'},'Location','southeast');

%Add all the slip ratio data to a new vector, for the statistical analysis
Slips4p05i01 = reshape(SRp.',1,[]);


%% Plot all means in one graph
figure
plot(1:length(meanSR2),meanSR2,'LineWidth',1,'color','#0072B2')
hold on
plot(1:length(meanSR4),meanSR4,'LineWidth',1,'color','#D55E00')
hold on
plot(1:length(meanSR03),meanSR03,'LineWidth',1,'color','#CC79A7')
hold on
plot(1:length(meanSR01),meanSR01,'LineWidth',1,'color','#009E73')
xlabel('Number of actuation cycles')
ylabel('Slip ratio')
ylim([0 1.4]);
legend('Mean S2-I05','Mean S4-I05','Mean S4-I03','Mean S4-I01','Location','southeast')
set(gca,'fontname','times','fontsize',14)  % Set font to times


%% Get the pressure differences for each cycle to compare them (S2)

data_dir = ('Data\xlsx\stroke experiments\2mm');
data_files = dir(fullfile(data_dir, '*.xlsx'));

for i = 1:length(data_files)
    data = xlsread(fullfile(data_dir,data_files(i).name));
    pressure2{i} = data(:,3);
        
    %Smooth the data with a Savitzky-Golay filter, order 4, frame length 11.
    pressure_smooth10{i} = sgolayfilt(pressure2{i},1,9);
    
    %Find peaks and valleys
    [pks_press{i},locs_ppress{i}] = findpeaks(pressure_smooth10{i},'MinPeakProminence',0.05); %Find peaks   
    [vall_press{i},locs_vpress{i}] = findpeaks(-pressure_smooth10{i},'MinPeakProminence',0.05); %Find valleys
    
    %calculate the difference in pressure between the valley and the peak
    
    for j = 1:length(vall_press{i})
    dP2(i,j) = pks_press{i}(j) - abs(vall_press{i}(j));
    end
    
end

%% Get the pressure differences for each cycle to compare them (S4)

data_dir = ('Data\xlsx\stroke experiments\4mm');
data_files = dir(fullfile(data_dir, '*.xlsx'));

for i = 1:length(data_files)
    data = xlsread(fullfile(data_dir,data_files(i).name));
    pressure2{i} = data(:,3);
        
    %Smooth the data with a Savitzky-Golay filter, order 4, frame length 11.
    pressure_smooth10{i} = sgolayfilt(pressure2{i},1,9);
    
    %Find peaks and valleys
    [pks_press{i},locs_ppress{i}] = findpeaks(pressure_smooth10{i},'MinPeakProminence',0.05); %Find peaks   
    [vall_press{i},locs_vpress{i}] = findpeaks(-pressure_smooth10{i},'MinPeakProminence',0.05); %Find valleys
    
    %calculate the difference in pressure between the valley and the peak
    
    for j = 1:length(vall_press{i})
    dP4(i,j) = pks_press{i}(j) - abs(vall_press{i}(j));
    end
     
end


%% Calculate the mean and standard deviations for the conditions

mean_s2p05i05 = mean(Slips2p05i05)
std_s2p05i05 = std(Slips2p05i05)

mean_s4p05i05 = mean(Slips4p05i05)
std_s4p05i05 = std(Slips4p05i05)

mean_s4p05i03 = mean(Slips4p05i03)
std_s4p05i03 = std(Slips4p05i03)

mean_s4p05i01 = mean(Slips4p05i01)
std_s4p05i01 = std(Slips4p05i01)
    

%% Make Boxplot for the slip ratio

figure
boxplot([Slips2p05i05',Slips4p05i05',Slips4p05i03',Slips4p05i01'],'Labels',{'S2-I05','S4-I05','S4-I03','S4-I01'},'Colors',[0,0.445,0.695; 0.832,0.367,0; 0.797,0.473,0.652; 0,0.617,0.449;],'PlotStyle','compact')
%title('Slip ratio for each of the conditions')
set(gca,'fontname','times','fontsize',14)  % Set font to times
xlabel('Condition')
ylabel('Slip ratio')
ylim([0 1.4]);


%% Unfiltered speed comparison

% Set time from ms to s
timeS4I05_sec = time4{3} / 1000; 
timeS4I03_sec = time03{8} / 1000; 
timeS4I01_sec = time01{1} /1000; 
timeS2I05_sec = time{4} / 1000; 

% Call individual distance measurements
distanceS4I05 = distance4;
distanceS4I03 = distance03;
distanceS4I01 = distance01;
distanceS2I05 = distance;

% Shift distance measurement to all start at (0,0)
first_distanceS4I05 = distanceS4I05{3}(1);
shifted_distanceS4I05 = distanceS4I05{3} - first_distanceS4I05;

first_distanceS4I03 = distanceS4I03{8}(1);
shifted_distanceS4I03 = distanceS4I03{8} - first_distanceS4I03;

first_distanceS4I01 = distanceS4I01{1}(1);
shifted_distanceS4I01 = distanceS4I01{1} - first_distanceS4I01;

first_distanceS2I05 = distanceS2I05{4}(1);
shifted_distanceS2I05 = distanceS2I05{4} - first_distanceS2I05;

new_shifted_distanceS4I05 = shifted_distanceS4I05(156:end);
new_timeS4I05_sec = timeS4I05_sec(156:end);

new_shifted_distanceS4I03 = shifted_distanceS4I03(88:718);
new_timeS4I03_sec = timeS4I03_sec(88:718);

new_shifted_distanceS4I01 = shifted_distanceS4I01(70:276);
new_timeS4I01_sec = timeS4I01_sec(70:276);

new_shifted_distanceS2I05 = shifted_distanceS2I05(63:1112);
new_timeS2I05_sec = timeS2I05_sec(63:1112);

new_shifted_timeS4I05_sec = new_timeS4I05_sec - new_timeS4I05_sec(1);
new_shifted_timeS4I03_sec = new_timeS4I03_sec - new_timeS4I03_sec(1);
new_shifted_timeS4I01_sec = new_timeS4I01_sec - new_timeS4I01_sec(1);
new_shifted_timeS2I05_sec = new_timeS2I05_sec - new_timeS2I05_sec(1);

% Extend times to t_end = 120 s
extended_timeS4I05 = [new_shifted_timeS4I05_sec; 120];
extended_distanceS4I05 = [new_shifted_distanceS4I05; new_shifted_distanceS4I05(end)];

extended_timeS4I03 = [new_shifted_timeS4I03_sec; 120];
extended_distanceS4I03 = [new_shifted_distanceS4I03; new_shifted_distanceS4I03(end)];

extended_timeS4I01 = [new_shifted_timeS4I01_sec; 120];
extended_distanceS4I01 = [new_shifted_distanceS4I01; new_shifted_distanceS4I01(end)];

extended_timeS2I05 = [new_shifted_timeS2I05_sec; 120];
extended_distanceS2I05 = [new_shifted_distanceS2I05; new_shifted_distanceS2I05(end)];

% Plot individual measurements of distance vs. time
figure
plot(extended_timeS2I05,extended_distanceS2I05,'LineWidth',1,'color','#0072B2')
hold on
plot(extended_timeS4I05,extended_distanceS4I05,'LineWidth',1,'color','#D55E00')
hold on
plot(extended_timeS4I03,extended_distanceS4I03,'LineWidth',1,'color','#CC79A7')
hold on
plot(extended_timeS4I01,extended_distanceS4I01, 'LineWidth',1,'color','#009E73')

xlabel('Time (s)')
ylabel('Measured traveled distance (mm)')
set(gca,'fontname','times','fontsize',14)  % Set font to times
ylim([0 15]);
legend('S2-I05','S4-I05','S4-I03','S4-I01','Location','southeast')


%% Do a normality check for all evaluations
close all

% With histograms
figure
histogram(Slips2p05i05)
ylabel('Frequency')
xlabel('Slip ratio per cycle')
title('s2-p05-i05')
figure
histogram(Slips4p05i05)
ylabel('Frequency')
xlabel('Slip ratio per cycle')
title('s4-p05-i05')
figure
histogram(Slips4p05i03)
ylabel('Frequency')
xlabel('Slip ratio per cycle')
title('s4-p05-i03')
figure
histogram(Slips4p05i01)
ylabel('Frequency')
xlabel('Slip ratio per cycle')
title('s4-p05-i01')

%with the normalitytest script from  ner & Deveci Kocakoc (2017)
normalitytest(Slips2p05i05)
%etc, etc. Manually check for all the data.


%% Statistical Analysis S2-I05 vs. S4-I05

mean_s2 = mean(Slips2p05i05)
std_s2 = std(Slips2p05i05)
mean_s4 = mean(Slips4p05i05)
std_s2 = std(Slips4p05i05)

%Do a Wilcoxon signed rank test, since there are two repeated groups of the
%same size that are not normally distributed

[p,h,stats] = signrank(Slips2p05i05,Slips4p05i05)


if h == 0
    sprintf('The null hypothesis is not rejected, the p-value is %s',num2str(p))
else
    sprintf('The null hypothesis is rejected, the p-value is %s',num2str(p))
end


%% Statistical Analysis Experiment S4-I05 vs. S4-I03 vs. S4-I01

%Perform a Friedman test for all three conditions
experiments = [Slips4p05i05' Slips4p05i03' Slips4p05i01'];

friedman(experiments)


%The friedman test rejects the null-hypotheis, and thus further examination
%is needed:

%Do a Wilcoxon signed rank test for the three combinations of conditions:
[p1,h1,stats1] = signrank(Slips4p05i05,Slips4p05i03)

[p2,h2,stats2] = signrank(Slips4p05i05,Slips4p05i01)

[p3,h3,stats3] = signrank(Slips4p05i03,Slips4p05i01)


%% Statistical analysis if the data would have been normally distributed

%the DOF
DOFbetween = 3;
DOFwithin = 1196;

%Calculate the means for the groups
x2mm = mean(Slips2p05i05);
x4mm = mean(Slips4p05i05);
xi03 = mean(Slips4p05i03);
xi01 = mean(Slips4p05i01);

%Put all slip ratios in one vector
allsr = [Slips2p05i05 Slips4p05i05 Slips4p05i03 Slips4p05i01];

%Calculate the grand mean
xg = mean(allsr);

%calculate the sum of squares
SStotal = sum((allsr-xg).^2);
SSwithin = sum((Slips2p05i05-x2mm).^2 + (Slips4p05i05-x4mm).^2 + (Slips4p05i03-xi03).^2 + (Slips4p05i01-xi01).^2);
SSbetween = SStotal - SSwithin;

%Calculate variance (mean square)
MSbetween = SSbetween/DOFbetween;
MSwithin = SSwithin/DOFwithin;

%Calculate the F-value
F = MSbetween/MSwithin



% Tukey's HSD test to find which mean(s) is/are different

q2mm4mm = (x2mm-x4mm)/sqrt(MSwithin/300)
q2mmi03 = (x2mm-xi03)/sqrt(MSwithin/300)
q2mmi01 = (x2mm-xi01)/sqrt(MSwithin/300)

q4mmi03 = (x4mm-xi03)/sqrt(MSwithin/300)
q4mmi01 = (x4mm-xi01)/sqrt(MSwithin/300)

qi03i01 = (xi03-xi01)/sqrt(MSwithin/300)

%% Do a two sample t-test to find if the mean pressures are statistically different

close all
%Check if the data is normally distributed with a histogram
figure
histogram(dP2)
xlabel('Pressure difference per cycle')
ylabel('Frequency')
title('s2-p05-i05')
figure
histogram(dP4)
xlabel('Pressure difference per cycle')
ylabel('Frequency')
title('s4-p05-i05')
figure
histogram(dP6)
xlabel('Pressure difference per cycle')
ylabel('Frequency')
title('s6-p05-i05')
figure
histogram(dP8)
xlabel('Pressure difference per cycle')
ylabel('Frequency')
title('s8-p05-i05')
figure
histogram(dP10)
xlabel('Pressure difference per cycle')
ylabel('Frequency')
title('s10-p05-i05')

%do the t-test. If h=1, the test rejects the null hypothesis at the 5%
%significance level, and 0 otherwise.

[h24,p24] = ttest2(reshape(dP2, 1,[]),reshape(dP4, 1,[]))
[h26,p26] = ttest2(reshape(dP2, 1,[]),reshape(dP6, 1,[]))
[h28,p28] = ttest2(reshape(dP2, 1,[]),reshape(dP8, 1,[]))
[h210,p210] = ttest2(reshape(dP2, 1,[]),reshape(dP10, 1,[]))
[h46,p46] = ttest2(reshape(dP4, 1,[]),reshape(dP6, 1,[]))
[h48,p48] = ttest2(reshape(dP4, 1,[]),reshape(dP8, 1,[]))
[h410,p410] = ttest2(reshape(dP4, 1,[]),reshape(dP10, 1,[]))
[h68,p68] = ttest2(reshape(dP6, 1,[]),reshape(dP8, 1,[]))
[h610,p610] = ttest2(reshape(dP6, 1,[]),reshape(dP10, 1,[]))
[h810,p810] = ttest2(reshape(dP8, 1,[]),reshape(dP10, 1,[]))
