function [p_T1, Sigma_T1value, p_T2, Sigma_T2value] = ConstrainedOptimization(constraints, parameters)
%This function completes the constrained optimization as described in the
%methods section of the paper
%% Parameters of Constrained Optimization
% Step 1 - Define starting point of the optimization. Store this in struct
% p_start
p_start(1,1)    = 0.63;     % friction coefficient of material
p_start(1,2)    = pi/4;     % initial value for phi_1
p_start(1,3)    = 3*pi/4;   % initial value for phi_2
p_start(1,4)    = 0.75;     % initial value for r_frac

% Step 2 - Define the constraints that the clutch configuraion needs to
% meet
% 2.a - Minimum difference between phi_2 and phi_1
phi_mindiff     = constraints(1); 
% 2.b - Amplification factor that must be achieved
xi_desired      = constraints(2);
% 2.c - Maximum value of r_frac
r_fracmax       = constraints(3);
% 2.d - Value of friction coefficient ofmaterial utilized
mu_material     = constraints(4);

% Step 3 - Extract the radius of the proposed design
r               = parameters(1);
%% Parameters for plot of objective function over parameter space
% Size of grid to plot objective function over
n_plot          = parameters(2);

%% Constrained Optimization
% Define a symbolic vector x_symb representing the vector of random
% variables and a symbolic vector p_symb representing a vector from
% parameter space P. 
%   x = [mu_f, a_0, a_N, a_f]
%   p = [mu_f, phi_1, phi_2, r_frac]
x_sym   = sym('x', [1 4]);
p_sym   = sym('p', [1 4]);

% x_sym and p_sym are real and positive
assume(x_sym, 'real')
assumeAlso(x_sym, 'positive')

assume(p_sym, 'real')
assumeAlso(p_sym, 'positive')

% Define a symbolic expression for the amplification factor in terms of
% vector x
xi      = AmplificationFactorX(x_sym);

% Calculate the jacobian and hessian of this expression
J_x     = jacobian(xi, x_sym);
H_x     = hessian(xi, x_sym);

% Express the jacobian and hessian in terms of values of p, to allow
% constraints such as phi_mindiff and r_fracmax to be easily implemented
J_x     = subs(J_x, x_sym, Config2Arms(p_sym, r));
H_x     = subs(H_x, x_sym, Config2Arms(p_sym, r));

% Calculate the covariance matrix using Equation 10. Since the expected
% value of x is equal to the nominal design values, this is done by:
Sigma_x     = SigmaX(Config2Arms(p_sym, r));

% Calculate the first and second order Taylor approximations of function f
Sigma_T1    = J_x * Sigma_x * transpose(J_x);
Sigma_T2    = Sigma_T1 + 0.5*trace(Sigma_x * H_x * Sigma_x * H_x);

% Constraint to ensure that the amplfication factor is indeed equal to the
% desired value. This is done by require the expected value of T_1 and T_2
% to be equal to xi_desired
mu_x        = Config2Arms(p_sym, r);
mu_T1       = AmplificationFactorX(mu_x);
mu_T2       = mu_T1 + 0.5*trace(H_x*Sigma_x);

%% Turn Sigma_T1 and Sigma_T2 into matlab functions
% Create a MATLAB function that accepts a vector from parameter space p and
% return the below values
Sigma_T1func    = matlabFunction(Sigma_T1, 'Vars', {p_sym});
Sigma_T2func    = matlabFunction(Sigma_T2, 'Vars', {p_sym});

mu_T1func       = matlabFunction(mu_T1, 'Vars', {p_sym});
mu_T2func       = matlabFunction(mu_T2, 'Vars', {p_sym});

%% Set constraint matrices for optimization
% Constraints to make sure phi_mindiff, min/max values of phi_1 and phi_2
% and r_fracmax are respected.
A = [   [0, 1,-1, 0]
        [0, 0, 1, 0]
        [0,-1, 0, 0]
        [0, 0, 0,-1]
        [0, 0, 0, 1]];
b = [-phi_mindiff; pi; 0; 0; r_fracmax];

Aeq = [1, 0, 0, 0];
beq = [mu_material];

lb = [];
ub = [];

%% Use interior-point algorithm
% Solve the optimization problem using the interior point algorighm with
% the below tolerances, as specified
nonlcon                         = @(p) AmplificationConstraint_T2(xi_desired, mu_T1func, p);
options                         = optimoptions('fmincon'); 
options.Display                 = 'none';
options.ConstraintTolerance     = 1e-10;
options.StepTolerance           = 1e-10;
options.OptimalityTolerance     = 1e-10;
options.Algorithm               = 'interior-point';
[p_T1, Sigma_T1value]               = fmincon(Sigma_T1func, p_start, ...
    A, b, Aeq, beq, lb, ub, nonlcon, options);

% Repeat the procedure with the second order approximation
% Set the start point to equal the solution of the first order
% approximation
nonlcon                         = @(p) AmplificationConstraint_T2(xi_desired, mu_T2func, p);
[p_T2, Sigma_T2value]           = fmincon(Sigma_T2func, p_start, ...
    A, b, Aeq, beq, lb, ub, nonlcon, options);

%% Plot objective function over parameter space P
% Define a symbolic vector p_symb representing a vector from parameter
% space P.
%   p = [mu_f, phi_1, phi_2, r_frac]
p_sym   = sym('p', [1 4]);

% p_sym is real and positive
assume(p_sym, 'real')
assumeAlso(p_sym, 'positive')

% Next, use the constraint that the amplification factor is fixed at a
% given value to calculate the value of r_frac given this value, phi_1 and
% phi_2
AmplificationConstraint     = AmplificationFactorP(p_sym) == xi_desired;
r_frac                      = isolate(AmplificationConstraint, 'p4');
r_fracfunc                  = matlabFunction(rhs(r_frac));

% Make an array of values for phi_1 and phi_2
phi_1           = linspace(0, pi, n_plot);
phi_2           = linspace(0, pi, n_plot);
[Phi_1, Phi_2]  = ndgrid(phi_1, phi_2);

% Set all values where phi_1 is larger than phi_2 to NaN for both grids
mask            = Phi_1 + phi_mindiff >= Phi_2;
Phi_1(mask)     = NaN;
Phi_2(mask)     = NaN;

% R_frac is a function of mu, phi_1 and phi_2
R_frac          = r_fracfunc(mu_material.*ones(n_plot,n_plot), Phi_1, Phi_2);

% Set all non-allowed values of r_frac to zero
mask            = R_frac > r_fracmax | R_frac < 0;
R_frac(mask)    = NaN;

% Calculate the objective function value using these arrays
J   = Sigma_T1func(reshape([mu_material.*ones(n_plot,n_plot), Phi_1, Phi_2, R_frac], [n_plot^2, 4]))';
J   = reshape(J, [n_plot, n_plot]);

% Calculate the values values in the parameter space meeting r_fracmax and
% xi_desired. First make a symbolic expression, and then create a matlab
% function that is solved using fsolve to solve the implicit function that
% finds a combination of phi_1 and phi_2 such that r_frac == r_fracmax and
% xi == xi_desired
phi_2_constraint        = subs(AmplificationConstraint, 'p1', mu_material);
phi_2_constraint        = subs(phi_2_constraint, 'p4', r_fracmax);
phi_2_constraint        = rhs(phi_2_constraint) - lhs(phi_2_constraint);
phi_2_constraintfunc    = matlabFunction(phi_2_constraint);

phi_1_rfrac     = linspace(0, p_T2(1,2), n_plot);
for i = 1:n_plot
    constraint      = @(x) phi_2_constraintfunc(phi_1_rfrac(i), x);
    phi_2_rfrac(i)  = fzero(constraint, pi/2);
end

% Plot these values in a figure
levels = [0.3, 0.25, 0.2, 0.19, 0.18, 0.1785];
figure(1)
hold('on')
plot(phi_1_rfrac, phi_2_rfrac, '--k', 'DisplayName', '$r_{\textnormal{frac}}$ constraint')
plot(phi_1_rfrac, phi_1_rfrac + phi_mindiff, '-.k', 'DisplayName', '$\phi_{\textnormal{diff}}$ constraint')
contour(Phi_1, Phi_2, J, levels, 'ShowText', 'on', 'DisplayName', 'Objective Function')
plot(p_T2(1,2), p_T2(1,3), 'r.', 'MarkerSize', 15, 'DisplayName', '$\vec{p_{T_2}}$')
hold('off')
xlabel('$\phi_1$', 'Interpreter', 'Latex', 'FontSize', 15)
ylabel('$\phi_2$', 'Interpreter', 'Latex', 'FontSize', 15)
legend('Location', 'southeast', 'Interpreter', 'Latex')
xlim([0 1.4])
ylim([0.6 3.0])
colorbar
end

