clc
clear
close all
%% Script Information
% Given the default values, this script performs the calculations mentioned
% in the design example. In order, this script:
%   1. Calculates an optimal design geometry given the dimension of the
%      clutch. It also plots the first-order approximation of the objective
%      function value over the parameter space P.
%   2. Performs a Monte Carlo simulation for the geometry returned by the
%      constrained optimization
% The MATLAB functions performing each of these calculations are stored in
% the folder 'Calculations'. Other supporting MATLAB functions are stored
% in the folder 'Functions'.
%
% This script contains the parameters and options for each of these
% calculations. These are stored in section 0
%% Copyright
% MIT License
% 
% Copyright (c) 2021 Alvaro Detailleur
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in all
% copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.
%% Add folders to path
% Add folders containing necessary MATLAB functions to path (see above)
addpath([pwd '\Calculations'])
addpath([pwd '\Functions'])
%% 0 - Clutch parameters and dimensions
% These values specify the nominal design values of the clutch
% configuration being examined. 
T    = 35;               % desired holding torque, in Nm
r    = 0.06;             % radius of drum pad, in m
d    = 0.025;            % depth of one clutch pad, in m
                         % (= 0.5 * 0.05 for design example)
tau  = 23e3;             % shear pressure of dielectric material, in Pa
mu   = 0.63;             % friction coefficient of dielectric material, in Pa
zeta = 0.9;              % normalized area of clutch pad covered, dimensionless

% Constraints to take into account during the constrained optimization
phi_mindiff = 2*pi/10;   % Minimum difference between phi_2 and phi_1 for
                         % optimal clutch configuration
xi_desired  = T / ...    % Desired amplification factor to meet the above 
  (2*pi*zeta*tau*r^2*d); % constraint
r_fracmax   = 0.925;     % Maximum value of r_frac, to ensure the clutch 
                         % can be manufactured

% Parameters of calculations
% Constrained optimization - size of grid to plot objective function over
n_plot          = 100;

% Monte Carlo Simulation - number of points in simulation and number of
% bins to be taken into account in histogram plot, respectively
n_MC            = 1e6;
n_bins          = 150;
%% 1 - ConstrainedOptimization
% Bundles these constraints in a vector. Since the friction coefficient of
% the material is also specified, bundle this in the array as well
constraints.optimization    = [phi_mindiff, xi_desired, r_fracmax, mu];

% Pass on the radius of the drum, to be able to calculate the arm length
% from a vector p in the parameter space P and the grid size of the plot
parameters.optimization     = [r, n_plot];

% Call the function that performs the constrained optimization
disp('Constrained Optimization results')
[p_T1, Sigma_T1, p_T2, Sigma_T2] = ...
    ConstrainedOptimization(constraints.optimization, parameters.optimization)
%% 2 - Monte Carlo simulation
% Pass on the radius of the drum, to be able to calculate the arm length
% from a vector p in the parameter space P, the number of points in the
% simulation, and the number of bins in the histogram
parameters.montecarlo       = [r, n_MC, n_bins];

disp([newline 'Monte Carlo results'])
Sigma_Yest  = MonteCarlo(p_T2, parameters.montecarlo)