function [variance] = MonteCarlo(p, parameters)
%This function performs a monte carlo simulation on the geometry p
%% Unpack parameters
% The following properties are stored in this array:
%  1. the radius of the geometry
%  2. the number of points in the simulation
%  3. the number of bins shown in the histogram
r       = parameters(1);
n_MC    = parameters(2);
n_bins  = parameters(3);

%% Monte Carlo Simulation
% First, for the given geometry calculate the variances for this
% configuration
Sigma_xMC     = SigmaX(Config2Arms(p, r));
x_T2          = Config2Arms(p, r);
x_T2MC        = x_T2.*ones(n_MC, 4);

% Create an array of 'noise' to be added to the nominal configuration, with
% the variance specified in Sigma_xMC
noise          = transpose(diag(Sigma_xMC.^0.5)) .* randn(n_MC, 4);

% Calculate the values of the amplification factor for each of these
% values
xi_MC          = AmplificationFactorX(x_T2MC + noise);

% Calculate the average value and variance of these values
average         = mean(xi_MC);
variance        = var(xi_MC);
stddev          = sqrt(variance);

% Calculate the first order approximation with these values, using a +/-
% 3-sigma interval
x_MCarray       = linspace(average - 3*stddev, average + 3*stddev, 1e3);
y_MCarray       = 1./(stddev .* sqrt(2.*pi)) .* exp(-0.5.* ...
                    ((x_MCarray - average) ./ stddev).^2);

% Plot these values in a histogram. Also plot the normal distribution
% based on the sample average and variance
figure(2)
hold('on')
histogram(xi_MC, n_bins, 'Normalization', 'pdf', 'DisplayName', ...
    join(['Monte Carlo (MC, N = ' num2str(n_MC, '%2.1e') ')']))
plot(x_MCarray, y_MCarray, 'r-', 'DisplayName', ...
    'Maximum Likelihood Gaussian Approximation of MC')
xlabel('$\xi$', 'Interpreter', 'Latex', 'FontSize', 15)
ylabel('probability density', 'Interpreter', 'Latex', 'FontSize', 15)
legend('Location', 'northeast')
end

