clear all
close all
clc

% This scripts tests performs a sensitivity analysis, where different
% values for the threshold (used in script "find_crests_troughs.m") are
% attempted.

% V1 - 02 May 2025

% Script written by: Sjoukje I. de Lange
% HKV lijn in water, Lelystad, the Netherlands

% This script belongs to the following publication:
% "Bimodality in subaqueous dune height suggests flickering behavior at high flow", published in Nature Communications (2025).
% Sjoukje I. de Lange(1,2,*), Roeland C. van de Vijsel(1), Paul J.J. F. Torfs(3), Nick P. Wallerstein(1), and A.J.F (Ton) Hoitink(1)
% 1 Wageningen University, Department of Environmental Sciences, Hydrology and Environmental Hydraulics Group, Wageningen, the Netherlands
% 2 HKV lijn in water, Lelystad, the Netherlands
% 3 independent researcher
% * Corresponding author: Sjoukje de Lange (delangesjoukje@gmail.com)

%%
% addpath output
load('..\Data\Data_experiments\Seatek\matlab output\20240710_1533 File 1_output.mat')
%%
% Extract data from the timeseries object
elevation_data = data_final0(:,sensor);  % Elevation values
time_data = time_final;       % Corresponding time values

% Define the threshold for filtering crests and troughs
tst = [0.01:0.01:0.99];

for q = 1:length(tst)
threshold = tst(q)*std(elevation_data);  % Example threshold value (adjust as needed)

% Find zero-crossings
sign_changes = diff(sign(elevation_data));  % Compute the difference in sign
zero_crossings = find(sign_changes);        % Find indices where sign changes

% Correct the zero-crossing indices to the time vector
zero_crossing_times = time_data(zero_crossings + 1); % +1 because 'diff' reduces length by 1

% Initialize arrays to store filtered crest and trough times and values
crest_times = [];
crest_values = [];
trough_times = [];
trough_values = [];

% Loop through each interval between zero-crossings
for i = 1:length(zero_crossings)-1
    % Define the interval
    interval_start = zero_crossings(i) + 1;
    interval_end = zero_crossings(i+1);
    
    % Extract the data for this interval
    interval_time = time_data(interval_start:interval_end);
    interval_elevation = elevation_data(interval_start:interval_end);
    
    % Find the crest (maximum) in this interval
    [crest_value, crest_index] = max(interval_elevation);
    crest_time = interval_time(crest_index);
    
    % Find the trough (minimum) in this interval
    [trough_value, trough_index] = min(interval_elevation);
    trough_time = interval_time(trough_index);
    
    % Check if the crest and trough meet the threshold criteria
    if crest_value > threshold
        crest_times = [crest_times; crest_time];
        crest_values = [crest_values; crest_value];
    end
    
    if trough_value < -threshold
        trough_times = [trough_times; trough_time];
        trough_values = [trough_values; trough_value];
    end
end

% Calculate dune heights and periods
dune_heights = [];
dune_periods = [];

% Loop through detected crests to calculate dune heights and periods
for i = 1:length(crest_times)-1
    % Find the trough corresponding to the current crest
    current_crest_time = crest_times(i);
    next_crest_time = crest_times(i+1);
    
    % Find the trough that is between the current crest and the next crest
    valid_troughs = trough_times(trough_times > current_crest_time & trough_times < next_crest_time);
    if ~isempty(valid_troughs)
        % Select the first trough in the interval
        current_trough_time = valid_troughs(1);
        % Calculate the height of the dune
        crest_index = find(crest_times == current_crest_time);
        trough_index = find(trough_times == current_trough_time);
        dune_height = crest_values(crest_index) - trough_values(trough_index);
        dune_heights = [dune_heights; dune_height];
        H_loop{q} = dune_heights;
        
        % Calculate the period of the dune
        dune_period = next_crest_time - current_crest_time;
        dune_periods = [dune_periods; dune_period];
    end
end

Hmean(q) = mean(dune_heights);
Hmed(q) = median(dune_heights); 
clear dune_height dune_heights
end

%% test mean/med
figure; hold on,
plot(tst, Hmean)
plot(tst, Hmed)
xlabel ('threshold constant'); ylabel ('\Delta [m]')
xline (0.5)
legend ('mean', 'median')

%% pdfs

for q = 1:length(H_loop)
y = H_loop{q};

% figure; hold on,
% h = histogram(y,20, 'Normalization', 'pdf');
% ksdensity(y, 'Bandwidth', 0.004)
% title (num2str(tst(q)))
% xlabel ('\Delta [m]'); 

[f, xi] = ksdensity(y, 'Bandwidth', 0.004);
[~, locs] = findpeaks(f);
% modi{q} = xi(locs)/0.15;
modi{q} = xi(locs);

clear pks locs f y xi
end

figure; hold on,
for q = 1:length(modi)
scatter (tst(q), modi{q}(1), 'filled', 'k')

    if length(modi{q})>1
                scatter (tst(q), modi{q}(2), 'filled', 'r')
    else
        continue
    end

    if length(modi{q})>2
            scatter (tst(q), modi{q}(3), 'filled', 'c')
    else
          continue
    end
    
%     if length(modi{q})>3
%          scatter (tst(q), modi{q}(4), 'filled', 'y')
%     else
%          continue
%     end

end
xlabel ('threshold x std');  
% ylabel ('H (m)')
ylabel ('\Delta/h')
xline (0.5)
legend ('first modus', 'second modus', 'third modus', 'Location', 'southeast')



%% figure for SI



figure; 
subplot (2,3,1), hold on,
q = 20;
y = H_loop{q};
h = histogram(y,20, 'Normalization', 'pdf');
[f, xi]  = ksdensity(y, 'Bandwidth', 0.004);
plot(xi,f, 'k', 'LineWidth', 2)
[pk, locs] = findpeaks(f);
scatter(xi(locs(1)), pk(1), 'k', 'filled')
scatter(xi(locs(2)), pk(2), 'r', 'filled')
scatter(xi(locs(3)), pk(3), 'c', 'filled')
title (strcat(num2str(tst(q)), ' * std'))
xlabel ('\Delta [m]'); 
xlim ([0 0.11]); ylim ([0 40]);

subplot (2,3,2), hold on,
q = 50;
y = H_loop{q};
h = histogram(y,20, 'Normalization', 'pdf');
[f, xi]  = ksdensity(y, 'Bandwidth', 0.004);
plot(xi,f, 'k', 'LineWidth', 2)
[pk, locs] = findpeaks(f);
scatter(xi(locs(1)), pk(1), 'k', 'filled')
scatter(xi(locs(2)), pk(2), 'r', 'filled')
title (strcat(num2str(tst(q)), ' * std'))
xlabel ('\Delta [m]'); 
xlim ([0 0.11]); ylim ([0 40]);


subplot (2,3,3), hold on,
q = 80;
y = H_loop{q};
h = histogram(y,20, 'Normalization', 'pdf');
[f, xi]  = ksdensity(y, 'Bandwidth', 0.004);
plot(xi,f, 'k', 'LineWidth', 2)
[pk, locs] = findpeaks(f);
scatter(xi(locs(1)), pk(1), 'k', 'filled')
scatter(xi(locs(2)), pk(2), 'r', 'filled')
title (strcat(num2str(tst(q)), ' * std'))
xlabel ('\Delta [m]'); 
xlim ([0 0.11]); ylim ([0 40]);

subplot(2,3,[4 5 6]), hold on,
for q = 1:length(modi)
scatter (tst(q), modi{q}(1), 'filled', 'k')

    if length(modi{q})>1
                scatter (tst(q), modi{q}(2), 'filled', 'r')
    else
        continue
    end

    if length(modi{q})>2
            scatter (tst(q), modi{q}(3), 'filled', 'c')
    else
          continue
    end
    
%     if length(modi{q})>3
%          scatter (tst(q), modi{q}(4), 'filled', 'y')
%     else
%          continue
%     end

end
xlabel ('threshold * std');  
ylabel ('\Delta [m]')
% ylabel ('\Delta/h')
xline (0.5); xline (0.2); xline (0.8)
legend ('first modus', 'second modus', 'third modus', 'Location', 'southeast')

set(gcf,'Units','Inches');
pos = get(gcf,'Position');
set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(gcf,['output/FigureS7.pdf'],'-dpdf','-r500')
% print(gcf,['SI_Figure_7.png'],'-dpng','-r500')

