%% PitchforkBifurcation_RK4_V3.m

% This script simulates flickering around a (modified) supercritical
% pitchfork bifurcation, i.e. dy/dt = ry-y^3, where y = -a+br^2+x was
% substituted to obtain a parabolic "central" equilibrium state. First, the
% equilibrium solutions of x are plotted. Then, using a fourth-order Runge
% Kutta scheme, the ordinary differential equation is time-integrated (with
% slowly varying control parameter r), and white noise is added at each
% time step. Finally, probability distributions are shown for different
% time windows along the flickering time series, to visualize that the
% system moves from one stable solution (unimodal distribution) to two
% stable solutions (bimodal distribution) once the pitchfork bifurcation
% point is exceeded.
% Updates in Version 2: fonts and labels finetuned for in the manuscript.

% Version 1 - 20 Nov 2023 - all OK.
% Version 2 - 14 Dec 2023 - all OK.
% Version 3 - 01 May 2025 - all OK.

% Script written by: Roeland C. van de Vijsel
% Hydrology & Environmental Hydraulics Group, Wageningen University, the Netherlands

% This script belongs to the following publication:
% "Bimodality in subaqueous dune height suggests flickering behavior at high flow", published in Nature Communications (2025).
% Sjoukje I. de Lange(1,2,*), Roeland C. van de Vijsel(1), Paul J.J. F. Torfs(3), Nick P. Wallerstein(1), and A.J.F (Ton) Hoitink(1)
% 1 Wageningen University, Department of Environmental Sciences, Hydrology and Environmental Hydraulics Group, Wageningen, the Netherlands
% 2 HKV lijn in water, Lelystad, the Netherlands
% 3 independent researcher
% * Corresponding author: Sjoukje de Lange (delangesjoukje@gmail.com)

clear all
close all
clc

VERSION = 3;

on = 1; off = 0;
SAVE = on; % Whether or not output and figures should be saved
LOADRESULTS = on; % Whether or not to load the output that was created earlier

% OK.

if LOADRESULTS == on % If so: use previously calculated output

    load('input/PitchforkBifurcation_RK4_a=1_b=1_sigma=0.05_drdt=0.001_dt=0.01.mat');

else % Else: do the calculations again

    %% Model settings

    % Control parameter r(t)
    drdt = 0.001;   % (fixed) rate of change of control parameter r
    r0 = -1;        % start value of r
    r1 = 1;         % end value of r
    r = @(t) (r0 + drdt.*t); % function r(t)
    
    % Time t
    dt = 0.01; % time step size
    t0 = 0; % start time
    t1 = (r1-r0)/drdt; % end time (such that r goes from r0 to r1)
    t_vec = t0:dt:t1; % corresponding time vector
    nt = length(t_vec); % number of time steps
    
    % Model parameters
    a = 1; % Constant in the r^0 term
    b = 1; % Constant in the r^2 term
    sigma = 0.05; % standard deviation of the Gaussian distribution (i.e., the white noise perturbation)
    
    % OK.
    
    %% Differential equation of a (modified) supercritical pitchfork bifurcation
    % Supercritical pitchfork bifurcation in normal form: dy/dt = ry-y^3
    % (see e.g., Strogatz 2018, Nonlinear Dynamics and Chaos)
    % Substitute y = -a+br^2+x, to obtain a modified bifurcation with a parabolic "center solution":
    % Assume dr/dt << dx/dt, such that dy/dt = dx/dt
    
    f  = @(T,X) ( r(T).*(-a + b*r(T).^2 + X) - (-a + b*r(T).^2 + X).^3 );
    
    % OK.
    
    %% Prepare arrays to store variables
    
    t_arr = -999*ones(1,nt); % time variable, t
    r_arr = -999*ones(1,nt); % control variable, r
    x_arr = -999*ones(1,nt); % state variable, x
    
    % OK.
    
    %% Fourth-order Runge-Kutta time integration
    
    % Define the four "trial steps" (see e.g., Strogatz 2018, Nonlinear Dynamics and Chaos)
    
    k1 = @(T,X   ) ( dt*f(T     , X     ) );
    k2 = @(T,X,K1) ( dt*f(T+dt/2, X+K1/2) );
    k3 = @(T,X,K2) ( dt*f(T+dt/2, X+K2/2) );
    k4 = @(T,X,K3) ( dt*f(T+dt  , X+K3  ) );
    
    % OK.
    
    %% Numerical time integration: initial time step
    
    i = 1; % Time vector index
    t = t_vec(i); % Specify the time
    
    dW = normrnd(0,sigma);  % white noise, to be added at each time step
    
    x = a - b*r(t).^2 + dW; % Initial state variable value: equilibrium solution plus white noise
    
    % Store initial values in variable arrays
    t_arr(i) = t;
    r_arr(i) = r(t);
    x_arr(i) = x;
    
    clc; disp(['i=',num2str(t),', t=',num2str(t),', r=',num2str(r(t)),', x=',num2str(x)]) % Show progress in the command window
    
    savename = ['PitchforkBifurcation_RK4_a=',num2str(a),'_b=',num2str(b),'_sigma=',num2str(sigma),'_drdt=',num2str(drdt),'_dt=',num2str(dt),'_V',num2str(VERSION)]; % Name for saving purposes
    
    % OK.
    
    %% Numerical time integration: loop over time steps
    
    for i = 2:nt
        
        dW = normrnd(0,sigma);  % white noise, to be added at each time step
    
        % State variable step
        dx = (1/6)*( k1(t,x) + 2*k2(t,x,k1(t,x)) + 2*k3(t,x,k2(t,x,k1(t,x))) + k4(t,x,k3(t,x,k2(t,x,k1(t,x)))) ) + dW;
    
        % Update x and t
        x = x + dx;
        t = t + dt;
    
        t_arr(i) = t;
        r_arr(i) = r(t);
        x_arr(i) = x;
    
        clc; disp(['i=',num2str(t),', t=',num2str(t),', r=',num2str(r(t)),', x=',num2str(x)]) % Show progress in the command window
    
    end
    
    % OK.

end

savename = ['PitchforkBifurcation_RK4_a=',num2str(a),'_b=',num2str(b),'_sigma=',num2str(sigma),'_drdt=',num2str(drdt),'_dt=',num2str(dt),'_V',num2str(VERSION)]; % Name for saving purposes

%% Plot the equilibrium states

lw = 1; %2; % LineWidth
ms = 4; %7; % MarkerSize
fs = 7; % 12; % FontSize

% The three solution branches
xsol1 = a - b.*r_arr.^2;
xsol2 = a - b.*r_arr.^2 - sqrt(r_arr);
xsol3 = a - b.*r_arr.^2 + sqrt(r_arr);

% Automatically check if solutions are real (re=1) or have an imaginary part (re=0)
re1 = ~imag(xsol1);
re2 = ~imag(xsol2);
re3 = ~imag(xsol3);

% Automatically check if solutions are stable (negative eigenvalue) or unstable (positive eigenvalue)
% The eigenvalue of xsol1 is r, the eigenvalue of xsol2 and xsol3 is -2r
stab1 = (r_arr < 0);
stab2 = (-2*r_arr < 0);
stab3 = (-2*r_arr < 0);

% Plot
close(figure(1));figure(1)
hold on

% Real and stable
plot(r_arr((re1 & stab1)),xsol1((re1 & stab1)),'b-','LineWidth',lw); 
plot(r_arr((re2 & stab2)),xsol2((re2 & stab2)),'c-','LineWidth',lw); 
plot(r_arr((re3 & stab3)),xsol3((re3 & stab3)),'c-','LineWidth',lw);

% Real and unstable
plot(r_arr((re1 & ~stab1)),xsol1((re1 & ~stab1)),'b--','LineWidth',lw); 
plot(r_arr((re2 & ~stab2)),xsol2((re2 & ~stab2)),'c--','LineWidth',lw); 
plot(r_arr((re3 & ~stab3)),xsol3((re3 & ~stab3)),'c--','LineWidth',lw);

plot(0,a,'or','MarkerFaceColor','r','MarkerSize',ms)

plot(r_arr,x_arr,'k')

axis([-1 1 -1 2])

xlabel('\it{r}')
ylabel('\it{x}')
title(['a = ',num2str(a),', b = ',num2str(b),', \sigma = ',num2str(sigma),', dr/dt = ',num2str(drdt),', dt = ',num2str(dt)],'FontSize',fs)
legend('dx/dt = r(-a+br^2+x) - (-a+br^2+x)^3','Location','SouthWest','FontSize',fs)
pbaspect([1.5,1,1])

% Save the plot
% set(gcf,'Units','Inches');
% pos = get(gcf,'Position');
% set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
% print(gcf,[savename,'.pdf'],'-dpdf','-r500')
% print(gcf,[savename,'.png'],'-dpng','-r500')

% OK.

%% Plot equilibrium states, with histograms to show flickering/bimodality

close(figure(2));figure(2)

set(0,'defaultAxesFontSize',fs)

window = 10000; % Length of the time windows for which histograms are plotted

xmin = floor(min(10*x_arr))/10; % Minimum (rounded down) of state variable x to be plotted
xmax = ceil(max(10*x_arr))/10;  % Maximum (rounded up) of state variable x to be plotted

histmax = 0.055; % Maximum probability to be shown in histograms

nbins = 50; % Number of histogram bins
nsx = 5; % Number of subplots in x-direction
nsy = 4; % Number of subplots in y-direction

% Choose a color map to color-code the different histograms
cmap = flipud(cmocean('thermal'));
col_i = round(linspace(1,length(cmap),nsx+1));
colors = cmap(col_i,:);

% Subplot
subplot(nsy,nsx,(nsy-1)*nsx + 1)

% Range of elements to be shown in histogram
el1a = 2*window + 1;
el1b = el1a+window;
el1m = round(mean([el1a,el1b]));

% Show the equilibrium state values as vertical lines in the histogram
if r_arr(el1m) <= 0
    xline(xsol1(el1m),'k','LineWidth',lw/2); hold on
else
    xline(xsol2(el1m),'k','LineWidth',lw/2); hold on
    xline(xsol3(el1m),'k','LineWidth',lw/2);
end

histogram(x_arr(el1a:el1b),nbins, 'Normalization','probability','FaceColor',colors(1,:), 'EdgeColor','none','FaceAlpha',1)
set(gca,'xlim',[xmin xmax],'ylim',[0 histmax],'FontSize',fs)
% xlabel('x')
ylabel('probability','FontSize',fs)
text(0.05,0.95,'b)','Units','Normalized','FontSize',fs)

% Subplot
subplot(nsy,nsx,(nsy-1)*nsx + 2)

% Range of elements to be shown in histogram
el2a = 6*window + 1;
el2b = el2a+window;
el2m = round(mean([el2a,el2b]));

if r_arr(el2m) <= 0
    xline(xsol1(el2m),'k','LineWidth',lw/2); hold on
else
    xline(xsol2(el2m),'k','LineWidth',lw/2); hold on
    xline(xsol3(el2m),'k','LineWidth',lw/2);
end

histogram(x_arr(el2a:el2b),nbins, 'Normalization','probability','FaceColor',colors(2,:), 'EdgeColor','none','FaceAlpha',1)
set(gca,'xlim',[xmin xmax],'ylim',[0 histmax],'YTickLabel',[])
% xlabel('x')
% ylabel('prob. dist.')
text(0.05,0.95,'c)','Units','Normalized','FontSize',fs)

% Subplot
subplot(nsy,nsx,(nsy-1)*nsx + 3)

% Range of elements to be shown in histogram
el3a = 10*window + 1;
el3b = el3a+window;
el3m = round(mean([el3a,el3b]));

if r_arr(el3m) <= 0
    xline(xsol1(el3m),'k','LineWidth',lw/2); hold on
else
    xline(xsol2(el3m),'k','LineWidth',lw/2); hold on
    xline(xsol3(el3m),'k','LineWidth',lw/2);
end
histogram(x_arr(el3a:el3b),nbins, 'Normalization','probability','FaceColor',colors(3,:), 'EdgeColor','none','FaceAlpha',1)
set(gca,'xlim',[xmin xmax],'ylim',[0 histmax],'YTickLabel',[])
xlabel('\it{x}')
% ylabel('prob. dist.')
text(0.05,0.95,'d)','Units','Normalized','FontSize',fs)

% Subplot
subplot(nsy,nsx,(nsy-1)*nsx + 4)

% Range of elements to be shown in histogram
el4a = 14*window + 1;
el4b = el4a+window;
el4m = round(mean([el4a,el4b]));

if r_arr(el4m) <= 0
    xline(xsol1(el4m),'k','LineWidth',lw/2); hold on
else
    xline(xsol2(el4m),'k','LineWidth',lw/2); hold on
    xline(xsol3(el4m),'k','LineWidth',lw/2);
end

histogram(x_arr(el4a:el4b),nbins, 'Normalization','probability','FaceColor',colors(4,:), 'EdgeColor','none','FaceAlpha',1)
set(gca,'xlim',[xmin xmax],'ylim',[0 histmax],'YTickLabel',[])
% xlabel('x')
% ylabel('prob. dist.')
text(0.05,0.95,'e)','Units','Normalized','FontSize',fs)

% Subplot
subplot(nsy,nsx,(nsy-1)*nsx + 5)

% Range of elements to be shown in histogram
el5a = 18*window + 1;
el5b = el5a+window;
el5m = round(mean([el5a,el5b]));

if r_arr(el5m) <= 0
    xline(xsol1(el5m),'k','LineWidth',lw/2); hold on
else
    xline(xsol2(el5m),'k','LineWidth',lw/2); hold on
    xline(xsol3(el5m),'k','LineWidth',lw/2);
end

histogram(x_arr(el5a:el5b),nbins, 'Normalization','probability','FaceColor',colors(5,:), 'EdgeColor','none','FaceAlpha',1)
set(gca,'xlim',[xmin xmax],'ylim',[0 histmax],'YTickLabel',[])
% xlabel('x')
% ylabel('prob. dist.')
text(0.05,0.95,'f)','Units','Normalized','FontSize',fs)

% Subplot
subplot(nsy,nsx,1:(nsy-1)*nsx)

% Show the flickering time series
plot(r_arr,x_arr,'Color',[150,150,150]/255); hold on

xlabel('\it{r}','FontSize',fs)
ylabel('\it{x}','FontSize',fs)

% % Highlight the corresponding parts from the flickering time series that
% % are shown in the histograms
plot(r_arr(el1a:el1b), x_arr(el1a:el1b), 'Color',colors(1,:) )
plot(r_arr(el2a:el2b), x_arr(el2a:el2b), 'Color',colors(2,:) )
plot(r_arr(el3a:el3b), x_arr(el3a:el3b), 'Color',colors(3,:) )
plot(r_arr(el4a:el4b), x_arr(el4a:el4b), 'Color',colors(4,:) )
plot(r_arr(el5a:el5b), x_arr(el5a:el5b), 'Color',colors(5,:) )
axis([-1 1 xmin xmax])

% Real and stable
plot(r_arr((re1 & stab1)),xsol1((re1 & stab1)),'k-','LineWidth',lw); 
plot(r_arr((re2 & stab2)),xsol2((re2 & stab2)),'k-','LineWidth',lw); 
plot(r_arr((re3 & stab3)),xsol3((re3 & stab3)),'k-','LineWidth',lw);

% Real and unstable
plot(r_arr((re1 & ~stab1)),xsol1((re1 & ~stab1)),'k--','LineWidth',lw); 
plot(r_arr((re2 & ~stab2)),xsol2((re2 & ~stab2)),'k--','LineWidth',lw); 
plot(r_arr((re3 & ~stab3)),xsol3((re3 & ~stab3)),'k--','LineWidth',lw);

% Plot the bifurcation point itself
plot(0,a,'ok','MarkerFaceColor','w','MarkerSize',ms,'LineWidth',lw)

pbaspect([2 1 1])

text(0.01,0.97,'a)','Units','Normalized','FontSize',fs)

% Save the plot and output

if SAVE == on

    set(gcf,'Units','Inches');
    pos = get(gcf,'Position');
    set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
    % print(gcf,[savename,'_hist_3.pdf'],'-dpdf','-r500')
    % print(gcf,[savename,'_hist_3.png'],'-dpng','-r500')
    print(gcf,['output/FigureS14.pdf'],'-dpdf','-r500')
    
    % save([savename,'.mat'])

end

% OK.