%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code on multi-dt approach for peak-locking error correction and  % 
% uncertainty quantification in PIV                                       %
% Authors: Sagar Adatrao, Michele Bertone and Andrea Sciacchitano         %
% Date: 10 December 2020                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Inputs:                                                                 %
%   - Parameters from PIV setup:                                          %
%       - Multiple dt values [micro second]                               %
%       - Number of image pairs for each dt                               %
%       - Camera pixel size [micro meter]                                 %
%       - Magnification factor                                            %
%   - Velocity fields from PIV                                            %
%   - t-statistic for desired confidence interval                         %
% Outputs:                                                                %
%   - Uncertainties in measured mean displacements (and velocities)       %
%   - Uncertainties in measured Reynolds stresses                         %
%   - Corrected mean displacements (and velocities) from regression       %
%   - Corrected Reynolds stresses from regression                         %
%   - Uncertainties in corrected mean displacements (and velocities)      %
%   - Uncertainties in corrected Reynolds stresses                        %
% Pseudo code:                                                            %
%   - Perform a least-squares regression of the measured mean             %
%     displacements from different dt acquisitions                        %
%   - Estimate the uncertainities in the measured and corrected           %
%     displacements (and velocities) as per the procedure described in    %
%     the corresponding article                                           %
%   - Similarly, perform a least-squares regression of the measured       %
%     Reynolds stresses from different dt acquisitions                    %
%   - Estimate the uncertainities in the measured and corrected           %
%     Reynolds stresses as per the procedure described in                 %
%     the corresponding article                                           %
%   - Export the results                                                  %
% Note:                                                                   %
%   - User should delete the script lines written for the reference       %
%     measurement                                                         %
%   - The reference measurement is only required for validation of the    % 
%     proposed uncertainty quantification approach                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clc; clear; close all;

%% Inputs

startPositionX=1; startPositionY=1; numColumn=10; numRow=10; 
                        % Region of analysis prescribed by X and Y indices 
                        % of velocity matrix from PIV  
numPoints=numColumn*numRow; % Number of points in the region of analysis
points=zeros(2,numPoints); n=1;
for j=1:numColumn
    for i=1:numRow        
        points(1,n)=startPositionX+j-1;
        points(2,n)=startPositionY+i-1;
        n=n+1;        
    end
end

dt=10:2:24; % Multiple dt's [micro second]
numDt=numel(dt); % Number of dt's
numImg=500; % Number of image pairs for each dt
s=20; % Camera pixel size [micro meter]
M=0.46; % Magnification factor
tValue=1; % t-statistic for desired confidence interval
zScore=3.3; % z-score for Chauvenet's criteria (devMax/SD)
            % z-score = 2.8 for N = 100 and 3.3 for N = 500 to 1000
datFile=('D:\reference\B (1).DAT'); % Data file of velocity field from PIV
sizeMatrix=sizeOfMatrix(datFile); % To retrieve size of the velocity matrix 

%% Reference measurement

dtAux=50; % Large dt [micro second] in auxiliary measurement
uAuxRaw=zeros(numImg,numPoints);                                                 
for i=1:numImg
    datFile=['D:\reference\B (',num2str(i),').DAT'];
    uAuxAll=readDatFile(datFile,sizeMatrix(1),sizeMatrix(2)); 
                                     % Auxiliary velocities in whole image                                          
    for p=1:numPoints
        uAuxRaw(i,p)=uAuxAll(points(2,p),points(1,p));
    end                   % Auxiliary velocities in the region of analysis
 end

dxAuxRaw=uAuxRaw*dtAux*M/s; 
                        % Auxiliary displacements in the region of analysis
uAux=zeros(numImg,numPoints); dxAux=zeros(numImg,numPoints);
for p=1:numPoints 
    uAux(:,p)=chauvenetCriteria(uAuxRaw(:,p),zScore);
    dxAux(:,p)=chauvenetCriteria(dxAuxRaw(:,p),zScore);
end % Outliers removal by Chauvenet's criterion

uAuxMean=nanmean(uAux); % Auxiliary mean velocity 
uAuxSD=nanstd(uAux); % Standard deviation of auxiliary velocities
dxAuxMean=nanmean(dxAux); % Auxiliary mean displacement
dxRef=zeros(numDt,numPoints);
for j=1:numDt
    dxRef(j,:)=(dxAuxMean/dtAux)*dt(j); 
                                % Reference displacement [pix] at each dt
end
uRef=uAuxMean; % Reference velocity [m/s]
RuuAux=uAuxSD.^2; % Auxiliary Reynolds stress [m^2/s^2]
RuuRef=RuuAux; % Reference Reynolds stress [m^2/s^2]

%% Actual measurements

uRaw=zeros(numImg,numDt,numPoints); u=zeros(numImg,numDt,numPoints);  
dxRaw=zeros(numImg,numDt,numPoints); dx=zeros(numImg,numDt,numPoints);  
uMean=zeros(numDt,numPoints); uSD=zeros(numDt,numPoints); 
dxMean=zeros(numDt,numPoints); dxSD=zeros(numDt,numPoints);          
dxNum=zeros(numDt,numPoints); Ruu=zeros(numDt,numPoints); 
dxFrac=zeros(numImg,numDt,numPoints); varSine=zeros(numDt,numPoints);

for j=1:numDt
    for i=1:numImg
        datFile=['D:\actual\dt',num2str(dt(j)),'\B (',num2str(i),').DAT'];
        uAll=readDatFile(datFile,sizeMatrix(1),sizeMatrix(2));  
                                      % Measured velocities in whole image
        for p=1:numPoints 
            uRaw(i,j,p)=uAll(points(2,p),points(1,p));
                           % Measured velocities in the region of analysis
        end
        dxRaw(i,j,:)=uRaw(i,j,:)*dt(j)*M/s; 
                        % Measured displacements in the region of analysis 
    end
    
    for p=1:numPoints 
        u(:,j,p)=chauvenetCriteria(uRaw(:,j,p),zScore);
        dx(:,j,p)=chauvenetCriteria(dxRaw(:,j,p),zScore);
                                % Outliers removal by Chauvenet's criterion
        uMean(j,p)=nanmean(u(:,j,p)); % Measured mean velocities                           
        uSD(j,p)=nanstd(u(:,j,p)); 
                               % Standard deviations of measured velocities
        dxMean(j,p)=nanmean(dx(:,j,p)); % Measured mean displacements                          
        dxSD(j,p)=nanstd(dx(:,j,p)); 
                            % Standard deviations of measured displacements
        dxNum(j,p)=sum(~isnan(dx(:,j,p)));
        Ruu(j,p)=uSD(j,p)^2; % Measured Reynolds stresses
        
        dxFrac(:,j,p)=dx(:,j,p)-round(dx(:,j,p)); % Sub-pixel displacements
        varSine(j,p)=nanvar(sin(2*pi.*dxFrac(:,j,p)));
    end
end

dxMeanFrac=dxMean-round(dxMean); % Sub-pixel measured mean displacements
coSine=cos(2*pi.*dxMeanFrac);

%% Regression for displacements

uRegr=zeros(1,numPoints); dxRegr=zeros(numDt,numPoints);
for p=1:numPoints
    f=fit(dt',dxMean(:,p),'poly1','Lower',[-Inf,0],'Upper',[+Inf,0]);
    uRegr(p)=f.p1;
    for j=1:numDt 
        dxRegr(j,p)=uRegr(p)*dt(j); % Displacement [pix] from regression
    end 
    uRegr(p)=uRegr(p)*s/M; % Velocity [m/s] from regression
end

%% Regression for Reynolds stresses

beta=zeros(4,numPoints); % Regression coefficients
numCoeff=size(beta,1);
p1=(1./(dt.^2))'; p2=zeros(numDt,numPoints); p3=zeros(numDt,numPoints);
P=zeros(numDt,numCoeff,numPoints);
AConstraints=[-1,0,0,0;0,-1,0,0;0,0,-1,0]; 
bConstraints=[0;0;0]; % For coefficients beta0, beta1, beta2 to be positive
RuuRegr=zeros(numDt,numPoints); RuuRegrTrue=zeros(1,numPoints); 
Regr2=zeros(numDt,numPoints); Regr3=zeros(numDt,numPoints);

for p=1:numPoints
    p2(:,p)=varSine(:,p)./(dt'.^2); p3(:,p)=coSine(:,p)./dt';
    P(:,:,p)=cat(2,ones(numDt,1),p1,p2(:,p),p3(:,p));
    beta(:,p)=lsqlin(P(:,:,p),Ruu(:,p),AConstraints,bConstraints);
    
    RuuRegrTrue(p)=beta(1,p); 
                        % Estimate of true Reynolds stress from regression
    Regr2(:,p)=beta(2,p)*p1+beta(3,p)*p2(:,p);
                        % Estimate of second term from regression
    Regr3(:,p)=beta(4,p)*p3(:,p);
                        % Estimate of third term from regression
    RuuRegr(:,p)=RuuRegrTrue(p)+Regr2(:,p)+Regr3(:,p); 
                        % Reynolds stress from regression
end

%% Uncertainty estimation for displacement and velocity

Ubdx=zeros(1,numPoints); Urdx=zeros(numDt,numPoints);
UdxMean=zeros(numDt,numPoints); errordxMean=zeros(numDt,numPoints);
UuMean=zeros(numDt,numPoints); erroruMean=zeros(numDt,numPoints);
UdxRegr=zeros(numDt,numPoints);

for p=1:numPoints
    Ubdx(p)=tValue*sqrt(sum((dxMean(:,p)-dxRegr(:,p)).^2)/(numDt-1));
                     % Bias uncertainty in mean measured displacement
    Urdx(:,p)=(tValue./sqrt(dxNum(:,p))).*dxSD(:,p);
                     % Precision uncertainty in mean measured displacement
    UdxMean(:,p)=sqrt(Ubdx(p)^2+Urdx(:,p).^2);
                     % Total uncertainty in mean measured displacement
    errordxMean(:,p)=dxMean(:,p)-dxRef(:,p);
                     % Error in mean measured displacement
    
    for j=1:numDt
        UuMean(j,p)=(UdxMean(j,p)./dt(j)).*s/M;
    end              % Total uncertainty in mean measured velocity [m/s]
    erroruMean(:,p)=uMean(:,p)- uRef(p); % Error in mean velocity
    
    coVarDt=sum(dt.^2)-(sum(dt)^2)/numDt; dtMean=mean(dt);
    UdxRegr(:,p)=Ubdx(p).*sqrt((1/numDt)+((dt(:)-dtMean).^2)./coVarDt);
                     % Total uncertainty in regression displacement [pix]
end

UuRegr=((Ubdx/sqrt(numDt))/dtMean)*s/M; 
                        % Total uncertainty in regression velocity [m/s]

%% Uncertainty estimation for Reynolds stress

UbRuu=zeros(1,numPoints); UrRuu=zeros(numDt,numPoints);
for p=1:numPoints
    UbRuu(p)=tValue*sqrt(sum((Ruu(:,p)-RuuRegrTrue(:,p)).^2)/...
        (numDt-numCoeff));  % Bias uncertainty in measured Reynolds stress
    UrRuu(:,p)=tValue.*sqrt(2./(dxNum(:,p)-1)).*Ruu(:,p);
                       % Precision uncertainty in measured Reynolds stress
end

URuu1=sqrt(UbRuu.^2+UrRuu.^2); URuu2=UrRuu;
            % Total uncertainties in measured Reynolds stress 
errorRuu=Ruu-RuuRef; % Error in measured Reynolds stress

URuuRegr=tValue*sqrt(sum((Ruu-RuuRegr).^2)/(numDt-numCoeff)); 
                        % Uncertainty in regression model
URuuRegrTrue=zeros(1,numPoints);
PP=zeros(numCoeff,numCoeff,numPoints);
PPInv=zeros(numCoeff,numCoeff,numPoints);
for p=1:numPoints
    PP(:,:,p)=P(:,:,p)'*P(:,:,p); PPInv(:,:,p)=inv(PP(:,:,p));
    URuuRegrTrue(p)=URuuRegr(p)*sqrt(PPInv(1,1,p));
         % Uncertainty in estimate of true Reynolds stress from regression
end

%% Exporting the results

v=1:numDt:numDt*numPoints;
t=1;
results=zeros(v(1,numel(v)),54);

for p=1:numPoints
    n=v(t);
    d=v(t);    
    for j=1:numDt        
        results(d,1)=points(1,p);
        results(d,2)=points(2,p);
        
        results(n,3)=dt(j);
        results(n,4)=dxRef(j,p);
        results(n,5)=dxMean(j,p);
        results(n,6)=dxRegr(j,p);
        results(d,7)=Ubdx(p);
        results(n,8)=Urdx(j,p);
        results(n,9)=UdxMean(j,p);
        results(n,10)=UdxRegr(j,p);
        results(n,11)=errordxMean(j,p);
        
        results(d,13)=uRef(p);
        results(n,14)=uMean(j,p);
        results(d,15)=uRegr(p);
        results(n,16)=UuMean(j,p);
        results(d,17)=UuRegr(p);
        results(n,18)=erroruMean(j,p);
        
        results(d,20)=RuuRef(p);
        results(n,21)=Ruu(j,p);
        results(n,22)=RuuRegr(j,p);
        results(n,23)=RuuRegrTrue(p);
        results(d,24)=UbRuu(p);
        results(n,25)=UrRuu(j,p);
        results(n,26)=URuu1(j,p);
        results(n,27)=URuu2(j,p);
        results(d,28)=URuuRegr(p);
        results(d,29)=URuuRegrTrue(p);
        results(n,30)=errorRuu(j,p);
                
        n = n +1;
    end
    t = t +1;
end

fid=fopen('results.txt','wt'); % Results file
for i=1:size(results,1)
    fprintf(fid,'%g\t',results(i,:));
    fprintf(fid,'\n');
end
fclose(fid);
