
% AccData_GoPro
% 1. CTS (milliseconds since first frame)
% 2. date
% 3. Accelerometer (z) (m/s^2)
% 4. Accelerometer (x) (m/s^2)
% 5. Accelerometer (y) (m/s^2)
% 6. temperature [°C]

% GPS Data GoPro
% 1 CTS (milliseconds since first frame)
% 2 date
% 3 GPS latitude (deg)
% 4 GPS longitude (deg)
% 5 GPS altitude (m)
% 6 GPS 2D speed (m/s)
% 7 GPS 3D speed (m/s)
% 8 fix
% 9 precision
% 10 altitude system
% 11 speedlimit_mapbox (km/h)
% 12 lon_mapbox (deg)
% 13 lat_mapbox (deg)

clear variables;close all;clc

% Class labels of the pre-trained YOLO model (COCO dataset)
Labels={'person'
    'bicycle'
    'car'
    'motorbike'
    'aeroplane'
    'bus'
    'train'
    'truck'
    'boat'
    'traffic light'
    'fire hydrant'
    'stop sign'
    'parking meter'
    'bench'
    'bird'
    'cat'
    'dog'
    'horse'
    'sheep'
    'cow'
    'elephant'
    'bear'
    'zebra'
    'giraffe'
    'backpack'
    'umbrella'
    'handbag'
    'tie'
    'suitcase'
    'frisbee'
    'skis'
    'snowboard'
    'sports ball'
    'kite'
    'baseball bat'
    'baseball glove'
    'skateboard'
    'surfboard'
    'tennis racket'
    'bottle'
    'wine glass'
    'cup'
    'fork'
    'knife'
    'spoon'
    'bowl'
    'banana'
    'apple'
    'sandwich'
    'orange'
    'broccoli'
    'carrot'
    'hot dog'
    'pizza'
    'donut'
    'cake'
    'chair'
    'sofa'
    'pottedplant'
    'bed'
    'diningtable'
    'toilet'
    'tvmonitor'
    'laptop'
    'mouse'
    'remote'
    'keyboard'
    'cell phone'
    'microwave'
    'oven'
    'toaster'
    'sink'
    'refrigerator'
    'book'
    'clock'
    'vase'
    'scissors'
    'teddy bear'
    'hair drier'
    'toothbrush'};

[bf,af] = butter(2,0.1); % filter parameters for Phone Acc data (recorded at 10 Hz); cut-off frequency: 0.5 Hz
[bf2,af2] = butter(2,0.005); % filter parameters for GoPro Acc data (recorded at 200 Hz); cut-off frequency: 0.5 Hz

files1=dir('Raw Data Phone\*.csv');
files2=dir('Acc Data GoPro\*acc_gopro.csv');
files3=dir('GPS Data GoPro\*gps_gopro.csv');

readraw=0;
% if readraw==1, then read .csv files in folder 1 and save as .mat file (Phone acceleration)
% if readraw==2, then read .csv files in folder 2 and save as .mat file (GoPro acceleration)
% if readraw==3, then read .csv files in folder 3 and save as .mat file (GoPro GPS)

rotate=0;
% if rotate==1, then rotate and filter Phone accelerometer data, and save as .mat file
% if rotate==2, then rotate and filter GoPro accelerometer data, and save at .mat file
%% Read .csv files and store as .mat files
if readraw==1 % set to 1 to read .csv files and save them as .mat files
    for i=1:length(files1)
        X=readtable(['Raw Data Phone\' files1(i).name]);

        disp([num2str(i) ', ' num2str(size(X,1)) ', ' num2str(size(X,2)) ', ' files1(i).name])
        clear x
        x(:,1)=datenum(table2array(X(:,1))); %#ok<*DATNM>
        x(:,2:13)=table2array(X(:,2:13));
        x=x(~isnan(x(:,11)),[1 11:13]); % only retain time, accx, accy, accz
        save(['AccData_Phone_' num2str(i) '.mat'],'x');
    end

elseif readraw==2
    for i=1:length(files2)
        X=readtable(['Acc Data GoPro\' files2(i).name]);
        disp([num2str(i) ', ' num2str(size(X,1)) ', ' num2str(size(X,2)) ', ' files2(i).name])
        clear x
        x(:,2)=datenum(table2array(X(:,2)));
        x(:,[1 3:6])=table2array(X(:,[1 3:6]));
        x(:,7:10)=NaN; % fill 4 columns with NaNs (for easy counting of variable numbers)
        save(['AccData_GoPro_' num2str(i) '.mat'],'x');
    end

elseif readraw==3
    for i=1:length(files3)
        X=readtable(['GPS Data GoPro\' files3(i).name]);
        disp([num2str(i) ', ' num2str(size(X,1)) ', ' num2str(size(X,2)) ', ' files2(i).name])
        clear x
        x(:,2) = datenum(table2array(X(:,2)));
        x(:,[1 3:9 11:13])=table2array(X(:,[1 3:9 11:13]));
        save(['GPSData_GoPro_' num2str(i) '.mat'],'x');
    end
end
%% Rotate phone accelerometer data
if rotate>0
    for i=1:length(files1)
        load(['AccData_Phone_' num2str(i) '.mat']);

        % definition according to NXP  % (Pedley et al., 2013; Figure 3))
        xa = -x(:,3); % longitudinal direction (Phone Y)
        ya = -x(:,2); % lateral direction (Phone X)
        za = x(:,4); % gravity (positive) (Phone Z)

        roll   = -atan2(mean(ya), mean(za)); % (Pedley et al., 2013; Eq. 25)
        pitch  = -atan2(-mean(xa), sqrt(mean(ya)^2 + mean(za)^2)); % (Pedley et al., 2013; Eq. 26)

        Rx = [1          0         0;            0 cos(roll)  sin(roll);    0          -sin(roll) cos(roll) ];
        Ry = [cos(pitch) 0         -sin(pitch);  0 1 0;                    sin(pitch) 0         cos(pitch)];
        x(:,[6 5 7])=(Rx*Ry*[xa ya za]')'; % Store in axis format as shown in Figure 1 of the paper
        x(:,5:7) = filtfilt(bf, af, x(:,5:7)); % apply filter
        x(:,7)=-x(:,7); % gravity negative
        % A positive x-value, x(:,5) corresponds with: turning to the left
        % A negative y-value, x(:,6) corresponds with: accelerating

        %%
        save(['AccData_Phone_Rotated_and_Filtered_' num2str(i) '.mat'],'x'); % same as the Data that was loaded, but now with rotated xyz
    end
end
%% Rotate GoPro accelerometer data
if rotate>0
    for i=1:length(files2)
        load(['AccData_GoPro_' num2str(i) '.mat']);

        % definition according to NXP, Figure 3
        xa = -x(:,4); % lateral direction (GoPro X)
        ya = x(:,5); % longitudinal direction (GoPro Y)
        za = -x(:,3); % gravity (positive) (GoPro Z)

        roll   = -atan2(mean(ya), mean(za)); % (Pedley et al., 2013; Eq. 25)
        pitch  = -atan2(-mean(xa), sqrt(mean(ya)^2 + mean(za)^2)); % (Pedley et al., 2013; Eq. 26)

        Rx = [1          0         0;            0 cos(roll)  sin(roll);    0          -sin(roll) cos(roll) ];
        Ry = [cos(pitch) 0         -sin(pitch);  0 1 0;                    sin(pitch) 0         cos(pitch)];
        x(:,7:9)=(Rx*Ry*[xa ya za]')'; % Store in axis format as shown in Figure 1 of the paper
        x(:,9)=-x(:,9); % gravity negative
        % A positive x-value, x(:,7) corresponds with: turning to the left
        % A negative y-value, x(:,8) means: accelerating

        save(['AccData_GoPro_Rotated_' num2str(i) '.mat'],'x'); % same as Data_acc_gopro but with rotated xyz
    end
end
%% Loop through 9 selected drives (i.e., drives for which Phone and GoPro data were available)
close all;clc

M = NaN(9,15); % pre-allocate matrix with NaNs (9 drives x 15 measures)
SpeedLimitDurationN = NaN(9,5); % 9 drives, 5 speed limit categories

for i=1:9 % loop over 9 drives
    load(['AccData_Phone_Rotated_and_Filtered_' num2str(i) '.mat']);
    x1=x; % store in x1
    x1(:,1)=24*3600*(x1(:,1)-x1(1,1)); % convert time vector to seconds
    x1(diff(x1(:,1))==0,:)=[]; % remove rows that have a duplicate time

    x1=interp1(x1(2:end-3,1),x1(2:end-3,:),0.1:0.1:max(x1(2:end-3,1)));  % interpolate x1 to 10 Hz
    x1(1,:)=x1(2,:);x1(1,1)=0.1; % make first row equal to second row, because it is sometimes missing. The elapsed time of the first row is 0.1 s, and of the second row 0.2 s.

    load(['AccData_GoPro_Rotated_' num2str(i) '.mat']); % load GoPro acceleration data
    x2=x; % store in x2
    x2(diff(x2(:,2))==0,:)=[]; % remove rows that have a duplicate time. Time is measured in Excel time (in days)

    goproacc_unfiltered=x2(:,[2 4 5 3 7:9]); % 1. time, 2-4: xyz non-rotated, 5-7: xyz rotated
    goproacc_unfiltered(:,1)=(goproacc_unfiltered(:,1)-goproacc_unfiltered(1,1))*24*3600; % convert time vector to seconds

    x2(:,7:9) = filtfilt(bf2, af2, x2(:,7:9)); % apply Butterworth filter to GoPro acceleration data

    xi=interp1(x2(:,2),x2(:,1),min(x2(:,2)):0.1/(24*3600):max(x2(:,2)),'nearest')'; % interpolate to 10 Hz. Use 'nearest' for CTS, x2(:,1)
    xi2=interp1(x2(:,2),x2(:,2:end),min(x2(:,2)):(0.1/(24*3600)):max(x2(:,2))); % interpolate to 10 Hz for the other variables
    x2=[xi xi2];

    load(['GPSData_GoPro_' num2str(i) '.mat']); % GoPro GPS
    x3=x; % store in x3
    x3(diff(x3(:,2))==0,:)=[];  % remove rows that have a duplicate time. Time is measured in Excel time (in days)
    xi=interp1(x3(:,2),x3(:,1),min(x2(:,2)):(0.1/(24*3600)):max(x2(:,2)),'nearest')'; % interpolate to 10 Hz. use 'nearest' for CTS, x3(:,1)
    xi2=interp1(x3(:,2),x3(:,[2:10 12:13]),min(x2(:,2)):(0.1/(24*3600)):max(x2(:,2))); % interpolate to 10 Hz
    xi3=interp1(x3(:,2),x3(:,11),min(x2(:,2)):(0.1/(24*3600)):max(x2(:,2)),'nearest')'; % interpolate to 10 Hz. use 'nearest' for speed limit, x3(:,11)

    clear x3
    x3(:,1)=xi;
    x3(:,[2:10 12:13])=xi2;
    x3(:,11)=xi3;

    x2(:,2)=24*3600*(x2(:,2)-x2(1,2)); % convert time vector to seconds
    x3(:,2)=24*3600*(x3(:,2)-x3(1,2)); % convert time vector to seconds

    x2=[x2 x3]; %#ok<AGROW> % combine GoPro data in a single matrix x2

    [a1,a2,D] = alignsignals(x1(:,6),x2(:,8),1000); % 5-7 is xyz of Phone acc (rotated and filtered), 7-9 is xyz of GoPro acc (rotated and filtered)

    x1 = [NaN(D,size(x1,2));x1]; %#ok<AGROW> % Add NaNs to data of Phone. This causes x1 and x2 to be aligned in time.
    time=linspace(0.1,size(a2,1)/10,size(a2,1));

    B=NaN(size(x2,1),1); % pre-allocate NaNs for bearing data

    x2(:,16)=fillmissing(x2(:,16),'linear'); % GoPro speed in m/s
    x2(:,16)=movmedian(x2(:,16),5); % filter GoPro speed
    x2(:,16)=filtfilt(bf,af,x2(:,16)); % filter GoPro speed

    if i==1
        % Figure 2. Illustration of the effect of low-pass filtering of the accelerometer data of the GoPro.
        figure
        set(gcf,'renderer','painters')

        yyaxis left
        plot(goproacc_unfiltered(:,1),goproacc_unfiltered(:,6),'color',[.5 .5 .5]);hold on
        plot(time,x2(:,8),'r-','Linewidth',3)
        xlabel('Elapsed time (s)')
        ylabel('Acceleration \it{y}\rm (m/s^2)')
        set(gca,'xlim',[1500 1550],'ycolor','r')
        grid on
        set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])

        yyaxis right
        grid on
        plot(time,3.6*x2(:,16),'b-','Linewidth',3)
        set(gca,'xlim',[1500 1550],'ycolor','b')
        ylabel('Speed (km/h)','color','b')
        legend('Acceleration, rotated and unfiltered','Acceleration, rotated and filtered', 'Speed','Location','northwest')
        h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')
        set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])
        set(gca,'ylim',[0 36],'ytick',0:6:36)
        set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
    end

    for j=1:size(x2,1)-1
        lat1 = deg2rad(x2(j,13));
        lat2 = deg2rad(x2(j+1,13));
        long1 = deg2rad(x2(j,14));
        long2 = deg2rad(x2(j+1,14));
        dLong = long2-long1;
        dLat =  lat2-lat1;
        B(j)= atan2(cos(lat1)*sin(lat2)-sin(lat1)*cos(lat2)*cos(long2-long1),sin(long2-long1)*cos(lat2)); % Compute bearing angle
    end

    % bearing
    B(3.6*x2(:,16)<5)=NaN;% set bearing to NaN when speed in km/h is smaller than threshold speed
    Bu=unwrap(B); % unwrap bearing
    B=movmedian(Bu,20,'includenan'); % moving median filter to remove spikes. NaNs are included so that edges are also cropped off
    B=fillmissing(B,'linear'); % linear interpolation of missing data
    B=filtfilt(bf,af,B); % Butterworth filtering

    % bearing rate
    Br=[0;10*diff(B)]; % compute bearing rate by taking subsequent differences
    Br=movmedian(Br,20); % moving median filter to remove spikes.
    Br=filtfilt(bf,af,Br); % Butterworth filtering

    % curvature
    Brc=Br./x2(:,16); % calculate curvature by dividing bearing rate by speed
    Brc=filtfilt(bf,af,Brc); % Butterworth filtering
    Brc(3.6*x2(:,16)<5)=NaN; % remove values when speed is smaller than 5 km/h

    if i==5
        % Figure 6. Calculated bearing before and after filtering, and path curvature, for ﻿Driving ﻿Test 5.

        figure;
        set(gcf,'renderer','painters')
        yyaxis left
        plot(time,rad2deg(Bu),'color',[.5 .5 .5],'Linewidth',3); hold on
        plot(time,rad2deg(B),'r-','Linewidth',3); hold on
        ylabel('Bearing (deg)')
        xlabel('Elapsed time (s)')
        set(gca,'ytick',-900:90:900,'ylim',[-90 360],'ycolor','r')
        grid on
        yyaxis right
        plot(time,Brc,'b-','Linewidth',3)
        ylabel('Curvature (1/m)','color','b')
        set(gca,'ylim',[-0.1 0.1],'ycolor','b')
        legend('Bearing, unfiltered','Bearing, filtered', 'Curvature','location','southeast')
        h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')
        set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])
        set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
    end

    validindices2=find(3.6*x2(:,16)>5); % only values above a certain speed
    validindices1=validindices2;
    validindices1(validindices1>size(x1,1))=[];
    validindices3=find(3.6*x2(:,16)>5 & ~isnan(x2(:,21))); % only values above a certain speed and speed limit data available

    ov=sqrt(x1(:,5).^2+x1(:,6).^2); % xy phone
    M(i,1)=mean(abs(ov(validindices1)),'omitnan');
    ovd=[NaN; abs(10*diff(ov))];
    M(i,2)=mean(ovd(validindices1),'omitnan'); % jerk
    peaks=findpeaks(ov);
    M(i,3)=3600*sum(peaks>3)./(length(validindices1)/10); % number of harsh accelerations per hour of driving

    ov=sqrt(x2(:,7).^2+x2(:,8).^2); % xy GoPro
    M(i,4:6)=std(x2(validindices2,7:9)); % SD Acc (7:9 xyz)
    ovd=[NaN; abs(10*diff(ov))];

    M(i,7) = mean(abs(ov(validindices2)));
    M(i,8) = mean(ovd(validindices2),'omitnan'); % jerk
    M(i,9) = mean(3.6*x2(validindices2,16)); % mean speed
    M(i,10) = mean(3.6*x2(validindices3,16)>x2(validindices3,21)); % speed limit exceedance

    [peaks,peaksi]=findpeaks(ov);
    [~,ia]=intersect(peaksi,validindices2); % retain indices where the car is driving faster than the threshold speed
    peaks=peaks(ia); % only peaks where the car is driving faster than the threshold speed
    M(i,11) = 3600*sum(peaks>3)./(length(validindices2)/10); % number of harsh accelerations

    [peaks,peaksi]=findpeaks(Brc); % find peaks in the curvature; note that length(Brc) == size(x2,1);
    [peaksi,ia]=intersect(peaksi,validindices2); % retain indices where the car is driving faster than the threshold speed

    peaks=abs(peaks(ia)); % only peaks in curvature where the car is driving faster than the threshold speed; take absolute value so that we do not distinguish between left and right curves
    mildcurves=find(peaks>0.005 & peaks < 0.05); % indices of mild curves among all identified peaks
    sharpcurves=find(peaks>=0.05); % indices of sharp curves among all identified peaks
    allcurves=find(peaks>=0.005); % indices of mild curves among all identified peaks

    M(i,12) = 3600*length(mildcurves)./(length(validindices2)/10); % number of mild curves per hour of driving
    M(i,13) = 3600*length(sharpcurves)./(length(validindices2)/10); % number of sharp curves per hour of driving
    M(i,14) = mean(ov(peaksi(ia(mildcurves))));
    M(i,15) = mean(ov(peaksi(ia(sharpcurves))));

    if i==1
        % Figure 3. Jerk based on combined acceleration in the xy-plane.
        figure
        set(gcf,'renderer','painters')

        yyaxis left
        ov=sqrt(x2(:,7).^2+x2(:,8).^2); % xy gopro
        plot(time,[NaN;10*abs(diff(ov))],'k-','Linewidth',3)
        xlabel('Elapsed time (s)')
        ylabel('Jerk (m/s^3)')
        set(gca,'ycolor','k')
        set(gca,'ylim',[0 3])
        grid on
        yyaxis right
        plot(time,ov,'r-','Linewidth',3)
        ylabel('Acceleration (m/s^2)','color','r')
        set(gca,'xlim',[1500 1550],'ycolor','r')
        set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])

        h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')
        set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
        set(gca,'ylim',[0 3])

    end
    vo=sqrt(x2(:,7).^2+x2(:,8).^2); % xy gopro

    if i==5
        % Figure 4. Mean absolute jerk in the xy-plane, vehicle speed recorded using GPS, and the speed limit for a portion of Driving Test 5.
        figure
        set(gcf,'renderer','painters')
        yyaxis left
        plot(time,ovd,'color',[255 165 0]/255,'Linewidth',1)
        xlabel('Elapsed time (s)')
        ylabel('Absolute jerk \it{xy}\rm combined (m/s^3)')
        set(gca,'ycolor',[255 165 0]/255)
        set(gca,'xlim',[800 1500])
        grid on
        set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])
        yyaxis right
        plot(time,3.6*x2(:,16),'k-','Linewidth',3); hold on
        plot(time,x2(:,21),'m-','Linewidth',3)
        set(gca,'ylim',[0 90])
        set(gca,'ycolor','k')
        ylabel('Speed (km/h)','color','k')
        legend('Absolute jerk','Speed','Speed limit','location','northeast')
        h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')
        set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
    end

    SpeedLimitDuration=0.1*histc(x2(validindices2,21),[30 50 60 80 100]); %#ok<HISTC> % number of seconds in each speed limit zone
    SpeedLimitDuration=SpeedLimitDuration./sum(SpeedLimitDuration); % normalize so that the sum equals 1
    SpeedLimitDurationN(i,:)=SpeedLimitDuration;
end

disp('Table 2. Dependent measure scores for the nine driving tests.')
disp(M(:,[1 2 3 7 8 9 10 11 ]))

disp('Table 3. Route statistics, computed from the GoPro GPS data.')
disp([SpeedLimitDurationN M(:,[12 13 14 15])])
%%
iv=1+find(diff(x2(:,1))<0); % find indexes where a new GoPro video starts (typically a driving test consists of 3 GoPro drives, so there will be 2 new videos that start)
for i=1:length(iv) % loop over these indexes
    % create new time vector:
    x2(iv(i):end,1)=x2(iv(i):end,1) + x2(iv(i)-1,1) - x2(iv(i),1);   % from jump:end + previous sample value - current sample value
end
%% Look up frame number
% b=6056; % index
% fn=round((x2(b,1)/1000)*29.97002997003); % compute frame number using CTS
% text_temp=(['Occurs at ' num2str(floor(x2(b,1)/1000/60)) ':' num2str(round(60*(x2(b,1)/1000/60-floor(x2(b,1)/1000/60)),2)) ' (frame number = ' num2str(fn) ').']);
% disp(text_temp)
%% Figure 5. Scatter plot of mean absolute acceleration in the xy-plane (left) and mean absolute jerk in the xy-plane (right) for the GoPro versus the phone
v1=M(:,1); % Macc phone
v2=M(:,7); % Macc GoPro
v3=M(:,2); % Mjerk phone
v4=M(:,8); % Mjerk GoPro
v5=M(:,3); % HarshA phone
v6=M(:,11); % HarshA GoPro
disp('Correlation Macc phone and Macc GoPro')
disp(round(corr(v1,v2),3))
disp('Correlation Mjerk phone and Mjerk GoPro')
disp(round(corr(v3,v4),3))
disp('Correlation HarshA phone and HarshA GoPro')
disp(round(corr(v5,v6),3))

figure
set(gcf,'renderer','painters')
subplot(1,2,1)
hold on;box on
plot([0.6 0.9],[0.6 0.9],'b--','Linewidth',3)
clear scatter1
for i=1:length(v1)
    scatter1(i)=scatter(v1(i),v2(i),700,'markerfacecolor','k','markeredgecolor','k');hold on;grid on %#ok<SAGROW>
    scatter1(i).MarkerFaceAlpha = .3; %#ok<SAGROW>
    text(v1(i),v2(i),num2str(i),'horizontalalignment','center','color','w')
end
set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])
set(gca,'pos',[0.063 0.09 0.42 0.89])
xlabel('Macc phone (m/s^2)')
ylabel('Macc GoPro (m/s^2)')
h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')

subplot(1,2,2)
hold on;box on
plot([0.2 0.45],[0.2 0.45],'b--','Linewidth',3)
clear scatter1
for i=1:length(v1)
    scatter1(i)=scatter(v3(i),v4(i),700,'markerfacecolor','k','markeredgecolor','k');hold on;grid on %#ok<SAGROW>
    scatter1(i).MarkerFaceAlpha = .3; %#ok<SAGROW>
    text(v3(i),v4(i),num2str(i),'horizontalalignment','center','color','w')
end
set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])

set(gca,'pos',[0.563 0.09 0.42 0.89])
xlabel('Mjerk phone (m/s^3)')
ylabel('Mjerk GoPro (m/s^3)')
h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')

%% Process computer-vision data for Figure 8
% https://goo.gl/maps/D8myyKCWY5rV7Ucm7
% Google Maps link
% We measured that 14 strips on the road correspond to 59.66 m (one stripe is 4.26 m)
% We measured, based on frame 36271, that the bounding box width is 61 pixels and the car is 6 stripes (25.56 m) away
% We measured, based on frame 36841, that the bounding box width is 39 pixels and the car is 9 stripes (47.34 m) away
% Headway distance in meters is then assumed to be: 1560/(bounding box width)

makevideo=0; % if set to 1, then make a video of the driving test with an overlay of bounding boxes

if makevideo==1
    vidObj=VideoReader('..\CBR - TU Delft\Dataset -- scenario''s - acceleration data - videos - event logs\Testdag 5 (11-4-2022)\Rit 5.2\Rit 5.2.mp4');
    try
        close(vidObj)
    catch error
    end
end

load YOLOData_5.mat
V=1:30:47083; %vidObj.NumFrames;

if makevideo==1
    try
        close(vr) % close video in case it still happens to be open
    catch error
    end
    vr = VideoWriter('YoloDrivingTest52b.mp4', 'Mpeg-4'); %#ok<*TNMLP>
    vr.Quality=100;vr.FrameRate=25;
    open(vr);
end
Xi=NaN(size(BB,2),3);

for j=V % loop over all video frames
    if makevideo==1
        load(['images\DrivingTest_5.2_second_image' num2str(j) '.mat']); % same as Data_acc_gopro but with rotated xyz
    end
    labels=LL{ceil(j/30)};
    bboxes=BB{ceil(j/30)}; % (bounding boxes [x,y,w,h], classification scores, and labels)

    scores=SS{ceil(j/30)};
    if ~isempty(bboxes) % if there are any bounding boxes
        highwidth=find(bboxes(:,3)>1800); % find instances which are very high width (the hood of the car itself is detected)
        bboxes(highwidth,:)=[]; % remove these instances
        labels(highwidth,:)=[]; % remove these instances
        scores(highwidth,:)=[]; % remove these instances
    end

    labelsn=NaN(size(scores));
    for l=1:length(labels)
        labelsn(l)=find(strcmp(Labels,char(labels(l))));
    end

    if makevideo==1
        if ~isempty(labels)
            annotations = string(labels);
            im = insertObjectAnnotation(im, 'rectangle', bboxes, annotations,'fontsize',30);
        end
        %    figure
        %   imshow(im)
        writeVideo(vr,im);
    end

    id1=[];id2=[];
    if ~isempty(bboxes)
        bboxes(:,5)=bboxes(:,1)+bboxes(:,3); % right x
        bboxes(:,6)=bboxes(:,2)+bboxes(:,4); % bottom y
        bboxes(:,7)=(bboxes(:,1)+bboxes(:,5))/2; % middle x

        id1=find(ismember(labelsn,1) & bboxes(:,4)>150 & bboxes(:,7)>995-200 & bboxes(:,7)<995+200); % indexes of 'person', height greater than 150 pixels
        id2=find(ismember(labelsn,[3 6 7 8]) & bboxes(:,7)>970 & bboxes(:,7)<1020); % indexes of car, bus, train, or truck
    end

    if ~isempty(id1) % if there is at least one person
        Xi(ceil(j/30),1)=length(id1); % store number of persons
    else
        Xi(ceil(j/30),1)=0;
    end

    if ~isempty(id2) % if there is at least one car, bus, train, or truck
        Xi(ceil(j/30),2)=1560/bboxes(id2(1),3); % store headway to vehicle in front, estimated from the width of the first bounding box
    else
        Xi(ceil(j/30),2)=NaN;
    end

end

if makevideo==1
    try
        close(vr) % close video in case it still happens to be open
    catch error
    end
end

%% Figure 8. Number of persons (top) and headway to the vehicle in front (bottom) measured at a frequency of 1 Hz for a portion of ﻿Driving ﻿Test 5

figure
set(gcf,'renderer','painters')

tiledlayout(2,1)
nexttile

[xb,yb]=stairs(Xi(:,1));
plot(xb,yb,'k-','linewidth',2,'markerfacecolor','k')

set(gca,'xlim',[800 1500])
xlabel('Elapsed time (s)')
ylabel('Person count')
grid on
set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])
h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])

nexttile
plot(linspace(1,length(Xi),length(Xi)),fillmissing(Xi(:,2),'spline','maxgap',5),'k-','Linewidth',2)
xlabel('Elapsed time (s)')
ylabel('Headway (m)')
set(gca,'xlim',[800 1500])
grid on
set(gca,'GridLineWidth',1,'GridColor',[0.15 0.15 0.15])
h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
