% This code creates Figures 6 and 7 (with subfigures) of the paper 'Departure Time Choice and Bottleneck Congestion with Automated Vehicles: Role of On-board Activities'.
% The running time of the entire code is about 20 minutes.
% When creating separate figures, start by once executing the 'General section' (just
% below), and then proceed to the section for desired figure.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% GENERAL SECTION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
syms t;
t_star = 50; % preferred arrival time
omega = 100; % end of morning interval
N=200; %# of travellers
s=5; % capacity of the bottleneck

gran = 0.1; % granularity of computation and graphs
% reducing granularity (e.g. gran = 1) will reduce the computation time, but
% introduce minor offset for some graphs due to rounding errors 

% Scheduling parameters:
alpha = 2;
beta = 1;
gamma = 4;
eh = 0.3;
ew = 0.3;
low = t_star-gamma*N/((beta+gamma)*s); % congestion start
up = t_star+beta*N/((beta+gamma)*s); % congestion end

% Conventional vehicle
wave_0 = t_star-beta*gamma*N/(alpha*(beta+gamma)*s);% undelayed departure time
r_0 = piecewise(low<=t<=wave_0, beta/(alpha-beta), wave_0<t<=up, -gamma/(alpha+gamma), 0); % flow rate
int_r_0 = @(u) int(r_0,t,0,u); % number of departed travellers until time u - a function
int_r_array_0 = arrayfun(int_r_0,[0:gran:omega]); % number of departed travellers, at time points of an interval

% Home AV
wave_1 = t_star-beta*gamma*N/(alpha*(1-eh)*(beta+gamma)*s);
r_1 = piecewise(low<=t<=wave_1, beta/(alpha*(1-eh)-beta), wave_1<t<=up, -gamma/(alpha*(1-eh)+gamma), 0);
int_r_1 = @(u) int(r_1,t,0,u);
int_r_array_1 = arrayfun(int_r_1,[0:gran:omega]);

% Universal AV
wave_2 = wave_1;
r_2 = piecewise(low<=t<=wave_2, beta/(alpha*(1-eh)-beta), wave_2<t<=t_star, alpha*(1-eh)/((alpha+gamma)*(1-ew))-1, t_star<t<=up, -gamma/((alpha+gamma)*(1-ew)), 0);
int_r_2 = @(u) int(r_2,t,0,u);
int_r_array_2 = arrayfun(int_r_2,[0:gran:omega]);

% Work AV
wave_3 = t_star-beta*gamma*N/((alpha*(1-ew)+beta*ew)*(beta+gamma)*s);
r_3 = piecewise(low<=t<=wave_3, beta/((alpha-beta)*(1-ew)), wave_3<t<=t_star, (alpha*(1-ew)+beta*ew)/((alpha+gamma)*(1-ew))-1, t_star<t<=up, (alpha*(1-ew)-gamma*ew)/((alpha+gamma)*(1-ew))-1, 0);
int_r_3 = @(u) int(r_3,t,0,u);
int_r_array_3 = arrayfun(int_r_3,[0:gran:omega]);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% END OF GENERAL SECTION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% FIGURE 6 - plot of ALL congestion graphs %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
figure(6);
plot([0:gran:100], int_r_array_0,'LineWidth',1,'Color','black')
hold on

% Frame 
xlim([15 60]);
xlabel('Departure time');
ylim([-3 30]);
ylabel('Queueing time');

% Supporting (faint) lines
plot([27.25 50], [22.75 0],'-','LineWidth',0.3,'Color', [0.8 0.8 0.8],'HandleVisibility','off');
plot([0 100], [0 0],'-','LineWidth',1,'Color', [0.6 0.6 0.6],'HandleVisibility','off');
plot([50 50], [-1 1],'-','LineWidth',1,'Color', [0.6 0.6 0.6],'HandleVisibility','off');
text(50.2,-1,'t*');

% Main graphs
plot([0:gran:100], int_r_array_1,'--','LineWidth',1.5,'Color','black');
plot([0:0.5:100], int_r_array_2(1:5:end),'o','MarkerSize',3,'LineWidth',1,'Color','black');
plot([0:gran:100], int_r_array_3,':','LineWidth',1.5,'Color','black');

hold off

legend({'Conventional vehicle','Home AV','Universal AV','Work AV'},'Location','northeast');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% END OF FIGURE 6 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%% FIGURE 7 (top left) - mixture of CONVENTIONAL vehicles and HOME AVs %%%%%%%%%%%%%%%%%%%%%
syms t tA tC;
syms lowx upx wavex;
e = 0.3; % For mixture graphs a single efficiency parameter is used. It applies to either eh or ew, or both
% The original graphs are taken from General section (using eh, ew defined therein).

% Finding the 2 points on x-axis, where both graphs cross: tA and tC
% # of conventional vehicle users departing before tA and after tC. Flow rate * intervals
eq1 = (alpha/(alpha-beta))*s*(tA-low)+(alpha/(alpha+gamma))*s*(up-tC);
% Queue lengths at tA and tC. Queue formula using conventional preferences
queA = beta*(tA-low)/(alpha-beta);
queC = beta*(wave_0-low)/(alpha-beta)+((alpha/(alpha+gamma))-1)*(tC-wave_0);
% Home AV costs (sum of travel time and schedule delay costs) at tA and tC should be equal. eq2 defines their difference
eq2 = queA*(alpha*(1-e)-beta)+(t_star-tA)*beta-queC*(alpha*(1-e)+gamma)-(tC-t_star)*gamma;

for share = 0.25:0.25:0.75 %Share of Home AVs
    syms lowx upx wavex;
    [tA,tC] = solve(eq1 == (200 * (1 - share)), eq2 == 0);
    % Convert results to numeric values
    tAx{share/0.25} = double(tA);
    tCx{share/0.25} = double(tC);
    queA = beta*(tAx{share/0.25}-low)/((alpha-beta)); %same formula as above, but here used for a numerical value
    queC = beta*(wave_0-low)/(alpha-beta)+(alpha/(alpha+gamma)-1)*(tCx{share/0.25}-wave_0);

    % Calculating parameters for the modified Home preferences: lowx (start of graph) ,upx (end of graph), wavex (undelayed time).
    % Queue at tA according to Home preferences
    eq3 = (beta/(alpha*(1-e)-beta))*(tAx{share/0.25}-lowx);
    % Queue at tC according to Home preferences (building up from the right)
    eq4 = (gamma/(alpha*(1-e)+gamma))*(upx - tCx{share/0.25});
    % Queue at undelayed time: equal when calculated from left and right
    eq5 = (wavex-lowx)*(beta/(alpha*(1-e)-beta))-(upx-wavex)*(gamma/(alpha*(1-e)+gamma));
    [lowx,upx,wavex] = solve(eq3 == queA, eq4 == queC, eq5 == 0);
    lowx = double(lowx);
    upx = double(upx);
    wavex = double(wavex);

    % Modified Home vector
    r_x = piecewise(low<=t<=tAx{share/0.25}, beta/(alpha-beta), tAx{share/0.25}<=t<=wavex, beta/(alpha*(1-e)-beta), wavex<t<=tCx{share/0.25}, -gamma/(alpha*(1-e)+gamma), tCx{share/0.25}<t<=up, -gamma/(alpha+gamma), 0);
    int_r_x = @(u) int(r_x,t,0,u);
    tef{share/0.25} = arrayfun(int_r_x,[0:gran:omega]);

end

%%%%%% PLOT %%%%%%
figure(71);
h(1) = plot([0:gran:100], int_r_array_0,'LineWidth',1.5,'Color',[0.85 0.85 0.85]);
hold on

% Frame
xlim([15 60]);
xlabel('Departure time');
ylim([-3 30]);
ylabel('Queueing time');

% Supporting (faint) lines
plot([0 100], [0 0],'-','LineWidth',1,'Color', [0.6 0.6 0.6],'HandleVisibility','off');
plot([50 50], [-1 1],'-','LineWidth',1,'Color', [0.6 0.6 0.6],'HandleVisibility','off');
text(50.2,-1,'t*');
plot([27.5 50], [22.5 0],'-','LineWidth',0.3,'Color', [0.8 0.8 0.8],'HandleVisibility','off');

% Draw the TEF with medium Home AV shares
for i=1:3
    plot([0:gran:tAx{i}], tef{i}(1:(tAx{i}/gran + 1)),[tCx{i}:gran:100], tef{i}((tCx{i}/gran + 1):(100/gran + 1)),'LineWidth',1.5,'Color',[1 1 1] - i/5 - 0.1);
    h(i+1) = plot([tAx{i}:gran:tCx{i}], tef{i}((tAx{i}/gran + 1):(tCx{i}/gran + 1)),'--','LineWidth',1.5,'Color',[1 1 1] - i/5 - 0.1);
end
    
% Draw the Home AV graph
plot([0:gran:low], zeros(length([0:gran:low]),1),[up:gran:100], zeros(length([up:gran:100]),1),'LineWidth',1.75,'Color',[0 0 0]);
h(5) = plot([low:gran:up], int_r_array_1((low/gran + 1):(up/gran+1)),'--','LineWidth',1.75,'Color',[0 0 0]);

hold off

legend(h([1:5]),{'Conventional vehicle','Home AV 25%','Home AV 50%','Home AV 75%','Home AV 100%'},'Location','northeast');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% END OF FIGURE 7 (top left) %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%% FIGURE 7 (top right) - mixture of CONVENTIONAL vehicles and WORK AVs %%%%%%%%%%%%%%%%%%%%%
syms t tA tC;
syms lowx upx wavex;
e = 0.3;

% Finding the 2 points on x-axis, where both graphs cross: tA and tC
% # of conventional vehicle users departing before tA and after tC. Flow rate * intervals
eq1 = (alpha/(alpha-beta))*s*(tA-low)+(alpha/(alpha+gamma))*s*(up-tC);
% Queue lengths at tA and tC. Queue formula using conventional preferences
queA = beta*(tA-low)/(alpha-beta);
queC = beta*(wave_0-low)/(alpha-beta)+((alpha/(alpha+gamma))-1)*(tC-wave_0);
% Work AV costs (sum of travel time and schedule delay costs) at tA and tC should be equal. eq2 defines their difference
costsA = queA*((alpha-beta)*(1-e))+(t_star-tA)*beta;
costsC = queC*((alpha+gamma)*(1-e))+(tC-t_star)*gamma;
eq2 = costsA-costsC;

for share = 0.25:0.25:0.75 %Share of Work AVs
    syms lowx upx wavex;
    [tA,tC] = solve(eq1 == (200 * (1 - share)), eq2 == 0);
    % Convert results to numeric values
    tAx{share/0.25} = double(tA);
    tCx{share/0.25} = double(tC);
    queA = beta*(tAx{share/0.25}-low)/((alpha-beta));
    queC = beta*(wave_0-low)/(alpha-beta)+(alpha/(alpha+gamma)-1)*(tCx{share/0.25}-wave_0);

    % Calculating parameters for the modified Work preferences: lowx (start of graph) ,upx (end of graph), wavex (undelayed time).
    % Queue at tA according to Work preferences
    eq3 = (beta/((alpha-beta)*(1-e)))*(tAx{share/0.25}-lowx);
    % Queue at tC according to Work preferences (building up from the right)
    eq4 = -(alpha/(alpha+gamma)-gamma*e/((alpha+gamma)*(1-e))-1)*(upx - tCx{share/0.25});
    % Queue at undelayed time: equal when calculated from left and right
    eq5 = (wavex-lowx)*(beta/((alpha-beta)*(1-e)))+(upx-t_star)*(alpha/(alpha+gamma)-gamma*e/((alpha+gamma)*(1-e))-1)+(t_star-wavex)*(alpha/(alpha+gamma)+beta*e/((alpha+gamma)*(1-e))-1);
    [lowx,upx,wavex] = solve(eq3 == queA, eq4 == queC, eq5 == 0);
    lowx = double(lowx);
    upx = double(upx);
    wavex = double(wavex);


    % Modified Work vector
    r_x = piecewise(low<=t<=tAx{share/0.25}, beta/(alpha-beta), tAx{share/0.25}<=t<=wavex, beta/((alpha-beta)*(1-e)), wavex<t<=t_star, alpha/(alpha+gamma)+beta*e/((alpha+gamma)*(1-e))-1, t_star<t<=tCx{share/0.25}, alpha/(alpha+gamma)-gamma*e/((alpha+gamma)*(1-e))-1, tCx{share/0.25}<t<=up, -gamma/(alpha+gamma), 0);
    int_r_x = @(u) int(r_x,t,0,u);
    tef{share/0.25} = arrayfun(int_r_x,[0:gran:omega]);

end


%%%%%% PLOT %%%%%%
figure(72);
h(1) = plot([0:gran:100], int_r_array_0,'LineWidth',1.5,'Color',[0.85 0.85 0.85]);
hold on

% Frame
xlim([15 60]);
xlabel('Departure time');
ylim([-3 30]);
ylabel('Queueing time');


% Supporting (faint) lines
plot([0 100], [0 0],'-','LineWidth',1,'Color', [0.6 0.6 0.6],'HandleVisibility','off');
plot([50 50], [-1 1],'-','LineWidth',1,'Color', [0.6 0.6 0.6],'HandleVisibility','off');
text(50.2,-1,'t*');
plot([31.5 50], [18.5 0],'-','LineWidth',0.3,'Color', [0.8 0.8 0.8],'HandleVisibility','off');

% Draw the TEF with medium Work AV shares
for i=1:3
    plot([0:gran:tAx{i}], tef{i}(1:(round(tAx{i}/gran) + 1)),[tCx{i}:gran:100], tef{i}(round((tCx{i}/gran + 1)):(100/gran + 1)),'LineWidth',1.5,'Color',[1 1 1] - i/5 - 0.1);
    h(i+1) = plot([tAx{i}:gran:tCx{i}], tef{i}(round((tAx{i}/gran + 1)):round((tCx{i}/gran + 1))),':','LineWidth',1.5,'Color',[1 1 1] - i/5 - 0.1);
end

% Draw the Work AV graph
plot([0:gran:low], zeros(length([0:gran:low]),1),[up:gran:100], zeros(length([up:gran:100]),1),'LineWidth',1.75,'Color',[0 0 0]);
h(5) = plot([low:gran:up], int_r_array_3((low/gran + 1):(up/gran+1)),':','LineWidth',1.75,'Color',[0 0 0]);

hold off

legend(h([1:5]),{'Conventional vehicle','Work AV 25%','Work AV 50%','Work AV 75%','Work AV 100%'},'Location','northeast');


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% END OF FIGURE 7 (top right) %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%% FIGURE 7 (bottom) - mixture of HOME vehicles and WORK AVs %%%%%%%%%%%%%%%%%%%%%%%%%%
syms t tA tC;
syms lowx upx wavex;
e = 0.3;

% Finding the 2 points on x-axis, where both graphs cross: tA and tC
% # of Work AV users departing before tA and after tC. Flow rate * intervals.
% 2 segments of flow rate need to be considered after tC
eq1 = (1+ beta/((alpha-beta)*(1-e)))*s*(tA-low)+(alpha/(alpha+gamma)+beta*e/((alpha+gamma)*(1-e)))*s*(t_star-tC)+(alpha/(alpha+gamma)-gamma*e/((alpha+gamma)*(1-e)))*s*(up-t_star);
% Queue lengths at tA and tC. Queue formula using Work preferences
queA = beta*(tA-low)/((alpha-beta)*(1-e));
queC = beta*(wave_3-low)/((alpha-beta)*(1-e))+(alpha/(alpha+gamma)+beta*e/((alpha+gamma)*(1-e))-1)*(tC-wave_3);
% Home AV costs (sum of travel time and schedule delay costs) at tA and tC should be equal. eq2 defines their difference
eq2 = queA*(alpha*(1-e)-beta)+(t_star-tA)*beta-queC*(alpha*(1-e)+gamma)-(tC-t_star)*gamma;

for share = 0.25:0.25:0.75 %Share of Work AVs
    syms lowx upx wavex;
    [tA,tC] = solve(eq1 == (200 * (1 - share)), eq2 == 0);
    % Convert results to numeric values
    tAx{share/0.25} = double(tA);
    tCx{share/0.25} = double(tC);
    queA = beta*(tAx{share/0.25}-low)/((alpha-beta)*(1-e));
    queC = beta*(wave_3-low)/((alpha-beta)*(1-e))+(alpha/(alpha+gamma)+beta*e/((alpha+gamma)*(1-e))-1)*(tCx{share/0.25}-wave_3);

    % Calculating parameters for the modified Home preferences: lowx (start of graph) ,upx (end of graph), wavex (undelayed time).
    % Queue at tA according to Home preferences
    eq3 = (beta/((alpha*(1-e)-beta)))*(tAx{share/0.25}-lowx);
    % Queue at tC according to Home preferences (building up from the right)
    eq4 = (gamma/(alpha*(1-e)+gamma))*(upx - tCx{share/0.25});
    % Queue at undelayed time: equal when calculated from left and right
    eq5 = (wavex-lowx)*(beta/(alpha*(1-e)-beta))-(upx-wavex)*(gamma/(alpha*(1-e)+gamma));
    [lowx,upx,wavex] = solve(eq3 == queA, eq4 == queC, eq5 == 0);
    lowx = double(lowx);
    upx = double(upx);
    wavex = double(wavex);

    % Modified Home vector; start and end from work graph
    r_x = piecewise(low<=t<=tAx{share/0.25}, beta/((alpha-beta)*(1-e)), tAx{share/0.25}<=t<=wavex, beta/(alpha*(1-e)-beta), wavex<t<=tCx{share/0.25}, -gamma/(alpha*(1-e)+gamma), tCx{share/0.25}<t<=t_star, (alpha*(1-e)+beta*e)/((alpha+gamma)*(1-e))-1, t_star<t<=up, (alpha*(1-e)-gamma*e)/((alpha+gamma)*(1-e))-1, 0);
    int_r_x = @(u) int(r_x,t,0,u);
    tef{share/0.25} = arrayfun(int_r_x,[0:gran:omega]);

end


%%%%%% PLOT %%%%%%
figure(73);
h(1) = plot([0:gran:100], int_r_array_3,':','LineWidth',1.5,'Color',[0.85 0.85 0.85]);
hold on

% Frame
xlim([15 60]);
xlabel('Departure time');
ylim([-3 30]);
ylabel('Queueing time');

% Supporting (faint) lines
plot([0 100], [0 0],'-','LineWidth',1,'Color', [0.6 0.6 0.6],'HandleVisibility','off');
plot([50 50], [-1 1],'-','LineWidth',1,'Color', [0.6 0.6 0.6],'HandleVisibility','off');
text(50.2,-1,'t*');
plot([27.25 50], [22.75 0],'-','LineWidth',0.3,'Color', [0.8 0.8 0.8],'HandleVisibility','off'); %line connecting the t-star

% Draw the TEF with medium Home AV shares
for i=1:3
    plot([low:gran:round(tAx{i},1)], tef{i}((low/gran + 1):(round(tAx{i}/gran) + 1)),':',[round(tCx{i},1):gran:up], tef{i}(round((tCx{i}/gran + 1)):(up/gran + 1)),':','LineWidth',1.5,'Color',[1 1 1] - i/5 - 0.1);
    h(i+1) = plot([round(tAx{i},1):gran:round(tCx{i},1)], tef{i}(round((tAx{i}/gran + 1)):round((tCx{i}/gran + 1))),'--','LineWidth',1.5,'Color',[1 1 1] - i/5 - 0.1);
end

% Draw the Home AV graph
plot([0:gran:low], zeros(length([0:gran:low]),1),[up:gran:100], zeros(length([up:gran:100]),1),'LineWidth',1.75,'Color',[0 0 0]);
h(5) = plot([low:gran:up], int_r_array_1((low/gran + 1):(up/gran+1)),'--','LineWidth',1.75,'Color',[0 0 0]);


hold off

legend(h([1:5]),{'Work AV','Home AV 25%','Home AV 50%','Home AV 75%','Home AV 100%'},'Location','northeast');


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% END OF FIGURE 7 (bottom) %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%