clear all
clc
set(0,'defaultfigurecolor',[1 1 1])

%% INFO

% Read and process friction data

% Version: 2-12-2020
% Matlab version: R2020a

% Cleaned to obtain figure 4b (overshoot ratio) in ESAFORM 2021

%% initialisation
figure(1), clf(1)
serie_range = [2 3];
series{1} = [1 2 3 4 5 6 7 8 9 10 11]; % experiments in range (relates to unique number of specimen)
series{2} = [1 2 4 5 6 7 8 9 10 11 12];

%% Go through the measurements
V_mean_vec = []; % store values in loop
tau_mean_vec = []; % store values in loop
tau_peak_vec = []; % store values in loop

for serie_i = 1:length(serie_range)
    
    serie = serie_range(serie_i);
    range_meas = series{serie_i};
    
    for meas_i = 1:length(range_meas)

        meas = range_meas(meas_i);

        %% Load data
        file_find = dir(['*_',num2str(serie),'-',num2str(meas),'.lvm']);
        fid = fopen(file_find.name);
        C = textscan(fid,'%f');
        data = zeros(int32(length(C{1})/13),13);
        for i = 0:length(C{1})-1
            i_row = 1+floor(i/13);
            i_col = 1+mod(i,13);
            data(int32(i_row),int32(i_col)) = C{1}(i+1);
        end
        fprintf(['File: ', file_find.name,'\n'])
        fclose(fid);

        d = data(1,13) - data(:,13); % displacement [mm]
        force_corr = mean(data(end-5:end,6)); % end value (end of experiment)
        f_pull = data(:,6) - force_corr; % pull force, corrected for end value [N]
        index = data(:,1);
        t = data(:,2) - data(1,2);

        %% Analysis

        %% index start/end using displacement data, mean velocity, shear stress
        
        % velocity from file name
        name_u = find(file_find.name=='u');
        name_u_end = name_u + 1;
        while file_find.name(name_u_end) ~= '_'
            name_u_end = name_u_end + 1;
        end
        vel = str2double(file_find.name(name_u+1:name_u_end-1)); % velocity from file name

        % determine window for smoothening based on velocity form file name
        if vel <= 1
            window = 100;
        elseif vel <= 3
            window = 75;
        elseif vel <= 5
            window = 50;
        elseif vel <= 8
            window = 25;
        elseif vel <= 15
            window = 15;
        else
            window = 15;
        end

        % smoothening data
        d_filt = smoothdata(d,'gaussian',window); % filter data met moving median filter

        v_vec = [0]; % velocity vector
        a_vec = [0]; % acceleration vector
        for i = 2:length(t)-1
            dt = t(i+1) - t(i);
            v = (d_filt(i+1)-d_filt(i-1))/(2*dt);
            a = (d_filt(i+1)-2*d_filt(i)+d_filt(i-1))/(dt^2);
            v_vec(i) = v;
            a_vec(i) = a;
        end
        v_vec(length(t)) = v;
        a_vec(length(t)) = a;
        
        % determine global start and end
        i_glob = 1;
        while d(i_glob) < 1 % deformation below 1 mm
            i_glob = i_glob + 1;
        end
       
        i_glob_end = i_glob;
        while t(i_glob_end) < t(i_glob) + d(end)/vel*60 % find index for time after full displacement
            i_glob_end = i_glob_end + 1;
        end
        idx_start = find(a_vec(1:i_glob_end)==max(a_vec(1:i_glob_end))); % start index, using max acceleration and global end index
        idx_end = find(a_vec(1:i_glob_end)==min(a_vec(1:i_glob_end))); % end index, using min acceleration and global end index

        % mean velocity
        V_mean = (d(idx_end)-d(idx_start))/(t(idx_end)-t(idx_start)); % mm/s
        V_mean_vec = [V_mean_vec, V_mean];
        
        % shear stress
        tau = f_pull/2500/2*1e3; % [N/mm] -> MPa, times 1e3 results in [kPa] (division by 2, as 2 slip interfaces are present)
        
        % determine range for mean shear stress
        ss_end = min(d(idx_end),10);  % max 10 mm slip length
        ss_start = ss_end - 2;  % 2 mm steady state region (~start at 8 mm)

        ss_idx_start = idx_start;
        while d(ss_idx_start) < ss_start
            ss_idx_start = ss_idx_start + 1;
        end
        ss_idx_end = ss_idx_start;
        while d(ss_idx_end) < ss_end
            ss_idx_end = ss_idx_end + 1;
        end
        tau_mean = mean(tau(ss_idx_start:ss_idx_end));
        
        % store values
        tau_mean_vec = [tau_mean_vec, tau_mean];
        idx_peak = find(tau(idx_start:ss_idx_end)==max(tau(idx_start:ss_idx_end)))+idx_start-1;
        tau_peak_vec = [tau_peak_vec, tau(idx_peak)];     
        
    end
end

%% Outside of the loop

% plot normalized peak shear stress with V
figure(1), hold on

i_store = 1;
marker_spec = ['o','o','o','o','o','o','o','o'];
for serie_i = 1:length(serie_range)
    
    serie = serie_range(serie_i);
    range_meas = series{serie_i};
    marker = marker_spec(serie_i);    
    
    for i_meas = 1:length(range_meas)
        plt = plot(V_mean_vec(i_store),(tau_peak_vec(i_store))/tau_mean_vec(i_store),marker,'color','black','markersize',8); %,'displayname',['Meas: ',num2str(serie),'-', num2str(range_meas(i_meas))]
        set( get( get( plt, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
        i_store = i_store + 1;
    end
end

xlabel('Velocity, $V$ [mm/s]','interpreter','Latex','fontsize',18)
ylabel('Overshoot ratio, $\tau_{p}/\tau_{\infty}$ [-]','interpreter','Latex','fontsize',18)

box on
grid on
axis([1e-2 0.8e1 0.5 3])
set(gca,'XScale', 'log', 'fontsize',18)


