clear all
clc
set(0,'defaultfigurecolor',[1 1 1])

%% INFO

% Read and process friction data

% Version: 2-12-2020
% Matlab version: R2020a

% Cleaned to obtain figure 1 (schematic pull force vs displacement) in
% ESAFROM 2021 conference paper


%% initialisation
figure(1), clf(1)
serie_range = [2]; 
series{1} = [5]; % experiments in range (relates to unique number of specimen)
mod_coeff = 0.90; %threshold for modulus fit, end; max difference local and average modulus

%% Go through the measurements

for serie_i = 1:length(serie_range)
    
    serie = serie_range(serie_i);
    range_meas = series{serie_i};
    
    for meas_i = 1:length(range_meas)

        meas = range_meas(meas_i);

        %% Load data
        file_find = dir(['*_',num2str(serie),'-',num2str(meas),'.lvm']);
        fid = fopen(file_find.name);
        C = textscan(fid,'%f');
        data = zeros(int32(length(C{1})/13),13);
        for i = 0:length(C{1})-1
            i_row = 1+floor(i/13);
            i_col = 1+mod(i,13);
            data(int32(i_row),int32(i_col)) = C{1}(i+1);
        end
        fprintf(['File: ', file_find.name,'\n'])
        fclose(fid);

        d = data(1,13) - data(:,13); % displacement [mm]
        force_corr = mean(data(end-5:end,6));  % end value (end of experiment)
        f_pull = data(:,6) - force_corr; % pull force, corrected for end value [N]
        t = data(:,2) - data(1,2); % time [s]

        %% Analysis

        %% index start/end using displacement data, mean velocity, shear stress
        
        % velocity from file name
        name_u = find(file_find.name=='u'); 
        name_u_end = name_u + 1; 
        while file_find.name(name_u_end) ~= '_'
            name_u_end = name_u_end + 1;
        end
        vel = str2double(file_find.name(name_u+1:name_u_end-1)); % velocity from file name
        
        % determine window for smoothening based on velocity form file name
        if vel <= 1
            window = 100;
        elseif vel <= 3
            window = 75;
        elseif vel <= 5
            window = 50;
        elseif vel <= 8
            window = 25;
        elseif vel <= 15
            window = 15;
        else
            window = 15;
        end
        
        % smoothening data
        d_filt = smoothdata(d,'gaussian',window); % filter data with moving median filter

        v_vec = [0]; % velocity vector
        a_vec = [0]; % acceleration vector
        for i = 2:length(t)-1
            dt = t(i+1) - t(i);
            v = (d_filt(i+1)-d_filt(i-1))/(2*dt);
            a = (d_filt(i+1)-2*d_filt(i)+d_filt(i-1))/(dt^2);
            v_vec(i) = v;
            a_vec(i) = a;
        end
        v_vec(length(t)) = v;
        a_vec(length(t)) = a;
        
        % determine global start and end
        i_glob = 1;
        while d(i_glob) < 1  % deformation below 1 mm
            i_glob = i_glob + 1;
        end
        
        i_glob_end = i_glob;
        while t(i_glob_end) < t(i_glob) + d(end)/vel*60 % find index for time after full displacement
            i_glob_end = i_glob_end + 1;
        end
        idx_start = find(a_vec(1:i_glob_end)==max(a_vec(1:i_glob_end))); % start index, using max acceleration and global end index
        idx_end = find(a_vec(1:i_glob_end)==min(a_vec(1:i_glob_end))); % end index, using min acceleration and global end index

        % mean velocity
        V_mean = (d(idx_end)-d(idx_start))/(t(idx_end)-t(idx_start)); % [mm/s]

        % shear stress
        tau = f_pull/2500/2*1e3; % [N/mm] -> MPa, times 1e3 results in [kPa] (division by 2, as 2 slip interfaces are present)
        
        %% modulus fit
        
        % filter displacement data (multiple inputs: averaging)
        d_red = []; % reduced displacement vector
        tau_red = []; % reduced shear stress vector
        idx_avg_end = idx_start; % find index for range for modulus fit
        while d(idx_avg_end) < 3  % stop after 3 mm
            idx_avg_end = idx_avg_end + 1;
        end
        d_prev = d(idx_start);
        i_count = 1;
        i_red = 1;
        tau_tot = tau(idx_start);
        for i = 1:(idx_avg_end-idx_start)  
            if d(i+idx_start) <= d_prev  % data for averaging (because same displacmeent input)
                tau_tot = tau_tot + tau(i+idx_start);
                i_count = i_count + 1;
            else
                d_red(i_red) = d_prev;  % include data point in reduced vector
                tau_red(i_red) = tau_tot/i_count;  % use average shear stress value
                d_prev = d(i+idx_start);
                i_count = 1;  % start again with counting  
                tau_tot = tau(i+idx_start); % tau_tot equals subsequent data point
                i_red = i_red + 1;  % index for reduced vector
            end
        end

        % derivative tau wrt displacement
        dtau_dd_first = [0 0];  % first derivative of tua to d (d tau / d d)
        for i_der = 3:length(tau_red)-2
            dtau_dd_first(i_der) = (-0.5*tau_red(i_der-1)+0.5*tau_red(i_der+1))/(0.5*(d_red(i_der)-d_red(i_der-1))+0.5*(d_red(i_der+1)-d_red(i_der)));
        end
        dtau_dd_first = smoothdata([dtau_dd_first, 0, 0],'gaussian',window);

        idx_mod_start = 1;
        while dtau_dd_first(idx_mod_start) < mod_coeff*max(dtau_dd_first)  % use fraction of max derivative for start range fit modulus
            idx_mod_start = idx_mod_start + 1;
        end
        idx_mod_start = idx_mod_start - 1;
        if idx_mod_start < 1
            idx_mod_start = 1;
        end
        idx_mod_end = find(dtau_dd_first==max(dtau_dd_first)); % start after the peak
        while dtau_dd_first(idx_mod_end) > mod_coeff*max(dtau_dd_first) % use fraction of max derivative for end range
            idx_mod_end = idx_mod_end + 1;
        end

        % modulus fit in selected range
        tau_mod_fit = tau_red(idx_mod_start:idx_mod_end);
        d_mod_fit = d_red(idx_mod_start:idx_mod_end);
        p = polyfit(d_mod_fit,tau_mod_fit,1);

        % correction for displacement
        d_corr = -p(2)/p(1);
        idx_corr = idx_start;
        while d(idx_corr) < d_corr
            idx_corr = idx_corr + 1;
        end
        d = d - d_corr;
        
        %% Show data

        figure(1), hold on
        plot(d,f_pull,'-k','displayname',['Meas: ',num2str(serie),'-', num2str(meas), ' V: ', num2str(V_mean*60,'%0.f'),' mm/min'])
        xlabel('Displacement, $d$ [mm]','interpreter','latex','fontsize',20)
        ylabel('Shear stress, $\tau$','interpreter','latex','fontsize',20) %NOTE: pull force is plotted, but its a schematic, so no problem to use 'shear stress' instead of pull force
        text(1.1,215,{['Shear stress']...
            [' overshoot, $\tau_p$']},'interpreter','latex','fontsize',20)
        text(5.5,120,{['Long-time']...
            ['shear stress, $\tau_\infty$']},'interpreter','latex','fontsize',20)
        plot(0.9637,203.2,'k.','markersize',36)
        plot([7.5, 9],[151.3, 151.3],'-k','linewidth',8)

    end
end

figure(1), axis([0 9 0 225]), set(gca,'YTick',[],'fontsize',20)

