clear all
clc
set(0,'defaultfigurecolor',[1 1 1])

%% INFO

% Read and process friction data

% Version: 2-12-2020
% Matlab version: R2020a

% Cleaned to obtain figure 5c (corrected long-term with V) in ESAFORM 2021


%% initialisation
figure(1), clf(1)
figure(2), clf(2)
figure(3), clf(3)
eta_0_cross = 589; % from fit on data (PEEK 150P), see fig5a_viscosity_curves.mat
tau_0_cross = 5.9887e4; % from fit on data (PEEK 150P), see fig5a_viscosity_curves.mat
n_cross = 0.39; % from fit on data (PEEK 150P), see fig5a_viscosity_curves.mat
h_film_cross = 2.5e-3;
serie_range = [2 3];
series{1} = [1 2 3 4 5 6 7 8 9 10 11]; % experiments in range (relates to unique number of specimen)
series{2} = [1 2 4 5 6 7 8 9 10 11 12];
ss_first = 4.5; % mm/min, threshold, steady state evaluated at first part of curve (not at end)
mod_coeff = 0.90; %threshold for modulus fit, end; max difference local and average modulus

%% Go through the measurements

V_mean_vec = []; % store values in loop
tau_mean_vec = []; % store values in loop
d_tau_end_vec = []; % store values in loop

for serie_i = 1:length(serie_range)
    
    serie = serie_range(serie_i);
    range_meas = series{serie_i};
    
    for meas_i = 1:length(range_meas)

        meas = range_meas(meas_i);

        %% Load data
        file_find = dir(['*_',num2str(serie),'-',num2str(meas),'.lvm']);
        fid = fopen(file_find.name);
        C = textscan(fid,'%f');
        data = zeros(int32(length(C{1})/13),13);
        for i = 0:length(C{1})-1
            i_row = 1+floor(i/13);
            i_col = 1+mod(i,13);
            data(int32(i_row),int32(i_col)) = C{1}(i+1);
        end
        fprintf(['File: ', file_find.name,'\n'])
        fclose(fid);

        d = data(1,13) - data(:,13); % displacement [mm]
        force_corr = mean(data(end-5:end,6)); % end value (end of experiment)
        f_pull = data(:,6) - force_corr; % pull force, corrected for end value [N]
        index = data(:,1);
        t = data(:,2) - data(1,2);


        %% Analysis

        %% index start/end using displacement data, mean velocity, shear stress
        
        % velocity from file name
        name_u = find(file_find.name=='u');
        name_u_end = name_u + 1;
        while file_find.name(name_u_end) ~= '_'
            name_u_end = name_u_end + 1;
        end
        vel = str2double(file_find.name(name_u+1:name_u_end-1)); % velocity from file name

        % determine window for smoothening based on velocity form file name
        if vel <= 1
            window = 100;
        elseif vel <= 3
            window = 75;
        elseif vel <= 5
            window = 50;
        elseif vel <= 8
            window = 25;
        elseif vel <= 15
            window = 15;
        else
            window = 15;
        end

        % smoothening data
        d_filt = smoothdata(d,'gaussian',window); % filter data met moving median filter

        v_vec = [0]; % velocity vector
        a_vec = [0]; % acceleration vector
        for i = 2:length(t)-1
            dt = t(i+1) - t(i);
            v = (d_filt(i+1)-d_filt(i-1))/(2*dt);
            a = (d_filt(i+1)-2*d_filt(i)+d_filt(i-1))/(dt^2);
            v_vec(i) = v;
            a_vec(i) = a;
        end
        v_vec(length(t)) = v;
        a_vec(length(t)) = a;
        
        % determine global start and end
        i_glob = 1;
        while d(i_glob) < 1 % deformation below 1 mm
            i_glob = i_glob + 1;
        end
        
        i_glob_end = i_glob;
        while t(i_glob_end) < t(i_glob) + d(end)/vel*60 % find index for time after full displacement
            i_glob_end = i_glob_end + 1;
        end
        idx_start = find(a_vec(1:i_glob_end)==max(a_vec(1:i_glob_end))); % start index, using max acceleration and global end index
        idx_end = find(a_vec(1:i_glob_end)==min(a_vec(1:i_glob_end))); % end index, using min acceleration and global end index

        % mean velocity
        V_mean = (d(idx_end)-d(idx_start))/(t(idx_end)-t(idx_start)); % mm/s
        V_mean_vec = [V_mean_vec, V_mean];
        
        % shear stress
        tau = f_pull/2500/2*1e3; % [N/mm] -> MPa, times 1e3 results in [kPa] (division by 2, as 2 slip interfaces are present)
        
        % determine range for mean shear stress
        if vel <= ss_first  % evaluate tau_mean in first part of the curve, as a small but gradual increase is seen with time at low velocity (perhaps due to the longer test times involved)
            t_ss_first = t(idx_start) + 1/vel*60;  %start after 1 mm 
            ss_idx_start = idx_start;
            while t(ss_idx_start) < t_ss_first
                ss_idx_start = ss_idx_start + 1;
            end
            t_ss_end_first = t_ss_first + 1*60/vel;    %end after another 1 mm extra
            ss_idx_end = ss_idx_start;
            while t(ss_idx_end) < t_ss_end_first
                ss_idx_end = ss_idx_end + 1;
            end
            tau_mean = mean(tau(ss_idx_start:ss_idx_end));
                        
        else  % steady state region at end of measurement
            ss_end = min(d(idx_end),10);  % max 10 mm slip length
            ss_start = ss_end - 2;  % 2 mm steady state region (~start at 8 mm)

            ss_idx_start = idx_start;
            while d(ss_idx_start) < ss_start
                ss_idx_start = ss_idx_start + 1;
            end
            ss_idx_end = ss_idx_start;
            while d(ss_idx_end) < ss_end
                ss_idx_end = ss_idx_end + 1;
            end
            tau_mean = mean(tau(ss_idx_start:ss_idx_end));
        end
        
        % store value
        tau_mean_vec = [tau_mean_vec, tau_mean];
        
        %% Relaxation
        % used to obtain end value after full relaxation (yield stress)
        
        % start and end indices
        idx_rel_start = idx_end - 10;
        while tau(idx_rel_start)-tau(idx_rel_start+1)<0.1 && idx_rel_start - idx_end < 10
            idx_rel_start = idx_rel_start + 1;
        end
        if idx_rel_start - idx_end >= 10
            idx_rel_start = idx_end;
            fprintf('Start relaxation not found \n')
        end
        idx_rel_end = idx_rel_start;
        while t(idx_rel_end) < t(idx_rel_start)+15  % 15 s relaxation time
            idx_rel_end = idx_rel_end + 1;
        end
        tau_rel = tau(idx_rel_start:idx_rel_end);
        t_rel = t(idx_rel_start:idx_rel_end)-t(idx_rel_start); 
        d_tau_end = mean(tau_rel(end-10:end));
        
        % store value
        d_tau_end_vec = [d_tau_end_vec, d_tau_end];
          
        %% modulus fit
        % used to correct data
        
        [t,d] = mod_fit_v1(idx_start,idx_end,t,d,tau,mod_coeff,window);
        
        %% Show data

        figure(2), hold on
        plot(d,f_pull,'-k')
        plot(d(ss_idx_start),f_pull(ss_idx_start),'xr','markersize',15);
        plot(d(ss_idx_end),f_pull(ss_idx_end),'xb','markersize',15);
        xlabel('Displacement [mm]')
        ylabel('Pull force [N]')
        box on
        grid on   

        figure(3), hold on
        plot(t_rel,tau_rel,'-k')
        xlabel('Recovery time [s]','interpreter','latex','fontsize',14)
        ylabel('Shear stress [kPa]','interpreter','latex','fontsize',14)
        box on
        grid on

    end
end

%% Outside of the loop

figure(2), legend('Friction data','Start steady-state','End steady-state')

% flow curve
figure(1), hold on
i_store = 1;
for serie_i = 1:length(serie_range)
    
    serie = serie_range(serie_i);
    range_meas = series{serie_i};
    
    for i_meas = 1:length(range_meas)
        plt = plot(V_mean_vec(i_store),tau_mean_vec(i_store)-d_tau_end_vec(i_store),'o','color','black','markersize',8); %,'displayname',['Meas: ',num2str(serie),'-', num2str(range_meas(i_meas))]
        set( get( get( plt, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
        i_store = i_store + 1;
    end
end

xlabel('Velocity, $V$ [mm/s]','interpreter','Latex','fontsize',18)
ylabel('Corr. stress, $\tau_\infty-\tau_y$ [kPa]','interpreter','Latex','fontsize',18)
box on
grid on

% shear stress plot from Cross model to compare rheometer and viscosity data
figure(1), hold on
tau_cross = @(h,C_shift,V) ((eta_0_cross)./(1+(eta_0_cross.*(V./h)./tau_0_cross).^(1-n_cross))).*(V./h).*C_shift;
V_cross = logspace(log10(0.1*h_film_cross),log10(200*h_film_cross),15);
tau_cross_fit = tau_cross(h_film_cross,1,V_cross);
plot(V_cross,tau_cross_fit/1e3,'-kx','displayname',['$\tau_{C}$, h = ', num2str(h_film_cross*1e3,'%0.1f'),' $\mu$m'])
legend('location','SouthEast','interpreter','latex','fontsize',16,'box','off')
axis([1e-2 0.5e1 2 50])
set(gca,'XScale','log','YScale','log','fontsize',18)


