clear all
clc
set(0,'defaultfigurecolor',[1 1 1])

%% INFO

% Read and process friction data

% Version: 2-12-2020
% Matlab version: R2020a

% Cleaned to obtain figure 4a (stress growth curves) in ESAFORM 2021

%% initialisation
figure(1), clf(1)
serie_range = [2 3 2];
series{1} = [10 9 3]; % experiments in range (relates to unique number of specimen)
series{2} = [4 1];
series{3} = [6]; 
mod_coeff = 0.90; %threshold for modulus fit, end; max difference local and average modulus

%% Go through the measurements

i_color = 1;
for serie_i = 1:length(serie_range)
    
    serie = serie_range(serie_i);
    range_meas = series{serie_i};
    
    for meas_i = 1:length(range_meas)

        meas = range_meas(meas_i);

        %% Load data
        file_find = dir(['*_',num2str(serie),'-',num2str(meas),'.lvm']);
        fid = fopen(file_find.name);
        C = textscan(fid,'%f');
        data = zeros(int32(length(C{1})/13),13);
        for i = 0:length(C{1})-1
            i_row = 1+floor(i/13);
            i_col = 1+mod(i,13);
            data(int32(i_row),int32(i_col)) = C{1}(i+1);
        end
        fprintf(['File: ', file_find.name,'\n'])
        fclose(fid);

        d = data(1,13) - data(:,13); % displacement [mm]
        force_corr = mean(data(end-5:end,6)); % end value (end of experiment)
        f_pull = data(:,6) - force_corr; % pull force, corrected for end value [N]
        t = data(:,2) - data(1,2); % time [s]

        %% Analysis

        %% index start/end using displacement data, mean velocity, shear stress
        
        % velocity from file name
        name_u = find(file_find.name=='u');
        name_u_end = name_u + 1;
        while file_find.name(name_u_end) ~= '_'
            name_u_end = name_u_end + 1;
        end
        vel = str2double(file_find.name(name_u+1:name_u_end-1)); % velocity from file name

        % determine window for smoothening based on velocity form file name
        if vel <= 1
            window = 100;
        elseif vel <= 3
            window = 75;
        elseif vel <= 5
            window = 50;
        elseif vel <= 8
            window = 25;
        elseif vel <= 15
            window = 15;
        else
            window = 15;
        end

        % smoothening data
        d_filt = smoothdata(d,'gaussian',window); % filter data met moving median filter

        v_vec = [0]; % velocity vector
        a_vec = [0]; % acceleration vector
        for i = 2:length(t)-1
            dt = t(i+1) - t(i);
            v = (d_filt(i+1)-d_filt(i-1))/(2*dt);
            a = (d_filt(i+1)-2*d_filt(i)+d_filt(i-1))/(dt^2);
            v_vec(i) = v;
            a_vec(i) = a;
        end
        v_vec(length(t)) = v;
        a_vec(length(t)) = a;
        
        % determine global start and end
        i_glob = 1;
        while d(i_glob) < 1 % deformation below 1 mm
            i_glob = i_glob + 1;
        end
        
        i_glob_end = i_glob;
        while t(i_glob_end) < t(i_glob) + d(end)/vel*60 % find index for time after full displacement
            i_glob_end = i_glob_end + 1;
        end
        idx_start = find(a_vec(1:i_glob_end)==max(a_vec(1:i_glob_end)));
        idx_end = find(a_vec(1:i_glob_end)==min(a_vec(1:i_glob_end)));

        % mean velocity
        V_mean = (d(idx_end)-d(idx_start))/(t(idx_end)-t(idx_start)); % mm/s
        
        % shear stress
        tau = f_pull/2500/2*1e3; % [N/mm] -> MPa, times 1e3 results in [kPa] (division by 2, as 2 slip interfaces are present)
           
        %% modulus fit
        
        [t,d] = mod_fit_v1(idx_start,idx_end,t,d,tau,mod_coeff,window);
        
        %% Show data
        
        figure(1), hold on 
        plot(t(1:idx_end),tau(1:idx_end),'-','linewidth',2,'color',[0 0 0]+0.12*i_color,'displayname',[num2str(vel), ' mm/min'])
        xlabel('Time, $t$ [s]','interpreter','Latex','fontsize',18)
        ylabel('Shear stress, $\tau$ [kPa]','interpreter','Latex','fontsize',18)
        box on
        grid on
        i_color = i_color + 1;
        
    end
end

figure(1)
axis([0 8 0 90])
legend('interpreter','latex','fontsize',14,'box','off')
set(gca,'fontsize',18)

