% 15/02/2021 run.m 
% Created by G. Egberts
% This script runs the finite element algorithm that solves the
% one-dimensional morhhoelastic system for skin contraction. The model,
% stablity analysis and numerical validation are described in the paper "A
% one-dimensional morphoelastic model for burn injuries - stability,
% numerical validation, and biological interpretation". 
% Comments describing the code are given as much as possible. The algorithm
% works in 6 steps:
% (1) Assemble chemical matrices and vectors and solve the system;
% (2) Perform inner Picard iteration for the chemical part of the model to
% account for the non-linearity of the model;
% (3) Assemble mechanical matrices and vectors and solve the system;
% (4) Record the displacement and displaced domain.
% (5) Record time-variables (i.e., iteration and day);
% (6) Record graphics (i.e., videos and figures).
% The script runs until day T is reached, that is the total simulation
% time. In some studies, we vary the time step.

% Initialize displacement vector and initial domain.
u = zeros(n,1);
x_init=x;

% Initialize time variables.
t = 1;
day = zeros(round(T/dt)-1,1);

% Initialize videos and figures.
InitiateVideo;
if study<10
    InitiateVideo;
    InitiateFigures;
end
while day(t) < T
    % 1) Assemble matrices and vectors and solve for the constituents.
    AssembleMass; % Mass matrix (lumped)
    AssembleN; % Fibroblasts
    AssembleM; % Myofibroblasts
    AssembleC; % Signaling molecules
    AssembleR; % Collagen molecules
    
    % Beforehand, store the distribution/density (z^n) at previous timepoint.
    N_n = N; M_n = M; c_n = c; rho_n = rho;
    
    F_n = zeros(n,1);                %
    F_n(:) = F_n(:) - S_n(:,1)*N_eq; %
    S_n(:,1) = 0;                    %
    S_n(1,:) = 0;                    % BC for Fibroblasts.
    S_n(1,1) = 1;                    %
    F_n(1) = N_eq;                   %
    N = (Mass + S_n*dt)\(Mass*N + F_n*dt); % Solve for fibroblasts (N^{n+1}_0).
    
    S_m(:,1) = 0; %
    S_m(1,:) = 0; % BC for myofibroblasts.
    S_m(1,1) = 1; %
    F_m(1)   = 0; %
    M = (Mass + S_m*dt)\(Mass*M + F_m*dt); % Solve for myofibroblasts (M^{n+1}_0).
    M(M<0)=0; % Correction for negative values (flux correction is not applied).
    
    S_c(:,1) = 0; %
    S_c(1,:) = 0; % BC for signaling molecules.
    S_c(1,1) = 1; %
    c = (Mass + S_c*dt)\(Mass*c); % Solve for signaling molecules (c^{n+1}_0).
    
    rho = (Mass + S_r*dt)\(Mass*rho + F_r*dt); % Sove for collagen molecules (rho^{n+1}_0).
    
    % 2) Perform inner Picard iterations to account for the nonlinearity.
    % In each iteration, we compute the relative 1-norm of the difference
    % between subsequent approximations per variable. The iterations stop if
    % the maximum of the relative 1-norms is less than 10^{-5}.
    
    max_norm = Inf;
    while max_norm > 1e-5
        AssembleN;
        F_n = zeros(n,1);
        F_n(:) = F_n(:) - S_n(:,1)*N_eq;
        S_n(:,1) = 0;
        S_n(1,:) = 0;
        S_n(1,1) = 1;
        F_n(1) = N_eq;
        N1 = (Mass + S_n*dt)\(Mass*N_n + F_n*dt); % N^{n+1}_{k+1}
        norm_N = norm(abs(N1-N)/abs(N1),1); % Relative 1-norm of N.
        
        AssembleM;
        S_m(:,1) = 0;
        S_m(1,:) = 0;
        S_m(1,1) = 1;
        F_m(1)   = 0;
        M1 = (Mass + S_m*dt)\(Mass*M_n + F_m*dt);
        M1(M1<0) = 0;
        norm_M = norm(abs(M1-M)/abs(M1+1e-40),1); % Relative 1-norm of M.
        
        AssembleC;
        S_c(:,1) = 0;
        S_c(1,:) = 0;
        S_c(1,1) = 1;
        c1 = (Mass + S_c*dt)\(Mass*c_n);
        norm_c = norm(abs(c1-c)/abs(c1+1e-40),1); % Relative 1-norm of c.
        
        AssembleR;
        rho1 = (Mass + S_r*dt)\(Mass*rho_n + F_r*dt);
        norm_rho = norm(abs(rho1-rho)/abs(rho1),1); % Relative 1-norm of rho.
        
        % In case the max-norm increases, we stop the simulation.
        if max([norm_N,norm_M,norm_c,norm_rho])>max_norm
            fprintf('no Picard convergence.\n');
            break;
        else
            max_norm = max([norm_N,norm_M,norm_c,norm_rho]);
        end
        
        % Update z^{n+1}
        N = N1; M = M1; c = c1; rho = rho1;
    end
    
    % 3) Assemble matrices and vectors and solve for v and epsilon.
    AssembleV; M_v=Mass*rho_t;
    AssembleE; M_e=Mass;
        
    % System of matrices and vectors.
    M1 = [M_v(2:n-1,2:n-1)+S_vv(2:n-1,2:n-1)*dt  ,  S_ve(2:n-1,:)*dt; ...
          S_ev(:,2:n-1)*dt                       ,  M_e+S_ee*dt];
    ve = M1 \ ([M_v(2:n-1,2:n-1)*v(2:n-1)+ f_v(2:n-1)*dt;M_e*epsilon]);
    v = vertcat(0,ve(1:n-2),0); epsilon = ve(n-1:length(ve));
        
    % 4) Compute the displacement and update the mesh.
    u = u + dt*v;
    x = x_init(1:n) + u';
    
    % 5) Record time-variables (i.e., iteration and day).
    t = t + 1; 
    waitbar(t/(10/dt))
    
    if study==5
        if (day(t-1) + dt)<=2
            day(t) = day(t-1) + dt;
        elseif (day(t-1) + dt)>2 && (day(t-1) + dt)<=50
            dt = 1;
            day(t) = round(day(t-1)) + dt;
        elseif (day(t-1) + dt)>50
            dt = 2;
            day(t) = day(t-1) + dt;
        end        
    else
        day(t) = day(t-1) + dt;
    end
    
    % (6) Graphics.
    WriteVideo;
    if study<10
        WriteVideo;
        if any(plotdays==round(day(t),3))
            SnapShot;
        end
    end
end
close(vid_v);
if study<10
    close(vid_N);
    close(vid_M);
    close(vid_c);
    close(vid_r);
    if study<8
        close(vid_v);
        close(vid_e);
    end
end