%%%%%%%%%%%%%%%%%%%%%%%%%%%% 2021-02-23  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
%  This script simulates the contraction of a healing burn wound. The     %
%  algorithm is as follows:                                               %
%                                                                         %
%  For each time step take the following steps.                           %
%  1) Assemble the lumped mass matrix.                                    %
%  2) Perform inner Picard iterations to account for the nonlinearity.    %
%  3) Assemble matrices and vectors and solve for the displacement velo-  %
%     city and the effective strain.                                      %
%  4) Compute the displacement, update the mesh and report RSAW.          %
%                                                                         %
%  The standard output is a figure showing the evolution of the relative  %
%  surface area over time until time T. If the user needs other outputs,  %
%  then this can be set in the input file.                                %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

addpath('InputOutput\Additional_scripts\MatricesAndVectors');
day=zeros(T/dt-1,1); 
RSAW=ones(T/dt,1);
while day(t)<T
    % 1) Assemble the lumped mass matrix.
    AssembleMass;
    
    % Beforehand, store the distribution/density (z^n) at previous timepoint.
    N_n=N; M_n=M; c_n=c; rho_n=rho;
        
    % 2) Perform inner Picard iterations to account for the nonlinearity.
    % In each iteration, we compute the relative 1-norm of the difference
    % between subsequent approximations per variable. The iterations stop if
    % the maximum of the relative 1-norms is less than 10^{-5}.

    max_norm=Inf;
    while max_norm>1e-5
        
        AssembleNM;
        N1=(Mass_L+S_n*dt)\(Mass_L*N_n+F_n*dt); % N^{n+1}_{k+1}
        norm_N=norm(abs(N1-N)/abs(N1),1); % Relative 1-norm of N.
        
        M1=(Mass_L+S_m*dt)\(Mass_L*M_n+F_m*dt);
        norm_M=norm(abs(M1-M)/abs(M1+1e-40),1); % Relative 1-norm of M.
        
        AssembleCR;
        c1=(Mass_L+S_c*dt)\(Mass_L*c_n);
        norm_c=norm(abs(c1-c)/abs(c1+1e-40),1); % Relative 1-norm of c.
        
        rho1=(Mass_L+S_r*dt)\(Mass_L*rho_n+F_r*dt);
        norm_rho=norm(abs(rho1-rho)/abs(rho1),1); % Relative 1-norm of rho.
        
        max_norm=max([norm_N,norm_M,norm_c,norm_rho]);
        
        % Update z^{n+1}
        N=N1; M=M1; c=c1; rho=rho1;
    end
        
    % 3) Assemble matrices and vectors and solve for v and epsilon.
    AssembleVE;
    
    % System of matrices and vectors.
    M1 = [rho_t*Mass(2:n-1,2:n-1)+S_vv(2:n-1,2:n-1)*dt, S_ve(2:n-1,1:n)*dt; ...
          S_ev(1:n,2:n-1)*dt                          , Mass+S_ee*dt];
    ve = M1 \ ([rho_t*Mass(2:n-1,2:n-1)*v(2:n-1)+F_v(2:n-1)*dt;Mass*epsilon]);
    v = vertcat(0,ve(1:n-2),0); epsilon = ve(n-1:length(ve));
    
    % 4) Compute the displacement, update the mesh and report RSAW.
    u=u+dt*v;
    x=x_init(1:n)+u';
        
    % Variables to record the day and time.
    t=t+1;
    day(t)=day(t-1)+dt;
    RSAW(t)=abs(x(lW))/Lw;
    
end
Ns{sim}=N;
Ms{sim}=M;
Cs{sim}=c;
Rs{sim}=rho;
Vs{sim}=v;
Es{sim}=epsilon;
Xs{sim}=x;
RSAWs(sim)=RSAW(end);