%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 2020-06-09  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
%  This script pre-processes the parameters, the mesh, the topology and   %
%  the initial conditions.                                                %
%                                                                         %
%  The user is able to write video files that show the evolution of the   %
%  constituents, the displacement velocity, the effective strain and      %
%  the displacement of the dermal layer. This is set by the following     %
%  parameter:                                                             %
%                                                                         %
%  Video = [0, 0, 0, 0, 0, 0, 0];    {N,M,c,rho,v,eps,u}                  %
%                                                                         %
%  Video's are written in case some of the values are set to 1. If the    %
%  user wants to combine the videos in one file, for example the fibro-   %
%  blasts, myofibroblasts, signaling molecules and collagen molecules,    %
%  then this can be set with 'Combine = 1'.                               %
%                                                                         %
%  One other option is to take snapshots of all constituents, the         %
%  displacement velocity, the effective strain and the displacement of    %
%  the dermal layer. These snapshots are taken at specified days:         %
%                                                                         %
%  ss_days = [day1, day2, day3, ...]                                      %
%                                                                         %
%  Videos and figures are automatically saved with the day stamp.         %      
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% % Specify videos Video = [N,M,c,rho,v,eps,u]
% Video = [1,1,1,1,1,1,1]; % example: [1,1,1,1,0,0,0]; for only the constituents
% %Video = [1,1,1,1,0,0,0];
% Combine = 1;
% 
% % Specify figures ss_days = [day1, day2, day3, ...]
% ss_days = 0; % example: [4,5,6,7,24,27,100];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%  Parameters %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
%   topology  : the topology of the model (one-dimesional)                %
%   n         : the number of elements + 2,                               %
%   s         : the steepness of the wound,                               %
%                                                                         %
%   N_eq      : the equlibrium of fibroblasts,                            %
%   c_eq      : the equlibrium of signaling molecules,                    %
%   rho_eq    : the equlibrium of collagen molecules,                     %
%   N_tilde   : the initial distribution of fibroblasts,                  %
%   c_tilde   : the initial density of signaling molecules,               %
%   rho_tilde : the initial density of collagen molecules,                %
%                                                                         %
%   R         : a fixed constant,                                         %
%   chi_F     : the chemotaxis of (myo)fibroblasts,                       %
%   r_F_max   : the enhancing proliferation maximum factor of (myo)fibro, %
%   eta_I     : ratio of myo-fibro in max-net-secretion of signaling mol.,%
%   eta_II    : ratio of myo-fibro in max-secretion of MMPs,              %
%   k_rho_max : the enhancing proliferation maximum factor of collagen,   %
%   delta_M   : apoptosis rate of myofibroblasts.                         %
%   D_F         : diffusion coefficient (myo)fibroblasts,                 %
%   r_F         : cell division rate of (myp)fibroblasts,                 %
%   a_c_I       : signaling concentration for half-max enhancement  of    %
%                   the division rate of (myo)fibroblasts,                %
%   kappa_F     : factor for reduction in cell-division due to crowding,  %
%   delta_N     : apoptosis rate of fibroblasts,                          %
%   k_F         : signaling dependent differentiation of fibro to myo,    %
%   D_c         : diffusion coefficient signalig molecule,                %
%   a_c_II      : signaling concentration for half-max secretion,         %
%   k_c         : max net ssecretion rate of signaling molecule,          %
%   delta_c     : proteolytic breakdown rate of signaling molecule,       %
%   a_c_III     : factor in inhibition of secretion of MMP,               %
%   a_c_IV      : signaling concentration for half-max secretion,         %
%   delta_rho   : proteolytic breakdown rate of collagen molecules,       %
%   rho_t       : total mass density of dermal tissues,                   %
%   mu          : viscosity,                                              %
%   E           : factor for Youngs Modulus (which depends on rho),       %
%   xi          : genereted stress (body force),                          %
%   zeta        : rate of morphoelastic change,                           %
%   q           : a constant depending on other parameter values,         %
%   k_rho       : secretion rate of collagen molecules, dependent.        %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

N_eq        = 1e4;
c_eq        = 0;
rho_eq      = 0.1125;
N_tilde     = 2000;
c_tilde     = 1e-8;
rho_tilde   = 0.01125;

R           = 0.995;
chi_F       = 2e-3;
r_F_max     = 2;
eta_I       = 2;
eta_II      = 0.5;
k_rho_max   = 10;
delta_M     = 6e-2;

D_F         = 1e-7;
r_F         = 0.924;
a_c_I       = 1e-8;
kappa_F     = 1e-6;
delta_N     = 0.02;
k_F         = 1.08e7;
D_c         = 2.88e-3;
a_c_II      = 1e-8;
k_c         = 4e-13;
delta_c     = 5e-4;
a_c_III     = 2e8;
a_c_IV      = 1e-9;
delta_rho   = 6e-6;
rho_t       = 1.02;
mu          = 100;
E           = 210;
xi          = 0.044;
zeta        = 4e2;
q           = (log(delta_N)-log(r_F.*(1-kappa_F*N_eq)))/log(N_eq);
k_rho       = delta_rho*rho_eq^2;

% Generate the mesh
addpath('InputOutput\Additional_scripts\MeshInitial');
GenerateMesh;
dt=(x_init(2)-x_init(1))^2;

% Process the initial conditions
InitialConditions;

t = 1;

% % Initiate videos
% if any(Video)
%     addpath('InputOutput\Additional_scripts\VideosAndFigures');
%     InitiateVideos;
% end
% 
% % Initiate snapshots
% if any(ss_days)
%     addpath('InputOutput\Additional_scripts\VideosAndFigures');
%     InitiateFigures;
% end