### Visualize validation CDFs
{
  ## Set working directory
  setwd("repository/scripts")
  
  ## Load packages
  library(data.table)
  library(ggplot2)
  
  ## Load simulation results
  files = list.files("../simulations/poacher_movement/",
                     full.names = TRUE,
                     recursive = TRUE,
                     include.dirs = FALSE)
  dat = list()
  pb = txtProgressBar(min=0, max=1, initial=0, char="-", style=3)
  for(i in seq_along(files)) {
    dat[[i]] = as.data.table(readRDS(files[i]))
    setTxtProgressBar(pb, i/length(files))
  }
  dat = rbindlist(dat)
  sum(is.na(dat$encounterdist))==0 #TRUE
  sum(is.infinite(dat$encounterdist))==0 #TRUE
  max(dat$encounterdist/dat$meanturndist)<unique(dat$nturns) #TRUE
  dat[, areaperrhino := ((knplength)*(knpwidth))/nrhinos]
  setkeyv(dat,c("nrhinos","rhinodis","distribution","meanturndist",
                "encounterradius","poacher"))
  setcolorder(dat)
  
  ## Load functions to compute CDF
  fun_a_o1 = function(a, r) {
    
    return(r^2*tan(.5*a))
  }
  
  fun_a_o2 = function(a, r, t) {
    
    return(t*(2*r-t/tan(.5*a))-tan(pi-a)*(r-t/tan(.5*a))^2)
  }
  
  fun_CDF = function(s, r, t, n, a_t = 360*54.5) {
    
    sum1 = integrate(fun_a_o1, 
                     lower = 0, upper = 2*atan(t/r), 
                     r = r)
    
    sum2 = integrate(fun_a_o2, 
                     lower = 2*atan(t/r), upper = pi, 
                     r = r, t = t)
    
    a_o = (sum1$value + sum2$value) / pi
    
    cdf = 1-(1-(pi*r^2 + 2*r*s - (
      (2*a_o*r*(2*r*t*(t*(2^(-s/t)*(a_o/(r*t))^(s/t)-1)+s)-a_o*s))/((a_o-2*r*t)^2)
    ))/a_t)^n
    
    return(cdf)
  }
  
  ## Subset simulations for visualizations
  dat_sel = dat[nrhinos %in% c(792,12672) &
                  meanturndist %in% c(500,5000) &
                  encounterradius %in% c(25,100),]
  
  ## Create output table for all selected parameter combinations
  dat_cmb = expand.grid(nrhinos = unique(dat_sel$nrhinos),
                        meanturndist = unique(dat_sel$meanturndist),
                        encounterradius = unique(dat_sel$encounterradius),
                        rhinodis = unique(dat_sel$rhinodis),
                        distribution = levels(dat_sel$distribution))
  
  ## Include the calculated CDF to the combinations
  dat_cmb = rbind(dat_cmb,
                  expand.grid(nrhinos = unique(dat_sel$nrhinos),
                              meanturndist = unique(dat_sel$meanturndist),
                              encounterradius = unique(dat_sel$encounterradius),
                              rhinodis = "uniform",
                              distribution = "CDF"))
  
  ## Compute CDF and encountered proportion of rhinos vs moved distance
  dat_cdf = list()
  
  pb = txtProgressBar(min=0, max=1, initial=0, char="-", style=3)
  for(i in seq_len(nrow(dat_cmb))) {
    if(dat_cmb$distribution[i]=="CDF") {
      dats = dat_sel[nrhinos==dat_cmb$nrhinos[i] &
                       meanturndist==dat_cmb$meanturndist[i] &
                       encounterradius==dat_cmb$encounterradius[i] &
                       rhinodis==dat_cmb$rhinodis[i],]
      dats[, encounterdist := as.integer(round(encounterdist))]
      
      dat_cdf[[i]] = 
        data.table(encounterdist = seq(0,
                                       max(dats$encounterdist),
                                       length.out = 1e3))
      dat_cdf[[i]] = cbind(dat_cdf[[i]],dat_cmb[i,])
      dat_cdf[[i]][, encounters := fun_CDF(s = dat_cdf[[i]]$encounterdist/1e3,
                                           r = dat_cdf[[i]]$encounterradius[1]/1e3,
                                           t = dat_cdf[[i]]$meanturndist[1]/1e3,
                                           n = dat_cdf[[i]]$nrhinos[1])]
      setkeyv(dat_cdf[[i]], c(colnames(dat_cmb),"encounterdist"))
      setcolorder(dat_cdf[[i]])
    } else {
      dats = dat_sel[nrhinos==dat_cmb$nrhinos[i] &
                       meanturndist==dat_cmb$meanturndist[i] &
                       encounterradius==dat_cmb$encounterradius[i] &
                       rhinodis==dat_cmb$rhinodis[i] &
                       distribution==dat_cmb$distribution[i],]
      dats[, encounterdist := as.integer(round(encounterdist))]
      dats = dats[, lapply(.SD, length), 
                  by=c("encounterdist"), 
                  .SDcols="poacher"]
      
      dat_cdf[[i]] = 
        data.table(encounterdist = 0:max(dats$encounterdist))
      
      setkeyv(dats, "encounterdist")
      setkeyv(dat_cdf[[i]], "encounterdist")
      dat_cdf[[i]] = dats[dat_cdf[[i]],]
      dat_cdf[[i]][is.na(poacher), poacher := 0]
      dat_cdf[[i]][, encounters := cumsum(poacher)/sum(poacher)]
      dat_cdf[[i]] = dat_cdf[[i]][,-"poacher"]
      dat_cdf[[i]] = cbind(dat_cdf[[i]],dat_cmb[i,])
      setkeyv(dat_cdf[[i]], c(colnames(dat_cmb),"encounterdist"))
      setcolorder(dat_cdf[[i]])
      
      dat_cdf1 = dat_cdf[[i]][!duplicated(encounters),]
      dat_cdf2 = dat_cdf[[i]][nrow(dat_cdf[[i]]):1,][!duplicated(encounters),][-1,]
      dat_cdf[[i]] = rbind(dat_cdf1,dat_cdf2)
      setkeyv(dat_cdf[[i]], c(colnames(dat_cmb),"encounterdist"))
    }
    setTxtProgressBar(pb, i/nrow(dat_cmb))
  }
  
  dat_cdf = rbindlist(dat_cdf)
  
  ## Process data for plotting
  dat_cdf[, `:=` ("encounterdist" = encounterdist/1e3,
                  "mean interturn distance (m)" = as.factor(meanturndist),
                  "observation radius (m)" = as.factor(encounterradius))]
  levels(dat_cdf$distribution)[levels(dat_cdf$distribution)=="folded_gaussian"] = 
    "folded Gaussian"
  levels(dat_cdf$distribution)[levels(dat_cdf$distribution)=="truncated_levy"] = 
    "truncated Lévy"
  dat_cdf[, distribution := 
            factor(as.character(distribution), 
                   levels = c("CDF",
                              "degenerate",
                              "folded Gaussian",
                              "truncated Lévy",
                              "uniform"))]
  
  ## Visualize figures
  pa = ggplot(dat_cdf[nrhinos==12672 & distribution != "CDF",], 
             aes(x=encounterdist, y=encounters,
                 color = distribution,
                 linetype = rhinodis,
                 group = interaction(distribution,rhinodis))) + 
    geom_line() +
    geom_line(data = dat_cdf[nrhinos==12672 & distribution == "CDF",], 
              mapping = aes(x=encounterdist, y=encounters,
                            color = distribution,
                            group = distribution),
              size = 1.05,
              show.legend = FALSE) +
    scale_color_manual(values=c("#000000",
                                "#E69F00",
                                "#56B4E9",
                                "#009E73",
                                "#0072B2")) +
    coord_cartesian(ylim=c(0,1),
                    expand=0) +
    facet_grid(`observation radius (m)`~`mean interturn distance (m)`,
               labeller = "label_both") +
    labs(x = "poacher movement distance (km)",
         y = "cumulative probability of first rhino encounter",
         linetype = "rhino distribution",
         color = "poacher movement") +
    theme(panel.spacing = unit(1, "lines"),
          legend.position="top")
  
  pb = ggplot(dat_cdf[nrhinos==792 & distribution != "CDF",], 
             aes(x=encounterdist, y=encounters,
                 color = distribution,
                 linetype = rhinodis,
                 group = interaction(distribution,rhinodis))) + 
    geom_line() +
    geom_line(data = dat_cdf[nrhinos==792 & distribution == "CDF",], 
              mapping = aes(x=encounterdist, y=encounters,
                            color = distribution,
                            group = distribution),
              size = 1.05,
              show.legend = FALSE) +
    scale_color_manual(values=c("#000000",
                                "#E69F00",
                                "#56B4E9",
                                "#009E73",
                                "#0072B2")) +
    coord_cartesian(ylim=c(0,1),
                    expand=0) +
    facet_grid(`observation radius (m)`~`mean interturn distance (m)`,
               labeller = "label_both") +
    labs(x = "poacher movement distance (km)",
         y = "cumulative probability of first rhino encounter",
         linetype = "rhino distribution",
         color = "poacher movement") +
    theme(panel.spacing = unit(1, "lines"),
          legend.position="top")
  
  ## Save figures
  ggsave("../plots/fig6a.pdf",
         pa,
         width = 3000,
         height = 2000,
         units = "px")
  ggsave("../plots/fig6b.pdf",
         pb,
         width = 3000,
         height = 2000,
         units = "px")
}
