### Load HPC packages
library(rslurm)

### Create parameter grid
params = expand.grid(
  'knplength' = 360*1e3, # KNP N-S length in meters
  'knpwidth' = 55*1e3, # KNP E-W width in meters
  'nrhinos' = (360*55)/c(1.25,2.5,5)^2, # number of rhinos, derived from distance between regular distributed rhinos
  'rhinodis' = c("uniform","regular"), # distribution of rhinos
  'encounterradius' = seq(from=5,to=.25,by=-.25)*1e2, # maximum distance in meters of poacher detecting rhino
  'npoachers' = 1e4, # number of poaching attempts
  'nturns' = 1e6, # max number of turns by poacher
  'steps' = 1, # poacher displacement in meters for which to compute distance to rhinos
  'meanturndist' = seq(from=5,to=.5,by=-.5)*1e3, # mean distance in meters between poacher turns
  'distribution' = c("degenerate","uniform","folded_gaussian","truncated_levy"), # distribution type of distance between poacher turns
  'maxturndist' = NA_real_, # distribution-specific maximum value given the mean distance
  'distributionpar' = NA_real_ # distribution-specific scale parameter given the mean distance
)

### Compute distribution parameter values
params[params$distribution=="degenerate",]$maxturndist = 
  params[params$distribution=="degenerate",]$meanturndist
params[params$distribution=="degenerate",]$distributionpar = 
  params[params$distribution=="degenerate",]$meanturndist

params[params$distribution=="uniform",]$maxturndist = 
  params[params$distribution=="uniform",]$meanturndist*2
params[params$distribution=="uniform",]$distributionpar = 
  params[params$distribution=="uniform",]$meanturndist*2

params[params$distribution=="folded_gaussian",]$maxturndist = 
  Inf
params[params$distribution=="folded_gaussian",]$distributionpar = 
  params[params$distribution=="folded_gaussian",]$meanturndist/sqrt(2/pi)

params[params$distribution=="truncated_levy",]$maxturndist = 
  3*(params[params$distribution=="truncated_levy",]$meanturndist/sqrt(2/pi))
for(i in seq_len(nrow(params[params$distribution=="truncated_levy",]))) {
  set.seed(i)
  params[params$distribution=="truncated_levy",]$distributionpar[i] = 
    optimise(
      f = function(pr, maxturndist, meanturndist) {
        temp = -pr + ((sqrt(pr*maxturndist) * exp(-pr/(2*maxturndist)) * sqrt(2/pi)) /
                        (2 * pnorm((sqrt(pr/maxturndist)/sqrt(2)) * sqrt(2), lower = FALSE)))
        return(abs(temp-meanturndist))
      },
      interval = c(0,params[params$distribution=="truncated_levy",]$maxturndist[i]),
      maxturndist = params[params$distribution=="truncated_levy",]$maxturndist[i],
      meanturndist = params[params$distribution=="truncated_levy",]$meanturndist[i]
    )$minimum
}
itot = nrow(params[params$distribution=="truncated_levy",])
iunq = length(unique(params[params$distribution=="truncated_levy",]$meanturndist))
ii = itot/iunq
for(i in seq_len(iunq)) {
  params[params$distribution=="truncated_levy",]$distributionpar[(1+(i-1)*ii):(i*ii)] =
    mean(params[params$distribution=="truncated_levy",]$distributionpar[(1+(i-1)*ii):(i*ii)])
}

### Create simulation function
poachersearch = function(knplength,
                         knpwidth,
                         nrhinos,
                         rhinodis,
                         encounterradius,
                         npoachers,
                         nturns,
                         steps,
                         meanturndist,
                         distribution,
                         maxturndist,
                         distributionpar) {
  
  ## Setup output table
  results = data.frame('encounterdist'=NA_real_,
                       'poacher'=seq_len(npoachers),
                       'knplength'=knplength,
                       'knpwidth'=knpwidth,
                       'nrhinos'=nrhinos,
                       'rhinodis'=rhinodis,
                       'encounterradius'=encounterradius,
                       'nturns'=nturns,
                       'steps'=steps,
                       'distribution'=distribution,
                       'distributionpar'=distributionpar,
                       'meanturndist'=meanturndist,
                       'maxturndist'=maxturndist)
  
  for(i in seq_len(npoachers)) {
    
    ## Unique simulation seed
    set.seed(i)
    
    ## Poacher movement
    poacher = matrix(rep(NA_real_,
                         nturns*2),
                     ncol = 2)
    
    ta = runif(nturns-1,
               0,
               2*pi)
    
    if(distribution=="degenerate") {
      par = distributionpar
      sl = rep(par,
               nturns-1)
    } else if(distribution=="uniform") {
      par = distributionpar
      sl = runif(nturns-1,
                 max = par)
    } else if(distribution=="folded_gaussian") {
      par = distributionpar
      sl = abs(rnorm(nturns-1,
                     sd = par))
    } else if(distribution=="truncated_levy") {
      par = distributionpar
      sl = (par/qnorm(1-runif(nturns-1,
                              max = (2*(1-pnorm((maxturndist/par)^(-.5)))))
                      /2)^2)
    }
    
    dx = sl*cos(ta)
    dy = sl*sin(ta)
    
    poacher[,1] = cumsum(c(knpwidth/2, dx)) %% knpwidth
    poacher[,2] = cumsum(c(knplength/2, dy)) %% knplength
    
    ## Rhino locations
    rhinos = matrix(rep(NA_real_,
                        nrhinos*2),
                    ncol = 2)
    
    if(rhinodis=="uniform") {
      rhinos[,1] = runif(nrhinos,
                         0,
                         knpwidth)
      rhinos[,2] = runif(nrhinos,
                         0,
                         knplength)
    } else if(rhinodis=="regular") {
      rhinos[1,1] = runif(1,
                          0,
                          knpwidth)
      rhinos[1,2] = runif(1,
                          0,
                          knplength)
      
      dist = sqrt((knplength*knpwidth)/nrhinos)
      x1 = dist*(1:(knpwidth/dist)) - dist
      y1 = dist*(1:(knplength/dist)) - dist
      
      rhinos[,1] = (rhinos[1,1] + rep(x1, times = length(y1))) %% knpwidth
      rhinos[,2] = (rhinos[1,2] + rep(y1, each = length(x1))) %% knplength
    }
    
    ## Compute encounter distance
    xr=rhinos[0,,drop=FALSE]
    j=1
    while(j<(nturns-1)) {
      x1 = poacher[j,1]
      y1 = poacher[j,2]
      
      ddx = abs(x1-rhinos[,1])
      ddx[ddx>(knpwidth/2)] = knpwidth - ddx[ddx>(knpwidth/2)]
      ddy = abs(y1-rhinos[,2])
      ddy[ddy>(knplength/2)] = knplength - ddy[ddy>(knplength/2)]
      
      dist = sqrt(ddx^2+ddy^2)
      mindist = min(dist)
      
      if(j==1) {
        if(mindist<encounterradius) {
          k = 0
          results[i,]$encounterdist = k
          break
        }
      }
      
      if(mindist<(encounterradius+sl[j])) {
        xr = rhinos[dist<(encounterradius+sl[j]),,drop=FALSE]
        
        ddx = (knpwidth/2)-x1
        x1 = x1+ddx
        xr[,1] = (xr[,1]+ddx) %% knpwidth
        ddy = (knplength/2)-y1
        y1 = y1+ddy
        xr[,2] = (xr[,2]+ddy) %% knplength
        
        x1 = x1 + seq(from=0,
                      to=dx[j],
                      length.out=ceiling(sl[j]/steps))
        y1 = y1 + seq(from=0,
                      to=dy[j],
                      length.out=ceiling(sl[j]/steps))
        
        par = rep(NA_real_, nrow(xr))
        
        for(k in seq_len(nrow(xr))) {
          par[k] = (which((sqrt((x1-xr[k,1])^2+(y1-xr[k,2])^2) - 
                             encounterradius)<0))[1]
        }
        
        par = sort(par)[1]
        
        if(!is.na(par)) {
          results[i,]$encounterdist = 
            sum(sl[seq_len(j-1)]) + (sl[j]/length(x1))*par
          break
        }
      }
      
      while(j<(nturns-1)) {
        mindist = mindist-sl[j]
        j=j+1
        if(mindist<(encounterradius+sl[j])) {
          break
        }
      }
    }
    
    ## Clear memory
    rm(j,k,dx,dy,x1,y1,ddx,ddy,dist,mindist,xr,
       ta,sl,par,poacher,rhinos)
    invisible(gc())
  }
  
  ## Return output
  return(results)
}

### Start HPC array job
sopt = list('time' = '0-8:0:0',
            'error' = 'error_%a.txt',
            'qos' = 'low',
            'mem-per-cpu' = '1500',
            'cpus-per-task' = '1')

sjob = slurm_apply(f = poachersearch,
                   params = params,
                   jobname = 'poacher_movement',
                   nodes = nrow(params),
                   cpus_per_node = 1,
                   job_array_task_limit = 100,
                   slurm_options = sopt,
                   submit = TRUE)
