### Visualize simulation PDFs
{
  ## Set working directory
  setwd("repository/scripts")
  
  ## Load packages
  library(data.table)
  library(ggplot2)
  
  ## Set up table over 0-1 CDF range for our four distributions
  dfs = as.data.table(expand.grid('distribution'=c("degenerate",
                                                   "uniform",
                                                   "folded Gaussian",
                                                   "truncated Lévy"),
                                  'meanturndist'=1,
                                  'maxturndist'=NA_real_,
                                  'distributionpar'=NA_real_,
                                  'sl'=NA_real_,
                                  'cdf'=seq(from = 0,
                                            to = 1,
                                            by = 1e-4)))
  
  ## Fill in parameter values per distribution type for mean=1
  dfs[distribution=="degenerate",]$maxturndist = 1
  dfs[distribution=="degenerate",]$distributionpar = 1
  
  dfs[distribution=="uniform",]$maxturndist = 2
  dfs[distribution=="uniform",]$distributionpar = 2
  
  dfs[distribution=="folded Gaussian",]$maxturndist = Inf
  dfs[distribution=="folded Gaussian",]$distributionpar = 1/sqrt(2/pi)
  
  dfs[distribution=="truncated Lévy",]$maxturndist = 3*(1/sqrt(2/pi))
  dfs[distribution=="truncated Lévy",]$distributionpar = 
    optimise(
      f = function(pr, maxturndist, meanturndist) {
        temp = -pr + ((sqrt(pr*maxturndist) * exp(-pr/(2*maxturndist)) * sqrt(2/pi)) /
                        (2 * pnorm((sqrt(pr/maxturndist)/sqrt(2)) * sqrt(2), lower = FALSE)))
        return(abs(temp-meanturndist))
      },
      interval = c(0,3*(1/sqrt(2/pi))),
      maxturndist = 3*(1/sqrt(2/pi)),
      meanturndist = 1
    )$minimum
  
  ## Calculate quantile function for these parameter values per distribution type
  dfs[distribution=="degenerate", 
      sl := distributionpar]
  dfs[distribution=="uniform", 
      sl := qunif(cdf,
                  max = distributionpar)]
  dfs[distribution=="folded Gaussian", 
      sl := abs(qnorm(cdf,
                      sd = distributionpar))]
  dfs[distribution=="truncated Lévy", 
      sl := ((distributionpar /
                qnorm(1 - (cdf *
                             (2 * (1 - pnorm((maxturndist/distributionpar)^(-.5)))))
                      /2)^2))]
  
  ## Approximate PDFs
  dfs[, pdf := c(NA_real_, 
                 1e-4/diff(sl)),
      by = distribution]
  dfs[distribution=="folded Gaussian", 
      pdf := 2*abs(pdf)]
  
  ## Replace correct Inf PDF of degenerate distribution at mean value
  ## with values suitable for plotting
  dfs = dfs[distribution!="degenerate",]
  dfs = rbind(dfs,
              data.table('distribution'="degenerate",
                         'meanturndist'=1,
                         'maxturndist'=1,
                         'distributionpar'=1,
                         'sl'=1,
                         'cdf'=1,
                         'pdf'=c(0,10)))
  
  ## Change distribution column to factor for plotting
  dfs[, distribution := 
        factor(as.character(distribution), 
               levels = c("degenerate",
                          "folded Gaussian",
                          "truncated Lévy",
                          "uniform"))]
  
  ## Visualize PDFs
  p = ggplot(dfs,
             aes(x=sl, y=pdf,
                 group=distribution,
                 color=distribution)) +
    geom_line() +
    coord_cartesian(xlim=c(0,3.5*(1/sqrt(2/pi))),
                    ylim=c(0,1.25),
                    expand=0) +
    labs(x = "interturn distance",
         y = "PDF",
         color = "interturn distribution") +
    scale_color_manual(values=c("#E69F00",
                                "#56B4E9",
                                "#009E73",
                                "#0072B2")) +
    scale_x_continuous(breaks = c(0,
                                  1,
                                  (1/sqrt(2/pi)),
                                  2,
                                  3*(1/sqrt(2/pi))),
                       minor_breaks = NULL,
                       labels = c(0,
                                  expression(mu),
                                  expression(mu/sqrt(2/pi)),
                                  expression(2*mu),
                                  expression(3*mu/sqrt(2/pi)))) +
    theme(legend.position="top")
  
  ## Save figure
  ggsave("../plots/fig5.pdf",
         p,
         width = 3000,
         height = 2000,
         units = "px")
}
