### Load functions
{
  ## Compute expected poacher movement distance
  
  totalDistance = function(
    a = 19623, # reserve area (km^2)
    r = .15, # poacher observation radius (km)
    n # number of rhinos
  ) {
    
    return(a / (2*r*(n+1)))
  }
  
  ## Compute poaching pressure and effort per year
  
  poachingPressureEffort = function(
    year, # vector of years in which rhinos were counted
    estimate, # vector of estimated rhino population sizes
    LCL, # vector of lower confidence limit rhino population sizes
    UCL, # vector of upper confidence limit rhino population sizes
    poached # vector of estimated rhino poaching incidents
    ) {
    
    # Load package if needed
    require(data.table)
    
    # Combine arguments in table
    rhino = data.table(year = year,
                       estimate = estimate,
                       LCL = LCL,
                       UCL = UCL,
                       poached = poached)
    
    # Reserve columns for output
    rhino[, poachingPressure := NA_real_]
    rhino[, poachingPressure_LCL := NA_real_]
    rhino[, poachingPressure_UCL := NA_real_]
    
    # Compute total distance moved per year
    for(i in seq_len(nrow(rhino)-1)) {
      rhino[(i+1), 
            poachingPressure := 
              sum(totalDistance(n = seq(from = rhino[i,]$estimate,
                                        to = rhino[(i+1),]$estimate,
                                        length.out = rhino[(i+1),]$poached)))]
      rhino[(i+1), 
            poachingPressure_LCL := 
              sum(totalDistance(n = seq(from = rhino[i,]$UCL,
                                        to = rhino[(i+1),]$UCL,
                                        length.out = rhino[(i+1),]$poached)))]
      rhino[(i+1), 
            poachingPressure_UCL := 
              sum(totalDistance(n = seq(from = rhino[i,]$LCL,
                                        to = rhino[(i+1),]$LCL,
                                        length.out = rhino[(i+1),]$poached)))]
    }
    
    # Compute average distance moved per poacher per year
    rhino[, `:=` (poachingEffort = poachingPressure/poached,
                  poachingEffort_LCL = poachingPressure_LCL/poached,
                  poachingEffort_UCL = poachingPressure_UCL/poached)]
    
    # Remove first year, which can't be computed
    rhino = rhino[-1,]
    
    # Return data
    return(rhino)
  }
}

### Example
{
  ## Set random seed for data generation
  set.seed(1)
  
  ## Run function with generated data
  examplePressureEffort = 
    poachingPressureEffort(year = 2010:2020,
                           estimate = round((12:2)*1000 + rnorm(11,0,100)),
                           LCL = round((11:1)*1000 + rnorm(11,0,10)),
                           UCL = round((13:3)*1000 + rnorm(11,0,10)),
                           poached = rep(1000,11))
  
  ## Send output to console
  print(examplePressureEffort)
}
