import math
import numpy as np
import scipy.stats as sps
import gurobipy as gp
from gurobipy import GRB,quicksum
import gc
from utility import Utility
from fleet import Vessel, Fleet
from network import Services, Network
from shipper import Shipment, Shipper

class IWToperator:
    def __init__(self, network, fleet, util, ocosts, vcosts, tsailod):
        self.network = network
        self.fleet = fleet
        self.util = util
        N_nodes=len(self.network.listTerminals)
        N = range(N_nodes)
        N_Vtypes = len(self.fleet.vesselnames)
        K=range(N_Vtypes)
        N_services = len(self.network.services.nameslist)
        S=range(N_services)
        fixed_costs=np.zeros((N_Vtypes,N_services))
        for k in K:
            for s in S:
                fcost=0
                for leg in self.network.services.legsmatrix[s]:
                    for i in N:
                        for j in N:
                            fcost+=leg[i][j]*ocosts[k][i][j]
                fixed_costs[k][s] = fcost
        self.fixcosts = fixed_costs
        
        var_costs=np.zeros((N_Vtypes,N_services,N_nodes,N_nodes))
        for k in K:
            for i in N:
                for j in N:
                    if i!=j:
                        for s in S:
                            vcost=0
                            for l in range(2*(len(self.network.services.nameslist[s])-1)):
                                vcost+=self.network.services.usagematrix[s][l][i][j]*sum(self.network.services.legsmatrix[s][l][a][b]*vcosts[k][a][b] for a in N for b in N)
                            if vcost==0:
                                var_costs[k][s][i][j]=100000
                            else:
                                var_costs[k][s][i][j]=vcost
        self.varcosts = var_costs

        t_sail=np.zeros((N_Vtypes,N_services))
        t_port=np.zeros((N_Vtypes,N_services))
        for k in K:
            for s in S:
                tstemp=0
                tptemp=0
                for l in range(2*(len(self.network.services.nameslist[s])-1)):
                    tstemp+=sum(self.network.services.legsmatrix[s][l][a][b]*tsailod[k][a][b] for a in N for b in N)
                    tptemp+=sum(self.network.services.legsmatrix[s][l][a][b]*(self.network.waitIWTterm[b]+2*round(self.fleet.vesselcaps[k]*0.024)) for a in N for b in N)
                t_sail[k][s] = tstemp
                t_port[k][s] = tptemp
        self.tsail = t_sail
        self.tport = t_port
                
        maxCycles=np.zeros((N_Vtypes,N_services), dtype=int)
        for k in K:
            for s in S:
                maxCycles[k][s]=math.floor(self.fleet.vesseloptime[k]/(self.tsail[k][s]+self.tport[k][s]))
        self.maxcycles = maxCycles
        
        Bound_f=np.zeros((N_nodes,N_nodes), dtype=int)
        for i in N:
            for j in N:
                Bound_f[i][j]=int(math.ceil(math.log2(35)))       
        self.bound_f = Bound_f
        
        initprices=np.zeros((N_nodes,N_nodes))
        for i in N:
            for j in N:
                if i != j:
                    initprices[i][j]=100
        self.pricesOD = initprices
        
        initfreqs=np.ones((N_services,N_Vtypes))
        initvesseluses=np.ones((N_services,N_Vtypes))
        self.freqs = initfreqs
        self.vesseluses = initvesseluses
        
        
    def getFreqOD(self):
        N_nodes=len(self.network.listTerminals)
        N = range(N_nodes)
        N_Vtypes = len(self.fleet.vesselnames)
        K=range(N_Vtypes)
        N_services = len(self.network.services.nameslist)
        S=range(N_services)
        freqod=np.zeros((N_nodes,N_nodes), dtype=int)
        for i in N:
            for j in N:
                if i!=j:
                    freqod[i][j]=sum(self.network.services.servODmatrix[s][i][j]*self.freqs[s][k] for s in S for k in K)
        return freqod
     
    def getCapOD(self):
        N_nodes=len(self.network.listTerminals)
        N = range(N_nodes)
        N_Vtypes = len(self.fleet.vesselnames)
        K=range(N_Vtypes)
        N_services = len(self.network.services.nameslist)
        S=range(N_services)
        capod=np.zeros((N_nodes,N_nodes), dtype=int)
        for i in N:
            for j in N:
                if i!=j:
                    capod[i][j]=sum(self.network.services.servODmatrix[s][i][j]*self.freqs[s][k]*self.fleet.vesselcaps[k] for s in S for k in K)
        return capod
           
    def solveSNDP(self, method, compet_prices, compet_freqs, N_draws, MU_b_c_inter, boundP=10000, boundF=10000, seedgen = 0, SIG_b_c_inter = 0, myvolumes=[], icvolumes=[], roadrailvolumes={}):
        N_nodes=len(self.network.listTerminals)
        N = range(N_nodes)
        N_Vtypes = len(self.fleet.vesselnames)
        K=range(N_Vtypes)
        N_services = len(self.network.services.nameslist)
        S=range(N_services)
        R=range(N_draws)
        maxLegs=2*(N_nodes-1)
        L=range(maxLegs)
        np.random.seed(seedgen)
        
        R_b_c_inter = []
        sysEpsilons = []
        if self.util.type == 'Deter':
            for r in R:
                R_b_c_inter.append(MU_b_c_inter)
                sysEpsilons.append(0)
        elif self.util.type == 'MNL':
            for r in R:
                R_b_c_inter.append(MU_b_c_inter)
                sysEpsilons.append(sps.gumbel_r.rvs())
        elif self.util.type == 'Mixed':
            for r in R:
                R_b_c_inter.append(-np.exp(MU_b_c_inter + SIG_b_c_inter * sps.norm.rvs()))
                sysEpsilons.append(sps.gumbel_r.rvs())
        Eps_IC=np.random.permutation(sysEpsilons)
        Eps_IWT=np.random.permutation(sysEpsilons)
        Eps_ROAD=np.random.permutation(sysEpsilons)
        Eps_RAIL=np.random.permutation(sysEpsilons)
                
        Vroad=np.zeros((N_nodes,N_nodes,N_draws))
        Vrail=np.zeros((N_nodes,N_nodes,N_draws))
        Vic=np.zeros((N_nodes,N_nodes,N_draws))
        OD_matrix_allM=np.zeros((N_nodes,N_nodes))
        acc=np.zeros((N_nodes,N_nodes))
        portif=np.zeros((N_nodes,N_nodes))
        VOTcostIWT=np.zeros((N_nodes,N_nodes))
        
        if roadrailvolumes!={}:
            deducedutils=self.computeCompUtils(myvolumes, icvolumes, roadrailvolumes, MU_b_c_inter, SIG_b_c_inter)
            for i in N:
                for j in N:
                    if i!=j:
                        odij=self.network.ODs[self.network.listTerminals[i]+self.network.listTerminals[j]]
                        OD_matrix_allM[i][j]=odij.totDemand
                        acc[i][j]=odij.accIWT
                        portif[i][j]=odij.portIWT
                        VOTcostIWT[i][j]=odij.VOTcIWT
                        for r in R:
                            Vroad[i][j][r]=deducedutils['Road'][i][j]+Eps_ROAD[r]
                            Vrail[i][j][r]=deducedutils['Rail'][i][j]+Eps_RAIL[r]
                            Vic[i][j][r]=deducedutils['IC'][i][j]+Eps_IC[r]
        else:
            for i in N:
                for j in N:
                    if i!=j:
                        odij=self.network.ODs[self.network.listTerminals[i]+self.network.listTerminals[j]]
                        OD_matrix_allM[i][j]=odij.totDemand
                        acc[i][j]=odij.accIWT
                        portif[i][j]=odij.portIWT
                        VOTcostIWT[i][j]=odij.VOTcIWT
                        for r in R:
                            Vroad[i][j][r]=self.util.a_road+self.util.b_c_road*(odij.cRoad+odij.VOTcRoad)/1000+self.util.b_acc_road*odij.accRoad+Eps_ROAD[r]
                            Vrail[i][j][r]=self.util.a_rail+R_b_c_inter[r]*(odij.cRail+odij.VOTcRail)/1000+self.util.b_acc_inter*odij.accRail+self.util.b_freq_inter*odij.fRail+Eps_RAIL[r]
                            Vic[i][j][r]=R_b_c_inter[r]*(compet_prices[i][j]+odij.VOTcIWT)/1000+self.util.b_acc_inter*odij.accIWT+self.util.b_freq_inter*compet_freqs[i][j]+self.util.b_port*odij.portIWT+Eps_IC[r]   

        servOD_matrix=self.network.services.servODmatrix
        usage_matrix=self.network.services.usagematrix
        
        if method == 'Heuristic':
            # fix_costsH=np.zeros((N_nodes,N_nodes))
            # var_costsH=np.zeros((N_nodes,N_nodes))

            # for i in N:
                # for j in N:
                    # if i!=j:
                        # fix_costsH[i][j]=self.fixcosts[0][i+j-1]/2
                        # var_costsH[i][j]=self.varcosts[0][i+j-1][i][j]


            maxV=np.zeros((N_nodes,N_nodes,N_draws))
            maxALT=np.zeros((N_nodes,N_nodes,N_draws))
            for i in N:
                for j in N:
                    if i!=j:
                        for r in R:
                            maxV[i][j][r]=max(Vroad[i][j][r],Vrail[i][j][r],Vic[i][j][r])
                            maxALT[i][j][r]=np.argmax([Vroad[i][j][r],Vrail[i][j][r],Vic[i][j][r]])

            N_prices=50
            lower_bound_price=0
            P=range(lower_bound_price,N_prices)
            N_freq=36
            F=range(N_freq)

            Viwtf=np.zeros((N_freq,N_nodes,N_nodes,N_draws,N_prices))
            Diwtf=np.zeros((N_prices,N_nodes,N_nodes,N_freq))
            profitf=np.zeros((N_freq,N_freq,N_nodes,N_nodes,N_prices))
            maxPROFf=np.zeros((N_nodes,N_nodes,N_freq,N_freq))
            optPf=np.zeros((N_nodes,N_nodes,N_freq,N_freq))

            for f in F:
                if f > 0:
                    for i in N:
                        for j in N:
                            if i!=j:
                                odij=self.network.ODs[self.network.listTerminals[i]+self.network.listTerminals[j]]
                                OD_matrix_allM[i][j]=odij.totDemand
                                for p in P:
                                    D=0
                                    for r in R:
                                        Viwtf[f][i][j][r][p]=R_b_c_inter[r]*(p*10+odij.VOTcIWT)/1000+self.util.b_freq_inter*f+self.util.b_acc_inter*odij.accIWT+self.util.b_port*odij.portIWT+Eps_IWT[r]
                                        if Viwtf[f][i][j][r][p]>=maxV[i][j][r]:
                                            D+=OD_matrix_allM[i][j]/N_draws
                                    Diwtf[p][i][j][f]=D
                                    for fsmall in range(self.fleet.nvessels[0]*math.ceil(self.maxcycles[0][i+j-1])):
                                        for fbig in range(self.fleet.nvessels[1]*math.ceil(self.maxcycles[1][i+j-1])):
                                            if fsmall+fbig==f:
                                                Ksmall=fsmall*self.fleet.vesselcaps[0]
                                                Kbig=fbig*self.fleet.vesselcaps[1]
                                                if D >= Ksmall+Kbig:
                                                    D=Ksmall+Kbig
                                                    dsmall=Ksmall
                                                    dbig=Kbig
                                                elif D >= Kbig:
                                                    dbig=Kbig
                                                    dsmall=D-dbig
                                                else:
                                                    dbig=D
                                                    dsmall=0
                                                profitf[fsmall][fbig][i][j][p]=D*p*10-(fsmall*self.fixcosts[0][i+j-1]/2+fbig*self.fixcosts[1][i+j-1]/2)-(dsmall*self.varcosts[0][i+j-1][i][j]+dbig*self.varcosts[1][i+j-1][i][j])
                                for fsmall in range(self.fleet.nvessels[0]*math.ceil(self.maxcycles[0][i+j-1])):
                                    for fbig in range(self.fleet.nvessels[1]*math.ceil(self.maxcycles[1][i+j-1])):
                                        if fsmall+fbig==f:
                                            optPf[i][j][fsmall][fbig]=10*np.argmax(profitf[fsmall][fbig][i][j])
                                            maxPROFf[i][j][fsmall][fbig]=np.max(profitf[fsmall][fbig][i][j])

            price1=np.zeros((N_nodes,N_nodes))
            f1=np.zeros((N_services,N_Vtypes))

            price2=np.zeros((N_nodes,N_nodes))
            f2=np.zeros((N_services,N_Vtypes))
            
            for i in N:
                for j in N:
                    if i!=j:
                        imax=np.unravel_index(np.argmax(maxPROFf[i][j]), maxPROFf[i][j].shape)
                        price2[i][j]=optPf[i][j][imax[0]][imax[1]]

            Flist=list(F)

            # del fix_costsH
            # del var_costsH
            del maxPROFf
            del Viwtf
            del profitf
            gc.collect()
            
            pvisit = []
            fvisit = []

            while (not price2.tolist() in pvisit) or (not f2.tolist() in fvisit):
                # print(price2)
                price1=np.copy(price2)
                pvisit.append(price1.tolist())
                f1=np.copy(f2)
                fvisit.append(f1.tolist())

                DemandMaxIWT=np.zeros((N_nodes,N_nodes,N_freq))
                for i in N:
                    for j in N:
                        DemandMaxIWT[i][j]=Diwtf[int(price1[i][j]/10)][i][j]

                n = gp.Model("SND_SAA.lp")

                freq = []
                bfreq=[]
                Fod=[]
                Dmax=[]
                flow = []
                vessel = []

                for s in S:
                    freq.append([])
                    vessel.append([])
                    for k in K:
                        freq[s].append(n.addVar(lb = 0, vtype = GRB.INTEGER, name = 'freq_sk('+str(s)+','+str(k)+')'))
                        vessel[s].append(n.addVar(lb = 0, vtype = GRB.INTEGER, name = 'vessel_sk('+str(s)+','+str(k)+')'))

                z = []

                for i in N:
                    flow.append([])
                    z.append([])
                    bfreq.append([])
                    Fod.append([])
                    Dmax.append([])
                    for j in N:
                        z[i].append([])
                        flow[i].append([])
                        bfreq[i].append([])
                        Fod[i].append(n.addVar(lb = 0, vtype = GRB.INTEGER, name = 'freq_ij('+str(i)+','+str(j)+')'))
                        Dmax[i].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'Dmax_ij('+str(i)+','+str(j)+')'))
                        for s in S:
                            flow[i][j].append([])
                            for k in K:
                                flow[i][j][s].append([])
                                for r in R:
                                    flow[i][j][s][k].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'x_ijskr('+str(i)+','+str(j)+','+str(s)+','+str(k)+','+str(r)+')'))
                        for r in R:
                            z[i][j].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'z1_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        for f in F:
                            bfreq[i][j].append(n.addVar(lb = 0, vtype = GRB.BINARY, name = 'bf_ijf('+str(i)+','+str(j)+','+str(f)+')'))

                n.setObjective((1/N_draws)*(quicksum(price1[i][j]*flow[i][j][s][k][r] for i in N for j in N for s in S for k in K for r in R)-quicksum(freq[s][k]*self.fixcosts[k][s] for s in S for k in K for r in R)-quicksum(flow[i][j][s][k][r]*(self.varcosts[k][s][i][j]) for k in K for s in S for i in N for j in N for r in R)))

                n.modelSense = GRB.MAXIMIZE
                n.update()


                for i in N:
                    for j in N:
                        for s in S:
                            for k in K:
                                for r in R:
                                    n.addConstr(flow[i][j][s][k][r] <= quicksum(usage_matrix[s][l][i][j] for l in L)*OD_matrix_allM[i][j])

                for k in K:     
                    for s in S:
                        n.addConstr(freq[s][k]<=self.maxcycles[k][s]*vessel[s][k])
                        n.addConstr(quicksum(freq[s][k] for k in K) >= quicksum(self.freqs[s][k] for k in K) - boundF)
                        n.addConstr(quicksum(freq[s][k] for k in K) <= quicksum(self.freqs[s][k] for k in K) + boundF)
                        for l in L:
                            n.addConstr(quicksum((1/N_draws)*flow[i][j][s][k][r]*usage_matrix[s][l][i][j] for i in N for j in N for r in R) <= freq[s][k]*self.fleet.vesselcaps[k])

                for k in K:
                    n.addConstr(quicksum(vessel[s][k] for s in S)<=self.fleet.nvessels[k])

                for i in N:
                    for j in N:
                        for r in R:
                            n.addConstr(quicksum(flow[i][j][s][k][r] for s in S for k in K)+z[i][j][r]==OD_matrix_allM[i][j])

                for i in N:
                    for j in N:
                        n.addConstr(quicksum(bfreq[i][j][f] for f in F)<=1)
                        n.addConstr(Dmax[i][j]==quicksum(bfreq[i][j][f]*DemandMaxIWT[i][j][f] for f in F))
                        n.addConstr(Fod[i][j]==quicksum(bfreq[i][j][f]*Flist[f] for f in F))
                        n.addConstr(Fod[i][j]==quicksum(servOD_matrix[s][i][j]*freq[s][k] for s in S for k in K))
                        n.addConstr(quicksum((1/N_draws)*flow[i][j][s][k][r] for s in S for k in K for r in R)<=Dmax[i][j])

                n.update()


                n.write('TEST_SNDpaper.lp')
                n.setParam('OutputFlag', False)
                n.optimize()
                #print(n.ObjVal)
                for i in N:
                    for j in N:
                        if i!=j:
                            fs=int(sum(servOD_matrix[s][i][j]*freq[s][0].x for s in S))
                            fb=int(sum(servOD_matrix[s][i][j]*freq[s][1].x for s in S))
                            price2[i][j]=optPf[i][j][fs][fb]
                    
            for i in N:
                for j in N:
                    self.pricesOD[i][j] = price2[i][j]
                    
            for k in K:
                for s in S:
                    self.freqs[s][k] = freq[s][k].x
                    self.vesseluses[s][k] = vessel[s][k].x
                    
            IWTvol=np.zeros((N_nodes,N_nodes))
            ICvol=np.zeros((N_nodes,N_nodes))
            for i in N:
                for j in N:
                    IWTvol[i][j]=sum(flow[i][j][s][k][r].x for s in S for k in K for r in R)/N_draws
                    volivola=0
                    for r in R:
                        if maxALT[i][j][r]==2:
                            volivola+=z[i][j][r].x
                    ICvol[i][j]=volivola/N_draws        
            
            results={'Profits': n.ObjVal, 'Prices':self.pricesOD, 'FreqsOD':self.getFreqOD(), 'CapsOD':self.getCapOD(), 'IWTvol': IWTvol, 'ICvol': ICvol}
            return results 


        if method=='Exact':

            n = gp.Model("SND_SAA.lp")

            freq = []
            freqbin= []
            flow = []
            vessel = []
            product = []

            for s in S:
                freq.append([])
                vessel.append([])
                for k in K:
                    freq[s].append(n.addVar(lb = 0, vtype = GRB.INTEGER, name = 'freq_sk('+str(s)+','+str(k)+')'))
                    vessel[s].append(n.addVar(lb = 0, vtype = GRB.INTEGER, name = 'vessel_sk('+str(s)+','+str(k)+')'))

            price = []
            zroad = []
            zrail = []
            zic=[]
            lagra = []
            yIiwt = []
            yIIiwt = []
            yIroad = []
            yIIroad = []
            yIrail = []
            yIIrail = []
            yIic = []
            yIIic = []

            for i in N:
                price.append([])
                freqbin.append([])
                flow.append([])
                zroad.append([])
                zrail.append([])
                zic.append([])
                lagra.append([])
                yIiwt.append([])
                yIIiwt.append([])
                yIroad.append([])
                yIIroad.append([])
                yIrail.append([])
                yIIrail.append([])
                yIic.append([])
                yIIic.append([])
                product.append([])
                for j in N:
                    price[i].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'p_ij('+str(i)+','+str(j)+')'))
                    freqbin[i].append([])
                    zroad[i].append([])
                    zrail[i].append([])
                    zic[i].append([])
                    lagra[i].append([])
                    yIiwt[i].append([])
                    yIIiwt[i].append([])
                    yIroad[i].append([])
                    yIIroad[i].append([])
                    yIrail[i].append([])
                    yIIrail[i].append([])
                    yIic[i].append([])
                    yIIic[i].append([])
                    flow[i].append([])
                    product[i].append([])
                    for b in range(self.bound_f[i][j]):
                        freqbin[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'freqbin_ijb('+str(i)+','+str(j)+','+str(b)+')'))
                    for s in S:
                        flow[i][j].append([])
                        product[i][j].append([]) 
                        for k in K:
                            flow[i][j][s].append([])
                            product[i][j][s].append([])
                            for b in range(self.bound_f[i][j]):
                                product[i][j][s][k].append([])
                                for r in R:
                                    product[i][j][s][k][b].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'a_ijskbr('+str(i)+','+str(j)+','+str(s)+','+str(k)+','+str(b)+','+str(r)+')'))
                            for r in R:
                                flow[i][j][s][k].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'x_ijskr('+str(i)+','+str(j)+','+str(s)+','+str(k)+','+str(r)+')'))
                    for r in R:
                        zroad[i][j].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'z1_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        zrail[i][j].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'z2_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        zic[i][j].append(n.addVar(lb = 0, vtype = GRB.CONTINUOUS, name = 'z3_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        lagra[i][j].append(n.addVar(lb = -GRB.INFINITY, vtype = GRB.CONTINUOUS, name = 'l_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        yIiwt[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'y0I_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        yIIiwt[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'y0II_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        yIroad[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'y1I_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        yIIroad[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'y1II_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        yIrail[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'y2I_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        yIIrail[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'y2II_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        yIic[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'y3I_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                        yIIic[i][j].append(n.addVar(vtype = GRB.BINARY, name = 'y3II_ijr('+str(i)+','+str(j)+','+str(r)+')'))
                    
            n.setObjective(quicksum((1000/(-R_b_c_inter[r]))*quicksum(OD_matrix_allM[i][j]/(N_draws)*lagra[i][j][r]+Vroad[i][j][r]*zroad[i][j][r]+Vrail[i][j][r]*zrail[i][j][r]+Vic[i][j][r]*zic[i][j][r]+quicksum((self.util.b_acc_inter*acc[i][j]+self.util.b_port*portif[i][j]+Eps_IWT[r])*flow[i][j][s][k][r] for s in S for k in K)+quicksum(self.util.b_freq_inter*quicksum((2**b)*product[i][j][s][k][b][r] for b in range(self.bound_f[i][j])) for s in S for k in K) for i in N for j in N) for r in R)-quicksum(freq[s][k]*self.fixcosts[k][s] for s in S for k in K)-quicksum(flow[i][j][s][k][r]*(self.varcosts[k][s][i][j]+VOTcostIWT[i][j]) for k in K for s in S for i in N for j in N for r in R))
            n.modelSense = GRB.MAXIMIZE
            n.update()


            for i in N:
                for j in N:
                    for s in S:
                        for k in K:
                            for r in R:
                                n.addConstr(flow[i][j][s][k][r] <= quicksum(usage_matrix[s][l][i][j] for l in L)*OD_matrix_allM[i][j]/(N_draws))
                        
            for k in K:     
                for s in S:
                    n.addConstr(freq[s][k]<=self.maxcycles[k][s]*vessel[s][k])
                    for l in L:
                        n.addConstr(quicksum(flow[i][j][s][k][r]*usage_matrix[s][l][i][j] for i in N for j in N for r in R) <= freq[s][k]*self.fleet.vesselcaps[k])

            for k in K:
                n.addConstr(quicksum(vessel[s][k] for s in S)<=self.fleet.nvessels[k])
                

            for i in N:
                for j in N:
                    n.addConstr(price[i][j] >= self.pricesOD[i][j] - boundP)
                    n.addConstr(price[i][j] <= self.pricesOD[i][j] + boundP)
                    n.addConstr(quicksum(freq[s][k] for k in K) >= quicksum(self.freqs[s][k] for k in K) - boundF)
                    n.addConstr(quicksum(freq[s][k] for k in K) <= quicksum(self.freqs[s][k] for k in K) + boundF)
                    for r in R:
                        n.addConstr(quicksum(flow[i][j][s][k][r] for s in S for k in K)+zroad[i][j][r]+zrail[i][j][r]+zic[i][j][r]==OD_matrix_allM[i][j]/(N_draws))
                        n.addConstr(lagra[i][j][r]<=-(R_b_c_inter[r]*(price[i][j]+VOTcostIWT[i][j])/1000+self.util.b_freq_inter*quicksum(servOD_matrix[s][i][j]*freq[s][k] for s in S for k in K)+self.util.b_acc_inter*acc[i][j]+self.util.b_port*portif[i][j]+Eps_IWT[r]))
                        n.addConstr(lagra[i][j][r]<=-Vroad[i][j][r])
                        n.addConstr(lagra[i][j][r]<=-Vrail[i][j][r])
                        n.addConstr(lagra[i][j][r]<=-Vic[i][j][r])
                        n.addConstr(-(R_b_c_inter[r]*(price[i][j]+VOTcostIWT[i][j])/1000+self.util.b_freq_inter*quicksum(servOD_matrix[s][i][j]*freq[s][k] for s in S for k in K)+self.util.b_acc_inter*acc[i][j]+self.util.b_port*portif[i][j]+Eps_IWT[r])-lagra[i][j][r]<=100000*yIiwt[i][j][r])
                        n.addConstr(quicksum(flow[i][j][s][k][r] for s in S for k in K)<=OD_matrix_allM[i][j]/(N_draws)*yIIiwt[i][j][r])
                        n.addConstr(yIiwt[i][j][r]+yIIiwt[i][j][r]<=1)
                        n.addConstr(-Vroad[i][j][r]-lagra[i][j][r]<=100000*yIroad[i][j][r])
                        n.addConstr(zroad[i][j][r]<=OD_matrix_allM[i][j]/(N_draws)*yIIroad[i][j][r])
                        n.addConstr(yIroad[i][j][r]+yIIroad[i][j][r]<=1)
                        n.addConstr(-Vrail[i][j][r]-lagra[i][j][r]<=100000*yIrail[i][j][r])
                        n.addConstr(zrail[i][j][r]<=OD_matrix_allM[i][j]/(N_draws)*yIIrail[i][j][r])
                        n.addConstr(yIrail[i][j][r]+yIIrail[i][j][r]<=1)
                        n.addConstr(-Vic[i][j][r]-lagra[i][j][r]<=100000*yIic[i][j][r])
                        n.addConstr(zic[i][j][r]<=OD_matrix_allM[i][j]/(N_draws)*yIIic[i][j][r])
                        n.addConstr(yIic[i][j][r]+yIIic[i][j][r]<=1)
                        
            for i in N:
                for j in N:        
                    n.addConstr(quicksum(servOD_matrix[s][i][j]*freq[s][k] for s in S for k in K)==quicksum((2**b)*freqbin[i][j][b] for b in range(self.bound_f[i][j])))
                    n.addConstr(quicksum(servOD_matrix[s][i][j]*freq[s][k] for s in S for k in K)<=35)
                    for s in S:
                        for k in K:
                            for b in range(self.bound_f[i][j]):
                                for r in R:
                                    n.addConstr(product[i][j][s][k][b][r]<=OD_matrix_allM[i][j]/(N_draws)*freqbin[i][j][b])
                                    n.addConstr(product[i][j][s][k][b][r]<=flow[i][j][s][k][r])
                                    n.addConstr(product[i][j][s][k][b][r]>=flow[i][j][s][k][r]-OD_matrix_allM[i][j]/(N_draws)*(1-freqbin[i][j][b])) 
                   
            n.update()

            n.write('TEST_SNDpaper.lp')
            n.setParam('OutputFlag', False)
            # n.computeIIS()
            # n.write("model.ilp")
            n.optimize()
         
            for k in K:
                for s in S:
                    self.freqs[s][k] = freq[s][k].x
                    self.vesseluses[s][k] = vessel[s][k].x
                    
            for i in N:
                for j in N:
                    self.pricesOD[i][j] = round(price[i][j].x)
            
            IWTvol=np.zeros((N_nodes,N_nodes))
            ICvol=np.zeros((N_nodes,N_nodes))
            ROADvol=np.zeros((N_nodes,N_nodes))
            RAILvol=np.zeros((N_nodes,N_nodes))
            IWTvolR=np.zeros((N_nodes,N_nodes,N_draws))
            ICvolR=np.zeros((N_nodes,N_nodes,N_draws))
            ROADvolR=np.zeros((N_nodes,N_nodes,N_draws))
            RAILvolR=np.zeros((N_nodes,N_nodes,N_draws))
            lagragaga=np.zeros((N_nodes,N_nodes,N_draws))
            mytiil=np.zeros((N_nodes,N_nodes,N_draws))
            for i in N:
                for j in N:
                    IWTvol[i][j]=sum(flow[i][j][s][k][r].x for s in S for k in K for r in R)
                    ICvol[i][j]=sum(zic[i][j][r].x for r in R)
                    ROADvol[i][j]=sum(zroad[i][j][r].x for r in R)
                    RAILvol[i][j]=sum(zrail[i][j][r].x for r in R)
                    # for r in R:
                        # lagragaga[i][j][r]=-lagra[i][j][r].x
                        # IWTvolR[i][j][r]=sum(flow[i][j][s][k][r].x for s in S for k in K)
                        # ICvolR[i][j][r]=zic[i][j][r].x
                        # ROADvolR[i][j][r]=zroad[i][j][r].x
                        # RAILvolR[i][j][r]=zrail[i][j][r].x
                        # mytiil[i][j][r]=R_b_c_inter[r]*(price[i][j].x+VOTcostIWT[i][j])/1000+self.util.b_freq_inter*sum(servOD_matrix[s][i][j]*freq[s][k].x for s in S for k in K)+self.util.b_acc_inter*acc[i][j]+self.util.b_port*portif[i][j]+Eps_IWT[r]
        
            results={'Profits': n.ObjVal, 'Prices':self.pricesOD, 'FreqsOD':self.getFreqOD(), 'CapsOD':self.getCapOD(), 'IWTvol': IWTvol, 'ICvol': ICvol}

            return results
    
    def computeActRevenues(self, transportedVolumes):
        N_nodes=len(self.network.listTerminals)
        N = range(N_nodes)
        return sum(self.pricesOD[i][j]*transportedVolumes[i][j] for i in N for j in N)
    
    def computeActFcosts(self):
        N_Vtypes = len(self.fleet.vesselnames)
        K=range(N_Vtypes)
        N_services = len(self.network.services.nameslist)
        S=range(N_services)
        return sum(self.freqs[s][k]*self.fixcosts[k][s] for s in S for k in K)
                
    def computeActVcosts(self, transportedVolumes):
        N_nodes=len(self.network.listTerminals)
        N = range(N_nodes)
        N_Vtypes = len(self.fleet.vesselnames)
        K=range(N_Vtypes)
        N_services = len(self.network.services.nameslist)
        S=range(N_services)
        usedSERVICE=np.zeros((N_nodes,N_nodes,N_services,N_Vtypes))
        for s in S:
            for i in N:
                for j in N:
                    if self.network.services.servODmatrix[s][i][j]==1:
                        for k in K:
                            if self.freqs[s][k]>0:
                                usedSERVICE[i][j][s][k]=1

        denominator=np.ones((N_nodes,N_nodes))
        for i in N:
            for j in N:
                sumserv=sum(usedSERVICE[i][j][s][k] for k in K for s in S)
                if sumserv>0:
                    denominator[i][j]=sumserv
        
        actvCosts=sum(transportedVolumes[i][j]*sum(usedSERVICE[i][j][s][k]*self.varcosts[k][s][i][j] for k in K for s in S)/(denominator[i][j]) for i in N for j in N)
        return actvCosts
        
    def computeActProfits(self, transportedVolumes):
        N_nodes=len(self.network.listTerminals)
        N = range(N_nodes)
        N_Vtypes = len(self.fleet.vesselnames)
        K=range(N_Vtypes)
        N_services = len(self.network.services.nameslist)
        S=range(N_services)
        revenue=sum(self.pricesOD[i][j]*transportedVolumes[i][j] for i in N for j in N)
        fcosts=sum(self.freqs[s][k]*self.fixcosts[k][s] for s in S for k in K)
        return revenue - fcosts - self.computeActVcosts(transportedVolumes)        

    def computeCompUtils(self, myvolumes, icvolumes, roadrailvolumes, MU_b_c_inter, SIG_b_c_inter):
        N_nodes=len(self.network.listTerminals)
        N = range(N_nodes)
        N_Vtypes = len(self.fleet.vesselnames)
        K=range(N_Vtypes)
        N_services = len(self.network.services.nameslist)
        S=range(N_services)
        Roadutil=np.zeros((N_nodes,N_nodes))
        Railutil=np.zeros((N_nodes,N_nodes))
        ICutil=np.zeros((N_nodes,N_nodes))
        capods=self.getCapOD()
        if (self.util.type == 'Deter') or (self.util.type == 'MNL'):
            b_c_inter = MU_b_c_inter
        elif self.util.type == 'Mixed':
            b_c_inter = -np.exp(MU_b_c_inter + ((SIG_b_c_inter**2)/2))
        for i in N:
            for j in N:
                if i!=j:
                    odij=self.network.ODs[self.network.listTerminals[i]+self.network.listTerminals[j]]
                    mydeterutil=b_c_inter*(self.pricesOD[i][j]+odij.VOTcIWT)/1000+self.util.b_freq_inter*sum(self.network.services.servODmatrix[s][i][j]*self.freqs[s][k] for s in S for k in K)+self.util.b_acc_inter*odij.accIWT+self.util.b_port*odij.portIWT
                    if myvolumes[i][j]>0:
                        if (myvolumes[i][j]/capods[i][j]<0.95) or (myvolumes[i][j]>=max(roadrailvolumes["ROAD"][i][j],roadrailvolumes["RAIL"][i][j],icvolumes[i][j])):
                            if roadrailvolumes["ROAD"][i][j]>0:
                                Roadutil[i][j]=mydeterutil + np.log(roadrailvolumes["ROAD"][i][j]/myvolumes[i][j])
                            else:
                                Roadutil[i][j]=mydeterutil + np.log(0.1/myvolumes[i][j])
                            if roadrailvolumes["RAIL"][i][j]>0:
                                Railutil[i][j]=mydeterutil + np.log(roadrailvolumes["RAIL"][i][j]/myvolumes[i][j])
                            else:
                                Railutil[i][j]=mydeterutil + np.log(0.1/myvolumes[i][j])
                            if icvolumes[i][j]>0:
                                ICutil[i][j]=mydeterutil + np.log(icvolumes[i][j]/myvolumes[i][j])
                            else:
                                ICutil[i][j]=mydeterutil + np.log(0.1/myvolumes[i][j])
                        else:
                            Roadutil[i][j]=mydeterutil #+ np.log(0.1/myvolumes[i][j])
                            Railutil[i][j]=mydeterutil #+ np.log(0.1/myvolumes[i][j])
                            ICutil[i][j]=mydeterutil #+ np.log(0.1/myvolumes[i][j])                                
                    else:
                        if roadrailvolumes["ROAD"][i][j]>0:
                            Roadutil[i][j]=mydeterutil + np.log(roadrailvolumes["ROAD"][i][j]/0.1)
                        else:
                            Roadutil[i][j]=mydeterutil + np.log(0.1/0.1)
                        if roadrailvolumes["RAIL"][i][j]>0:
                            Railutil[i][j]=mydeterutil + np.log(roadrailvolumes["RAIL"][i][j]/0.1)
                        else:
                            Railutil[i][j]=mydeterutil + np.log(0.1/0.1)
                        if icvolumes[i][j]>0:
                            ICutil[i][j]=mydeterutil + np.log(icvolumes[i][j]/0.1)
                        else:
                            ICutil[i][j]=mydeterutil + np.log(0.1/0.1)
        return {"Road":Roadutil, "Rail":Railutil, "IC":ICutil}
                   
                
                
                