%% GW_LFV_Exper.m
close all
clear
clc

disp('Experimental Analysis of the Vibration Effects on GW Propagation');
disp(' ');

%% Select folder of combined data files
curdir = pwd;
data_dir_1 = uigetdir(curdir, 'Select Data Directory 1 - LFV');
data_dir_2 = uigetdir(curdir, 'Select Data Directory 2 - SW');
if data_dir_1 == 0
    hb = errordlg('LFV data folder not selected.','Execution Aborted');
    return
elseif data_dir_2 == 0
    hb = errordlg('SW data folder not selected.','Execution Aborted');
    return
end
dir_names = {data_dir_1,data_dir_2};
result_dir = curdir;

%% Controling parameters
opt_filter = 'narrow';
param = {[1,2,123];[2,2,123];[1,6,123];[1,6,213];[1,6,335]};
Npzt = 18;  % Pre-defined

%% PZT transducer properties
pzt_rad = 10e-3;
pzt_th = 0.4e-3;
Apzt = pi*(pzt_rad^2);
e0 = 8.85e-12;
e33T = 1900*e0;
k31 = 0.36;
d31 = -175e-12;
Y11E = 6.3e10;
s11E = 1/Y11E;
nu_pzt = 0.35;
Cpzt = e33T * (Apzt/pzt_th);
Zosc = 1e6;
Yosc = 1/Zosc;

%% memory allocation
res_mat = zeros(51,15); % Pre-defined according to analysis needs
baseline = cell(3,3);
counter = 0; % Total counter of the analysis

%% Main loop - For the desired actuator and frequency
% For each frequency
for p = 1:length(param)
    
    % Extract loop controlling parameters
    dir_num = param{p}(1);
    data_dir = dir_names{dir_num};
    act = param{p}(2);
    freq = param{p}(3);
    
    % Create file list
    if p == 1 || (p > 1 && (act ~= param{p-1}(2) || freq ~= param{p-1}(3)))
        disp(['Reading files for A',sprintf('%02d',act),', ',...
            num2str(freq),' kHz ...']);
    end
    
    files2search = [data_dir,'\*',num2str(freq),...
        'kHz*A',sprintf('%02d',act),'*.mat'];
    file_list = dir(files2search);
    
    % For each file in the file list
    for c = 1:length(file_list)
        
        % Open selected GW test file
        % The files are already ordered chronologically
        fdir = file_list(c).folder;
        fname = file_list(c).name;
        disp(fname);
        load(fullfile(fdir,fname));
        
        %% Extract information from file name string
        get_filename_info
        
        %% Extract variables from '.mat' files
        pLFV = strfind(fname,'_durLFV_');
        pSW = strfind(fname,'_cooling_system_');
        
        if isempty(pLFV) == 1 %%% NOT a LFV case %%%
            
            if isempty(pSW) == 1 %%% Baseline case %%%
                Nosc = 2;
                state_str = dmg_str;
                
                get_comb_data
                
                % Clear memory
                clearvars test_data
                
            else %%% Cooling system case %%%
                Nosc = 1;
                state_str = [dmg_str,'+SW',...
                    fname(pSW+length('cooling_system')+2)];
                
                get_pico_data
                
                % Clear memory
                clearvars Length RequestedLength Tinterval Tstart 
            end
            
        else %%% LFV case %%%
            Nosc = 1;
            state_str = [dmg_str,',',...
                fname(pLFV+1:pLFV+length('durLFV'))];
            
            get_pico_data
            
            % Clear memory
            clearvars Length RequestedLength Tinterval Tstart
            
        end
        
        % Grphical object array
        hf = gobjects(1,length(s_num));
                
        %% For each signal in each file
        for h = 1:Nosc
            dt_x = dt(h);
            dt_a = dt(1);
            t0_x = t0(h);
            t0_a = t0(1);
            L_x = sigL(h);
            L_a = sigL(1);
            
            t_sen = (t0_x:dt_x:(t0_x + ((L_x - 1) * dt_x)))';
            t_a_osc = (t0_a:dt_a:(t0_a + ((L_a - 1) * dt_a)))';
            
            si = s_dist{h}(1);
            sf = s_dist{h}(2);
            
            for s = si:sf
                
                %%% Correct 'state_str' for plotting
                p2 = strfind(state_str,',durLFV');
                if isempty(p2) == 0
                    state_str = strrep(state_str,',durLFV','+LFV');
                end
                
                %%% Filtering (CWT or other) and windowing (if necessary)
                if strcmp(state_str,'ND') == 1
                    tlim_max = t_sen(end);
                    t_tot_b = t_sen(end) - t_sen(1);
                end
                
                %tlim_max = 0.002; % from inspection
                %pend = find(t > tlim_max,1);
                %if isempty(pend) == 0
                t_tot = t_sen(end) - t_sen(1);
                if t_tot > t_tot_b
                    pend = find(t_sen > t_sen(1) + t_tot_b,1);
                    t = t_sen(1:pend-1);
                    t_a = t_a_osc(1:pend-1);
                    x_orig = x_sen{s}{2}(1:pend-1);
                    x_act = x_act(1:pend-1);
                    L_x = length(x_orig);
                    L_a = length(x_act);
                else
                    x_orig = x_sen{s}{2};
                    t = t_sen;
                    t_a = t_a_osc;
                end
                bd_lev = 3; % dB
                x_rec = cwtGWfilter(x_orig,x_act,L_a,dt_x,dt_a,...
                    bd_lev,opt_filter);
                
                %% Compute quantities for analysis of each signal
                
                %%% Cumultive energy
                % For the signal
                cum_e = cumtrapz(t,abs(x_rec).^2);
                cum_e_n = cum_e./max(cum_e);
                dcum_e = diff(cum_e)./diff(t);
                dcum_e_n = dcum_e./max(dcum_e);
                % For the excitation signal
                cum_e_a = cumtrapz(t_a,abs(x_act).^2);
                cum_e_a_n = cum_e_a./max(cum_e_a);
                dcum_e_a = diff(cum_e_a)./diff(t_a);
                dcum_e_a_n = dcum_e_a./max(dcum_e_a);
                
                %%% Time-of-Flight
                p0 = find(cum_e_n(1:end-1) >= 1e-2 & dcum_e_n >= 1e-1 ,1);
                %p0_a = find(cum_e_a_n(1:end-1) >= 1e-2 & ...
                %    dcum_e_a_n >= 1e-1 ,1);
                p0_a = find(cum_e_a_n(1:end-1) >= 1e-2,1);
                %tof = abs(t(p0) - t(p0_a));
                
                % Time-of-Flight (approach 2)
                [~,idx_a] = max(x_act);
                [max_xs,idx_s] = max(x_rec);
                tof = t(idx_s) - t(idx_a);
                
                %%% 10-percent duration time
                idx_10_perc = find(x_rec(idx_s:end) <= 0.1*max_xs,1);
                T10_perc = t(idx_10_perc + (idx_s-1)) - t(idx_s);
                
                %%% Estimate of signal-to-noise ratio
                %inst_rms = zeros(1,length(x_rec));
                %inst_rms(1) = x_rec(1);
                %for i = 2:length(x_rec)
                %   inst_rms(i) = rms(x_rec(2:i));
                %end
                %snr = (max(abs(x_rec))/rms(x_rec(round((2/3)*L_x):end)))^2;
                %snr = (max(inst_rms) / rms(x_rec(round((2/3)*L_x):end)))^2;
                snr = 20*log10(max(x_rec) / ...
                    rms(x_rec(round((2/3)*L_x):end)));
                
                %%% FFT spectrum
                fft_x = abs(fftshift(fft(x_rec)));                
                Fs = 1/dt_x;
                dF = Fs/L_x;
                f_vec = -Fs/2:dF:Fs/2-dF;
                % Find index of freq = 0
                ki = find(f_vec >= 0,1); % Find index of freq = 0
                kf = find(f_vec >= 1e6,1); % Find index of freq = 1 MHz
                fft_x = fft_x(ki:kf);
                f_vec = f_vec(ki:kf);
                fft_x_n = fft_x./max(fft_x);
                
                % Characteristic frequency
                fch = sum(fft_x' .* f_vec) / sum(fft_x);
                
                % Area under normalized FFT spectrum
                Afft = trapz(fft_x_n);
                Esfft = trapz(f_vec,abs(fft_x_n).^2);
                % Frequency of maximum FFT magnitude
                [max_mag,idx] = max(fft_x);
                fmain = f_vec(idx);
                
                %%% Frequency bandwidth
                bw_lev = 20;
                [f_bw,flo,fhi,~] = powerbw(x_rec,Fs,[],bw_lev);
                %fc = (flo+fhi)/2;
                
                %%% Intantaneous phase
                %ht_x = hilbert(x_rec);
                %phi = unwrap(angle(ht_x));
                %Aphi = trapz(phi);
                
                %%% Max amplitude of raw signal
                [mV,i_mV] = max(abs(x_orig));
                maxV = x_orig(i_mV);
                
                %%% Max amplitude of filtered signal
                [maxA,i_mA] = max(abs(x_rec));
                t_maxA = t(i_mA);
                
                %% Subplots of quantities for each signal
                hf(s) = figure('Units','normalized',...
                    'Position',[0 0 1/3 1]);
                %Nqt = 5;
                Nqt = 3;
                
                % Convert voltage to strain (approximation)
                Ypzt = 1i*(2*pi*fch)*Cpzt;
                x_strain = x_orig * ...
                    ((4*pi*e33T*(1-nu_pzt)) / (d31*Y11E*pzt_th)) * 1e6;
                
                %%% Plot raw voltage signal
                subplot(Nqt,1,1);
                plot(t*1e3,x_orig);
                xlim(1e3*[t(1) tlim_max]);
                ylabel('Amp [V]');
                xlabel('Time [ms]');
                title(['PZT ',num2str(act),' - PZT ',...
                    num2str(s_num(s)),', ',state_str,', ',freq_str,'kHz']);
                grid on
                %y_lims = ylim;
                %dy = abs(y_lims(2) - y_lims(1));
                %text(0.7*tlim_max*1e3,min(y_lims)+(0.4*dy),'Raw signal');
                %text(0.7*tlim_max*1e3,min(y_lims)+(0.8*dy),...
                %    ['maxV = ',sprintf('%0.2f',maxV), ' V']);
                text(0.7,0.4,'Raw signal','Units','normalized');
                text(0.7,0.8,['maxV = ',sprintf('%0.2f',maxV), ' V'],...
                    'Units','normalized');
                text(0.015,0.9,'A','Units','normalized',...
                    'FontWeight','bold','FontSize',12);
                
                %%% Plot raw strain signal
%                 subplot(Nqt,1,2);
%                 plot(t*1e3,x_strain);
%                 xlim(1e3*[t(1) tlim_max]);
%                 ylabel('Strain [\mu\epsilon]');
%                 xlabel('Time [ms]');
%                 grid on
%                 %y_lims = ylim;
%                 %dy = abs(y_lims(2) - y_lims(1));
%                 %text(0.7*tlim_max*1e3,min(y_lims)+(0.2*dy),'Raw signal');
%                 text(0.7,0.2,'Raw signal','Units','normalized');
%                 text(0.015,0.9,'B','Units','normalized',...
%                     'FontWeight','bold','FontSize',12);
                
                %%% Plot filtered signal (voltage)
                %subplot(Nqt,1,3);
                subplot(Nqt,1,2);
                plot(t*1e3,x_rec);
                xlim(1e3*[t(1) tlim_max]);
                ylabel('Amp [V]');
                xlabel('Time [ms]');
                grid on
                %y_lims = ylim;
                %dy = abs(y_lims(2) - y_lims(1));
                %text(0.7*tlim_max*1e3,min(y_lims)+(0.2*dy),...
                %    'Filtered signal');
                %text(0.7*tlim_max*1e3,min(y_lims)+(0.8*dy),...
                %    ['SBR \approx ',sprintf('%0.2f',snr),' dB']);
                text(0.7,0.2,'Filtered signal','Units','normalized');
                text(0.7,0.8,['SBR \approx ',sprintf('%0.2f',snr),' dB'],...
                    'Units','normalized');
                text(0.015,0.9,'C','Units','normalized',...
                    'FontWeight','bold','FontSize',12);
                %text(0.7*tlim_max*1e3,min(y_lims)+(0.7*dy),...
                %    ['maxA = ',sprintf('%0.2f',maxA*1e3), ' mV']);
                %text(0.7*tlim_max*1e3,min(y_lims)+(0.9*dy),...
                %    ['TmaxA = ',sprintf('%0.2f',t_maxA*1e3),' ms']);
                
                %%% Plot instantaneous root mean square value
                %subplot(Nqt,1,4);
                %plot(t*1e3,inst_rms);
                %xlim(1e3*[t(1) tlim_max]);
                %ylabel('RMS [V]');
                %grid on
                %y_lims = ylim;
                %dy = abs(y_lims(2) - y_lims(1));
                %text(0.7*tlim_max*1e3,min(y_lims)+(0.8*dy),...
                %    ['SBR \approx ',sprintf('%0.2f',snr)]);
                
                %%% Plot cumulative energy
%                 subplot(Nqt,1,4);
%                 plot(t*1e3,cum_e);
%                 hold on
%                 plot(t(p0)*1e3,cum_e(p0),'k*');
%                 xlim(1e3*[t(1) tlim_max]);
%                 ylabel('Es [V^{2}.s]');
%                 xlabel('Time [ms]');
%                 grid on
%                 %y_lims = ylim;
%                 %dy = abs(y_lims(2) - y_lims(1));
%                 %text(0.7*tlim_max*1e3,min(y_lims)+(0.2*dy),...
%                 %    ['ToF = ',sprintf('%0.2f',tof*1e6), ' \mus']);
%                 %text(0.7*tlim_max*1e3,min(y_lims)+(0.5*dy),...
%                 %    ['Es = ',sprintf('%0.2f',cum_e(end)*1e9),...
%                 %    ' nV^{2}.s']);
%                 text(0.7,0.2,['ToF = ',sprintf('%0.2f',tof*1e6), ' \mus'],...
%                     'Units','normalized');
%                 text(0.7,0.5,['Es = ',sprintf('%0.2f',cum_e(end)*1e9),...
%                     ' nV^{2}.s'],'Units','normalized');
%                 text(0.015,0.9,'D','Units','normalized',...
%                     'FontWeight','bold','FontSize',12);
                
                %%% Plot FFT spectrum
                %subplot(Nqt,1,5);
                subplot(Nqt,1,3);
                plot(1e-3*f_vec,fft_x_n);
                hold on
                %plot(fmain*1e-3,fft_x_n(idx),'r*');
                plot([fch fch]*1e-3,[0 1],'--k');
                xlim([50 2*freq]);
                ylabel('Mag [V/V]');
                xlabel('Frequency [kHz]');
                grid on
                txtpos = 50 + (0.7*((2*freq)-50));
                text(txtpos,0.2,...
                    ['Area = ',sprintf('%0.2f',Afft),' V/V']);
                %text(txtpos,0.3,...
                %    ['Es = ',sprintf('%0.2f',Esfft*1e-3),' k(V/V)^{2}.Hz']);
                text(txtpos,0.5,...
                    ['Fch = ',sprintf('%0.2f',fch*1e-3),' kHz']);
                text(txtpos,0.8,...
                    ['FFTmax = ',sprintf('%0.2f',max_mag),' V/V']);
                text(0.015,0.9,'E','Units','normalized',...
                    'FontWeight','bold','FontSize',12);
                %text(txtpos,0.9,...
                %    ['Fc = ',sprintf('%0.2f',fc*1e-3),' kHz']);
                %text(txtpos,0.7,...
                %    ['bw = ',sprintf('%0.2f',f_bw*1e-3),' kHz']);
                
                %% Extra operations
                if s_num(s) ~= 1 && s_num(s) ~= 10
                    
                    %%% Attribution of baseline
                    if strcmp(state_str,'ND') == 1
                        if s_num(s) == 3 || s_num(s) == 7
                            i_nd = 1;
                        elseif s_num(s) == 4 || s_num(s) == 8
                            i_nd = 2;
                        elseif s_num(s) == 5 || s_num(s) == 9
                            i_nd = 3;
                        end
                        baseline{i_nd,1} = s_num(s);
                        baseline{i_nd,2} = {x_rec;t};
                        baseline{i_nd,3} = {fft_x_n;(f_vec)'};
                    end
                    
                    %%% Copmute DIs - RMSD and CC
                    % Downsampling if necessary
                    % Time-domain
                    i_nd = find(cell2mat(baseline(:,1)) == s_num(s),1);
                    if length(baseline{i_nd,2}{1}) > L_x
                        [t_b,xt_b] = downsampsig(baseline{i_nd,2}{1,1},...
                            baseline{i_nd,2}{2,1},t);
                    else
                        xt_b = baseline{i_nd,2}{1};
                    end
                    % Frequency-domain
                    if length(baseline{i_nd,3}{1}) > length(fft_x_n)
                        [f_b,xf_b] = downsampsig(baseline{i_nd,3}{1,1},...
                            (baseline{i_nd,3}{2,1}),(f_vec)');
                    else
                        xf_b = baseline{i_nd,3}{1};
                    end
                    
                    
                    cc_td = corr2(xt_b,x_rec);
                    rmsd_td = sqrt(sum((x_rec - xt_b).^2) / sum(xt_b.^2));
                    N_dmg = x_rec / sqrt(sum(x_rec.^2));
                    N_ref = (xt_b * sum(xt_b .* N_dmg)) / sum(xt_b.^2);
                    cc_fd = corr2(xf_b,fft_x_n);
                    rmsd_fd = sqrt(sum((fft_x_n - xf_b).^2) / sum(xf_b.^2));
                    
                    %%% Store results in matrix
                    % Increment total counter
                    counter = counter + 1;
                    
                    res_mat(counter,1) = act;
                    res_mat(counter,2) = s_num(s);
                    res_mat(counter,3) = freq;
                    if strcmp(state_str,'ND') == 1
                        case_num = 0;
                    elseif strcmp(state_str,'D1') == 1 ||...
                            strcmp(state_str,'D2') == 1
                        case_num = 1;
                    elseif isempty(strfind(state_str,'LFV')) == 0
                        case_num = 2;
                    elseif isempty(strfind(state_str,'SW')) == 0
                        case_num = 3;
                    end
                    res_mat(counter,4) = case_num;
                    res_mat(counter,5) = T10_perc*1e6; %cum_e(end)*1e9;
                    res_mat(counter,6) = tof*1e6;
                    res_mat(counter,7) = Afft;
                    res_mat(counter,8) = fch*1e-3;
                    res_mat(counter,9) = maxV;
                    res_mat(counter,10) = 1 - cc_td;
                    res_mat(counter,11) = rmsd_td;
                    res_mat(counter,12) = sum((N_dmg - N_ref).^2);
                    res_mat(counter,13) = 1 - cc_fd;
                    res_mat(counter,14) = rmsd_fd;
                    res_mat(counter,15) = snr;
                end
                
            end
        end
        %%% Save and close figures for each file
        figname = ['LFV_effect_A',sprintf('%02d',act),...
            '_',freq_str,'kHz_',state_str,'.fig'];
        fig_fname = fullfile(result_dir,figname);
        savefig(hf,fig_fname);
        close(hf);
        
    end
    
end
%%% Save result matrix
res_filename = 'LFV_effect_results.mat';
save(fullfile(result_dir,res_filename),'res_mat','-mat');
