% Program to compute creep parameters from a seismic model and olivine flow law parameters
% 
% Wouter van der Wal, Delft University of Technology, 2022
%
% Calculation of flow law parameters as in van der Wal et al. (EPSL, 2015)
% Olivine flow law of Hirth and Kohlsted (AGU Monograph 2003)
% Seismic model: Lloyd et al. (JGR, 2020) for Antarctica, elsewhere Smean2
%  http://www-udc.ig.utexas.edu/external/becker/tdata.html
% interpolation is done with either the Matlab routine 'griddata' between two 3D cartesian grids, or using average of 6 nearest points weighted by spherical distances
% moving average is applied on the boundary between the Antarctic seismic model and Smean2 used for the rest of the world
% Creep parameters are outputed which can be used in the finite element software ABAQUS with use subroutine CREEP.f
% Viscosity values are computed for a selected stress to provide output that is more meaningful
%
% USES flowlawolivinewet, flowlawolivine, makelayer_pressure
% 
%
clear all
close all
clc
%%
w = warning ('off','all');

% Parameters
plcheck             = true; % if true, extra plots are produced to help check the program
interpolate         = false; % if true, interpolate from the original data, otherwise read the file with interpolated data (faster)
method              = 'xyz'; % xyz: griddata to interpolate between 2 3D cartesian grids, otherwise distance weighted average of the 6 nearest points is used
interpolate_smean   = false; % if true, interpolate from the original data, otherwise read the file with interpolated data (faster) 
method_smean        = 'xyz'; % see 'method'
read_smean          = false; 
r_E                 = 6371; % radius of the Earth
S = referenceSphere('unit','km');
lat_boundary        = -55;  % boundary between Antarctica grid and the rest of the world
d                   = 4000; % grain size in micrometer
water               = 500;  % water content in ppm
phi_melt            = 0;    % no melt is assumed, always set to 0
stressMPa           = 0.1;  % Von Mises stress level in MPa
load coastlines;
depth_interval = 50;
filename_output = 'elno_Bdiff_Bdisl_w500_d4.dat';

raw = load('Coordinates.dat'); % File with coordinates of centroids of elements
a = sortrows(raw);
disp('Elements loaded');
elements = a(:,1);
x = a(:,2);
y = a(:,3);
z = a(:,4);
[lon_out, lat_out, r_out]=cart2sph(z,x,y); % z,x,y due to turned coordinate system in ABAQUS
lat_out = lat_out*180/pi; lon_out = lon_out*180/pi;
depth_out = r_E-r_out/1e3;
figure; scatter(1:length(depth_out),sort(depth_out),'.'); grid minor; ylabel('Depth [km]'); title('Centroids directly from Abaqus'); 
figure; scatter(1:length(depth_out),sort(depth_out),'.'); grid minor; ylabel('Depth [km]'); title('Centroids directly from Abaqus'); ylim([0 700])

%% Use only elements shallower than 700 km
selection = find(depth_out<700);
lon_out = lon_out(selection); lat_out = lat_out(selection); depth_out = depth_out(selection); elements = elements(selection);
r_out = (r_E-depth_out);
T_out = zeros(size(lon_out));
x_out = r_out.*cosd(lat_out).*cosd(lon_out);
y_out = r_out.*cosd(lat_out).*sind(lon_out);
z_out = r_out.*sind(lat_out);

%% Load Smean
x_in_smean_total = []; y_in_smean_total = []; z_in_smean_total = []; dV_smean_total = [];
depth_smean = load('smean2\depths.dat');
prefilename = 'smean2\lonlatvs';
if read_smean % if true, read the model from the original files
    for ii = 45:58
        disp(['read Smean file ii = ',num2str(ii)]);
        filename = [prefilename,num2str(ii),'.dat'];
        a = load(filename);
        dV_smean = 0.01*a(:,3);
        lon_smean = a(:,1);
        lon_smean(lon_smean>180) = -( 360-lon_smean(lon_smean>180) );
        lat_smean = a(:,2);
        r_smean = 6371-depth_smean(ii)*ones(size(lon_smean));
        x_in_smean = r_smean.*cosd(lat_smean).*cosd(lon_smean);
        y_in_smean = r_smean.*cosd(lat_smean).*sind(lon_smean);
        z_in_smean = r_smean.*sind(lat_smean);
        x_in_smean_total = [x_in_smean_total; x_in_smean];
        y_in_smean_total = [y_in_smean_total; y_in_smean];
        z_in_smean_total = [z_in_smean_total; z_in_smean];
        dV_smean_total = [dV_smean_total; dV_smean];
        arr_smean = [x_in_smean_total y_in_smean_total z_in_smean_total dV_smean_total];
        save('x_y_z_smean.mat','arr_smean')
    end
else
    load('x_y_z_smean.mat');
    x_in_smean_total = arr_smean(:,1);
    y_in_smean_total = arr_smean(:,2);
    z_in_smean_total = arr_smean(:,3);
    dV_smean_total = arr_smean(:,4);
    clear arr_smean;
end
if plcheck
    depth_plot = 125;
    r_plot = sqrt(x_in_smean_total.^2+y_in_smean_total.^2+z_in_smean_total.^2);
    indplot = find(  abs(r_plot-(6371-depth_plot))<1  );
    [lon_plot,lat_plot,r_plot] = cart2sph(x_in_smean_total(indplot),y_in_smean_total(indplot),z_in_smean_total(indplot));
    [Z, refvec] = geoloc2grid(rad2deg(lat_plot),rad2deg(lon_plot),dV_smean_total(indplot),0.5);
    figure(20);
    latlim = [-90 90];
    lonlim = [-180 180];
    oldcmap = colormap('jet');
    colormap( flipud(oldcmap) );
    caxis([-0.05 0.05]);
    h = colorbar; set(h, 'ylim', [-0.05 0.05]);
    ax = axesm('mollweid','MapLatLimit',latlim,...
        'MapLonLimit',lonlim,'Grid','on',...
        'MeridianLabel','off','ParallelLabel','on');
    set(ax,'Visible','off');
    geoshow(Z, refvec, 'DisplayType', 'texture');
    plotm(coastlat,coastlon);
    h=colorbar('v');
    set(get(h,'ylabel'),'string','Vv (km/s)');
    title(['delta Voight in % @ ',num2str(depth_plot),' km FE grid']);
    set(findall(gca, 'type', 'text'), 'visible', 'on')
end

% Derivatives of temperature to shear wave velocity anomaly from Karato p. 376
depth_kar =  [24.4; 40;   60;   80;    115;   185;   219.9; 220.1; 265;   310;   355;   400; ...
    400.1; 450; 500; 549.9; 550.1; 600; 635; 669.9; 670.1; 721; 771; 871; 971; 1071; 1171];   % [km]
dvs_dT    = -[2*7.25; 2*7.71; 2*8.09; 16.84; 17.17; 17.32; 17.32; 13.32; 13.02; 12.34; 12.04; 11.8; ...
    8.92; 8.50; 8.16; 7.85; 7.85; 7.49; 7.49; 7.44; 8.46; 8.26; 8.07; 7.70; 7.35; 7.18; 6.87];  % [10^-5 K^-1]


if interpolate_smean % interpolate smean to the element centroids
    % Load finite elements with centroid coordinates
    indout = find(lat_out>lat_boundary-0.5);
    lon_out_cropped = lon_out(indout);
    lat_out_cropped = lat_out(indout);
    r_out_cropped = r_out(indout);
    depth_out_cropped = depth_out(indout);
    dvs_dT_int_cropped = interp1(depth_kar,dvs_dT,depth_out_cropped,'linear'); % still 52.5 km first in depth_tur and dvs_dT_int
    dvs_dT_int_cropped(depth_out_cropped<depth_kar(1)) = dvs_dT(1); % still 52.5 km first in depth_tur and dvs_dT_int
    x_out = r_out_cropped.*cosd(lat_out_cropped).*cosd(lon_out_cropped);
    y_out = r_out_cropped.*cosd(lat_out_cropped).*sind(lon_out_cropped);
    z_out = r_out_cropped.*sind(lat_out_cropped);
    if strcmp(method_smean,'xyz')
        disp('Start interpolating Smean GRIDDATA');
        dVv_out_cropped = griddata(x_in_smean_total,y_in_smean_total,z_in_smean_total,dV_smean_total,x_out,y_out,z_out,'nearest');
        dVv_out(indout) = dVv_out_cropped;
        dT_out(indout) = dVv_out_cropped ./ (dvs_dT_int_cropped*1e-5);
        save(['dVv_out_xyz_smean.mat'],'dVv_out');
        save(['dT_out_xyz_smean.mat'],'dT_out');
        disp('Finished interpolating Smean');
    elseif strcmp(method_smean,'nn')
        error('check nn interpolation smean!')
        %%
        disp('Find 6 closest points in x,y,z space')
        p = knnsearch([x_in y_in z_in],[x_out,y_out,z_out],'k',6);
        disp('Calculate great circle distances for the 6 points');
        S.Radius = r_E-mean(depth_out_cropped);
        arclentot = zeros(size(lat_out_cropped));
        dVv_out_cropped = zeros(size(lat_out_cropped));
        for ii = 1:6
            arclentot = arclentot + distance( lat(p(:,ii)),lon(p(:,ii)),lat_out_cropped,lon_out_cropped,S ); % same unit as S, so km
        end
        for ii = 1:6
            arclen = distance( lat(p(:,ii)),lon(p(:,ii)),lat_out_cropped,lon_out_cropped,S );
            dVv_out_cropped = dVv_out_cropped+dVv(p(:,ii)).*(arclen./arclentot);
        end
        dVv_out(indout) = dVv_out_cropped;
        dT_out(indout) = dVv_out_cropped ./ (dvs_dT_int_cropped*1e-5);
        disp('Finished interpolating inverse distance');
        save(['dVv_out_invdist_smean.mat'],'dVv_out');
        save(['dT_out_invdist_smean.mat'],'dT_out');
    else
        error('choose interpolation method')
    end
else
    if strcmp(method_smean,'xyz')
        load(['dT_out_xyz_smean.mat']);
    elseif strcmp(method_smean,'nn')
        load(['dT_out_invdist_smean.mat']);
    end
end
%%
depthplot = 188;
indplot = find(abs(depth_out-depthplot)<depth_interval);
if plcheck
    [Z, refvec] = geoloc2grid(lat_out(indplot),lon_out(indplot),dT_out(indplot)',0.5);
    figure(25);
    latlim = [-90 90];
    lonlim = [-180 180];
    colormap('jet');
    caxis([-200 200]);
    h = colorbar; set(h, 'ylim', [-200 200]);
    ax = axesm('mollweid','MapLatLimit',latlim,...
        'MapLonLimit',lonlim,'Grid','on',...
        'MeridianLabel','off','ParallelLabel','on');
    set(ax,'Visible','off');
    geoshow(Z, refvec, 'DisplayType', 'texture');
    plotm(coastlat,coastlon);
    h=colorbar('v');
    set(get(h,'ylabel'),'string','Vv (km/s)');
    title(['Temp. in [K] ',num2str(depthplot),' km FE grid']);
    set(findall(gca, 'type', 'text'), 'visible', 'on')
end


%% ----------------------------------- Interpolate Antarctica
depths_fem = [52 85 113 138 188 226 263 330 361 190 453 487 518 580 614 644];
if interpolate
    qq = 0;
    for depth = depths_fem
        if depth < min(depth_kar)
            dvs_dT_int = dvs_dT(1);
        else
            dvs_dT_int = interp1(depth_kar,dvs_dT,depth,'linear');
        end
        qq = qq+1;
        disp(['depth = ',num2str(depth)])
        prefilename1 = 'ANT-20_GEN-1_SPHERE_SL\ANT-20_GEN-1_SPHERE_SL_voigtVs_';
        prefilename2 = 'STW015_GEN-1_SPHERE_SL\STW105_GEN-1_SPHERE_SL_voigtVs_';
        % 2. The model parameter, in this case it is the Voigt average shear wave speed with units are km/s
        if round(depth,-1)> depth
            depth1 = round(depth,-1);
            depth2 = round(depth,-1)-5;
        elseif round(depth,-1) < depth
            depth1 = round(depth,-1)+5;
            depth2 = round(depth,-1);
        end
        filename = [prefilename1,num2str(depth1),'.xypdrll'];
        a = load(filename);
        filename = [prefilename1,num2str(depth2),'.xypdrll'];
        b = load(filename);
        Vv = (b(:,3)+a(:,3))/2.;
        r = (b(:,5)+a(:,5))/2.;
        filename = [prefilename2,num2str(depth1),'.xypdrll'];
        a = load(filename);
        filename = [prefilename2,num2str(depth2),'.xypdrll'];
        b = load(filename);
        Vvref = (a(:,3)+b(:,3))/2.;
        Vv = Vv(~isnan(Vv));
        Vvref = Vvref(~isnan(Vvref));
        filename = [prefilename1,num2str(20),'.xypdrll'];
        a = load(filename);
        lon = a(:,6);
        lon = lon(~isnan(r));
        lat = a(:,7);
        lat = lat(~isnan(r));
        r = r(~isnan(r));
        dVv = log(Vv)-log(Vvref);
        % Create x,y,z coordinate for input seismic grid
        x_in = r*1e3.*cosd(lat).*cosd(lon);
        y_in = r*1e3.*cosd(lat).*sind(lon);
        z_in = r*1e3.*sind(lat);
        % Plot for Antarctica
        if plcheck
            %%
            [Z, refvec] = geoloc2grid(lat,lon,dVv,0.5);
            figure(30+qq);
            latlim = [-90 lat_boundary];
            lonlim = [-180 180];
            colormap(flipud(jet));
                caxis([-0.05 0.05]);
            h = colorbar;
                set(h, 'ylim', [0.05 0.5]);
            ax = axesm('stereo','MapLatLimit',latlim,...
                'MapLonLimit',lonlim,'Grid','on',...
                'MeridianLabel','on','ParallelLabel','on');
            set(ax,'Visible','off');
            geoshow(Z, refvec, 'DisplayType', 'texture');
            plotm(coastlat,coastlon);
            h=colorbar('v');
            set(get(h,'ylabel'),'string','dV (-)');
            title([num2str(depth),' km']);
            set(findall(gca, 'type', 'text'), 'visible', 'on')
            %%
            
        end
        %%
        
        inddepth = find(abs(depth_out-depth)<depth_interval & lat_out<lat_boundary-0.5);
        lon_out_cropped = lon_out(inddepth);
        lat_out_cropped = lat_out(inddepth);
        r_out_cropped = r_out(inddepth);
        depth_out_cropped = depth_out(inddepth);

        % Interpolate
        if interpolate
            % Load finite elements with centroid coordinates
            % Find
            mean_depth_out_cropped_km = mean(depth_out_cropped);
            x_out = r_out_cropped.*cosd(lat_out_cropped).*cosd(lon_out_cropped);
            y_out = r_out_cropped.*cosd(lat_out_cropped).*sind(lon_out_cropped);
            z_out = r_out_cropped.*sind(lat_out_cropped);
            if strcmp(method,'xyz')
                disp('Start interpolating Antarctica GRIDDATA');
                dVv_out_cropped = griddata(x_in,y_in,z_in,dVv,x_out,y_out,z_out,'nearest');
                dT_out(inddepth) = dVv_out_cropped / (dvs_dT_int*1e-5);
                disp('Finished interpolating Antarctica ');
            elseif strcmp(method,'nn')
                %%
                disp('Find 6 closest points in x,y,z space')
                p = knnsearch([x_in y_in z_in],[x_out,y_out,z_out],'k',6);
                disp('Calculate great circle distances for the 6 points');
                S.Radius = r_E-mean(depth_out_cropped);
                arclentot = zeros(size(lat_out_cropped));
                dVv_out_cropped = zeros(size(lat_out_cropped));
                for ii = 1:6
                    arclentot = arclentot + distance( lat(p(:,ii)),lon(p(:,ii)),lat_out_cropped,lon_out_cropped,S ); % same unit as S, so km
                end
                for ii = 1:6
                    arclen = distance( lat(p(:,ii)),lon(p(:,ii)),lat_out_cropped,lon_out_cropped,S );
                    dVv_out_cropped = dVv_out_cropped+dVv(p(:,ii)).*(arclen./arclentot);
                end
                dT_out(inddepth) = dVv_out_cropped / (dvs_dT_int*1e-5);
                disp('Finished interpolating inverse distance');
            else
                error('choose interpolation method')
            end
        end
        %%
        if plcheck
            figure(40);
            [Z, refvec] = geoloc2grid(lat_out_cropped,lon_out_cropped,dT_out(inddepth)',0.5);
            latlim = [-90 lat_boundary];
            lonlim = [-180 180];
            colormap('jet');
            caxis([-200 200]);
            h = colorbar; set(h, 'ylim', [-200 200]);
            ax = axesm('stereo','MapLatLimit',latlim,...
                'MapLonLimit',lonlim,'Grid','on',...
                'MeridianLabel','on','ParallelLabel','on');
            set(ax,'Visible','off');
            geoshow(Z, refvec, 'DisplayType', 'texture');
            plotm(coastlat,coastlon);
            h=colorbar('v');
            set(get(h,'ylabel'),'string','Vv (km/s)');
            title(['\Delta Temp [K] ',num2str(depth),' km FE grid']);
            set(findall(gca, 'type', 'text'), 'visible', 'on')
        end
        %%
        
        %% Smooth
        % Make array
        % Sort latitudes for each longitude
        inddepth = find(  abs(depth_out-depth)<depth_interval  );
        ind60 = find(  abs(lat_out-lat_boundary)<4 & abs(depth_out-depth)<depth_interval );
        arclentot = 0;
        dT_out_temp = 0*dT_out;
        for ii = 1:length(ind60)
            temp = distance( lat_out(ind60(ii)),lon_out(ind60(ii)),lat_out(inddepth),lon_out(inddepth),S );
            sorttemp = sort(temp);
            arclentot = sum(sorttemp(1:5)); % same unit as S, so km
            dTtemp = 0;
            for jj = 1:5
                arr = [temp dT_out(inddepth)'];
                temparr = sortrows(arr,[1 2]);
                dTtemp =  dTtemp + temparr(jj,2)*(temparr(jj,1)./arclentot);
                dT_out_temp(ind60(ii)) = dTtemp;
            end
        end
        
        dT_out(ind60) = dT_out_temp(ind60);

        if plcheck
            depth_plot = depth;
            indplot = find(  abs(depth_out-depth_plot)<depth_interval);
            [Z, refvec] = geoloc2grid(lat_out(indplot),lon_out(indplot),dT_out(indplot)',0.5);
            figure(50+qq);
            latlim = [-90 90];
            lonlim = [-180 180];
            colormap('jet');
            caxis([-200 200]);
            h = colorbar; set(h, 'ylim', [-200 200]);
            ax = axesm('mollweid','MapLatLimit',latlim,...
                'MapLonLimit',lonlim,'Grid','on',...
                'MeridianLabel','off','ParallelLabel','on');
            set(ax,'Visible','off');
            geoshow(Z, refvec, 'DisplayType', 'texture');
            plotm(coastlat,coastlon);
            h=colorbar('v');
            set(get(h,'ylabel'),'string','Vv (km/s)');
            title(['\Delta Temp. [K]',num2str(depth_plot),' km FE grid']);
            set(findall(gca, 'type', 'text'), 'visible', 'on')
        end
        
    end
    if strcmp(method,'xyz')
        save(['dT_out_xyz.mat'],'dT_out');
    elseif strcmp(method,'nn')
        save(['dT_out_invdist.mat'],'dT_out');
    end
else
    if strcmp(method,'xyz')
        load(['dT_out_xyz.mat']);
    elseif strcmp(method,'nn')
        load(['dT_out_invdist.mat']);
    end
end

%%

depth_plot = 188;
indplot = find(  abs(depth_out-depth_plot)<depth_interval  );
if plcheck
    [Z, refvec] = geoloc2grid(lat_out(indplot),lon_out(indplot),dT_out(indplot)',0.5);
    figure(70);
    colormap('jet');
    ax = axesm('mollweid','MapLatLimit',latlim,...
        'MapLonLimit',lonlim,'Grid','on',...
        'MeridianLabel','off','ParallelLabel','on');
    set(ax,'Visible','off');
    geoshow(Z, refvec, 'DisplayType', 'texture');
    plotm(coastlat,coastlon);
    h=colorbar('v');
    set(get(h,'ylabel'),'string','T [K]');
    title(['Temperature @ ',num2str(depth_plot),' km FE grid']);
    set(findall(gca, 'type', 'text'), 'visible', 'on')
end


%% Compute pressure at the current depth
% p has size 3001
rlay = [3480000;  4340000; 5200000;  5450000;  5701000;  5835000;  5971000;  6046000;  6096000;  6146000; 6196000; 6246000; 6296000; 6346000; 6371000];
depthlay = 6371-rlay/1e3;
depthlay = flipud(depthlay);
[dvec_int,p,mu,rho] = makelayer_pressure(depthlay,3000,1); % Pa
pressurevec = [];
for ii = 1:length(depth_out)
    x = p(min(find(dvec_int==round(depth_out(ii)))));
    pressurevec = [pressurevec; x];
end
if plcheck; figure; plot(pressurevec); hold on; plot(depth_out.*0.0333e9,'r'); hold off; end


%% Compute background temperature
r_stacey = [5400; 5600; 5700.5; 5701.5; 5771; 5871; 5970.5; 5971.5; 6061; 6150.9; 6151.1; 6256; 6331.5; 6332.5; 6371];
T_stacey = [2060; 1974;  1931; 2010; 1985; 1948; 1907; 1853; 1817; 1780; 1719; 1282;  880;  880;  300];
depth_stacey = r_E-r_stacey;
coefficients = polyfit([depth_stacey(end-4),depth_stacey(end-3)],[T_stacey(end-4),T_stacey(end-3)],1);
a = coefficients(1);
b = coefficients(2);
intersection = (b-1315-270)/(0.4-a);
disp(['intersection = ',num2str(intersection)])
% interpolate to current depth.
for ii = 1:length(depth_out)
    if depth_out(ii)<=intersection
        T_background(ii) = interp1(depth_stacey,T_stacey,depth_out(ii),'linear');
    else
        T_background(ii) = 1315. + 270. + 0.4*depth_out(ii);
    end
end
depthtemp = [0:10:600];
T_convective = 1315. + 270. + 0.4*depthtemp;


%% Compute absolute temperature
disp('Compute temperatures')
T_out_new = T_background + dT_out; % units Kelvin for flow law
depth_plot = 188;
indplot = find(abs(depth_out-depth_plot)<depth_interval);
if plcheck
    figure(80);
    [Z, refvec] = geoloc2grid(lat_out(indplot),lon_out(indplot),T_out_new(indplot)',0.5);
    colormap('jet');
    ax = axesm('mollweid','MapLatLimit',latlim,...
        'MapLonLimit',lonlim,'Grid','on',...
        'MeridianLabel','off','ParallelLabel','on');
    set(ax,'Visible','off');
    geoshow(Z, refvec, 'DisplayType', 'texture');
    plotm(coastlat,coastlon);
    h=colorbar('v');
    set(get(h,'ylabel'),'string','T [K]');
    title(['Temperature [K] @ ',num2str(depth_plot),' km FE grid']);
    set(findall(gca, 'type', 'text'), 'visible', 'on')
end


%% Compute creep parameters
if water>0
    [Bdisl,Bdiff] = flowlawolivinewet(T_out_new',d,phi_melt,pressurevec,water);
else
    [Bdisl,Bdiff] = flowlawolivine(T_out_new',d,phi_melt,pressurevec);
end
visc_out = 1e6*stressMPa./ (   3.*(Bdiff.*stressMPa+Bdisl.*stressMPa.^3.5)   );
visc_out = log10(visc_out);

Bdiff = Bdiff*1e-6*(3./2.);
Bdisl = Bdisl*(1e-6^3.5)*(3./2.);

arr = [elements Bdiff Bdisl];
fidw = fopen(filename_output,'w');
fprintf(fidw,'%5i %12.4e %12.4e\n',arr');
fclose(fidw);

q = load(filename_output);
if plcheck
    figure; scatter(q(:,1),q(:,2),10,q(:,3)); colorbar
end

%% Plot viscosities
plotdepth = 263;
indplot = find(abs(depth_out-plotdepth)<depth_interval);
[Z, refvec] = geoloc2grid(lat_out(indplot),lon_out(indplot),visc_out(indplot),0.5);
%%
if plcheck
    figure(90);
    latlim = [-90 90];lonlim = [-180 180];
    cmap = colormap('jet'); cmap = flipud(cmap);
    colormap(cmap);
    ax = axesm('mollweid','MapLatLimit',latlim,'MapLonLimit',lonlim,'Grid','on','MeridianLabel','off','ParallelLabel','on');
    set(ax,'Visible','off');
    geoshow(Z, refvec, 'DisplayType', 'texture');
    plotm(coastlat,coastlon);
    h=colorbar('v');
    set(get(h,'ylabel'),'string','visc [Pa s]');
    title(['Viscosity [Pa s] @ ',num2str(round(plotdepth)),' km FE grid']);
    set(findall(gca, 'type', 'text'), 'visible', 'on')
end
%%
figure();
cmap = colormap('jet'); cmap = flipud(cmap);
colormap(cmap);
caxis([18 25]);
h = colorbar('v'); set(h, 'ylim', [18 25]);
set(get(h,'ylabel'),'string','Visc (Pa s)');
latlim = [-90 lat_boundary];lonlim = [-180 180];
ax = axesm('stereo','MapLatLimit',latlim,'MapLonLimit',lonlim,'Grid','on','MeridianLabel','on','ParallelLabel','on');
set(ax,'Visible','off');
set(findall(gca, 'type', 'text'), 'visible', 'on')
geoshow(Z, refvec, 'DisplayType', 'texture');
plotm(coastlat,coastlon);
title(['Viscosity [Pa s] @ ',num2str(plotdepth),' km FE grid']);
set(findall(gca, 'type', 'text'), 'visible', 'on')
fsave = ['visc_Ant_w10_newcoordinates',num2str(round(plotdepth)),'km_xyz'];
saveas(gcf,fsave, 'fig');

%%

