## 3D GIA model using ABAQUS2019: developed at the Technical University of Delft, Astrodynamics and Space Missions. 
## Author 1: Bas Blank
# Author 2: Haiyang Hu
# Author 3: username van Calcar

# "Model_gen" is the multi-layer model generator for Abaqus-CAE. All parameters are set in Model_data.py
# 2014 - 2019: Haiyang Hu and Bas Blank developed the script that: builts the model that consists of multiple layers, generates the mesh including a high resolution area at the southpole, 
# sets boundary conditions and the foundation, generate steps, and defines the load for all steps, the job and the output (Blank et al., 2021).
# 2019 - 2020: username van Calcar separated the model_gen file and moved parts of it to the Iter_ult file. The Model_gen now only: builts the model that consists of multiple layers, 
# generates the mesh including a high resolution area at the southpole and sets boundary conditions and the foundation (Van Calcar, 2020).

# References: 
	# Blank, B., Barletta, V., Hu, H., Pappa, F., & van der Wal, W. (2021). Effect of Lateral and Stress‐Dependent Viscosity Variations on GIA Induced Uplift Rates in the Amundsen Sea Embayment. Geochemistry, Geophysics, Geosystems, 22(9), e2021GC009807.
	# C.J. van Calcar (2020). Interactions between Ice Sheet Dynamics and Glacial Isostatic Adjustment. Master thesis Delft University of Technology. http://resolver.tudelft.nl/uuid:8a3a4e21-a6f0-4874-a41f-e26ac95f03be

from abaqus import *
from abaqusConstants import *
from caeModules import *
from odbAccess import*
import numpy as np
import os
import __main__


if 'Model_data_coupling' in (sys.modules.keys()):
    reload(Model_data_coupling)
    from Model_data_coupling import *
else:
    Dir=r"/home/username/HetGroteKoppelScript/GIA_Model/"
    os.chdir(Dir)
    import Model_data_coupling
    from Model_data_coupling import *

os.chdir(Dir)

#### Model Building 

Mdb()
mdb.models.changeKey('Model-1',Model_name)

##  Generate Core
s = mdb.models[Model_name].ConstrainedSketch(name='__profile__',sheetSize=10000.0)
g, v, d= s.geometry, s.vertices, s.dimensions
s.setPrimaryObject(option=STANDALONE)
s.ConstructionLine(point1=(0.0, -5000.0), point2=(0.0, 5000.0))
s.FixedConstraint(entity=g[2])

s.ArcByCenterEnds(center=(0.0, 0.0), point1=(0.0, Radius[0]), point2=(0.0, -Radius[0]), direction=CLOCKWISE)

s.Line(point1=(0.0, -Radius[0]), point2=(0.0, Radius[0]))
p = mdb.models[Model_name].Part(name='L0', dimensionality=THREE_D, type=DEFORMABLE_BODY)
p = mdb.models[Model_name].parts['L0']
p.BaseSolidRevolve(sketch=s, angle=360.0, flipRevolveDirection=OFF)

p.PartitionCellByPlaneThreePoints(point1=(0,0,0),point2=(0,1e3,0), point3=(0,0,1e3),cells=p.cells)                                                     
p.PartitionCellByPlaneThreePoints(point1=(0,0,0),point2=(0,1e3,0), point3=(1e3,0,0),cells=p.cells)
p.PartitionCellByPlaneThreePoints(point1=(0,0,0),point2=(0,0,1e3), point3=(1e3,0,0),cells=p.cells)   
v=p.vertices

if N_layer==1:
        p.Set(faces=p.faces.findAt((0,Radius[0],0),).getFacesByFaceAngle(10), name='IF0')    #node set of inner layer which is outside of core (Interface0)                                             
p.Set(vertices=v.findAt(((0,0,0),)),name='Center')   #Center of core                               
p.Surface(side1Faces=p.faces.findAt((0,Radius[0],0),).getFacesByFaceAngle(10), name='Sout0') #Core Surface                          
                              
                              
##  Generate Outer Layers
for i in range(1,N_layer):                                
    s = mdb.models[Model_name].ConstrainedSketch(name='__profile__',    sheetSize=10000.0)
    g, v, d, c = s.geometry, s.vertices, s.dimensions, s.constraints
    s.setPrimaryObject(option=STANDALONE)
    s.ConstructionLine(point1=(0.0, -5000.0), point2=(0.0, 5000.0))
 
    s.ArcByCenterEnds(center=(0.0, 0.0), point1=(0.0, Radius[i-1]), point2=(0.0, -Radius[i-1]), direction=CLOCKWISE)
    s.ArcByCenterEnds(center=(0.0, 0.0), point1=(0.0, Radius[i]), point2=(0.0, -Radius[i]), direction=CLOCKWISE)
    s.Line(point1=(0.0, -Radius[i-1]), point2=(0.0, -Radius[i]))                                                    
    s.Line(point1=(0.0, Radius[i-1]), point2=(0.0, Radius[i]))
    
    p = mdb.models[Model_name].Part(name='L'+str(i), dimensionality=THREE_D, type=DEFORMABLE_BODY)
    p.BaseSolidRevolve(sketch=s, angle=360.0, flipRevolveDirection=OFF)
    s.unsetPrimaryObject()
    del mdb.models[Model_name].sketches['__profile__']
                                                    
    p.PartitionCellByPlaneThreePoints(point1=(0,0,0),point2=(0,1e3,0), point3=(0,0,1e3),cells=p.cells)                                                     
    p.PartitionCellByPlaneThreePoints(point1=(0,0,0),point2=(0,1e3,0), point3=(1e3,0,0),cells=p.cells)
    p.PartitionCellByPlaneThreePoints(point1=(0,0,0),point2=(0,0,1e3), point3=(1e3,0,0),cells=p.cells)   
    
    if i==1:
        p.Set(faces=p.faces.findAt((0,Radius[0],0),).getFacesByFaceAngle(10), name='IF0')    #node set of inner layer which is outside of core (Interface0)   
    p.Set(faces=p.faces.findAt((0,Radius[i],0),).getFacesByFaceAngle(10), name='IF'+str(i))    #node set of outer layer                                                   
    p.Surface(side1Faces=p.faces.findAt((0,Radius[i-1],0),).getFacesByFaceAngle(10), name='Sin'+str(i)) #Inner L1 Surface                                                    
    p.Surface(side1Faces=p.faces.findAt((0,Radius[i],0),).getFacesByFaceAngle(10), name='Sout'+str(i)) #Outter L1 Surface          
   
##Material, section
for i in range(N_layer):
    mdb.models[Model_name].Material(name='M'+str(i))
    mdb.models[Model_name].materials['M'+str(i)].Elastic(moduli=INSTANTANEOUS,table=((Y_mod[i], Poi), ))
    mdb.models[Model_name].materials['M'+str(i)].Density(table=((Density[i], ), ))
    if Vis[i]!=0 and Vis[i]<1e40:
        mdb.models[Model_name].materials['M'+str(i)].Viscoelastic(domain=TIME, time=PRONY,     table=((1-1e-10, 0.0, Vis[i]/Y_mod[i]*3), ))
    mdb.models[Model_name].HomogeneousSolidSection(name='S'+str(i),  material='M'+str(i), thickness=None)
    p = mdb.models[Model_name].parts['L'+str(i)]   
    p.SectionAssignment(region=regionToolset.Region(cells=p.cells), sectionName='S'+str(i), offset=0.0, offsetType=MIDDLE_SURFACE, offsetField='', thicknessAssignment=FROM_SECTION)   
    
## Assemblely and mesh
a = mdb.models[Model_name].rootAssembly
if flag_lc==0:
    start=0
else:
    start=1
        
    a.DatumCsysByThreePoints(name='Coor', coordSysType=SPHERICAL, origin=(0.0, 0.0, 0.0), line1=(0.0, 0.0, 1.0), line2=(1.0, 0.0, 0.0))

if N_layer>2 or (N_layer==2 and flag_lc==0):
    for i in range(0,N_layer):
        p = mdb.models[Model_name].parts['L'+str(i)]
        a.Instance(name='I'+str(i), part=p, dependent=ON)
        if i==0:
            region = a.instances['I0'].sets['Center']
    if Coarse_Layers == 1:
        mdb.models[Model_name].parts.changeKey('L1',Model_name + '_Low')
        p = mdb.models[Model_name].parts[Model_name + '_Low']
        a.Instance(name=Model_name + '_Low', part=p, dependent=ON)
    else:       
        a.InstanceFromBooleanMerge(name=Model_name + '_Low', instances=list(a.instances['I'+str(j)] for j in range(1,Coarse_Layers+1)), keepIntersections=ON, originalInstances=SUPPRESS, mergeNodes=ALL, nodeMergingTolerance=1e-06, domain=BOTH)
        a.instances.changeKey(Model_name+'_Low-1',Model_name + '_Low')
    a.InstanceFromBooleanMerge(name=Model_name, instances=list(a.instances['I'+str(j)] for j in range(Coarse_Layers+1,N_layer)), keepIntersections=ON, originalInstances=SUPPRESS, mergeNodes=ALL, nodeMergingTolerance=1e-06, domain=BOTH)
    a.instances.changeKey(Model_name+'-1',Model_name)
elif N_layer==2:
    mdb.models[Model_name].parts.changeKey('L1',Model_name)
    p = mdb.models[Model_name].parts[Model_name]
    a.Instance(name=Model_name, part=p, dependent=ON)
else:
    mdb.models[Model_name].parts.changeKey('L0',Model_name)
    p = mdb.models[Model_name].parts[Model_name]
    a.Instance(name=Model_name, part=p, dependent=ON)

elemType1 = mesh.ElemType(elemCode=C3D8H, elemLibrary=STANDARD)                                                               
elemType2 = mesh.ElemType(elemCode=C3D6H, elemLibrary=STANDARD,     secondOrderAccuracy=OFF, distortionControl=DEFAULT)
elemType3 = mesh.ElemType(elemCode=C3D4H, elemLibrary=STANDARD)
p = mdb.models[Model_name].parts[Model_name]
c = p.cells
p.setMeshControls(regions=c,elemShape=HEX_DOMINATED, technique=SWEEP,  algorithm=ADVANCING_FRONT)

#### edge 1

##Top of surface mesh adjustment
p.seedPart(size=Seeds, deviationFactor=0.05, minSizeFactor=0.1)
Par = PolarCirkelRadiusDeg/90.0
e = p.edges
pickedEdges1=e.getClosest(coordinates=((Radius[-1]*np.sin((PolarCirkelRadiusDeg/2)/180*np.pi),-Radius[-1]*np.cos((PolarCirkelRadiusDeg/2)/180*np.pi),0),))[0][0]
p.PartitionEdgeByParam(edges=pickedEdges1, parameter=(1-Par))

for i in range(Coarse_Layers+1,N_layer):
    print("layer")
    print(i)
    if i==Coarse_Layers+1:
        pickedEdges=e.getClosest(coordinates=((Radius[1]*np.sin((PolarCirkelRadiusDeg/2)/180*np.pi),-Radius[Coarse_Layers]*np.cos((PolarCirkelRadiusDeg/2)/180*np.pi),0),))[0][0]
        p.PartitionEdgeByParam(edges=pickedEdges, parameter=Par)

    pickedEdges=e.getClosest(coordinates=((Radius[i]*np.sin(alpha),-Radius[i]*np.cos(alpha),0.0),))[0][0]
    
    if i<(N_layer-1):
        p.PartitionEdgeByParam(edges=pickedEdges, parameter=(1-Par))
    R_av = (Radius[i-1]+Radius[i])/2
    v1= p.vertices.getClosest(coordinates=((Radius[i-1]*sin(alpha), -Radius[i-1]*cos(alpha),0,),))[0][0]
    v2= p.vertices.getClosest(coordinates=((Radius[i]*sin(alpha), -Radius[i]*cos(alpha),0,),))[0][0]
    F1 = p.faces.getClosest(coordinates =((R_av*np.sin(alpha),-R_av*np.cos(alpha),0.0,),))[0][0]
    p.PartitionFaceByShortestPath(point1=v1,point2=v2, faces=F1)
    

    pickedCells = p.cells.getByBoundingSphere((0.0,0.0,0.0), 1.1*Radius[-1])                       
    pickedEdges = e.getClosest(coordinates=((R_av*np.sin(alpha),-R_av*np.cos(alpha),0.0,),))[0][0]                       
    sweepEdge = e.findAt((Radius[i-1]*np.cos(1.0/4.0*np.pi),0.0,-Radius[i-1]*np.sin(1.0/4.0*np.pi)))                    
    p.PartitionCellBySweepEdge(sweepPath=sweepEdge, cells=pickedCells, edges=(pickedEdges, ))
    
    pickedCells = p.cells.getByBoundingSphere((0.0,0.0,0.0), 1.1*Radius[-1])
    pickedEdges = e.getClosest(coordinates=((0.0,-R_av*np.cos(alpha),-R_av*np.sin(alpha),),))[0][0]
    sweepEdge = e.findAt((-Radius[i-1]*np.cos(1.0/4.0*np.pi),0.0,-Radius[i-1]*np.sin(1.0/4.0*np.pi)))
    p.PartitionCellBySweepEdge(sweepPath=sweepEdge, cells=pickedCells, edges=(pickedEdges, ))
    
    pickedCells = p.cells.getByBoundingSphere((0.0,0.0,0.0), 1.1*Radius[-1])
    pickedEdges = e.getClosest(coordinates=((-R_av*np.sin(alpha),-R_av*np.cos(alpha),0.0,),))[0][0]
    sweepEdge = e.findAt((-Radius[i-1]*np.cos(1.0/4.0*np.pi),0.0,Radius[i-1]*np.sin(1.0/4.0*np.pi)))
    p.PartitionCellBySweepEdge(sweepPath=sweepEdge, cells=pickedCells, edges=(pickedEdges, ))
    
    pickedCells = p.cells.getByBoundingSphere((0.0,0.0,0.0), 1.1*Radius[-1])
    pickedEdges = e.getClosest(coordinates=((0.0,-R_av*np.cos(alpha),R_av*np.sin(alpha),),))[0][0]
    sweepEdge = e.findAt((Radius[i-1]*np.cos(1.0/4.0*np.pi),0.0,Radius[i-1]*np.sin(1.0/4.0*np.pi)))
    p.PartitionCellBySweepEdge(sweepPath=sweepEdge, cells=pickedCells, edges=(pickedEdges, ))
    
    pickedEdges = e.findAt((0.0,-R_av,0.0))
    p.seedEdgeBySize(edges=(pickedEdges,), size=R_Target_seed, deviationFactor=0.05,minSizeFactor=0.1, constraint=FINER)
    pickedEdges1 = e.findAt((-Radius[i-1]*np.sin(alpha/2),-Radius[i-1]*np.cos(alpha/2),0))
    pickedEdges2 = e.findAt((Radius[i-1]*np.sin(alpha/2),-Radius[i-1]*np.cos(alpha/2),0))
    pickedEdges3 = e.findAt((0,-Radius[i-1]*np.cos(alpha/2),-Radius[i-1]*np.sin(alpha/2)))
    pickedEdges4 = e.findAt((0,-Radius[i-1]*np.cos(alpha/2),Radius[i-1]*np.sin(alpha/2)))
    p.seedEdgeBySize(edges=(pickedEdges1,pickedEdges2,pickedEdges3,pickedEdges4,), size=Plane_Target_seed, deviationFactor=0.05,minSizeFactor=0.1, constraint=FINER)

pickedEdges1 = e.findAt((-Radius[-1]*np.sin(alpha/2),-Radius[-1]*np.cos(alpha/2),0))
pickedEdges2 = e.findAt((Radius[-1]*np.sin(alpha/2),-Radius[-1]*np.cos(alpha/2),0))
pickedEdges3 = e.findAt((0,-Radius[-1]*np.cos(alpha/2),-Radius[-1]*np.sin(alpha/2)))
pickedEdges4 = e.findAt((0,-Radius[-1]*np.cos(alpha/2),Radius[-1]*np.sin(alpha/2)))
p.seedEdgeBySize(edges=(pickedEdges1,pickedEdges2,pickedEdges3,pickedEdges4,), size=Plane_Target_seed, deviationFactor=0.05,minSizeFactor=0.1, constraint=FINER)

pickedEdges = e.getByBoundingCylinder((0.0,-1.1*Radius[-1],0.0),(0.0,-0.9*Radius[0],0.0), 1000.0)
p.seedEdgeBySize(edges=pickedEdges, size=75000.0, deviationFactor=0.05,minSizeFactor=0.1, constraint=FINER)
   
cells=c.getByBoundingBox(-Radius[-1]-100,-Radius[-1]-100,-Radius[-1]-100,Radius[-1]+100,Radius[-1]+100,Radius[-1]+100)
pickedRegions =(cells, )

# generate mesh
p.setElementType(regions=pickedRegions, elemTypes=(elemType1, elemType2, elemType3))  
p.generateMesh()
p.setElementType(regions=pickedRegions, elemTypes=(elemType1, elemType2, elemType3))  

## generate core
p = mdb.models[Model_name].parts['L0']
c = p.cells
cells=c.getByBoundingBox(-Radius[0]-100,-Radius[0]-100,-Radius[0]-100,Radius[0]+100,Radius[0]+100,Radius[0]+100)
pickedRegions =(cells, )
p.setMeshControls(regions=c,elemShape=HEX, technique=STRUCTURED,  algorithm=DEFAULT)
p.seedPart(size=Seeds*LM_res, deviationFactor=0.1, minSizeFactor=0.1)
p.generateMesh()
p.setElementType(regions=pickedRegions, elemTypes=(elemType1, elemType2, elemType3))
a = mdb.models[Model_name].rootAssembly
region1=a.instances['I0'].surfaces['Sout0']
region2=a.instances[Model_name + '_Low'].surfaces['Sin1']
mdb.models[Model_name].Tie(name='Constraint-1', master=region1, slave=region2, 
    positionToleranceMethod=COMPUTED, adjust=ON, tieRotations=ON, thickness=ON)

# for lower mantle
a = mdb.models[Model_name].rootAssembly
region1=a.instances[Model_name+'_Low'].surfaces['Sout'+ str(Coarse_Layers)] 
a = mdb.models[Model_name].rootAssembly
region2=a.instances[Model_name].surfaces['Sin' + str(Coarse_Layers+1)]
mdb.models[Model_name].Tie(name='Constraint-2', master=region1, slave=region2, 
    positionToleranceMethod=COMPUTED, adjust=ON, tieRotations=ON, thickness=ON)
p = mdb.models[Model_name].parts[Model_name + '_Low']
p.setMeshControls(regions=c,elemShape=HEX, technique=STRUCTURED,  algorithm=DEFAULT)
p.seedPart(size=Seeds*LM_res, deviationFactor=0.1, minSizeFactor=0.1)                                                             
p.generateMesh()
p.setElementType(regions=pickedRegions, elemTypes=(elemType1, elemType2, elemType3))
p = mdb.models[Model_name].parts[Model_name]

if fixed_point ==1:
    ## Boundary condition.
    a = mdb.models[Model_name].rootAssembly
    region_core = a.instances['I0'].sets['Center']
    mdb.models[Model_name].EncastreBC(name='BC-1', createStepName='Initial',     region=region_core, localCsys=None) ##  fixing center of Earth
                                                       
## Foundation
for i in range(N_layer-1):
    if Density[i]>Density[i+1]:
        if i < Coarse_Layers:
            region=a.instances[Model_name + '_Low'].surfaces['Sin'+str(i+1)]
        else:
            region=a.instances[Model_name].surfaces['Sin'+str(i+1)]
        mdb.models[Model_name].ElasticFoundation(name='F'+str(i), createStepName='Initial', surface=region, stiffness=(Density[i]-Density[i+1])*Gacc[i])
region=a.instances[Model_name].surfaces['Sout'+str(N_layer-1)]
mdb.models[Model_name].ElasticFoundation(name='F'+str(N_layer-1), createStepName='Initial', surface=region, stiffness=Density[N_layer-1]*Gacc[N_layer-1])
  
mdb.saveAs(r'/home/username/HetGroteKoppelScript/GIA_Model/' +Model_name+ '.cae')
mdb.saveAs(r'/home/username/HetGroteKoppelScript/GIA_Model/' +Model_name+'0.cae')
