# 3D GIA model using ABAQUS2019: developed at the Technical University of Delft, Astrodynamics and Space Missions. 
# Author 1: Bas Blank
# Author 2: Haiyang Hu
# Author 3: username van Calcar

# "Iter_ult" is a model iterator for Abaqus-CAE, apply after the the model generated.
# 2014 - 2019: Haiyang Hu and Bas Blank developed the script that: applies the load on the model, computes the deformation and iterates to include the correction for centre of gravity (Initial_CoG_correction), 
	# self gravitation and the sea level equation (Blank et al., 2021). 
# 2019 - 2021: The original model ran all timesteps subsequently before starting the next iteration, username van Calcar adjusted the script to be able to run all iterations for one step before moving on to the next timestep (van Calcar, 2020).
	# This allows to restart the model at any timestep. To do that, the following adjustments have been made: add the generation of the timestep, define the load for the the step, create a analysis or restart job dependent on the timestep, 
	# add the output field and change the stucture of each variable from including all timesteps to include only one timestep. The correction for centre of gravity and the sea level equation are removed for simplicity.

# References: 
	# Blank, B., Barletta, V., Hu, H., Pappa, F., & van der Wal, W. (2021). Effect of Lateral and Stress‐Dependent Viscosity Variations on GIA Induced Uplift Rates in the Amundsen Sea Embayment. Geochemistry, Geophysics, Geosystems, 22(9), e2021GC009807.
	# C.J. van Calcar (2020). Interactions between Ice Sheet Dynamics and Glacial Isostatic Adjustment. Master thesis Delft University of Technology. http://resolver.tudelft.nl/uuid:8a3a4e21-a6f0-4874-a41f-e26ac95f03be


from abaqus import *
from abaqusConstants import *
from caeModules import *
from odbAccess import*
import numpy as np
import math
import os
import sys

if 'Model_data_coupling' in (sys.modules.keys()):
    reload(Model_data_coupling)
    from Model_data_coupling import *
else:
    Dir=r"/home/username/HetGroteKoppelScript/GIA_Model/"
    os.chdir(Dir)
    import Model_data_coupling
    from Model_data_coupling import *
    

os.chdir(Dir)

# Total number of iterations
Res=2
# Start at iteration number Num
Num=1
# Step
i = np.loadtxt('Step.dat'); i=int(i) 

## Remove lock-file if present
if os.path.isfile(Model_name+'.lck'):
   os.remove(Model_name+'.lck')
   
openMdb(pathName='/home/username/HetGroteKoppelScript/GIA_Model/Earth.cae')

## Steps                                             
if i==0:
    if Rampload_enabled==1:
        mdb.models[Model_name].ViscoStep(name='Step'+str(i+1), previous='Initial',    timePeriod=Time[i], maxNumInc=500, initialInc=Time[i]/1e2,     minInc=Time[i]/1e6, maxInc=Time[i]/25, cetol=CETOL, amplitude=RAMP)
    else:
        mdb.models[Model_name].ViscoStep(name='Step'+str(i+1), previous='Initial',    timePeriod=Time[i], maxNumInc=500, initialInc=Time[i]/1e2,     minInc=Time[i]/1e6, maxInc=Time[i]/25, cetol=CETOL)
else:
    Gap=Time[i]-Time[i-1]
    if Rampload_enabled==1:
        mdb.models[Model_name].ViscoStep(name='Step'+str(i+1), previous='Step'+str(i),    timePeriod=Gap, maxNumInc=500, initialInc=Gap/1e2,     minInc=Gap/1e6, maxInc=Gap/25, cetol=CETOL, amplitude=RAMP)
    else:
        mdb.models[Model_name].ViscoStep(name='Step'+str(i+1), previous='Step'+str(i),    timePeriod=Gap, maxNumInc=500, initialInc=Gap/1e2,     minInc=Gap/1e6, maxInc=Gap/25, cetol=CETOL)

# Disable spherical harmonics 0 and 1
if DisableSH_0_1 ==1:
    print 'Start initial sub-Python process to remove SHdegree 0 and 1\n'
    CHECK=os.system('python2.7 sph_tools_initial_load.py') ##1
    print 'Finish initial sub-Python process\n'
    
    DataLoad_noSPH01=np.loadtxt('DataLoad_in.dat')
    DataLoad = DataLoad_noSPH01
else:
    DataLoad =  DataIce

# Create force field 
Coor_sph=mdb.models[Model_name].rootAssembly.datums.values()[0]
theta=np.linspace(0,180,grid1+1)
theta=np.hstack((0,theta))
theta=theta.reshape(theta.size,1)
lam=np.linspace(0,360,grid1*2+1)
lam=np.hstack((lam[lam<=180],np.array([J-360 for J in lam if J>180])))   #in Abaqus, Th start from 0~pi then -pi~0.
New_Force_field=np.vstack((lam,DataLoad))
New_Force_field=np.hstack((theta,New_Force_field)) 
Force_field= tuple([tuple(J) for J in New_Force_field])
     
## apply force field
a = mdb.models[Model_name].rootAssembly
region = a.instances[Model_name].surfaces['Sout'+str(N_layer-1)]
mdb.models[Model_name].MappedField(name='TotalLoad' +str(i)+'_field', description='',
   regionType=POINT, partLevelData=False, localCsys=Coor_sph,
   pointDataFormat=GRID, fieldDataType=SCALAR, gridPointPlane=PLANE32,
   gridPointData={str(Radius[N_layer-1]):Force_field}) 
mdb.models[Model_name].Pressure(name='TotalLoad_'+str(i), createStepName='Step'+str(i+1),region=region,
   distributionType=FIELD, field='TotalLoad'+str(i)+'_field', magnitude=Density_Ice*Gacc[N_layer-1],amplitude=UNSET)

if i>0:
    mdb.models[Model_name].loads['TotalLoad_'+str(i-1)].deactivate('Step'+str(i+1))
            
#Define Output
mdb.models[Model_name].fieldOutputRequests['F-Output-1'].setValues(variables=('U','S','COORD'))

# define job and output
# restart output request for the current timestep
mdb.models[Model_name].steps['Step'+str(i+1)].Restart(frequency=300, numberIntervals=0, overlay=ON, timeMarks=OFF) 
if i>0:
    mdb.models[Model_name].setValues(restartJob=Model_name+str(i-1), restartStep='Step'+str(i))
    
if i==0:
    # analysis job for the first time step
    mdb.Job(name=Model_name+str(i), model=Model_name, description='', type=ANALYSIS, atTime=None, waitMinutes=0, waitHours=0, queue=None, memory=90, memoryUnits=PERCENTAGE, getMemoryFromAnalysis=True, explicitPrecision=SINGLE, nodalOutputPrecision=SINGLE, echoPrint=OFF, modelPrint=OFF, contactPrint=OFF, historyPrint=OFF, userSubroutine='/home/username/HetGroteKoppelScript/GIA_Model/user_2.f', scratch='', multiprocessingMode=DEFAULT, numCpus=CPUs,numDomains=CPUs)  
else:
    # restart jobs for the coming time steps
    mdb.Job(name=Model_name+str(i), model=Model_name, description='', type=RESTART, atTime=None, waitMinutes=0, waitHours=0, queue=None, memory=90, memoryUnits=PERCENTAGE, getMemoryFromAnalysis=True, explicitPrecision=SINGLE, nodalOutputPrecision=SINGLE, echoPrint=OFF, modelPrint=OFF, contactPrint=OFF, historyPrint=OFF, userSubroutine='/home/username/HetGroteKoppelScript/GIA_Model/user_2.f', scratch='', multiprocessingMode=DEFAULT, numCpus=CPUs,numDomains=CPUs)    

################# END MODEL GEN ####################

while Num<=Res: 
   
    print 'Start FEM\n,round '+str(Num)
    mdb.jobs[Model_name+str(i)].submit(consistencyChecking=OFF)
    mdb.jobs[Model_name+str(i)].waitForCompletion()
    print 'Finish FEM\n'
    
    print 'Start odb process\n'
    odb=openOdb(path=Dir +Model_name+str(i)+ '.odb')        
    
    Node_set=[0]*N_layer
    for j in range(N_layer):
        if j > Coarse_Layers:
            Node_set[j]=odb.rootAssembly.instances[Model_name.upper()].nodeSets['IF'+str(j)]
        else:
            Node_set[j]=odb.rootAssembly.instances[Model_name.upper()+'_LOW'].nodeSets['IF'+str(j)]
  
    
    #### con2
    nodes_length=np.zeros(N_layer)
    fileout=open('Data_output.dat','w')
    
    if include_oceanload==1:
        DataIce_ant = np.loadtxt('DataIce_in.dat')
        DataIce = DataIce_ant + dRSL*Density_water
    else:
        DataIce = np.loadtxt('DataIce_in.dat')
    
    gridIce=DataIce.shape[1]/2
         
    ### Compute initial deflection
    U=odb.steps['Step'+str(i+1)].frames[-1].fieldOutputs['U']
    U1=[0]*N_layer
    for j in range(N_layer):

        if Density[j+1]-Density[j]<0:
            print 'Density[j+1]-Density[j]<0'
            U1=U.getSubset(region=Node_set[j])
            L1=len(Node_set[j].nodes)
            nodes_length[j]=L1
            Data_out=np.zeros([L1,4])
            for k in range(L1):
                Data_out[k,0:3]=Node_set[j].nodes[k].coordinates
                X=Node_set[j].nodes[k].coordinates
                a=U1.values[k].data
                radius = np.linalg.norm(X)
                Data_out[k,3]=(X*a).sum()/radius                    
        else:
            print 'Density[j+1]-Density[j]<!0' 
            L1=len(Node_set[j].nodes)
            Data_out=np.zeros([L1,4])
            nodes_length[j]=L1
        np.savetxt(fileout,Data_out,fmt='%1.4e')
    fileout.close()
    np.savetxt('Nodes_config.dat', nodes_length,fmt='%1.4e')
    session.odbs[Model_name+str(i)+'.odb'].close()
    
    ## process the output
    print 'Start sub-Python process\n'
    CHECK=os.system('python2.7 sph_tools.py') ##1
    print 'Finish sub-Python process\n'    

    print 'finished odb process\n'
    Num=Num+1  
    
mdb.saveAs(r'/home/username/HetGroteKoppelScript/GIA_Model/' +Model_name+ '.cae')