! File name: surface_height_module.f90
!
! Copyright (C) 2014 Thomas Reerink
!
! This file is part of the ICEDYN-package
!
! IMAU, Utrecht University, The Netherlands
!

MODULE surface_height_module
  IMPLICIT NONE

CONTAINS
  SUBROUTINE surface_height(sealevel, Hi, Hb, Hs)
    ! Compute surface height Hs from sealevel, Hi and Hb.
    USE configuration_module , ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: sealevel  ! Relative to pd_sealevel [m]
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: Hb        ! Relative to pd_sealevel [m]
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: Hi        ! [m]

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: Hs        ! Relative to pd_sealevel [m]

    ! The surface height Hs depends on the floatation criterion, see equation (3.1):
    Hs = Hi + Max(sealevel - C%ice_density / C%seawater_density * Hi, Hb)
  END SUBROUTINE surface_height
  
  

  SUBROUTINE adapt_Hi_and_Hs_for_no_shelf(mask, sealevel, Hi, Hs, dHi_dt, Us, Vs)
    ! In case of no shelf, the Hs and the Hi are resetted at shelf points
    USE configuration_module , ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(IN)    :: mask      ! Relative to pd_sealevel [m]
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)    :: sealevel  ! Relative to pd_sealevel [m]

    ! In/Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(INOUT) :: Hi        ! [m]
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(INOUT) :: Hs        ! Relative to pd_sealevel [m]
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(INOUT) :: dHi_dt    ! [m yr^-1]
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(INOUT) :: Us        ! [m yr^-1]
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(INOUT) :: Vs        ! [m yr^-1]

    ! Local variables:
    REAL(dp), DIMENSION(C%NY,C%NX)                :: Hs_shelf
    INTEGER,  DIMENSION(C%NY,C%NX)                :: lake_mask

    ! Output: lake_mask
    CALL determine_lake_mask(mask, lake_mask)
    
    ! The surface heigth of the very thin floating ice layer with thickness C%Hi_min:
    Hs_shelf = sealevel + (1._dp - C%ice_density / C%seawater_density) * C%Hi_min
    WHERE(mask == C%type_shelf .AND. lake_mask /= 1)
     Hs     = Hs_shelf
     Hi     = C%Hi_min
     dHi_dt = 0._dp
     Us     = 0._dp
     Vs     = 0._dp
    END WHERE
  END SUBROUTINE adapt_Hi_and_Hs_for_no_shelf



  SUBROUTINE determine_lake_mask(mask, lake_mask)
    ! This routine calculates a lake mask, i.e. lake points have value 1 the rest of the points have value 0.
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(IN)            :: mask

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: lake_mask

    ! Local variables:
    INTEGER                                               :: i, j
    LOGICAL                                               :: need_to_search
    INTEGER,  DIMENSION(C%NY,C%NX)                        :: open_ocean

    ! Mark lake points on an ice sheet:
    lake_mask  = 0
    open_ocean = 0                                                        ! as a start, no ocean point is identified
    WHERE(mask(1   ,   :) == C%type_shelf) open_ocean(1   ,   :) = 1      ! identify all shelf points at domain boundary as open ocean
    WHERE(mask(C%NY,   :) == C%type_shelf) open_ocean(C%NY,   :) = 1      ! identify all shelf points at domain boundary as open ocean
    WHERE(mask(:   ,   1) == C%type_shelf) open_ocean(:   ,   1) = 1      ! identify all shelf points at domain boundary as open ocean
    WHERE(mask(:   ,C%NX) == C%type_shelf) open_ocean(:   ,C%NX) = 1      ! identify all shelf points at domain boundary as open ocean
    need_to_search = .TRUE.                                               ! set initial need to search for closed shelf points
    DO WHILE (need_to_search)
     need_to_search = .FALSE.
     DO i = 2, C%NX - 1                                                   ! loop over domain
     DO j = 2, C%NY - 1
       IF(mask(j,i) == C%type_shelf) THEN                                 ! for each shelf point
        IF(MAX(open_ocean(j+1,i  ), open_ocean(j-1,i  ), &                ! search for a neighboring open ocean point
               open_ocean(j-1,i-1), open_ocean(j+1,i+1), &
               open_ocean(j-1,i+1), open_ocean(j+1,i-1), &
               open_ocean(j  ,i+1), open_ocean(j  ,i-1)) == 1) THEN
         IF(open_ocean(j,i) /= 1) need_to_search = .TRUE.                 ! if such neighboring point is found
         open_ocean(j,i) = 1                                              ! the grid point also belongs to the open ocean
        END IF
       END IF
     END DO
     END DO
    END DO
    WHERE((open_ocean == 0) .AND. (mask == C%type_shelf)) lake_mask = 1   ! all remaining shelf points must be closed shelf points
  END SUBROUTINE determine_lake_mask

END MODULE surface_height_module
