! File name: smooth_module.f90
!
! Copyright (C) 2014 Thomas Reerink
!
! This file is part of the ICEDYN-package
!
! IMAU, Utrecht University, The Netherlands
!

MODULE smooth_module

CONTAINS
  SUBROUTINE smooth(NX, NY, smooth_radius, exponent, V, V_smooth, optional_mask)
    ! This subroutine smooths a field V within a smooth_radius with the
    ! Shepard weigthing. The domain points are handled by extending the domain
    ! of the original points. Optionally only points with a TRUE mask are taken
    ! into account for the smoothing.
    USE configuration_module, ONLY: dp
    IMPLICIT NONE

    ! Input variables:
    INTEGER,                              INTENT(IN)                                        :: NX, NY         ! Must be positive integers
    INTEGER,                              INTENT(IN)                                        :: smooth_radius  ! The radius (in grid units) within points are taken into acoount in the smoothing
    REAL(dp),                             INTENT(IN)                                        :: exponent       ! The distance weighting exponent in the Shepard weighting
    REAL(dp), DIMENSION(NY,NX)          , INTENT(IN)                                        :: V              ! Variable V
    LOGICAL,  DIMENSION(NY,NX), OPTIONAL, INTENT(IN)                                        :: optional_mask  ! mask

    ! Output variables:
    REAL(dp), DIMENSION(NY,NX)          , INTENT(OUT)                                       :: V_smooth       ! Smoothed Variable V_smooth

    ! Local variables:
    INTEGER                                                                                 :: i, j           ! Indices over the whole domain
    INTEGER                                                                                 :: k, l           ! Indices over local smoothing square around a point (j,i)
    REAL(dp)                                                                                :: ShepNumSum     ! The sum of the numerators in the Shepard weighting
    REAL(dp)                                                                                :: ShepDenSum     ! The sum of the denumerators in the Shepard weighting
    REAL(dp)                                                                                :: distance       ! in gridsize units
    REAL(dp), DIMENSION(1-smooth_radius:NY+smooth_radius,1-smooth_radius:NX+smooth_radius)  :: V_extended     ! Array with V, but also defined in a margin outside the NX-NY domain 
    LOGICAL,  DIMENSION(1-smooth_radius:NY+smooth_radius,1-smooth_radius:NX+smooth_radius)  :: mask           ! mask

    IF(PRESENT(optional_mask)) THEN
     ! Filling the usual domain:
     mask(1:NY,1:NX) = optional_mask

     ! Filling the right and left margin of the extended domain:
     DO j = 1 , NY
      mask(j,1-smooth_radius:0)     = mask(j, 1)
      mask(j,NX+1:NX+smooth_radius) = mask(j,NX)
     END DO

     ! Filling the bottom and top margin (including the margin corners) of the extended domain:
     DO i = 1 - smooth_radius, NX + smooth_radius
      mask(1-smooth_radius:0,i)     = mask( 1,i)
      mask(NY+1:NY+smooth_radius,i) = mask(NY,i)
     END DO
    ELSE
     mask = .TRUE.
    END IF

    ! Filling the usual domain:
    V_extended(1:NY,1:NX) = V

    ! Filling the right and left margin of the extended domain:
    DO j = 1 , NY
     V_extended(j,1-smooth_radius:0)     = V(j, 1)
     V_extended(j,NX+1:NX+smooth_radius) = V(j,NX)
    END DO

    ! Filling the bottom and top margin (including the margin corners) of the extended domain:
    DO i = 1 - smooth_radius, NX + smooth_radius
     V_extended(1-smooth_radius:0,i)     = V_extended( 1,i)
     V_extended(NY+1:NY+smooth_radius,i) = V_extended(NY,i)
    END DO

    DO i = 1, NX
    DO j = 1, NY
      ShepNumSum = 0
      ShepDenSum = 0
      DO k = i - smooth_radius, i + smooth_radius
      DO l = j - smooth_radius, j + smooth_radius
        distance = SQRT(REAL((l-j)**2 + (k-i)**2))
        IF(distance <= smooth_radius .AND. distance > 0 .AND. mask(l,k)) THEN
          ShepNumSum = ShepNumSum + (V_extended(l,k) / (distance**exponent))
          ShepDenSum = ShepDenSum + (          1._dp / (distance**exponent))
        END IF
      END DO
      END DO
      V_smooth(j,i) = ShepNumSum / ShepDenSum
    END DO
    END DO
  END SUBROUTINE smooth

END MODULE smooth_module
