! File name: robin_solution_module.f90
!
! Copyright (C) 2014 Thomas Reerink and Michiel Helsen
!
! This file is part of the ICEDYN-package
!
! IMAU, Utrecht University, The Netherlands
!

MODULE robin_solution_module

CONTAINS
  SUBROUTINE calculate_Ti_with_robin_solution(Ts, Hi, netto_mass_balance, mask, Ti)
    ! This subroutine calculates equilibrium temperature profiles using fields of surface temperature 
    ! and the geothermal heat flux as boundary conditions, see Robin solution in:
    ! Cuffey & Paterson 2010, 4th ed, chapter 9, eq. (9.13) - (9.22).
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(IN)  :: Ts                   ! Surface temperature [K]
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(IN)  :: Hi                   ! Ice thickness [m]
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(IN)  :: netto_mass_balance   ! Netto mass balance [meter ice equivalent per year]
    INTEGER,  DIMENSION(     C%NY,C%NX), INTENT(IN)  :: mask                 ! surface type
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX), INTENT(OUT) :: Ti                   ! The calculated new ice temperature distribution
    
    ! Local variables:
    INTEGER                                          :: i, j

    ! Calculation of temperature profile:
    DO i = 1, C%NX
    DO j = 1, C%NY
      Ti(:,j,i) = robin_solution(Ts(j,i), Hi(j,i), netto_mass_balance(j,i), mask(j,i))
    END DO
    END DO
  END SUBROUTINE calculate_Ti_with_robin_solution



  FUNCTION robin_solution(Ts, Hi, netto_mass_balance, mask) RESULT(Ti)
    ! This function calculates for one horizontal grid point the temperature profiles
    ! using the surface temperature and the geothermal heat flux as boundary conditions.
    ! See Robin solution in: Cuffey & Paterson 2010, 4th ed, chapter 9, eq. (9.13) - (9.22).
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp),                  INTENT(IN)  :: Ts                   ! Surface temperature [K]
    REAL(dp),                  INTENT(IN)  :: Hi                   ! Ice thickness [m]
    REAL(dp),                  INTENT(IN)  :: netto_mass_balance   ! Netto mass balance [meter ice equivalent per year]
    INTEGER,                   INTENT(IN)  :: mask                 ! surface type

    ! Result variables:
    REAL(dp), DIMENSION(C%NZ)              :: Ti                   ! The calculated new ice temperature distribution

    ! Local variables:
    INTEGER                                :: k
    REAL(dp)    thermal_length_scale                               ! thermal length scale [m] (the z*)
    REAL(dp)    distance_above_bed                                 ! distance above bed [m]
    REAL(dp)    error_function_1                                   ! error function
    REAL(dp)    error_function_2                                   ! error function

    ! Calculation of temperature profile:
    IF(Hi > C%Hi_min .AND. netto_mass_balance > 0 .AND. mask /= C%type_shelf) THEN
     ! The Robin solution can be used to estimate the subsurface temperature profile
     thermal_length_scale = sqrt(2._dp * C%thermal_diffusivity_robin * Hi / netto_mass_balance)
     DO k = 1, C%NZ
       distance_above_bed = (1._dp - C%zeta(k)) * Hi
       CALL error_function(distance_above_bed / thermal_length_scale, error_function_1)
       CALL error_function(Hi / thermal_length_scale, error_function_2)
       Ti(k) = Ts + sqrt(C%pi) / 2._dp * thermal_length_scale * C%bottom_temperature_gradient_robin * (error_function_1 - error_function_2)
     END DO
    ELSE IF(netto_mass_balance <= 0 .AND. Hi > C%Hi_min) THEN
     ! Ablation area: use linear temperature profile from Ts to (offset below) T_pmp
     Ti(:) = Ts + ((C%triple_point_of_water - C%Claus_Clap_gradient * Hi - C%delta_Ti_pmp_bottom) - Ts) * C%zeta(:)
    ELSE IF(mask == C%type_shelf .AND. Hi > C%Hi_min) THEN
     ! This option initializes the temperature at shelf points linear between Ts and seawater temperature:
     Ti(:) = Ts + C%zeta(:) * (C%seawater_temperature - Ts)
    ELSE
     ! Only very thin layer: use Ts for entire ice profile
     Ti(:) = Ts
    END IF

    ! Correct all temperatures above T_pmp:
    DO k = 1, C%NZ
      IF(Ti(k) > C%triple_point_of_water - C%Claus_Clap_gradient * Hi * C%zeta(k)) &
         Ti(k) = C%triple_point_of_water - C%Claus_Clap_gradient * Hi * C%zeta(k)
    END DO

    RETURN
  END FUNCTION robin_solution



  SUBROUTINE error_function(X, ERR)
    ! Purpose: Compute error function erf(x)
    ! Input:   x   --- Argument of erf(x)
    ! Output:  ERR --- erf(x)
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), INTENT(IN)  :: X
    
    ! Output variables:
    REAL(dp), INTENT(OUT) :: ERR
    
    ! Local variables:
    REAL(dp)              :: EPS
    REAL(dp)              :: X2
    REAL(dp)              :: ER
    REAL(dp)              :: R
    REAL(dp)              :: C0
    INTEGER               :: k
    
    EPS = 1.0E-15_dp
    X2  = X * X
    IF(ABS(X) < 3.5_dp) THEN
     ER = 1.0_dp
     R  = 1.0_dp
     DO k = 1, 50
       R  = R * X2 / (REAL(k, dp) + 0.5_dp)
       ER = ER+R
       IF(ABS(R) < ABS(ER) * EPS) THEN
        C0  = 2.0_dp / SQRT(C%pi) * X * EXP(-X2)
        ERR = C0 * ER
        EXIT
       END IF
     END DO
    ELSE
     ER = 1.0_dp
     R  = 1.0_dp
     DO k = 1, 12
       R  = -R * (REAL(k, dp) - 0.5_dp) / X2
       ER = ER + R
       C0  = EXP(-X2) / (ABS(X) * SQRT(C%pi))
       ERR = 1.0_dp - C0 * ER
       IF(X < 0.0_dp) ERR = -ERR
     END DO
    ENDIF

    RETURN
  END SUBROUTINE error_function

END MODULE robin_solution_module
