! File name: restart_file_module.f90
!
! Copyright (C) 2014 Thomas Reerink
!
! This file is part of the ICEDYN-package
!
! IMAU, Utrecht University, The Netherlands
!

MODULE restart_file_module

CONTAINS
  SUBROUTINE initialize_restart_fields(file_name, nc) 
    USE configuration_module, ONLY: C
    USE netcdf_module, ONLY: netcdf_file_type
    USE netcdf, ONLY: nf90_int, nf90_double, nf90_unlimited
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*),       INTENT(IN)  :: file_name

    ! Output variables:
    TYPE(netcdf_file_type), INTENT(OUT) :: nc

    ! Local variables:
    INTEGER                             :: stat

    nc%NDIM = 4
    nc%NVAR = nc%NDIM + 7

    ! Setting the number of loops in which the dimensions + variables are written to the netcdf file:
    nc%N_loop = nc%NVAR

    ALLOCATE(nc%field_name       (nc%NVAR), &
             nc%field_unit       (nc%NVAR), &
             nc%field_longname   (nc%NVAR), &
             nc%id               (nc%NVAR), &
             nc%type             (nc%NVAR), &
             nc%case             (nc%NVAR), &
             nc%LEN_DIM          (nc%NDIM), &
             nc%grid_size        (nc%NDIM), &
             STAT=stat)
    IF(stat /= 0) STOP ' STOP message from: initialize_restart_fields:  Could not allocate memory for the netcdf file struct.'

    ! The file name is stored for error messaging:
    nc%file_name = file_name

    nc%field_name(        1) = 'x          '
    nc%field_name(        2) = 'y          '
    nc%field_name(        3) = 'zeta       '
    nc%field_name(nc%NDIM  ) = 'time       '
    nc%field_name(nc%NDIM+1) = 'Hi         '
    nc%field_name(nc%NDIM+2) = 'Hb         '
    nc%field_name(nc%NDIM+3) = 'Hs         '
    nc%field_name(nc%NDIM+4) = 'Us         '
    nc%field_name(nc%NDIM+5) = 'Vs         '
    nc%field_name(nc%NDIM+6) = 'bottom_melt'
    nc%field_name(nc%NDIM+7) = 'Ti         '

    nc%field_unit(        1) = 'meter    ' ! x          
    nc%field_unit(        2) = 'meter    ' ! y          
    nc%field_unit(        3) = 'unitless ' ! zeta       
    nc%field_unit(nc%NDIM  ) = 'year     ' ! time       
    nc%field_unit(nc%NDIM+1) = 'm        ' ! Hi         
    nc%field_unit(nc%NDIM+2) = 'm        ' ! Hb         
    nc%field_unit(nc%NDIM+3) = 'm        ' ! Hs         
    nc%field_unit(nc%NDIM+4) = 'm / y    ' ! Us         
    nc%field_unit(nc%NDIM+5) = 'm / y    ' ! Vs         
    nc%field_unit(nc%NDIM+6) = 'mieq / y ' ! bottom_melt
    nc%field_unit(nc%NDIM+7) = 'K        ' ! Ti         

    nc%field_longname(        1) = 'horizontal x-axis (meter)'
    nc%field_longname(        2) = 'horizontal y-axis (meter)'
    nc%field_longname(        3) = 'scaled vertical zeta-axis (unitless)'
    nc%field_longname(nc%NDIM  ) = 'Time (year)'
    nc%field_longname(nc%NDIM+1) = 'Ice Thickness (meter)'
    nc%field_longname(nc%NDIM+2) = 'Height of Bedrock (meter) - relative to presentday sea level'
    nc%field_longname(nc%NDIM+3) = 'Surface Height (meter) - relative to presentday sea level'
    nc%field_longname(nc%NDIM+4) = 'Depth averaged velocity in x-direction (meter / year)'
    nc%field_longname(nc%NDIM+5) = 'Depth averaged velocity in y-direction (meter / year)'
    nc%field_longname(nc%NDIM+6) = 'Bottom melt (meter ice equivalent / year)'
    nc%field_longname(nc%NDIM+7) = 'Ice Temperature (Kelvin)'

    nc%type(         1:nc%NVAR )    = nf90_double

    nc%case(                  1)    =    1 ! The dimension: X
    nc%case(                  2)    =    2 ! The dimension: Y
    nc%case(                  3)    =    3 ! The dimension: zeta
    nc%case(nc%NDIM            )    =    6 ! The dimension: time
    nc%case(nc%NDIM+1:nc%NDIM+6)    =    7 ! the NX, NY     , time case  [Default case]
    nc%case(nc%NDIM+7:nc%NDIM+7)    =    8 ! the NX, NY, NZ , time case

    nc%LEN_DIM(1      )             = C%NX
    nc%LEN_DIM(2      )             = C%NY
    nc%LEN_DIM(3      )             = C%NZ
    nc%LEN_DIM(nc%NDIM)             = nf90_unlimited

    nc%grid_size(1      )           = C%dx
    nc%grid_size(2      )           = C%dy
   !nc%grid_size(3      )           =        ! non equidistant
   !nc%grid_size(nc%NDIM)           = C%dt   ! but the time step can be dynamical: see G%dt
  END SUBROUTINE initialize_restart_fields



  SUBROUTINE write_restart_fields(nc, record_number, time, Hi, Hb, Hs, Us, Vs, bottom_melt, Ti)
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type, handle_error
    USE netcdf, ONLY: nf90_put_var, nf90_sync
    IMPLICIT NONE

    ! Input variables:
    TYPE(netcdf_file_type),                                         INTENT(IN)         :: nc
    INTEGER,                                                        INTENT(IN)         :: record_number
    REAL(dp),                                                       INTENT(IN)         :: time
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hi
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hb
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hs
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Us
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Vs
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: bottom_melt
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Ti

    ! Local variables:
    INTEGER                                                                            :: m, n        ! Grid point counters
    INTEGER                                                                            :: i           ! Counter over the dimensions or the variables
    REAL(dp), DIMENSION(nc%NDIM+1:nc%NDIM+6,nc%LEN_DIM(1),nc%LEN_DIM(2)              ) :: var_data_2D ! The 2D data of the variables in one array
    REAL(dp), DIMENSION(nc%NDIM+7:nc%NDIM+7,nc%LEN_DIM(1),nc%LEN_DIM(2),nc%LEN_DIM(3)) :: var_data_3D ! The 3D data of the variables in one array

    ! Transpose:
    DO m = 1, nc%LEN_DIM(1)
    DO n = 1, nc%LEN_DIM(2) 
      var_data_2D(nc%NDIM+1,m,n  ) = Hi         (  n,m)
      var_data_2D(nc%NDIM+2,m,n  ) = Hb         (  n,m)
      var_data_2D(nc%NDIM+3,m,n  ) = Hs         (  n,m)
      var_data_2D(nc%NDIM+4,m,n  ) = Us         (  n,m)
      var_data_2D(nc%NDIM+5,m,n  ) = Vs         (  n,m)
      var_data_2D(nc%NDIM+6,m,n  ) = bottom_melt(  n,m)
      var_data_3D(nc%NDIM+7,m,n,:) = Ti         (:,n,m)
    END DO           
    END DO

    ! Data writing mode:
    DO i = nc%NDIM, nc%N_loop
      SELECT CASE(nc%case(i))
      ! Write the data for the unlimited (time) dimension (the other dimension data is written by the subroutine create_netcdf_file):
      CASE(6)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ time /)          , start=(/         record_number/)), '. [1] From write_restart_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)

      ! Write the data of the variables:
      CASE(7)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_2D(i,:,:  ), start=(/1, 1,    record_number/)), '. [2] From write_restart_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(8)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_3D(i,:,:,:), start=(/1, 1, 1, record_number/)), '. [3] From write_restart_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE DEFAULT
       ! Do nothing: Ignore variables with another mode
      END SELECT
    END DO

    ! Synchronize the disk copy of a netcdf dataset: to minimize data loss in case of an abnormal termination:
    IF(C%synchronize_netcdf_writing) CALL handle_error(nf90_sync(nc%ncid)                                       , '. [4] From write_restart_fields(): while synchronizing the file: '//TRIM(nc%file_name))
  END SUBROUTINE write_restart_fields
  


  SUBROUTINE read_restart_file(file_name, record_number, Hi, Hb, Hs, Us, Vs, bottom_melt, Ti, time)
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type, handle_error, open_netcdf_file, inquire_netcdf_file, count_netcdf_record_size, close_netcdf_file
    USE netcdf, ONLY: nf90_get_var
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*),                     INTENT(IN)            :: file_name
    INTEGER,                              INTENT(IN)            :: record_number

    ! Output variables:
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hi
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hb
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hs
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Us
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Vs
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: bottom_melt
    REAL(dp), DIMENSION(C%NZ ,C%NY,C%NX), INTENT(OUT)           :: Ti
    REAL(dp),                             INTENT(OUT), OPTIONAL :: time

    ! Local variables:
    TYPE(netcdf_file_type)                                      :: nc      
    INTEGER                                                     :: m, n          ! Grid point counters
    INTEGER                                                     :: i             ! Counter over the dimensions or the variables
    INTEGER                                                     :: record_size
    INTEGER                                                     :: used_record   ! The record number which is used for reading
    REAL(dp), DIMENSION(:,:   ,:         ), ALLOCATABLE         :: T_var_2D
    REAL(dp), DIMENSION(:,:   ,:   ,:    ), ALLOCATABLE         :: T_var_3D

    ! Output: netcdf 
    CALL initialize_restart_fields(file_name, nc)

    ALLOCATE(T_var_2D(nc%NDIM+1:nc%NDIM+6,C%NX,C%NY     ))
    ALLOCATE(T_var_3D(nc%NDIM+7:nc%NDIM+7,C%NX,C%NY,C%NZ))

    ! In/Output: netcdf
    CALL open_netcdf_file(nc) 

    ! In/Output: netcdf
    CALL inquire_netcdf_file(nc)

    ! Output: record_size
    CALL count_netcdf_record_size(nc, record_size)

    IF(0 < record_number .AND. record_number <= record_size) THEN
     used_record = record_number
    ELSE IF(record_number == 0) THEN
     ! In this case the variables from the last record:
     used_record = record_size
    ELSE
     WRITE(UNIT=*,FMT='(A,I6,A)') 'ERROR: Record number ', record_number, ' does not exist in the file:  '//TRIM(nc%file_name)
     STOP
    END IF

    DO i = nc%NDIM, nc%NVAR
      SELECT CASE(nc%case(i))
       CASE(6)
       IF(PRESENT(time)) &
        CALL handle_error(nf90_get_var(nc%ncid, nc%id(i), time             , start=(/         used_record/) ), '. [1] From read_restart_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(7)
        CALL handle_error(nf90_get_var(nc%ncid, nc%id(i), T_var_2D(i,:,:  ), start=(/1, 1,    used_record/) ), '. [2] From read_restart_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(8)
        CALL handle_error(nf90_get_var(nc%ncid, nc%id(i), T_var_3D(i,:,:,:), start=(/1, 1, 1, used_record/) ), '. [3] From read_restart_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE DEFAULT
        !  Do nothing: Ignore variables with another mode
      END SELECT
    END DO

    ! Finish and closing writing output:
    CALL close_netcdf_file(nc)

    ! Transpose the read fields from {x,y,*} to {*,y,x}:
    DO m = 1, C%NX
    DO n = 1, C%NY 
      Hi         (  n,m) = T_var_2D(nc%NDIM+1,m,n  )
      Hb         (  n,m) = T_var_2D(nc%NDIM+2,m,n  )
      Hs         (  n,m) = T_var_2D(nc%NDIM+3,m,n  )
      Us         (  n,m) = T_var_2D(nc%NDIM+4,m,n  )
      Vs         (  n,m) = T_var_2D(nc%NDIM+5,m,n  )
      bottom_melt(  n,m) = T_var_2D(nc%NDIM+6,m,n  )
      Ti         (:,n,m) = T_var_3D(nc%NDIM+7,m,n,:)
    END DO
    END DO

    DEALLOCATE(T_var_2D)
    DEALLOCATE(T_var_3D)
  END SUBROUTINE read_restart_file



  SUBROUTINE initialize_help_fields_file(file_name, nc) 
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type
    USE netcdf, ONLY: nf90_int, nf90_double, nf90_unlimited
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*),       INTENT(IN)  :: file_name

    ! Output variables:
    TYPE(netcdf_file_type), INTENT(OUT) :: nc

    ! Local variables:
    INTEGER                             :: i          ! Counter over the ignored help fields
    INTEGER                             :: stat

    nc%NDIM = 5
    nc%NVAR = nc%NDIM + 200

    ! Setting the number of loops in which the dimensions + variables are written to the netcdf file:
    nc%N_loop = MIN(nc%NDIM + C%which_help_fields, nc%NVAR)

    ALLOCATE(nc%field_name       (nc%NVAR), &
             nc%field_unit       (nc%NVAR), &
             nc%field_longname   (nc%NVAR), &
             nc%id               (nc%NVAR), &
             nc%type             (nc%NVAR), &
             nc%case             (nc%NVAR), &
             nc%LEN_DIM          (nc%NDIM), &
             nc%grid_size        (nc%NDIM), &
             STAT=stat)
    IF(stat/=0) STOP ' STOP message from: initialize_help_fields_file:  Could not allocate memory for the netcdf file struct.'

    ! The file name is stored for error messaging:
    nc%file_name = file_name

    nc%field_name(          1) = 'x                               '
    nc%field_name(          2) = 'y                               '
    nc%field_name(          3) = 'zeta                            '
    nc%field_name(          4) = 'z                               '
    nc%field_name(nc%NDIM    ) = 'time                            '
    nc%field_name(nc%NDIM+  1) = 'smb                             '
    nc%field_name(nc%NDIM+  2) = 'bmb                             '
    nc%field_name(nc%NDIM+  3) = 'MB_applied                      '
    nc%field_name(nc%NDIM+  4) = 'forcing_concentration           '
    nc%field_name(nc%NDIM+  5) = 'A_flow_mean                     '
    nc%field_name(nc%NDIM+  6) = 'D_2D                            '
    nc%field_name(nc%NDIM+  7) = 'number_of_D_uv_3D_interventions '
    nc%field_name(nc%NDIM+  8) = 'q_bottom                        '
    nc%field_name(nc%NDIM+  9) = 'q_rock_to_ice                   '
    nc%field_name(nc%NDIM+ 10) = 'q_friction                      '
    nc%field_name(nc%NDIM+ 11) = 'water_level                     '
    nc%field_name(nc%NDIM+ 12) = 'sliding_factor                  '
    nc%field_name(nc%NDIM+ 13) = 'mask                            '
    nc%field_name(nc%NDIM+ 14) = 'mask_changes                    '
    nc%field_name(nc%NDIM+ 15) = 'calve_mask                      '
    nc%field_name(nc%NDIM+ 16) = 'hi_x_mask                       '
    nc%field_name(nc%NDIM+ 17) = 'x_discretization_mask           '
    nc%field_name(nc%NDIM+ 18) = 'y_discretization_mask           '
    nc%field_name(nc%NDIM+ 19) = 'bottom_level                    '
    nc%field_name(nc%NDIM+ 20) = 'db_dx                           '
    nc%field_name(nc%NDIM+ 21) = 'db_dy                           '
    nc%field_name(nc%NDIM+ 22) = 'dHi_dxy                         '
    nc%field_name(nc%NDIM+ 23) = 'dHs_dxy                         '
    nc%field_name(nc%NDIM+ 24) = 'dHi_dx                          '
    nc%field_name(nc%NDIM+ 25) = 'dHi_dy                          '
    nc%field_name(nc%NDIM+ 26) = 'dHi_dxx                         '
    nc%field_name(nc%NDIM+ 27) = 'dHi_dyy                         '
    nc%field_name(nc%NDIM+ 28) = 'dHs_dx                          '
    nc%field_name(nc%NDIM+ 29) = 'dHs_dy                          '
    nc%field_name(nc%NDIM+ 30) = 'dHs_dxx                         '
    nc%field_name(nc%NDIM+ 31) = 'dHs_dyy                         '
    nc%field_name(nc%NDIM+ 32) = 'dzeta_dz                        '
    nc%field_name(nc%NDIM+ 33) = 'dzeta_dxz                       '
    nc%field_name(nc%NDIM+ 34) = 'dzeta_dyz                       '
    nc%field_name(nc%NDIM+ 35) = 'Hi                              '
    nc%field_name(nc%NDIM+ 36) = 'Hb                              '
    nc%field_name(nc%NDIM+ 37) = 'Hs                              '
    nc%field_name(nc%NDIM+ 38) = 'dHi_dt                          '
    nc%field_name(nc%NDIM+ 39) = 'dHb_dt                          '
    nc%field_name(nc%NDIM+ 40) = 'dHs_dt                          '
    nc%field_name(nc%NDIM+ 41) = 'Us                              '
    nc%field_name(nc%NDIM+ 42) = 'Vs                              '
    nc%field_name(nc%NDIM+ 43) = 'intervention_cover              '
    nc%field_name(nc%NDIM+ 44) = 'coverage_ti_pmp                 '
    nc%field_name(nc%NDIM+ 45) = 'cover_adjusted_temperatures     '
    nc%field_name(nc%NDIM+ 46) = 'longitude                       '
    nc%field_name(nc%NDIM+ 47) = 'latitude                        '
    nc%field_name(nc%NDIM+ 48) = 'hi_y_mask                       '
    nc%field_name(nc%NDIM+ 49) = 'critical_time_step_pattern      '
    nc%field_name(nc%NDIM+ 50) = 'bottom_melt                     '
    nc%field_name(nc%NDIM+ 51) = 'sor_residue_x                   '
    nc%field_name(nc%NDIM+ 52) = 'sor_residue_y                   '
   !nc%field_name(nc%NDIM+ 51) = 'dummy_2_2D                      '
   !nc%field_name(nc%NDIM+ 52) = 'dummy_1_2D                      '
    nc%field_name(nc%NDIM+ 53) = 'U                               '   
    nc%field_name(nc%NDIM+ 54) = 'V                               '
    nc%field_name(nc%NDIM+ 55) = 'W                               '
    nc%field_name(nc%NDIM+ 56) = 'Ti                              '
    nc%field_name(nc%NDIM+ 57) = 'Ti_pmp                          '
    nc%field_name(nc%NDIM+ 58) = 'A_flow                          '
    nc%field_name(nc%NDIM+ 59) = 'Cpi                             '
    nc%field_name(nc%NDIM+ 60) = 'Ki                              '
    nc%field_name(nc%NDIM+ 61) = 'D_uv_3D                         '
    nc%field_name(nc%NDIM+ 62) = 'concentration                   '
    nc%field_name(nc%NDIM+ 63) = 'dTi_dx                          '
    nc%field_name(nc%NDIM+ 64) = 'dTi_dy                          '
    nc%field_name(nc%NDIM+ 65) = 'dU_dx                           '
    nc%field_name(nc%NDIM+ 66) = 'dV_dx                           '
    nc%field_name(nc%NDIM+ 67) = 'dW_dx                           '
    nc%field_name(nc%NDIM+ 68) = 'dU_dy                           '
    nc%field_name(nc%NDIM+ 69) = 'dV_dy                           '
    nc%field_name(nc%NDIM+ 70) = 'dW_dy                           '
    nc%field_name(nc%NDIM+ 71) = 'dU_dzeta                        '
    nc%field_name(nc%NDIM+ 72) = 'dV_dzeta                        '
    nc%field_name(nc%NDIM+ 73) = 'dW_dzeta                        '
    nc%field_name(nc%NDIM+ 74) = 'dzeta_dt                        '
    nc%field_name(nc%NDIM+ 75) = 'dzeta_dx                        '
    nc%field_name(nc%NDIM+ 76) = 'dzeta_dy                        '
    nc%field_name(nc%NDIM+ 77) = 'dzeta_dxx                       '
    nc%field_name(nc%NDIM+ 78) = 'dzeta_dxy                       '
    nc%field_name(nc%NDIM+ 79) = 'dzeta_dyy                       '
    nc%field_name(nc%NDIM+ 80) = 'Eta                             '
    nc%field_name(nc%NDIM+ 81) = 'dEta_dx                         '
    nc%field_name(nc%NDIM+ 82) = 'dEta_dy                         '
    nc%field_name(nc%NDIM+ 83) = 'dEta_dzeta                      '
    nc%field_name(nc%NDIM+ 84) = 'Eta_x                           '
    nc%field_name(nc%NDIM+ 85) = 'Eta_y                           '
    nc%field_name(nc%NDIM+ 86) = 'Eta_zeta                        '
    nc%field_name(nc%NDIM+ 87) = 'rhs_eq1                         '
    nc%field_name(nc%NDIM+ 88) = 'rhs_eq2                         '
    nc%field_name(nc%NDIM+ 89) = 'rhs_eq1_calving_bc              '
    nc%field_name(nc%NDIM+ 90) = 'U_linear                        ' 
    nc%field_name(nc%NDIM+ 91) = 'V_linear                        '
    nc%field_name(nc%NDIM+ 92) = 'W_linear                        '
    nc%field_name(nc%NDIM+ 93) = 'Ti_linear                       '
    nc%field_name(nc%NDIM+ 94) = 'Ti_pmp_linear                   '
    nc%field_name(nc%NDIM+ 95) = 'A_flow_linear                   '
    nc%field_name(nc%NDIM+ 96) = 'Cpi_linear                      '
    nc%field_name(nc%NDIM+ 97) = 'Ki_linear                       '
    nc%field_name(nc%NDIM+ 98) = 'D_uv_3D_linear                  '
    nc%field_name(nc%NDIM+ 99) = 'concentration_linear            '
    nc%field_name(nc%NDIM+100) = 'dTi_dx_linear                   '
    nc%field_name(nc%NDIM+101) = 'dTi_dy_linear                   '
    nc%field_name(nc%NDIM+102) = 'dU_dx_linear                    '
    nc%field_name(nc%NDIM+103) = 'dV_dx_linear                    '
    nc%field_name(nc%NDIM+104) = 'dW_dx_linear                    '
    nc%field_name(nc%NDIM+105) = 'dU_dy_linear                    '
    nc%field_name(nc%NDIM+106) = 'dV_dy_linear                    '
    nc%field_name(nc%NDIM+107) = 'dW_dy_linear                    '
    nc%field_name(nc%NDIM+108) = 'dU_dzeta_linear                 '
    nc%field_name(nc%NDIM+109) = 'dV_dzeta_linear                 '
    nc%field_name(nc%NDIM+110) = 'dW_dzeta_linear                 '
    nc%field_name(nc%NDIM+111) = 'dzeta_dt_linear                 '
    nc%field_name(nc%NDIM+112) = 'dzeta_dx_linear                 '
    nc%field_name(nc%NDIM+113) = 'dzeta_dy_linear                 '
    nc%field_name(nc%NDIM+114) = 'dzeta_dxx_linear                '
    nc%field_name(nc%NDIM+115) = 'dzeta_dxy_linear                '
    nc%field_name(nc%NDIM+116) = 'dzeta_dyy_linear                '
    nc%field_name(nc%NDIM+117) = 'Eta_linear                      '
    nc%field_name(nc%NDIM+118) = 'dEta_dx_linear                  '
    nc%field_name(nc%NDIM+119) = 'dEta_dy_linear                  '
    nc%field_name(nc%NDIM+120) = 'dEta_dzeta_linear               '
    nc%field_name(nc%NDIM+121) = 'Eta_x_linear                    '
    nc%field_name(nc%NDIM+122) = 'Eta_y_linear                    '
    nc%field_name(nc%NDIM+123) = 'Eta_zeta_linear                 '
    nc%field_name(nc%NDIM+124) = 'rhs_eq1_linear                  '
    nc%field_name(nc%NDIM+125) = 'rhs_eq2_linear                  '
    nc%field_name(nc%NDIM+126) = 'rhs_eq1_calving_bc_linear       '
    nc%field_name(nc%NDIM+127) = 'U_linear_xz_cs                  '   ! cs = cross section,  xz means 'cs of the xz-plane'
    nc%field_name(nc%NDIM+128) = 'V_linear_xz_cs                  '
    nc%field_name(nc%NDIM+129) = 'W_linear_xz_cs                  '
    nc%field_name(nc%NDIM+130) = 'Ti_linear_xz_cs                 '
    nc%field_name(nc%NDIM+131) = 'Ti_pmp_linear_xz_cs             '
    nc%field_name(nc%NDIM+132) = 'A_flow_linear_xz_cs             '
    nc%field_name(nc%NDIM+133) = 'Cpi_linear_xz_cs                '
    nc%field_name(nc%NDIM+134) = 'Ki_linear_xz_cs                 '
    nc%field_name(nc%NDIM+135) = 'D_uv_3D_linear_xz_cs            '
    nc%field_name(nc%NDIM+136) = 'concentration_linear_xz_cs      '
    nc%field_name(nc%NDIM+137) = 'dTi_dx_linear_xz_cs             ' 
    nc%field_name(nc%NDIM+138) = 'dTi_dy_linear_xz_cs             '
    nc%field_name(nc%NDIM+139) = 'dU_dx_linear_xz_cs              '
    nc%field_name(nc%NDIM+140) = 'dV_dx_linear_xz_cs              '
    nc%field_name(nc%NDIM+141) = 'dW_dx_linear_xz_cs              '
    nc%field_name(nc%NDIM+142) = 'dU_dy_linear_xz_cs              '
    nc%field_name(nc%NDIM+143) = 'dV_dy_linear_xz_cs              '
    nc%field_name(nc%NDIM+144) = 'dW_dy_linear_xz_cs              '
    nc%field_name(nc%NDIM+145) = 'dU_dzeta_linear_xz_cs           '
    nc%field_name(nc%NDIM+146) = 'dV_dzeta_linear_xz_cs           '
    nc%field_name(nc%NDIM+147) = 'dW_dzeta_linear_xz_cs           '
    nc%field_name(nc%NDIM+148) = 'dzeta_dt_linear_xz_cs           '
    nc%field_name(nc%NDIM+149) = 'dzeta_dx_linear_xz_cs           '
    nc%field_name(nc%NDIM+150) = 'dzeta_dy_linear_xz_cs           '
    nc%field_name(nc%NDIM+151) = 'dzeta_dxx_linear_xz_cs          '
    nc%field_name(nc%NDIM+152) = 'dzeta_dxy_linear_xz_cs          '
    nc%field_name(nc%NDIM+153) = 'dzeta_dyy_linear_xz_cs          '
    nc%field_name(nc%NDIM+154) = 'Eta_linear_xz_cs                '
    nc%field_name(nc%NDIM+155) = 'dEta_dx_linear_xz_cs            '
    nc%field_name(nc%NDIM+156) = 'dEta_dy_linear_xz_cs            '
    nc%field_name(nc%NDIM+157) = 'dEta_dzeta_linear_xz_cs         '
    nc%field_name(nc%NDIM+158) = 'Eta_x_linear_xz_cs              '
    nc%field_name(nc%NDIM+159) = 'Eta_y_linear_xz_cs              '
    nc%field_name(nc%NDIM+160) = 'Eta_zeta_linear_xz_cs           '
    nc%field_name(nc%NDIM+161) = 'rhs_eq1_linear_xz_cs            '
    nc%field_name(nc%NDIM+162) = 'rhs_eq2_linear_xz_cs            '
    nc%field_name(nc%NDIM+163) = 'rhs_eq1_calving_bc_linear_xz_cs '
    nc%field_name(nc%NDIM+164) = 'U_linear_yz_cs                  '   ! cs = cross section,  yz means 'cs of the yz-plane'
    nc%field_name(nc%NDIM+165) = 'V_linear_yz_cs                  '
    nc%field_name(nc%NDIM+166) = 'W_linear_yz_cs                  '
    nc%field_name(nc%NDIM+167) = 'Ti_linear_yz_cs                 '
    nc%field_name(nc%NDIM+168) = 'Ti_pmp_linear_yz_cs             '
    nc%field_name(nc%NDIM+169) = 'A_flow_linear_yz_cs             '
    nc%field_name(nc%NDIM+170) = 'Cpi_linear_yz_cs                '
    nc%field_name(nc%NDIM+171) = 'Ki_linear_yz_cs                 '
    nc%field_name(nc%NDIM+172) = 'D_uv_3D_linear_yz_cs            '
    nc%field_name(nc%NDIM+173) = 'concentration_linear_yz_cs      '
    nc%field_name(nc%NDIM+174) = 'dTi_dx_linear_yz_cs             ' 
    nc%field_name(nc%NDIM+175) = 'dTi_dy_linear_yz_cs             '
    nc%field_name(nc%NDIM+176) = 'dU_dx_linear_yz_cs              '
    nc%field_name(nc%NDIM+177) = 'dV_dx_linear_yz_cs              '
    nc%field_name(nc%NDIM+178) = 'dW_dx_linear_yz_cs              '
    nc%field_name(nc%NDIM+179) = 'dU_dy_linear_yz_cs              '
    nc%field_name(nc%NDIM+180) = 'dV_dy_linear_yz_cs              '
    nc%field_name(nc%NDIM+181) = 'dW_dy_linear_yz_cs              '
    nc%field_name(nc%NDIM+182) = 'dU_dzeta_linear_yz_cs           '
    nc%field_name(nc%NDIM+183) = 'dV_dzeta_linear_yz_cs           '
    nc%field_name(nc%NDIM+184) = 'dW_dzeta_linear_yz_cs           '
    nc%field_name(nc%NDIM+185) = 'dzeta_dt_linear_yz_cs           '
    nc%field_name(nc%NDIM+186) = 'dzeta_dx_linear_yz_cs           '
    nc%field_name(nc%NDIM+187) = 'dzeta_dy_linear_yz_cs           '
    nc%field_name(nc%NDIM+188) = 'dzeta_dxx_linear_yz_cs          '
    nc%field_name(nc%NDIM+189) = 'dzeta_dxy_linear_yz_cs          '
    nc%field_name(nc%NDIM+190) = 'dzeta_dyy_linear_yz_cs          '
    nc%field_name(nc%NDIM+191) = 'Eta_linear_yz_cs                '
    nc%field_name(nc%NDIM+192) = 'dEta_dx_linear_yz_cs            '
    nc%field_name(nc%NDIM+193) = 'dEta_dy_linear_yz_cs            '
    nc%field_name(nc%NDIM+194) = 'dEta_dzeta_linear_yz_cs         '
    nc%field_name(nc%NDIM+195) = 'Eta_x_linear_yz_cs              '
    nc%field_name(nc%NDIM+196) = 'Eta_y_linear_yz_cs              '
    nc%field_name(nc%NDIM+197) = 'Eta_zeta_linear_yz_cs           '
    nc%field_name(nc%NDIM+198) = 'rhs_eq1_linear_yz_cs            '
    nc%field_name(nc%NDIM+199) = 'rhs_eq2_linear_yz_cs            '
    nc%field_name(nc%NDIM+200) = 'rhs_eq1_calving_bc_linear_yz_cs '

    nc%field_unit(          1) = 'meter            ' ! x
    nc%field_unit(          2) = 'meter            ' ! y
    nc%field_unit(          3) = 'unitless         ' ! zeta
    nc%field_unit(          4) = 'meter            ' ! z
    nc%field_unit(nc%NDIM    ) = 'year             ' ! time
    nc%field_unit(nc%NDIM+  1) = 'mieq / y         ' ! smb
    nc%field_unit(nc%NDIM+  2) = 'mieq / y         ' ! bmb
    nc%field_unit(nc%NDIM+  3) = 'mieq / y         ' ! MB_applied
    nc%field_unit(nc%NDIM+  4) = 'particles / m^2  ' ! forcing_concentration
    nc%field_unit(nc%NDIM+  5) = 'Pa^-n y^-1       ' ! A_flow_mean
    nc%field_unit(nc%NDIM+  6) = 'm / y            ' ! D_2D
    nc%field_unit(nc%NDIM+  7) = 'unitless         ' ! number_of_D_uv_3D_interventions
    nc%field_unit(nc%NDIM+  8) = 'J m^-2 y^-1      ' ! q_bottom
    nc%field_unit(nc%NDIM+  9) = 'J m^-2 y^-1      ' ! q_rock_to_ice
    nc%field_unit(nc%NDIM+ 10) = 'J m^-2 y^-1      ' ! q_friction
    nc%field_unit(nc%NDIM+ 11) = 'm                ' ! water_level
    nc%field_unit(nc%NDIM+ 12) = 'unitless         ' ! sliding_factor
    nc%field_unit(nc%NDIM+ 13) = 'unitless         ' ! mask
    nc%field_unit(nc%NDIM+ 14) = 'unitless         ' ! mask_changes
    nc%field_unit(nc%NDIM+ 15) = 'unitless         ' ! calve_mask
    nc%field_unit(nc%NDIM+ 16) = 'unitless         ' ! hi_x_mask
    nc%field_unit(nc%NDIM+ 17) = 'unitless         ' ! x_discretization_mask
    nc%field_unit(nc%NDIM+ 18) = 'unitless         ' ! y_discretization_mask
    nc%field_unit(nc%NDIM+ 19) = 'm                ' ! bottom_level
    nc%field_unit(nc%NDIM+ 20) = 'm / m            ' ! db_dx
    nc%field_unit(nc%NDIM+ 21) = 'm / m            ' ! db_dy
    nc%field_unit(nc%NDIM+ 22) = 'm / m^2          ' ! dHi_dxy
    nc%field_unit(nc%NDIM+ 23) = 'm / m^2          ' ! dHs_dxy
    nc%field_unit(nc%NDIM+ 24) = 'm / m            ' ! dHi_dx
    nc%field_unit(nc%NDIM+ 25) = 'm / m            ' ! dHi_dy
    nc%field_unit(nc%NDIM+ 26) = 'm / m^2          ' ! dHi_dxx
    nc%field_unit(nc%NDIM+ 27) = 'm / m^2          ' ! dHi_dyy
    nc%field_unit(nc%NDIM+ 28) = 'm / m            ' ! dHs_dx
    nc%field_unit(nc%NDIM+ 29) = 'm / m            ' ! dHs_dy
    nc%field_unit(nc%NDIM+ 30) = 'm / m^2          ' ! dHs_dxx
    nc%field_unit(nc%NDIM+ 31) = 'm / m^2          ' ! dHs_dyy
    nc%field_unit(nc%NDIM+ 32) = 'm^-1             ' ! dzeta_dz
    nc%field_unit(nc%NDIM+ 33) = 'm^-2             ' ! dzeta_dxz
    nc%field_unit(nc%NDIM+ 34) = 'm^-2             ' ! dzeta_dyz
    nc%field_unit(nc%NDIM+ 35) = 'm                ' ! Hi
    nc%field_unit(nc%NDIM+ 36) = 'm                ' ! Hb
    nc%field_unit(nc%NDIM+ 37) = 'm                ' ! Hs
    nc%field_unit(nc%NDIM+ 38) = 'm / y            ' ! dHi_dt
    nc%field_unit(nc%NDIM+ 39) = 'm / y            ' ! dHb_dt
    nc%field_unit(nc%NDIM+ 40) = 'm / y            ' ! dHs_dt
    nc%field_unit(nc%NDIM+ 41) = 'm / y            ' ! Us
    nc%field_unit(nc%NDIM+ 42) = 'm / y            ' ! Vs
    nc%field_unit(nc%NDIM+ 43) = 'unitless         ' ! intervention_cover
    nc%field_unit(nc%NDIM+ 44) = 'unitless         ' ! coverage_ti_pmp
    nc%field_unit(nc%NDIM+ 45) = 'unitless         ' ! cover_adjusted_temperatures
    nc%field_unit(nc%NDIM+ 46) = 'degree           ' ! longitude
    nc%field_unit(nc%NDIM+ 47) = 'degree           ' ! latitude
    nc%field_unit(nc%NDIM+ 48) = 'unitless         ' ! hi_y_mask
    nc%field_unit(nc%NDIM+ 49) = 'year             ' ! critical_time_step_pattern
    nc%field_unit(nc%NDIM+ 50) = '?                ' ! bottom_melt
    nc%field_unit(nc%NDIM+ 51) = 'm / y            ' ! sor_residue_x
    nc%field_unit(nc%NDIM+ 52) = 'm / y            ' ! sor_residue_y
   !nc%field_unit(nc%NDIM+ 51) = '?                ' ! dummy_2_2D
   !nc%field_unit(nc%NDIM+ 52) = '?                ' ! dummy_1_2D
    nc%field_unit(nc%NDIM+ 53) = 'm / y            ' ! U
    nc%field_unit(nc%NDIM+ 54) = 'm / y            ' ! V
    nc%field_unit(nc%NDIM+ 55) = 'm / y            ' ! W
    nc%field_unit(nc%NDIM+ 56) = 'K                ' ! Ti
    nc%field_unit(nc%NDIM+ 57) = 'K                ' ! Ti_pmp
    nc%field_unit(nc%NDIM+ 58) = 'Pa^-n y^-1       ' ! A_flow
    nc%field_unit(nc%NDIM+ 59) = 'J kg^-1 K^-1     ' ! Cpi
    nc%field_unit(nc%NDIM+ 60) = 'J m^-1 K^-1 y^-1 ' ! Ki
    nc%field_unit(nc%NDIM+ 61) = 'm / y            ' ! D_uv_3D
    nc%field_unit(nc%NDIM+ 62) = 'm^-3             ' ! concentration
    nc%field_unit(nc%NDIM+ 63) = 'K / m            ' ! dTi_dx
    nc%field_unit(nc%NDIM+ 64) = 'K / m            ' ! dTi_dy
    nc%field_unit(nc%NDIM+ 65) = 'y^-1             ' ! dU_dx
    nc%field_unit(nc%NDIM+ 66) = 'y^-1             ' ! dV_dx
    nc%field_unit(nc%NDIM+ 67) = 'y^-1             ' ! dW_dx
    nc%field_unit(nc%NDIM+ 68) = 'y^-1             ' ! dU_dy
    nc%field_unit(nc%NDIM+ 69) = 'y^-1             ' ! dV_dy
    nc%field_unit(nc%NDIM+ 70) = 'y^-1             ' ! dW_dy
    nc%field_unit(nc%NDIM+ 71) = 'm / y            ' ! dU_dzeta
    nc%field_unit(nc%NDIM+ 72) = 'm / y            ' ! dV_dzeta
    nc%field_unit(nc%NDIM+ 73) = 'm / y            ' ! dW_dzeta
    nc%field_unit(nc%NDIM+ 74) = 'y^-1             ' ! dzeta_dt
    nc%field_unit(nc%NDIM+ 75) = 'm^-1             ' ! dzeta_dx
    nc%field_unit(nc%NDIM+ 76) = 'm^-1             ' ! dzeta_dy
    nc%field_unit(nc%NDIM+ 77) = 'm^-2             ' ! dzeta_dxx
    nc%field_unit(nc%NDIM+ 78) = 'm^-2             ' ! dzeta_dxy
    nc%field_unit(nc%NDIM+ 79) = 'm^-2             ' ! dzeta_dyy
    nc%field_unit(nc%NDIM+ 80) = 'Pa y             ' ! Eta
    nc%field_unit(nc%NDIM+ 81) = 'Pa y m^-1        ' ! dEta_dx
    nc%field_unit(nc%NDIM+ 82) = 'Pa y m^-1        ' ! dEta_dy
    nc%field_unit(nc%NDIM+ 83) = 'Pa y             ' ! dEta_dzeta
    nc%field_unit(nc%NDIM+ 84) = 'm^-1             ' ! Eta_x
    nc%field_unit(nc%NDIM+ 85) = 'm^-1             ' ! Eta_y
    nc%field_unit(nc%NDIM+ 86) = 'm^-1             ' ! Eta_zeta
    nc%field_unit(nc%NDIM+ 87) = '?                ' ! rhs_eq1
    nc%field_unit(nc%NDIM+ 88) = '?                ' ! rhs_eq2
    nc%field_unit(nc%NDIM+ 89) = '?                ' ! rhs_eq1_calving_bc
    nc%field_unit(nc%NDIM+ 90) = 'm / y            ' ! U_linear
    nc%field_unit(nc%NDIM+ 91) = 'm / y            ' ! V_linear
    nc%field_unit(nc%NDIM+ 92) = 'm / y            ' ! W_linear
    nc%field_unit(nc%NDIM+ 93) = 'K                ' ! Ti_linear
    nc%field_unit(nc%NDIM+ 94) = 'K                ' ! Ti_pmp_linear
    nc%field_unit(nc%NDIM+ 95) = 'Pa^-n y^-1       ' ! A_flow_linear
    nc%field_unit(nc%NDIM+ 96) = 'J kg^-1 K^-1     ' ! Cpi_linear
    nc%field_unit(nc%NDIM+ 97) = 'J m^-1 K^-1 y^-1 ' ! Ki_linear
    nc%field_unit(nc%NDIM+ 98) = 'm / y            ' ! D_uv_3D_linear
    nc%field_unit(nc%NDIM+ 99) = 'm^-3             ' ! concentration_linear
    nc%field_unit(nc%NDIM+100) = 'K / m            ' ! dTi_dx_linear
    nc%field_unit(nc%NDIM+101) = 'K / m            ' ! dTi_dy_linear
    nc%field_unit(nc%NDIM+102) = 'y^-1             ' ! dU_dx_linear
    nc%field_unit(nc%NDIM+103) = 'y^-1             ' ! dV_dx_linear
    nc%field_unit(nc%NDIM+104) = 'y^-1             ' ! dW_dx_linear
    nc%field_unit(nc%NDIM+105) = 'y^-1             ' ! dU_dy_linear
    nc%field_unit(nc%NDIM+106) = 'y^-1             ' ! dV_dy_linear
    nc%field_unit(nc%NDIM+107) = 'y^-1             ' ! dW_dy_linear
    nc%field_unit(nc%NDIM+108) = 'm / y            ' ! dU_dzeta_linear
    nc%field_unit(nc%NDIM+109) = 'm / y            ' ! dV_dzeta_linear
    nc%field_unit(nc%NDIM+110) = 'm / y            ' ! dW_dzeta_linear
    nc%field_unit(nc%NDIM+111) = 'y^-1             ' ! dzeta_dt_linear
    nc%field_unit(nc%NDIM+112) = 'm^-1             ' ! dzeta_dx_linear
    nc%field_unit(nc%NDIM+113) = 'm^-1             ' ! dzeta_dy_linear
    nc%field_unit(nc%NDIM+114) = 'm^-2             ' ! dzeta_dxx_linear
    nc%field_unit(nc%NDIM+115) = 'm^-2             ' ! dzeta_dxy_linear
    nc%field_unit(nc%NDIM+116) = 'm^-2             ' ! dzeta_dyy_linear
    nc%field_unit(nc%NDIM+117) = 'Pa y             ' ! Eta_linear
    nc%field_unit(nc%NDIM+118) = 'Pa y m^-1        ' ! dEta_dx_linear
    nc%field_unit(nc%NDIM+119) = 'Pa y m^-1        ' ! dEta_dy_linear
    nc%field_unit(nc%NDIM+120) = 'Pa y             ' ! dEta_dzeta_linear
    nc%field_unit(nc%NDIM+121) = 'm^-1             ' ! Eta_x_linear
    nc%field_unit(nc%NDIM+122) = 'm^-1             ' ! Eta_y_linear
    nc%field_unit(nc%NDIM+123) = 'm^-1             ' ! Eta_zeta_linear
    nc%field_unit(nc%NDIM+124) = '?                ' ! rhs_eq1_linear
    nc%field_unit(nc%NDIM+125) = '?                ' ! rhs_eq2_linear
    nc%field_unit(nc%NDIM+126) = '?                ' ! rhs_eq1_calving_bc_linear
    nc%field_unit(nc%NDIM+127) = 'm / y            ' ! U_linear_xz_cs
    nc%field_unit(nc%NDIM+128) = 'm / y            ' ! V_linear_xz_cs
    nc%field_unit(nc%NDIM+129) = 'm / y            ' ! W_linear_xz_cs
    nc%field_unit(nc%NDIM+130) = 'K                ' ! Ti_linear_xz_cs
    nc%field_unit(nc%NDIM+131) = 'K                ' ! Ti_pmp_linear_xz_cs
    nc%field_unit(nc%NDIM+132) = 'Pa^-n y^-1       ' ! A_flow_linear_xz_cs
    nc%field_unit(nc%NDIM+133) = 'J kg^-1 K^-1     ' ! Cpi_linear_xz_cs
    nc%field_unit(nc%NDIM+134) = 'J m^-1 K^-1 y^-1 ' ! Ki_linear_xz_cs
    nc%field_unit(nc%NDIM+135) = 'm / y            ' ! D_uv_3D_linear_xz_cs
    nc%field_unit(nc%NDIM+136) = 'm^-3             ' ! concentration_linear_xz_cs
    nc%field_unit(nc%NDIM+137) = 'K / m            ' ! dTi_dx_linear_xz_cs
    nc%field_unit(nc%NDIM+138) = 'K / m            ' ! dTi_dy_linear_xz_cs
    nc%field_unit(nc%NDIM+139) = 'y^-1             ' ! dU_dx_linear_xz_cs
    nc%field_unit(nc%NDIM+140) = 'y^-1             ' ! dV_dx_linear_xz_cs
    nc%field_unit(nc%NDIM+141) = 'y^-1             ' ! dW_dx_linear_xz_cs
    nc%field_unit(nc%NDIM+142) = 'y^-1             ' ! dU_dy_linear_xz_cs
    nc%field_unit(nc%NDIM+143) = 'y^-1             ' ! dV_dy_linear_xz_cs
    nc%field_unit(nc%NDIM+144) = 'y^-1             ' ! dW_dy_linear_xz_cs
    nc%field_unit(nc%NDIM+145) = 'm / y            ' ! dU_dzeta_linear_xz_cs
    nc%field_unit(nc%NDIM+146) = 'm / y            ' ! dV_dzeta_linear_xz_cs
    nc%field_unit(nc%NDIM+147) = 'm / y            ' ! dW_dzeta_linear_xz_cs
    nc%field_unit(nc%NDIM+148) = 'y^-1             ' ! dzeta_dt_linear_xz_cs
    nc%field_unit(nc%NDIM+149) = 'm^-1             ' ! dzeta_dx_linear_xz_cs
    nc%field_unit(nc%NDIM+150) = 'm^-1             ' ! dzeta_dy_linear_xz_cs
    nc%field_unit(nc%NDIM+151) = 'm^-2             ' ! dzeta_dxx_linear_xz_cs
    nc%field_unit(nc%NDIM+152) = 'm^-2             ' ! dzeta_dxy_linear_xz_cs
    nc%field_unit(nc%NDIM+153) = 'm^-2             ' ! dzeta_dyy_linear_xz_cs
    nc%field_unit(nc%NDIM+154) = 'Pa y             ' ! Eta_linear_xz_cs
    nc%field_unit(nc%NDIM+155) = 'Pa y m^-1        ' ! dEta_dx_linear_xz_cs
    nc%field_unit(nc%NDIM+156) = 'Pa y m^-1        ' ! dEta_dy_linear_xz_cs
    nc%field_unit(nc%NDIM+157) = 'Pa y             ' ! dEta_dzeta_linear_xz_cs
    nc%field_unit(nc%NDIM+158) = 'm^-1             ' ! Eta_x_linear_xz_cs
    nc%field_unit(nc%NDIM+159) = 'm^-1             ' ! Eta_y_linear_xz_cs
    nc%field_unit(nc%NDIM+160) = 'm^-1             ' ! Eta_zeta_linear_xz_cs
    nc%field_unit(nc%NDIM+161) = '?                ' ! rhs_eq1_linear_xz_cs
    nc%field_unit(nc%NDIM+162) = '?                ' ! rhs_eq2_linear_xz_cs
    nc%field_unit(nc%NDIM+163) = '?                ' ! rhs_eq1_calving_bc_linear_xz_cs
    nc%field_unit(nc%NDIM+164) = 'm / y            ' ! U_linear_yz_cs
    nc%field_unit(nc%NDIM+165) = 'm / y            ' ! V_linear_yz_cs
    nc%field_unit(nc%NDIM+166) = 'm / y            ' ! W_linear_yz_cs
    nc%field_unit(nc%NDIM+167) = 'K                ' ! Ti_linear_yz_cs
    nc%field_unit(nc%NDIM+168) = 'K                ' ! Ti_pmp_linear_yz_cs
    nc%field_unit(nc%NDIM+169) = 'Pa^-n y^-1       ' ! A_flow_linear_yz_cs
    nc%field_unit(nc%NDIM+170) = 'J kg^-1 K^-1     ' ! Cpi_linear_yz_cs
    nc%field_unit(nc%NDIM+171) = 'J m^-1 K^-1 y^-1 ' ! Ki_linear_yz_cs
    nc%field_unit(nc%NDIM+172) = 'm / y            ' ! D_uv_3D_linear_yz_cs
    nc%field_unit(nc%NDIM+173) = 'm^-3             ' ! concentration_linear_yz_cs
    nc%field_unit(nc%NDIM+174) = 'K / m            ' ! dTi_dx_linear_yz_cs
    nc%field_unit(nc%NDIM+175) = 'K / m            ' ! dTi_dy_linear_yz_cs
    nc%field_unit(nc%NDIM+176) = 'y^-1             ' ! dU_dx_linear_yz_cs
    nc%field_unit(nc%NDIM+177) = 'y^-1             ' ! dV_dx_linear_yz_cs
    nc%field_unit(nc%NDIM+178) = 'y^-1             ' ! dW_dx_linear_yz_cs
    nc%field_unit(nc%NDIM+179) = 'y^-1             ' ! dU_dy_linear_yz_cs
    nc%field_unit(nc%NDIM+180) = 'y^-1             ' ! dV_dy_linear_yz_cs
    nc%field_unit(nc%NDIM+181) = 'y^-1             ' ! dW_dy_linear_yz_cs
    nc%field_unit(nc%NDIM+182) = 'm / y            ' ! dU_dzeta_linear_yz_cs
    nc%field_unit(nc%NDIM+183) = 'm / y            ' ! dV_dzeta_linear_yz_cs
    nc%field_unit(nc%NDIM+184) = 'm / y            ' ! dW_dzeta_linear_yz_cs
    nc%field_unit(nc%NDIM+185) = 'y^-1             ' ! dzeta_dt_linear_yz_cs
    nc%field_unit(nc%NDIM+186) = 'm^-1             ' ! dzeta_dx_linear_yz_cs
    nc%field_unit(nc%NDIM+187) = 'm^-1             ' ! dzeta_dy_linear_yz_cs
    nc%field_unit(nc%NDIM+188) = 'm^-2             ' ! dzeta_dxx_linear_yz_cs
    nc%field_unit(nc%NDIM+189) = 'm^-2             ' ! dzeta_dxy_linear_yz_cs
    nc%field_unit(nc%NDIM+190) = 'm^-2             ' ! dzeta_dyy_linear_yz_cs
    nc%field_unit(nc%NDIM+191) = 'Pa y             ' ! Eta_linear_yz_cs
    nc%field_unit(nc%NDIM+192) = 'Pa y m^-1        ' ! dEta_dx_linear_yz_cs
    nc%field_unit(nc%NDIM+193) = 'Pa y m^-1        ' ! dEta_dy_linear_yz_cs
    nc%field_unit(nc%NDIM+194) = 'Pa y             ' ! dEta_dzeta_linear_yz_cs
    nc%field_unit(nc%NDIM+195) = 'm^-1             ' ! Eta_x_linear_yz_cs
    nc%field_unit(nc%NDIM+196) = 'm^-1             ' ! Eta_y_linear_yz_cs
    nc%field_unit(nc%NDIM+197) = 'm^-1             ' ! Eta_zeta_linear_yz_cs
    nc%field_unit(nc%NDIM+198) = '?                ' ! rhs_eq1_linear_yz_cs
    nc%field_unit(nc%NDIM+199) = '?                ' ! rhs_eq2_linear_yz_cs
    nc%field_unit(nc%NDIM+200) = '?                ' ! rhs_eq1_calving_bc_linear_yz_cs

    nc%field_longname(          1) = 'horizontal x-axis (meter)                                       ' ! x
    nc%field_longname(          2) = 'horizontal y-axis (meter)                                       ' ! y
    nc%field_longname(          3) = 'scaled vertical zeta-axis (unitless)                            ' ! zeta
    nc%field_longname(          4) = 'linear vertical z-axis in ice (meter)                           ' ! z
    nc%field_longname(nc%NDIM    ) = 'Time (year)                                                     ' ! time
    nc%field_longname(nc%NDIM+  1) = 'Surface Mass Balance (meter ice equivalent / year)              ' ! smb
    nc%field_longname(nc%NDIM+  2) = 'Basal Mass Balance (meter ice equivalent / year)                ' ! bmb
    nc%field_longname(nc%NDIM+  3) = 'Applied Mass Balance (meter ice equivalent / year)              ' ! MB_applied
    nc%field_longname(nc%NDIM+  4) = 'forcing concentration (patricles / m^2)                         ' ! forcing_concentration
    nc%field_longname(nc%NDIM+  5) = 'A_flow_mean, the depth average flow parameter (Pa^-n y^-1)      ' ! A_flow_mean
    nc%field_longname(nc%NDIM+  6) = 'D_2D (meter / year)                                             ' ! D_2D
    nc%field_longname(nc%NDIM+  7) = 'number of D_uv_3D interventions (per horzontal grid box)        ' ! number_of_D_uv_3D_interventions
    nc%field_longname(nc%NDIM+  8) = 'q_bottom (J m^-2 y^-1)                                          ' ! q_bottom
    nc%field_longname(nc%NDIM+  9) = 'q_rock_to_ice (J m^-2 y^-1)                                     ' ! q_rock_to_ice
    nc%field_longname(nc%NDIM+ 10) = 'q_friction (J m^-2 y^-1)                                        ' ! q_friction
    nc%field_longname(nc%NDIM+ 11) = 'water_level (meter)                                             ' ! water_level
    nc%field_longname(nc%NDIM+ 12) = 'sliding coverage (unitless), 1 if sliding, 0 else               ' ! sliding_factor
    nc%field_longname(nc%NDIM+ 13) = 'mask, the continental type (unitless)                           ' ! mask
    nc%field_longname(nc%NDIM+ 14) = 'mask changes (unitless), the changes of the continental type    ' ! mask_changes
    nc%field_longname(nc%NDIM+ 15) = 'calving mask (unitless), the extended continental type          ' ! calve_mask
    nc%field_longname(nc%NDIM+ 16) = 'hi_x_mask (unitless)                                            ' ! hi_x_mask
    nc%field_longname(nc%NDIM+ 17) = 'mask for discretizations in the x-direction (unitless)          ' ! x_discretization_mask
    nc%field_longname(nc%NDIM+ 18) = 'mask for discretizations in the y-direction (unitless)          ' ! y_discretization_mask
    nc%field_longname(nc%NDIM+ 19) = 'bottom level of the ice (meter)                                 ' ! bottom_level
    nc%field_longname(nc%NDIM+ 20) = 'db_dx   (m / m  )                                               ' ! db_dx
    nc%field_longname(nc%NDIM+ 21) = 'db_dy   (m / m  )                                               ' ! db_dy
    nc%field_longname(nc%NDIM+ 22) = 'dHi_dxy (m / m^2)                                               ' ! dHi_dxy
    nc%field_longname(nc%NDIM+ 23) = 'dHs_dxy (m / m^2)                                               ' ! dHs_dxy
    nc%field_longname(nc%NDIM+ 24) = 'dHi_dx  (m / m  )                                               ' ! dHi_dx
    nc%field_longname(nc%NDIM+ 25) = 'dHi_dy  (m / m  )                                               ' ! dHi_dy
    nc%field_longname(nc%NDIM+ 26) = 'dHi_dxx (m / m^2)                                               ' ! dHi_dxx
    nc%field_longname(nc%NDIM+ 27) = 'dHi_dyy (m / m^2)                                               ' ! dHi_dyy
    nc%field_longname(nc%NDIM+ 28) = 'dHs_dx  (m / m  )                                               ' ! dHs_dx
    nc%field_longname(nc%NDIM+ 29) = 'dHs_dy  (m / m  )                                               ' ! dHs_dy
    nc%field_longname(nc%NDIM+ 30) = 'dHs_dxx (m / m^2)                                               ' ! dHs_dxx
    nc%field_longname(nc%NDIM+ 31) = 'dHs_dyy (m / m^2)                                               ' ! dHs_dyy
    nc%field_longname(nc%NDIM+ 32) = 'dzeta_dz (m^-1)                                                 ' ! dzeta_dz
    nc%field_longname(nc%NDIM+ 33) = 'dzeta_dxz (m^-2)                                                ' ! dzeta_dxz
    nc%field_longname(nc%NDIM+ 34) = 'dzeta_dyz (m^-2)                                                ' ! dzeta_dyz
    nc%field_longname(nc%NDIM+ 35) = 'Ice Thickness (meter)                                           ' ! Hi
    nc%field_longname(nc%NDIM+ 36) = 'Height of Bedrock (meter) - relative to presentday sea level    ' ! Hb
    nc%field_longname(nc%NDIM+ 37) = 'Surface Height (meter) - relative to presentday sea level       ' ! Hs
    nc%field_longname(nc%NDIM+ 38) = 'Time derivative of Ice Thickness (meter / year)                 ' ! dHi_dt
    nc%field_longname(nc%NDIM+ 39) = 'Time derivative of bedrock level (meter / year)                 ' ! dHb_dt
    nc%field_longname(nc%NDIM+ 40) = 'Time derivative of surface level (meter / year)                 ' ! dHs_dt
    nc%field_longname(nc%NDIM+ 41) = 'Depth averaged velocity in x-direction (meter / year)           ' ! Us
    nc%field_longname(nc%NDIM+ 42) = 'Depth averaged velocity in y-direction (meter / year)           ' ! Vs
    nc%field_longname(nc%NDIM+ 43) = 'Points at which the flux routine intervented (unitless)         ' ! intervention_cover
    nc%field_longname(nc%NDIM+ 44) = 'Amount of vertical points at pressure melting point (unitless)  ' ! coverage_ti_pmp
    nc%field_longname(nc%NDIM+ 45) = 'Amount of vertical points adjusted for too low values (unitless)' ! cover_adjusted_temperatures
    nc%field_longname(nc%NDIM+ 46) = 'longitude (degree)                                              ' ! longitude
    nc%field_longname(nc%NDIM+ 47) = 'latitude (degree)                                               ' ! latitude
    nc%field_longname(nc%NDIM+ 48) = 'hi_y_mask (unitless)                                            ' ! hi_y_mask
    nc%field_longname(nc%NDIM+ 49) = 'critical_time_step_pattern (year)                               ' ! critical_time_step_pattern
    nc%field_longname(nc%NDIM+ 50) = 'bottom_melt (?)                                                 ' ! bottom_melt
    nc%field_longname(nc%NDIM+ 51) = 'sor_residue_x (m / y)                                           ' ! sor_residue_x
    nc%field_longname(nc%NDIM+ 52) = 'sor_residue_y (m / y)                                           ' ! sor_residue_y
   !nc%field_longname(nc%NDIM+ 51) = 'dummy_2_2D (?)                                                  ' ! dummy_2_2D
   !nc%field_longname(nc%NDIM+ 52) = 'dummy_1_2D (?)                                                  ' ! dummy_1_2D
    nc%field_longname(nc%NDIM+ 53) = 'U, the 3D velocity in the x-direction (meter / year)            ' ! U
    nc%field_longname(nc%NDIM+ 54) = 'V, the 3D velocity in the y-direction (meter / year)            ' ! V
    nc%field_longname(nc%NDIM+ 55) = 'W, the 3D velocity in the zeta-direction (meter / year)         ' ! W
    nc%field_longname(nc%NDIM+ 56) = 'Ice Temperature (Kelvin)                                        ' ! Ti
    nc%field_longname(nc%NDIM+ 57) = 'Ti_pmp, pressure melting point temperature (Kelvin)             ' ! Ti_pmp
    nc%field_longname(nc%NDIM+ 58) = 'A_flow, the flow parameter (Pa^-n y^-1)                         ' ! A_flow
    nc%field_longname(nc%NDIM+ 59) = 'Cpi, the specific heat capacity of ice (J kg^-1 K^-1)           ' ! Cpi
    nc%field_longname(nc%NDIM+ 60) = 'Ki, the conductivity of ice (J m^-1 K^-1 y^-1)                  ' ! Ki
    nc%field_longname(nc%NDIM+ 61) = 'D_uv_3D (meter / year)                                          ' ! D_uv_3D
    nc%field_longname(nc%NDIM+ 62) = 'Concentration (m^-3)                                            ' ! concentration
    nc%field_longname(nc%NDIM+ 63) = 'dTi_dx (K / m)                                                  ' ! dTi_dx
    nc%field_longname(nc%NDIM+ 64) = 'dTi_dy (K / m)                                                  ' ! dTi_dy
    nc%field_longname(nc%NDIM+ 65) = 'dU_dx (y^-1)                                                    ' ! dU_dx
    nc%field_longname(nc%NDIM+ 66) = 'dV_dx (y^-1)                                                    ' ! dV_dx
    nc%field_longname(nc%NDIM+ 67) = 'dW_dx (y^-1)                                                    ' ! dW_dx
    nc%field_longname(nc%NDIM+ 68) = 'dU_dy (y^-1)                                                    ' ! dU_dy
    nc%field_longname(nc%NDIM+ 69) = 'dV_dy (y^-1)                                                    ' ! dV_dy
    nc%field_longname(nc%NDIM+ 70) = 'dW_dy (y^-1)                                                    ' ! dW_dy
    nc%field_longname(nc%NDIM+ 71) = 'dU_dzeta (y^-1)                                                 ' ! dU_dzeta
    nc%field_longname(nc%NDIM+ 72) = 'dV_dzeta (y^-1)                                                 ' ! dV_dzeta
    nc%field_longname(nc%NDIM+ 73) = 'dW_dzeta (y^-1)                                                 ' ! dW_dzeta
    nc%field_longname(nc%NDIM+ 74) = 'dzeta%t (y^-1)                                                  ' ! dzeta_dt
    nc%field_longname(nc%NDIM+ 75) = 'dzeta%x (m^-1)                                                  ' ! dzeta_dx
    nc%field_longname(nc%NDIM+ 76) = 'dzeta%y (m^-1)                                                  ' ! dzeta_dy
    nc%field_longname(nc%NDIM+ 77) = 'dzeta%xx (m^-2)                                                 ' ! dzeta_dxx
    nc%field_longname(nc%NDIM+ 78) = 'dzeta%xy (m^-2)                                                 ' ! dzeta_dxy
    nc%field_longname(nc%NDIM+ 79) = 'dzeta%yy (m^-2)                                                 ' ! dzeta_dyy
    nc%field_longname(nc%NDIM+ 80) = 'Eta (Pa y)                                                      ' ! Eta
    nc%field_longname(nc%NDIM+ 81) = 'dEta_dx (Pa y m^-1)                                             ' ! dEta_dx
    nc%field_longname(nc%NDIM+ 82) = 'dEta_dy (Pa y m^-1)                                             ' ! dEta_dy
    nc%field_longname(nc%NDIM+ 83) = 'dEta_dzeta (Pa y)                                               ' ! dEta_dzeta
    nc%field_longname(nc%NDIM+ 84) = 'Eta_x (m^-1)                                                    ' ! Eta_x
    nc%field_longname(nc%NDIM+ 85) = 'Eta_y (m^-1)                                                    ' ! Eta_y
    nc%field_longname(nc%NDIM+ 86) = 'Eta_zeta (m^-1)                                                 ' ! Eta_zeta
    nc%field_longname(nc%NDIM+ 87) = 'rhs_eq1 ( )                                                     ' ! rhs_eq1
    nc%field_longname(nc%NDIM+ 88) = 'rhs_eq2 ( )                                                     ' ! rhs_eq2
    nc%field_longname(nc%NDIM+ 89) = 'rhs_eq1_calving_bc ( )                                          ' ! rhs_eq1_calving_bc
    nc%field_longname(nc%NDIM+ 90) = 'U_linear (meter / year)                                         ' ! U_linear
    nc%field_longname(nc%NDIM+ 91) = 'V_linear (meter / year)                                         ' ! V_linear
    nc%field_longname(nc%NDIM+ 92) = 'W_linear (meter / year)                                         ' ! W_linear
    nc%field_longname(nc%NDIM+ 93) = 'Ti_linear (Kelvin)                                              ' ! Ti_linear
    nc%field_longname(nc%NDIM+ 94) = 'Ti_pmp_linear (Kelvin)                                          ' ! Ti_pmp_linear
    nc%field_longname(nc%NDIM+ 95) = 'A_flow_linear (Pa^-n y^-1)                                      ' ! A_flow_linear
    nc%field_longname(nc%NDIM+ 96) = 'Cpi_linear (J kg^-1 K^-1)                                       ' ! Cpi_linear
    nc%field_longname(nc%NDIM+ 97) = 'Ki_linear (J m^-1 K^-1 y^-1)                                    ' ! Ki_linear
    nc%field_longname(nc%NDIM+ 98) = 'D_uv_3D_linear (meter / year)                                   ' ! D_uv_3D_linear
    nc%field_longname(nc%NDIM+ 99) = 'concentration_linear (m^-3)                                     ' ! concentration_linear
    nc%field_longname(nc%NDIM+100) = 'dTi_dx_linear (K / m)                                           ' ! dTi_dx_linear
    nc%field_longname(nc%NDIM+101) = 'dTi_dy_linear (K / m)                                           ' ! dTi_dy_linear
    nc%field_longname(nc%NDIM+102) = 'dU_dx_linear (y^-1)                                             ' ! dU_dx_linear
    nc%field_longname(nc%NDIM+103) = 'dV_dx_linear (y^-1)                                             ' ! dV_dx_linear
    nc%field_longname(nc%NDIM+104) = 'dW_dx_linear (y^-1)                                             ' ! dW_dx_linear
    nc%field_longname(nc%NDIM+105) = 'dU_dy_linear (y^-1)                                             ' ! dU_dy_linear
    nc%field_longname(nc%NDIM+106) = 'dV_dy_linear (y^-1)                                             ' ! dV_dy_linear
    nc%field_longname(nc%NDIM+107) = 'dW_dy_linear (y^-1)                                             ' ! dW_dy_linear
    nc%field_longname(nc%NDIM+108) = 'dU_dzeta_linear (meter / year)                                  ' ! dU_dzeta_linear
    nc%field_longname(nc%NDIM+109) = 'dV_dzeta_linear (meter / year)                                  ' ! dV_dzeta_linear
    nc%field_longname(nc%NDIM+110) = 'dW_dzeta_linear (meter / year)                                  ' ! dW_dzeta_linear
    nc%field_longname(nc%NDIM+111) = 'dzeta_dt_linear (y^-1)                                          ' ! dzeta_dt_linear
    nc%field_longname(nc%NDIM+112) = 'dzeta_dx_linear (m^-1)                                          ' ! dzeta_dx_linear
    nc%field_longname(nc%NDIM+113) = 'dzeta_dy_linear (m^-1)                                          ' ! dzeta_dy_linear
    nc%field_longname(nc%NDIM+114) = 'dzeta_dxx_linear (m^-2)                                         ' ! dzeta_dxx_linear
    nc%field_longname(nc%NDIM+115) = 'dzeta_dxy_linear (m^-2)                                         ' ! dzeta_dxy_linear
    nc%field_longname(nc%NDIM+116) = 'dzeta_dyy_linear (m^-2)                                         ' ! dzeta_dyy_linear
    nc%field_longname(nc%NDIM+117) = 'Eta_linear (Pa y)                                               ' ! Eta_linear
    nc%field_longname(nc%NDIM+118) = 'dEta_dx_linear (Pa y m^-1)                                      ' ! dEta_dx_linear
    nc%field_longname(nc%NDIM+119) = 'dEta_dy_linear (Pa y m^-1)                                      ' ! dEta_dy_linear
    nc%field_longname(nc%NDIM+120) = 'dEta_dzeta_linear (Pa y)                                        ' ! dEta_dzeta_linear
    nc%field_longname(nc%NDIM+121) = 'Eta_x_linear (m^-1)                                             ' ! Eta_x_linear
    nc%field_longname(nc%NDIM+122) = 'Eta_y_linear (m^-1)                                             ' ! Eta_y_linear
    nc%field_longname(nc%NDIM+123) = 'Eta_zeta_linear (m^-1)                                          ' ! Eta_zeta_linear
    nc%field_longname(nc%NDIM+124) = 'rhs_eq1_linear ( )                                              ' ! rhs_eq1_linear
    nc%field_longname(nc%NDIM+125) = 'rhs_eq2_linear ( )                                              ' ! rhs_eq2_linear
    nc%field_longname(nc%NDIM+126) = 'rhs_eq1_calving_bc_linear ( )                                   ' ! rhs_eq1_calving_bc_linear
    nc%field_longname(nc%NDIM+127) = 'U_linear (meter / year), cross section of xz-plane              ' ! U_linear_xz_cs
    nc%field_longname(nc%NDIM+128) = 'V_linear (meter / year), cross section of xz-plane              ' ! V_linear_xz_cs
    nc%field_longname(nc%NDIM+129) = 'W_linear (meter / year), cross section of xz-plane              ' ! W_linear_xz_cs
    nc%field_longname(nc%NDIM+130) = 'Ti_linear (Kelvin), cross section of xz-plane                   ' ! Ti_linear_xz_cs
    nc%field_longname(nc%NDIM+131) = 'Ti_pmp_linear (Kelvin), cross section of xz-plane               ' ! Ti_pmp_linear_xz_cs
    nc%field_longname(nc%NDIM+132) = 'A_flow_linear (Pa^-n y^-1), cross section of xz-plane           ' ! A_flow_linear_xz_cs
    nc%field_longname(nc%NDIM+133) = 'Cpi_linear (J kg^-1 K^-1), cross section of xz-plane            ' ! Cpi_linear_xz_cs
    nc%field_longname(nc%NDIM+134) = 'Ki_linear (J m^-1 K^-1 y^-1), cross section of xz-plane         ' ! Ki_linear_xz_cs
    nc%field_longname(nc%NDIM+135) = 'D_uv_3D_linear (m y^-1), cross section of xz-plane              ' ! D_uv_3D_linear_xz_cs
    nc%field_longname(nc%NDIM+136) = 'concentration_linear (m^-3), cross section of xz-plane          ' ! concentration_linear_xz_cs
    nc%field_longname(nc%NDIM+137) = 'dTi_dx_linear (K / m), cross section of xz-plane                ' ! dTi_dx_linear_xz_cs
    nc%field_longname(nc%NDIM+138) = 'dTi_dy_linear (K / m), cross section of xz-plane                ' ! dTi_dy_linear_xz_cs
    nc%field_longname(nc%NDIM+139) = 'dU_dx_linear (y^-1), cross section of xz-plane                  ' ! dU_dx_linear_xz_cs
    nc%field_longname(nc%NDIM+140) = 'dV_dx_linear (y^-1), cross section of xz-plane                  ' ! dV_dx_linear_xz_cs
    nc%field_longname(nc%NDIM+141) = 'dW_dx_linear (y^-1), cross section of xz-plane                  ' ! dW_dx_linear_xz_cs
    nc%field_longname(nc%NDIM+142) = 'dU_dy_linear (y^-1), cross section of xz-plane                  ' ! dU_dy_linear_xz_cs
    nc%field_longname(nc%NDIM+143) = 'dV_dy_linear (y^-1), cross section of xz-plane                  ' ! dV_dy_linear_xz_cs
    nc%field_longname(nc%NDIM+144) = 'dW_dy_linear (y^-1), cross section of xz-plane                  ' ! dW_dy_linear_xz_cs
    nc%field_longname(nc%NDIM+145) = 'dU_dzeta_linear (meter / year), cross section of xz-plane       ' ! dU_dzeta_linear_xz_cs
    nc%field_longname(nc%NDIM+146) = 'dV_dzeta_linear (meter / year), cross section of xz-plane       ' ! dV_dzeta_linear_xz_cs
    nc%field_longname(nc%NDIM+147) = 'dW_dzeta_linear (meter / year), cross section of xz-plane       ' ! dW_dzeta_linear_xz_cs
    nc%field_longname(nc%NDIM+148) = 'dzeta_dt_linear (y^-1), cross section of xz-plane               ' ! dzeta_dt_linear_xz_cs
    nc%field_longname(nc%NDIM+149) = 'dzeta_dx_linear (m^-1), cross section of xz-plane               ' ! dzeta_dx_linear_xz_cs
    nc%field_longname(nc%NDIM+150) = 'dzeta_dy_linear (m^-1), cross section of xz-plane               ' ! dzeta_dy_linear_xz_cs
    nc%field_longname(nc%NDIM+151) = 'dzeta_dxx_linear (m^-2), cross section of xz-plane              ' ! dzeta_dxx_linear_xz_cs
    nc%field_longname(nc%NDIM+152) = 'dzeta_dxy_linear (m^-2), cross section of xz-plane              ' ! dzeta_dxy_linear_xz_cs
    nc%field_longname(nc%NDIM+153) = 'dzeta_dyy_linear (m^-2), cross section of xz-plane              ' ! dzeta_dyy_linear_xz_cs
    nc%field_longname(nc%NDIM+154) = 'Eta_linear (Pa y), cross section of xz-plane                    ' ! Eta_linear_xz_cs
    nc%field_longname(nc%NDIM+155) = 'dEta_dx_linear (Pa y m^-1), cross section of xz-plane           ' ! dEta_dx_linear_xz_cs
    nc%field_longname(nc%NDIM+156) = 'dEta_dy_linear (Pa y m^-1), cross section of xz-plane           ' ! dEta_dy_linear_xz_cs
    nc%field_longname(nc%NDIM+157) = 'dEta_dzeta_linear (Pa y), cross section of xz-plane             ' ! dEta_dzeta_linear_xz_cs
    nc%field_longname(nc%NDIM+158) = 'Eta_x_linear (m^-1), cross section of xz-plane                  ' ! Eta_x_linear_xz_cs
    nc%field_longname(nc%NDIM+159) = 'Eta_y_linear (m^-1), cross section of xz-plane                  ' ! Eta_y_linear_xz_cs
    nc%field_longname(nc%NDIM+160) = 'Eta_zeta_linear (m^-1), cross section of xz-plane               ' ! Eta_zeta_linear_xz_cs
    nc%field_longname(nc%NDIM+161) = 'rhs_eq1_linear ( ), cross section of xz-plane                   ' ! rhs_eq1_linear_xz_cs
    nc%field_longname(nc%NDIM+162) = 'rhs_eq2_linear ( ), cross section of xz-plane                   ' ! rhs_eq2_linear_xz_cs
    nc%field_longname(nc%NDIM+163) = 'rhs_eq1_calving_bc_linear ( ), cross section of xz-plane        ' ! rhs_eq1_calving_bc_linear_xz_cs
    nc%field_longname(nc%NDIM+164) = 'U_linear (meter / year), cross section of yz-plane              ' ! U_linear_yz_cs
    nc%field_longname(nc%NDIM+165) = 'V_linear (meter / year), cross section of yz-plane              ' ! V_linear_yz_cs
    nc%field_longname(nc%NDIM+166) = 'W_linear (meter / year), cross section of yz-plane              ' ! W_linear_yz_cs
    nc%field_longname(nc%NDIM+167) = 'Ti_linear (Kelvin), cross section of yz-plane                   ' ! Ti_linear_yz_cs
    nc%field_longname(nc%NDIM+168) = 'Ti_pmp_linear (Kelvin), cross section of yz-plane               ' ! Ti_pmp_linear_yz_cs
    nc%field_longname(nc%NDIM+169) = 'A_flow_linear (Pa^-n y^-1), cross section of yz-plane           ' ! A_flow_linear_yz_cs
    nc%field_longname(nc%NDIM+170) = 'Cpi_linear (J kg^-1 K^-1), cross section of yz-plane            ' ! Cpi_linear_yz_cs
    nc%field_longname(nc%NDIM+171) = 'Ki_linear (J m^-1 K^-1 y^-1), cross section of yz-plane         ' ! Ki_linear_yz_cs
    nc%field_longname(nc%NDIM+172) = 'D_uv_3D_linear (m y^-1), cross section of yz-plane              ' ! D_uv_3D_linear_yz_cs
    nc%field_longname(nc%NDIM+173) = 'concentration_linear (m^-3), cross section of yz-plane          ' ! concentration_linear_yz_cs
    nc%field_longname(nc%NDIM+174) = 'dTi_dx_linear (K / m), cross section of yz-plane                ' ! dTi_dx_linear_yz_cs
    nc%field_longname(nc%NDIM+175) = 'dTi_dy_linear (K / m), cross section of yz-plane                ' ! dTi_dy_linear_yz_cs
    nc%field_longname(nc%NDIM+176) = 'dU_dx_linear (y^-1), cross section of yz-plane                  ' ! dU_dx_linear_yz_cs
    nc%field_longname(nc%NDIM+177) = 'dV_dx_linear (y^-1), cross section of yz-plane                  ' ! dV_dx_linear_yz_cs
    nc%field_longname(nc%NDIM+178) = 'dW_dx_linear (y^-1), cross section of yz-plane                  ' ! dW_dx_linear_yz_cs
    nc%field_longname(nc%NDIM+179) = 'dU_dy_linear (y^-1), cross section of yz-plane                  ' ! dU_dy_linear_yz_cs
    nc%field_longname(nc%NDIM+180) = 'dV_dy_linear (y^-1), cross section of yz-plane                  ' ! dV_dy_linear_yz_cs
    nc%field_longname(nc%NDIM+181) = 'dW_dy_linear (y^-1), cross section of yz-plane                  ' ! dW_dy_linear_yz_cs
    nc%field_longname(nc%NDIM+182) = 'dU_dzeta_linear (meter / year), cross section of yz-plane       ' ! dU_dzeta_linear_yz_cs
    nc%field_longname(nc%NDIM+183) = 'dV_dzeta_linear (meter / year), cross section of yz-plane       ' ! dV_dzeta_linear_yz_cs
    nc%field_longname(nc%NDIM+184) = 'dW_dzeta_linear (meter / year), cross section of yz-plane       ' ! dW_dzeta_linear_yz_cs
    nc%field_longname(nc%NDIM+185) = 'dzeta_dt_linear (y^-1), cross section of yz-plane               ' ! dzeta_dt_linear_yz_cs
    nc%field_longname(nc%NDIM+186) = 'dzeta_dx_linear (m^-1), cross section of yz-plane               ' ! dzeta_dx_linear_yz_cs
    nc%field_longname(nc%NDIM+187) = 'dzeta_dy_linear (m^-1), cross section of yz-plane               ' ! dzeta_dy_linear_yz_cs
    nc%field_longname(nc%NDIM+188) = 'dzeta_dxx_linear (m^-2), cross section of yz-plane              ' ! dzeta_dxx_linear_yz_cs
    nc%field_longname(nc%NDIM+189) = 'dzeta_dxy_linear (m^-2), cross section of yz-plane              ' ! dzeta_dxy_linear_yz_cs
    nc%field_longname(nc%NDIM+190) = 'dzeta_dyy_linear (m^-2), cross section of yz-plane              ' ! dzeta_dyy_linear_yz_cs
    nc%field_longname(nc%NDIM+191) = 'Eta_linear (Pa y), cross section of yz-plane                    ' ! Eta_linear_yz_cs
    nc%field_longname(nc%NDIM+192) = 'dEta_dx_linear (Pa y m^-1), cross section of yz-plane           ' ! dEta_dx_linear_yz_cs
    nc%field_longname(nc%NDIM+193) = 'dEta_dy_linear (Pa y m^-1), cross section of yz-plane           ' ! dEta_dy_linear_yz_cs
    nc%field_longname(nc%NDIM+194) = 'dEta_dzeta_linear (Pa y), cross section of yz-plane             ' ! dEta_dzeta_linear_yz_cs
    nc%field_longname(nc%NDIM+195) = 'Eta_x_linear (m^-1), cross section of yz-plane                  ' ! Eta_x_linear_yz_cs
    nc%field_longname(nc%NDIM+196) = 'Eta_y_linear (m^-1), cross section of yz-plane                  ' ! Eta_y_linear_yz_cs
    nc%field_longname(nc%NDIM+197) = 'Eta_zeta_linear (m^-1), cross section of yz-plane               ' ! Eta_zeta_linear_yz_cs
    nc%field_longname(nc%NDIM+198) = 'rhs_eq1_linear ( ), cross section of yz-plane                   ' ! rhs_eq1_linear_yz_cs
    nc%field_longname(nc%NDIM+199) = 'rhs_eq2_linear ( ), cross section of yz-plane                   ' ! rhs_eq2_linear_yz_cs
    nc%field_longname(nc%NDIM+200) = 'rhs_eq1_calving_bc_linear ( ), cross section of yz-plane        ' ! rhs_eq1_calving_bc_linear_yz_cs

    nc%type(          1:nc%NVAR    )  = nf90_double

    nc%case(                      1)  =    1 ! The dimension: X
    nc%case(                      2)  =    2 ! The dimension: Y
    nc%case(                      3)  =    3 ! The dimension: zeta
    nc%case(                      4)  =    5 ! The dimension: z
    nc%case(nc%NDIM                )  =    6 ! The dimension: time
    nc%case(nc%NDIM+  1:nc%NDIM+ 52)  =    7 ! The NX, NY,       time case
    nc%case(nc%NDIM+ 53:nc%NDIM+ 89)  =    8 ! The NX, NY , NZ,  time case 
    nc%case(nc%NDIM+ 90:nc%NDIM+126)  =   10 ! The NX, NY , NZL, time case 
    nc%case(nc%NDIM+127:nc%NDIM+163)  =   11 ! The NX, NZL,      time case
    nc%case(nc%NDIM+164:nc%NDIM+200)  =   12 ! The NY, NZL,      time case

    ! Ignore mode:
    DO i = 1, C%N_ignore - 1, 2
      nc%case(MAX(C%ignore_some_help_fields(i) + nc%NDIM, nc%NDIM+1):MIN(C%ignore_some_help_fields(i+1) + nc%NDIM, nc%NVAR)) = 1000  
    END DO
    nc%case(MIN(C%which_help_fields + nc%NDIM + 1, nc%NVAR + 1):nc%NVAR) = 1000

    nc%LEN_DIM(1      )               = C%NX
    nc%LEN_DIM(2      )               = C%NY
    nc%LEN_DIM(3      )               = C%NZ
    nc%LEN_DIM(4      )               = C%NZL
    nc%LEN_DIM(nc%NDIM)               = nf90_unlimited

    nc%grid_size(1      )             = C%dx
    nc%grid_size(2      )             = C%dy
   !nc%grid_size(3      )             =          ! non equidistant
    nc%grid_size(4      )             = C%z_step
   !nc%grid_size(nc%NDIM)             = C%dt     ! but the time step can be dynamical: see G%dt
  END SUBROUTINE initialize_help_fields_file



  SUBROUTINE write_help_fields(nc                              , &
                               record_number                   , &
                               t                               , &
                               smb                             , &
                               bmb                             , &
                               MB_applied                      , &
                               forcing_concentration           , &
                               A_flow_mean                     , &
                               D_2D                            , &
                               number_of_D_uv_3D_interventions , &
                               q_bottom                        , &
                               q_rock_to_ice                   , &
                               q_friction                      , &
                               water_level                     , &
                               sliding_factor                  , &
                               mask                            , &
                               mask_changes                    , &
                               calve_mask                      , &
                               hi_x_mask                       , &
                               x_discretization_mask           , &
                               y_discretization_mask           , &
                               bottom_level                    , &
                               db_dx                           , &
                               db_dy                           , &
                               dHi_dxy                         , &
                               dHs_dxy                         , &
                               dHi_dx                          , &
                               dHi_dy                          , &
                               dHi_dxx                         , &
                               dHi_dyy                         , &
                               dHs_dx                          , &
                               dHs_dy                          , &
                               dHs_dxx                         , &
                               dHs_dyy                         , &
                               dzeta_dz                        , &
                               dzeta_dxz                       , &
                               dzeta_dyz                       , &
                               Hi                              , &
                               Hb                              , &
                               Hs                              , &
                               dHi_dt                          , &
                               dHb_dt                          , &
                               dHs_dt                          , &
                               Us                              , &
                               Vs                              , &
                               intervention_cover              , &
                               coverage_ti_pmp                 , &
                               cover_adjusted_temperatures     , &
                               longitude                       , &
                               latitude                        , &
                               hi_y_mask                       , &
                               critical_time_step_pattern      , &
                               bottom_melt                     , &
                               sor_residue_x                   , &
                               sor_residue_y                   , &
                              !dummy_2_2D                      , &
                              !dummy_1_2D                      , &
                               U                               , &
                               V                               , &
                               W                               , &
                               Ti                              , &
                               Ti_pmp                          , &
                               A_flow                          , &
                               Cpi                             , &
                               Ki                              , &
                               D_uv_3D                         , &
                               concentration                   , &
                               dTi_dx                          , &
                               dTi_dy                          , &
                               dU_dx                           , &
                               dV_dx                           , &
                               dW_dx                           , &
                               dU_dy                           , &
                               dV_dy                           , &
                               dW_dy                           , &
                               dU_dzeta                        , &
                               dV_dzeta                        , &
                               dW_dzeta                        , &
                               dzeta_dt                        , &
                               dzeta_dx                        , &
                               dzeta_dy                        , &
                               dzeta_dxx                       , &
                               dzeta_dxy                       , &
                               dzeta_dyy                       , &
                               Eta                             , &
                               dEta_dx                         , &
                               dEta_dy                         , &
                               dEta_dzeta                      , &
                               Eta_x                           , &
                               Eta_y                           , &
                               Eta_zeta                        , &
                               rhs_eq1                         , &
                               rhs_eq2                         , &
                               rhs_eq1_calving_bc              )  
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type, handle_error
    USE netcdf, ONLY: nf90_put_var, nf90_sync
    IMPLICIT NONE

    ! Input variables:
    TYPE(netcdf_file_type),                                         INTENT(IN)           :: nc
    INTEGER,                                                        INTENT(IN)           :: record_number
    REAL(dp),                                                       INTENT(IN)           :: t
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: smb                     
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: bmb                       
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: MB_applied                     
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: forcing_concentration          
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: A_flow_mean                    
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: D_2D                           
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: number_of_D_uv_3D_interventions
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: q_bottom                       
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: q_rock_to_ice                  
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: q_friction                     
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: water_level                    
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: sliding_factor               
    INTEGER,  DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: mask                           
    INTEGER,  DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: mask_changes                   
    INTEGER,  DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: calve_mask                     
    INTEGER,  DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: hi_x_mask                      
    INTEGER,  DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: x_discretization_mask          
    INTEGER,  DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: y_discretization_mask          
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: bottom_level                   
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: db_dx                          
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: db_dy                          
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHi_dxy                        
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHs_dxy                        
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHi_dx                         
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHi_dy                         
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHi_dxx                        
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHi_dyy                        
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHs_dx                         
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHs_dy                         
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHs_dxx                        
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHs_dyy                        
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dz                       
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dxz
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dyz
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Hi                                ! necesarry in cross section calculation
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Hb                                ! necesarry in cross section calculation                      
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Hs                                ! necesarry in cross section calculation                      
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHi_dt                         
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHb_dt                         
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dHs_dt                         
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Us                             
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Vs                             
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: intervention_cover
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: coverage_ti_pmp
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: cover_adjusted_temperatures
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: longitude
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: latitude
    INTEGER,  DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: hi_y_mask
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: critical_time_step_pattern
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: bottom_melt
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: sor_residue_x
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: sor_residue_y
   !REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dummy_2_2D
   !REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dummy_1_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: U                              
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: V                              
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: W                              
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Ti                             
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Ti_pmp                         
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: A_flow                         
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Cpi                            
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Ki                             
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: D_uv_3D                        
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: concentration                  
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dTi_dx                         
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dTi_dy                         
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dU_dx                          
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dV_dx                          
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dW_dx                          
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dU_dy                          
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dV_dy                          
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dW_dy                          
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dU_dzeta                       
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dV_dzeta                       
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dW_dzeta                       
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dt                       
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dx                       
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dy                       
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dxx                      
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dxy                      
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dzeta_dyy                      
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Eta                            
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dEta_dx                        
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dEta_dy                        
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: dEta_dzeta                     
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Eta_x                          
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Eta_y                          
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: Eta_zeta                       
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: rhs_eq1                        
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: rhs_eq2                        
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: rhs_eq1_calving_bc             

    ! Local variables:
    INTEGER                                                                                :: m, n            ! Grid point counters
    INTEGER                                                                                :: i               ! Counter over the dimensions or the variables
    INTEGER                                                                                :: iL              ! Counter over the i-Loop
    REAL(dp)                                                                               :: time            ! The time or the relative time since the start of the run
    REAL(dp), DIMENSION(nc%NDIM+  1:nc%NDIM+ 52,nc%LEN_DIM(1),nc%LEN_DIM(2)              ) :: var_data_2D_a   ! The 2D transposed NX * NY       arrays
    REAL(dp), DIMENSION(nc%NDIM+ 53:nc%NDIM+ 89,nc%LEN_DIM(1),nc%LEN_DIM(2),nc%LEN_DIM(3)) :: var_data_3D_a   ! The 3D transposed NX * NY * NZ  arrays
    REAL(dp), DIMENSION(nc%NDIM+ 90:nc%NDIM+126,nc%LEN_DIM(1),nc%LEN_DIM(2),nc%LEN_DIM(4)) :: var_data_3D_b   ! The 3D transposed NX * NY * NZL arrays
    REAL(dp), DIMENSION(nc%NDIM+127:nc%NVAR+163,nc%LEN_DIM(1),nc%LEN_DIM(4)              ) :: var_data_2D_b   ! The 2D transposed NX * NZL      arrays
    REAL(dp), DIMENSION(nc%NDIM+164:nc%NDIM+200,nc%LEN_DIM(2),nc%LEN_DIM(4)              ) :: var_data_2D_c   ! The 2D transposed NY * NZL      arrays

    REAL(dp), DIMENSION(nc%LEN_DIM(1),nc%LEN_DIM(2))                                       :: Hi_transposed   ! necesarry in cross section calculation
    REAL(dp), DIMENSION(nc%LEN_DIM(1),nc%LEN_DIM(2))                                       :: Hb_transposed   ! necesarry in cross section calculation
    REAL(dp), DIMENSION(nc%LEN_DIM(1),nc%LEN_DIM(2))                                       :: Hs_transposed   ! necesarry in cross section calculation

    REAL(dp), DIMENSION(nc%LEN_DIM(1),nc%LEN_DIM(4)              )                         :: x_cross_section ! The 2D transposed NX * NZL      arrays
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(4)              )                         :: y_cross_section ! The 2D transposed NY * NZL      arrays
    REAL(dp), DIMENSION(nc%LEN_DIM(1),nc%LEN_DIM(2),nc%LEN_DIM(3))                         :: scaled_3D_field ! The 3D transposed NX * NY * NZ  arrays
    REAL(dp), DIMENSION(nc%LEN_DIM(1),nc%LEN_DIM(2),nc%LEN_DIM(4))                         :: linear_3D_field ! The 3D transposed NX * NY * NZL arrays

    ! In case an optional variable is not present, but is written: 
    var_data_2D_a = 0._dp
    IF(C%which_help_fields >=  53) var_data_3D_a = 0._dp
    IF(C%which_help_fields >=  90) var_data_3D_b = 0._dp ! Saves large zero-array assignment, and can prevent memory problems with a too large but not used NZL value
    IF(C%which_help_fields >= 127) var_data_2D_b = 0._dp
    IF(C%which_help_fields >= 164) var_data_2D_c = 0._dp

    ! Transpose the read fields from {*,y,x} to {x,y,*}:
    DO m = 1, nc%LEN_DIM(1)
    DO n = 1, nc%LEN_DIM(2) 
      IF(PRESENT(smb                             )) var_data_2D_a(nc%NDIM+  1,m,n  ) =       smb                             (  n,m)      
      IF(PRESENT(bmb                             )) var_data_2D_a(nc%NDIM+  2,m,n  ) =       bmb                             (  n,m)      
      IF(PRESENT(MB_applied                      )) var_data_2D_a(nc%NDIM+  3,m,n  ) =       MB_applied                      (  n,m)      
      IF(PRESENT(forcing_concentration           )) var_data_2D_a(nc%NDIM+  4,m,n  ) =       forcing_concentration           (  n,m)      
      IF(PRESENT(A_flow_mean                     )) var_data_2D_a(nc%NDIM+  5,m,n  ) =       A_flow_mean                     (  n,m)      
      IF(PRESENT(D_2D                            )) var_data_2D_a(nc%NDIM+  6,m,n  ) =       D_2D                            (  n,m)      
      IF(PRESENT(number_of_D_uv_3D_interventions )) var_data_2D_a(nc%NDIM+  7,m,n  ) =       number_of_D_uv_3D_interventions (  n,m)      
      IF(PRESENT(q_bottom                        )) var_data_2D_a(nc%NDIM+  8,m,n  ) =       q_bottom                        (  n,m)      
      IF(PRESENT(q_rock_to_ice                   )) var_data_2D_a(nc%NDIM+  9,m,n  ) =       q_rock_to_ice                   (  n,m)      
      IF(PRESENT(q_friction                      )) var_data_2D_a(nc%NDIM+ 10,m,n  ) =       q_friction                      (  n,m)      
      IF(PRESENT(water_level                     )) var_data_2D_a(nc%NDIM+ 11,m,n  ) =       water_level                     (  n,m)      
      IF(PRESENT(sliding_factor                  )) var_data_2D_a(nc%NDIM+ 12,m,n  ) =       sliding_factor                  (  n,m)      
      IF(PRESENT(mask                            )) var_data_2D_a(nc%NDIM+ 13,m,n  ) = REAL( mask                            (  n,m) , dp)
      IF(PRESENT(mask_changes                    )) var_data_2D_a(nc%NDIM+ 14,m,n  ) = REAL( mask_changes                    (  n,m) , dp)
      IF(PRESENT(calve_mask                      )) var_data_2D_a(nc%NDIM+ 15,m,n  ) = REAL( calve_mask                      (  n,m) , dp) 
      IF(PRESENT(hi_x_mask                       )) var_data_2D_a(nc%NDIM+ 16,m,n  ) = REAL( hi_x_mask                       (  n,m) , dp) 
      IF(PRESENT(x_discretization_mask           )) var_data_2D_a(nc%NDIM+ 17,m,n  ) = REAL( x_discretization_mask           (  n,m) , dp)
      IF(PRESENT(y_discretization_mask           )) var_data_2D_a(nc%NDIM+ 18,m,n  ) = REAL( y_discretization_mask           (  n,m) , dp)
      IF(PRESENT(bottom_level                    )) var_data_2D_a(nc%NDIM+ 19,m,n  ) =       bottom_level                    (  n,m)
      IF(PRESENT(db_dx                           )) var_data_2D_a(nc%NDIM+ 20,m,n  ) =       db_dx                           (  n,m)      
      IF(PRESENT(db_dy                           )) var_data_2D_a(nc%NDIM+ 21,m,n  ) =       db_dy                           (  n,m)      
      IF(PRESENT(dHi_dxy                         )) var_data_2D_a(nc%NDIM+ 22,m,n  ) =       dHi_dxy                         (  n,m)      
      IF(PRESENT(dHs_dxy                         )) var_data_2D_a(nc%NDIM+ 23,m,n  ) =       dHs_dxy                         (  n,m)      
      IF(PRESENT(dHi_dx                          )) var_data_2D_a(nc%NDIM+ 24,m,n  ) =       dHi_dx                          (  n,m)      
      IF(PRESENT(dHi_dy                          )) var_data_2D_a(nc%NDIM+ 25,m,n  ) =       dHi_dy                          (  n,m)      
      IF(PRESENT(dHi_dxx                         )) var_data_2D_a(nc%NDIM+ 26,m,n  ) =       dHi_dxx                         (  n,m)      
      IF(PRESENT(dHi_dyy                         )) var_data_2D_a(nc%NDIM+ 27,m,n  ) =       dHi_dyy                         (  n,m)      
      IF(PRESENT(dHs_dx                          )) var_data_2D_a(nc%NDIM+ 28,m,n  ) =       dHs_dx                          (  n,m)      
      IF(PRESENT(dHs_dy                          )) var_data_2D_a(nc%NDIM+ 29,m,n  ) =       dHs_dy                          (  n,m)      
      IF(PRESENT(dHs_dxx                         )) var_data_2D_a(nc%NDIM+ 30,m,n  ) =       dHs_dxx                         (  n,m)      
      IF(PRESENT(dHs_dyy                         )) var_data_2D_a(nc%NDIM+ 31,m,n  ) =       dHs_dyy                         (  n,m)      
      IF(PRESENT(dzeta_dz                        )) var_data_2D_a(nc%NDIM+ 32,m,n  ) =       dzeta_dz                        (  n,m)      
      IF(PRESENT(dzeta_dxz                       )) var_data_2D_a(nc%NDIM+ 33,m,n  ) =       dzeta_dxz                       (  n,m)      
      IF(PRESENT(dzeta_dyz                       )) var_data_2D_a(nc%NDIM+ 34,m,n  ) =       dzeta_dyz                       (  n,m)      
      IF(PRESENT(Hi                              )) var_data_2D_a(nc%NDIM+ 35,m,n  ) =       Hi                              (  n,m)
      IF(PRESENT(Hb                              )) var_data_2D_a(nc%NDIM+ 36,m,n  ) =       Hb                              (  n,m)
      IF(PRESENT(Hs                              )) var_data_2D_a(nc%NDIM+ 37,m,n  ) =       Hs                              (  n,m)
      IF(PRESENT(dHi_dt                          )) var_data_2D_a(nc%NDIM+ 38,m,n  ) =       dHi_dt                          (  n,m)
      IF(PRESENT(dHb_dt                          )) var_data_2D_a(nc%NDIM+ 39,m,n  ) =       dHb_dt                          (  n,m)
      IF(PRESENT(dHs_dt                          )) var_data_2D_a(nc%NDIM+ 40,m,n  ) =       dHs_dt                          (  n,m)
      IF(PRESENT(Us                              )) var_data_2D_a(nc%NDIM+ 41,m,n  ) =       Us                              (  n,m)
      IF(PRESENT(Vs                              )) var_data_2D_a(nc%NDIM+ 42,m,n  ) =       Vs                              (  n,m)
      IF(PRESENT(intervention_cover              )) var_data_2D_a(nc%NDIM+ 43,m,n  ) =       intervention_cover              (  n,m) 
      IF(PRESENT(coverage_ti_pmp                 )) var_data_2D_a(nc%NDIM+ 44,m,n  ) =       coverage_ti_pmp                 (  n,m) 
      IF(PRESENT(cover_adjusted_temperatures     )) var_data_2D_a(nc%NDIM+ 45,m,n  ) =       cover_adjusted_temperatures     (  n,m)
      IF(PRESENT(longitude                       )) var_data_2D_a(nc%NDIM+ 46,m,n  ) =       longitude                       (  n,m)
      IF(PRESENT(latitude                        )) var_data_2D_a(nc%NDIM+ 47,m,n  ) =       latitude                        (  n,m)
      IF(PRESENT(hi_y_mask                       )) var_data_2D_a(nc%NDIM+ 48,m,n  ) = REAL( hi_y_mask                       (  n,m) , dp)
      IF(PRESENT(critical_time_step_pattern      )) var_data_2D_a(nc%NDIM+ 49,m,n  ) =       critical_time_step_pattern      (  n,m)
      IF(PRESENT(bottom_melt                     )) var_data_2D_a(nc%NDIM+ 50,m,n  ) =       bottom_melt                     (  n,m)
      IF(PRESENT(sor_residue_x                   )) var_data_2D_a(nc%NDIM+ 51,m,n  ) =       sor_residue_x                   (  n,m)
      IF(PRESENT(sor_residue_y                   )) var_data_2D_a(nc%NDIM+ 52,m,n  ) =       sor_residue_y                   (  n,m)
     !IF(PRESENT(dummy_2_2D                      )) var_data_2D_a(nc%NDIM+ 51,m,n  ) =       dummy_2_2D                      (  n,m)
     !IF(PRESENT(dummy_1_2D                      )) var_data_2D_a(nc%NDIM+ 52,m,n  ) =       dummy_1_2D                      (  n,m)
      IF(PRESENT(U                               )) var_data_3D_a(nc%NDIM+ 53,m,n,:) =       U                               (:,n,m)      
      IF(PRESENT(V                               )) var_data_3D_a(nc%NDIM+ 54,m,n,:) =       V                               (:,n,m)      
      IF(PRESENT(W                               )) var_data_3D_a(nc%NDIM+ 55,m,n,:) =       W                               (:,n,m)      
      IF(PRESENT(Ti                              )) var_data_3D_a(nc%NDIM+ 56,m,n,:) =       Ti                              (:,n,m)
      IF(PRESENT(Ti_pmp                          )) var_data_3D_a(nc%NDIM+ 57,m,n,:) =       Ti_pmp                          (:,n,m)
      IF(PRESENT(A_flow                          )) var_data_3D_a(nc%NDIM+ 58,m,n,:) =       A_flow                          (:,n,m)      
      IF(PRESENT(Cpi                             )) var_data_3D_a(nc%NDIM+ 59,m,n,:) =       Cpi                             (:,n,m)      
      IF(PRESENT(Ki                              )) var_data_3D_a(nc%NDIM+ 60,m,n,:) =       Ki                              (:,n,m)      
      IF(PRESENT(D_uv_3D                         )) var_data_3D_a(nc%NDIM+ 61,m,n,:) =       D_uv_3D                         (:,n,m)      
      IF(PRESENT(concentration                   )) var_data_3D_a(nc%NDIM+ 62,m,n,:) =       concentration                   (:,n,m)      
      IF(PRESENT(dTi_dx                          )) var_data_3D_a(nc%NDIM+ 63,m,n,:) =       dTi_dx                          (:,n,m)
      IF(PRESENT(dTi_dy                          )) var_data_3D_a(nc%NDIM+ 64,m,n,:) =       dTi_dy                          (:,n,m)
      IF(PRESENT(dU_dx                           )) var_data_3D_a(nc%NDIM+ 65,m,n,:) =       dU_dx                           (:,n,m)      
      IF(PRESENT(dV_dx                           )) var_data_3D_a(nc%NDIM+ 66,m,n,:) =       dV_dx                           (:,n,m)      
      IF(PRESENT(dW_dx                           )) var_data_3D_a(nc%NDIM+ 67,m,n,:) =       dW_dx                           (:,n,m)      
      IF(PRESENT(dU_dy                           )) var_data_3D_a(nc%NDIM+ 68,m,n,:) =       dU_dy                           (:,n,m)      
      IF(PRESENT(dV_dy                           )) var_data_3D_a(nc%NDIM+ 69,m,n,:) =       dV_dy                           (:,n,m)      
      IF(PRESENT(dW_dy                           )) var_data_3D_a(nc%NDIM+ 70,m,n,:) =       dW_dy                           (:,n,m)      
      IF(PRESENT(dU_dzeta                        )) var_data_3D_a(nc%NDIM+ 71,m,n,:) =       dU_dzeta                        (:,n,m)      
      IF(PRESENT(dV_dzeta                        )) var_data_3D_a(nc%NDIM+ 72,m,n,:) =       dV_dzeta                        (:,n,m)      
      IF(PRESENT(dW_dzeta                        )) var_data_3D_a(nc%NDIM+ 73,m,n,:) =       dW_dzeta                        (:,n,m)      
      IF(PRESENT(dzeta_dt                        )) var_data_3D_a(nc%NDIM+ 74,m,n,:) =       dzeta_dt                        (:,n,m)      
      IF(PRESENT(dzeta_dx                        )) var_data_3D_a(nc%NDIM+ 75,m,n,:) =       dzeta_dx                        (:,n,m)      
      IF(PRESENT(dzeta_dy                        )) var_data_3D_a(nc%NDIM+ 76,m,n,:) =       dzeta_dy                        (:,n,m)      
      IF(PRESENT(dzeta_dxx                       )) var_data_3D_a(nc%NDIM+ 77,m,n,:) =       dzeta_dxx                       (:,n,m)      
      IF(PRESENT(dzeta_dxy                       )) var_data_3D_a(nc%NDIM+ 78,m,n,:) =       dzeta_dxy                       (:,n,m)      
      IF(PRESENT(dzeta_dyy                       )) var_data_3D_a(nc%NDIM+ 79,m,n,:) =       dzeta_dyy                       (:,n,m)      
      IF(PRESENT(Eta                             )) var_data_3D_a(nc%NDIM+ 80,m,n,:) =       Eta                             (:,n,m)      
      IF(PRESENT(dEta_dx                         )) var_data_3D_a(nc%NDIM+ 81,m,n,:) =       dEta_dx                         (:,n,m)      
      IF(PRESENT(dEta_dy                         )) var_data_3D_a(nc%NDIM+ 82,m,n,:) =       dEta_dy                         (:,n,m)      
      IF(PRESENT(dEta_dzeta                      )) var_data_3D_a(nc%NDIM+ 83,m,n,:) =       dEta_dzeta                      (:,n,m)      
      IF(PRESENT(Eta_x                           )) var_data_3D_a(nc%NDIM+ 84,m,n,:) =       Eta_x                           (:,n,m)      
      IF(PRESENT(Eta_y                           )) var_data_3D_a(nc%NDIM+ 85,m,n,:) =       Eta_y                           (:,n,m)      
      IF(PRESENT(Eta_zeta                        )) var_data_3D_a(nc%NDIM+ 86,m,n,:) =       Eta_zeta                        (:,n,m)      
      IF(PRESENT(rhs_eq1                         )) var_data_3D_a(nc%NDIM+ 87,m,n,:) =       rhs_eq1                         (:,n,m)      
      IF(PRESENT(rhs_eq2                         )) var_data_3D_a(nc%NDIM+ 88,m,n,:) =       rhs_eq2                         (:,n,m)      
      IF(PRESENT(rhs_eq1_calving_bc              )) var_data_3D_a(nc%NDIM+ 89,m,n,:) =       rhs_eq1_calving_bc              (:,n,m)      
    END DO           
    END DO

    ! Making the linear fields:
    IF(C%which_help_fields >= 90) THEN
     Hi_transposed = var_data_2D_a(nc%NDIM+35,:,:)
     Hb_transposed = var_data_2D_a(nc%NDIM+36,:,:)
     Hs_transposed = var_data_2D_a(nc%NDIM+37,:,:)

     iL = 0
     DO i = nc%NDIM+53, nc%NDIM+89
      if(iL< 5) iL = iL + 1                              ! For setting the fake_value field counter (5 is the default)

      IF(nc%case(i+37) == 10 .OR. nc%case(i+74) == 11 .OR. nc%case(i+111) == 12) THEN
       scaled_3D_field = var_data_3D_a(i,:,:,:)

       ! Output: x_cross_section, y_cross_section, linear_3D_field
       CALL cross_section(iL, C%x_fixed, C%y_fixed, Hi_transposed, Hb_transposed, Hs_transposed, scaled_3D_field, x_cross_section, y_cross_section, var_linear = linear_3D_field)

       var_data_3D_b(i+ 37,:,:,:) = linear_3D_field      !  37 =  90 - 53
       var_data_2D_b(i+ 74,:,:  ) = x_cross_section      !  74 = 127 - 53
       var_data_2D_c(i+111,:,:  ) = y_cross_section      ! 111 = 164 - 53
      END IF
     END DO
    END IF

    IF(C%use_grid_units_in_dimension) THEN
     IF(C%dynamic_time_step_addaption) THEN
      ! In this case only the shifted time is used
      time =  t - C%start_time_of_run
     ELSE
      time = (t - C%start_time_of_run) / C%dt
     END IF
    ELSE
     time = t
    END IF

    ! Data writing mode:
    DO i = nc%NDIM, nc%N_loop
      SELECT CASE(nc%case(i))
      ! Write the data for the unlimited (time) dimension (the other dimension data is written by the subroutine create_netcdf_file):
      CASE(6)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ time             /), start=(/         record_number/)), '. [1] From write_help_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)

      ! Write the data of the variables:
      CASE(7)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_2D_a(i,:,:  ), start=(/1, 1,    record_number/)), '. [2] From write_help_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(8)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_3D_a(i,:,:,:), start=(/1, 1, 1, record_number/)), '. [3] From write_help_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(10)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_3D_b(i,:,:,:), start=(/1, 1, 1, record_number/)), '. [4] From write_help_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(11)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_2D_b(i,:,:  ), start=(/1, 1,    record_number/)), '. [5] From write_help_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(12)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_2D_c(i,:,:  ), start=(/1, 1,    record_number/)), '. [6] From write_help_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE DEFAULT
       ! Do nothing: Ignore variables with another mode
      END SELECT
    END DO

    ! Synchronize the disk copy of a netcdf dataset: to minimize data loss in case of an abnormal termination:
    IF(C%synchronize_netcdf_writing) CALL handle_error(nf90_sync(nc%ncid)                                         , '. [7] From write_help_fields(): while synchronizing the file: '//TRIM(nc%file_name))
  END SUBROUTINE write_help_fields



  SUBROUTINE initialize_extra_netcdf_file(choice_predefined_extra_fields, number_of_fields, nc) 
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type
    USE netcdf, ONLY: nf90_int, nf90_double, nf90_unlimited
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*)     ,  INTENT(IN)  :: choice_predefined_extra_fields      ! With this choice the programmer selects which predefined extra fields are used
    INTEGER              ,  INTENT(IN)  :: number_of_fields

    ! Output variables:
    TYPE(netcdf_file_type), INTENT(OUT) :: nc

    ! Local variables:
    INTEGER                             :: stat

    IF(number_of_fields > 17) STOP ' The maximum number_of_fields in routine  initialize_extra_netcdf_file()  is 17 currently. If you need more fields the routine  write_extra_netcdf_fields()  has to be changed.'
    
    nc%NDIM = 3
    nc%NVAR = nc%NDIM + number_of_fields

    ! Setting the number of loops in which the dimensions + variables are written to the netcdf file:
    nc%N_loop = nc%NVAR

    ALLOCATE(nc%field_name       (nc%NVAR), &
             nc%field_unit       (nc%NVAR), &
             nc%field_longname   (nc%NVAR), &
              nc%id               (nc%NVAR), &
             nc%type             (nc%NVAR), &
             nc%case             (nc%NVAR), &
             nc%LEN_DIM          (nc%NDIM), &
             nc%grid_size        (nc%NDIM), &
             STAT=stat)
    IF(stat/=0) STOP ' STOP message from: initialize_extra_netcdf_file:  Could not allocate memory for the netcdf file struct.'

    ! The file name is stored for error messaging:
    nc%file_name = C%extra_fields_filename

    nc%field_name       (      1) = 'x   '
    nc%field_name       (      2) = 'y   '
    nc%field_name       (nc%NDIM) = 'time'

    nc%field_unit       (      1) = 'meter'
    nc%field_unit       (      2) = 'meter'
    nc%field_unit       (nc%NDIM) = 'year '

    nc%field_longname   (      1) = 'horizontal x-axis (meter)'
    nc%field_longname   (      2) = 'horizontal y-axis (meter)'
    nc%field_longname   (nc%NDIM) = 'Time (year)              '

    SELECT CASE(choice_predefined_extra_fields)
    CASE('extra_field_choice_1')
     nc%field_name       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_1_name       (1:number_of_fields)
     nc%field_unit       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_1_unit       (1:number_of_fields)
     nc%field_longname   (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_1_longname   (1:number_of_fields)
    CASE('extra_field_choice_2')
     nc%field_name       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_2_name       (1:number_of_fields)
     nc%field_unit       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_2_unit       (1:number_of_fields)
     nc%field_longname   (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_2_longname   (1:number_of_fields)
    CASE('extra_field_choice_3')
     nc%field_name       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_3_name       (1:number_of_fields)
     nc%field_unit       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_3_unit       (1:number_of_fields)
     nc%field_longname   (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_3_longname   (1:number_of_fields)
    CASE('extra_field_choice_4')
     nc%field_name       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_4_name       (1:number_of_fields)
     nc%field_unit       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_4_unit       (1:number_of_fields)
     nc%field_longname   (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_4_longname   (1:number_of_fields)
    CASE('extra_field_choice_5')
     nc%field_name       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_5_name       (1:number_of_fields)
     nc%field_unit       (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_5_unit       (1:number_of_fields)
     nc%field_longname   (nc%NDIM+1:nc%NDIM+number_of_fields) = C%extra_field_choice_5_longname   (1:number_of_fields)
    CASE DEFAULT
     WRITE(UNIT=*, FMT='(/2A/)') ' STOP: programmer error for: choice_predefined_extra_fields = ', TRIM(choice_predefined_extra_fields)
     STOP
    END SELECT

    nc%type(1:nc%NVAR        )       = nf90_double

    nc%case(1                )       = 1 ! The dimension: X
    nc%case(2                )       = 2 ! The dimension: Y
    nc%case(nc%NDIM          )       = 6 ! The dimension: time
    nc%case(nc%NDIM+1:nc%NVAR)       = 7 ! The NX, NY, time case

    nc%LEN_DIM(1      )              = C%NX
    nc%LEN_DIM(2      )              = C%NY
    nc%LEN_DIM(nc%NDIM)              = nf90_unlimited

    nc%grid_size(1      )            = C%dx
    nc%grid_size(2      )            = C%dy
   !nc%grid_size(nc%NDIM)            = C%dt     ! but the time step can be dynamical: see G%dt
  END SUBROUTINE initialize_extra_netcdf_file



  SUBROUTINE write_extra_netcdf_fields(nc            , &
                                       record_number , &
                                       time          , &
                                       variable_01_2D, &
                                       variable_02_2D, &
                                       variable_03_2D, &
                                       variable_04_2D, &
                                       variable_05_2D, &
                                       variable_06_2D, &
                                       variable_07_2D, &
                                       variable_08_2D, &
                                       variable_09_2D, &
                                       variable_10_2D, &
                                       variable_11_2D, &
                                       variable_12_2D, &
                                       variable_13_2D, &
                                       variable_14_2D, &
                                       variable_15_2D, &
                                       variable_16_2D, &
                                       variable_17_2D, &
                                       variable_18_2D, &
                                       variable_19_2D, &
                                       variable_20_2D)  
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type, handle_error
    USE netcdf, ONLY: nf90_put_var, nf90_sync
    IMPLICIT NONE

    ! Input variables:
    TYPE(netcdf_file_type),                           INTENT(IN)           :: nc
    INTEGER,                                          INTENT(IN)           :: record_number
    REAL(dp),                                         INTENT(IN)           :: time
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_01_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_02_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_03_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_04_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_05_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_06_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_07_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_08_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_09_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_10_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_11_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_12_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_13_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_14_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_15_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_16_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_17_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_18_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_19_2D
    REAL(dp), DIMENSION(nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN), OPTIONAL :: variable_20_2D

    ! Local variables:
    INTEGER                                                                :: m, n            ! Grid point counters
    INTEGER                                                                :: i               ! Counter over the dimensions or the variables
    REAL(dp), DIMENSION(nc%NVAR,nc%LEN_DIM(1),nc%LEN_DIM(2))               :: var_data_2D     ! The 2D transposed NX * NY       arrays

    ! In case an optional variable is not present, but is written: 
    var_data_2D = 0._dp

    ! Transpose the read fields from {*,y,x} to {x,y,*}:
    DO m = 1, nc%LEN_DIM(1)
    DO n = 1, nc%LEN_DIM(2) 
      IF(PRESENT(variable_01_2D)) var_data_2D(nc%NDIM+ 1,m,n) = variable_01_2D(n,m)      
      IF(PRESENT(variable_02_2D)) var_data_2D(nc%NDIM+ 2,m,n) = variable_02_2D(n,m)      
      IF(PRESENT(variable_03_2D)) var_data_2D(nc%NDIM+ 3,m,n) = variable_03_2D(n,m)      
      IF(PRESENT(variable_04_2D)) var_data_2D(nc%NDIM+ 4,m,n) = variable_04_2D(n,m)      
      IF(PRESENT(variable_05_2D)) var_data_2D(nc%NDIM+ 5,m,n) = variable_05_2D(n,m)      
      IF(PRESENT(variable_06_2D)) var_data_2D(nc%NDIM+ 6,m,n) = variable_06_2D(n,m)      
      IF(PRESENT(variable_07_2D)) var_data_2D(nc%NDIM+ 7,m,n) = variable_07_2D(n,m)      
      IF(PRESENT(variable_08_2D)) var_data_2D(nc%NDIM+ 8,m,n) = variable_08_2D(n,m)      
      IF(PRESENT(variable_09_2D)) var_data_2D(nc%NDIM+ 9,m,n) = variable_09_2D(n,m)      
      IF(PRESENT(variable_10_2D)) var_data_2D(nc%NDIM+10,m,n) = variable_10_2D(n,m)      
      IF(PRESENT(variable_11_2D)) var_data_2D(nc%NDIM+11,m,n) = variable_11_2D(n,m)      
      IF(PRESENT(variable_12_2D)) var_data_2D(nc%NDIM+12,m,n) = variable_12_2D(n,m)      
      IF(PRESENT(variable_13_2D)) var_data_2D(nc%NDIM+13,m,n) = variable_13_2D(n,m)      
      IF(PRESENT(variable_14_2D)) var_data_2D(nc%NDIM+14,m,n) = variable_14_2D(n,m)      
      IF(PRESENT(variable_15_2D)) var_data_2D(nc%NDIM+15,m,n) = variable_15_2D(n,m)      
      IF(PRESENT(variable_16_2D)) var_data_2D(nc%NDIM+16,m,n) = variable_16_2D(n,m)      
      IF(PRESENT(variable_17_2D)) var_data_2D(nc%NDIM+17,m,n) = variable_17_2D(n,m)      
      IF(PRESENT(variable_18_2D)) var_data_2D(nc%NDIM+18,m,n) = variable_18_2D(n,m)      
      IF(PRESENT(variable_19_2D)) var_data_2D(nc%NDIM+19,m,n) = variable_19_2D(n,m)      
      IF(PRESENT(variable_20_2D)) var_data_2D(nc%NDIM+20,m,n) = variable_20_2D(n,m)      
    END DO           
    END DO

    ! Data writing mode:
    DO i = nc%NDIM, nc%N_loop
      SELECT CASE(nc%case(i))
      ! Write the data for the unlimited (time) dimension (the other dimension data is written by the subroutine create_netcdf_file):
      CASE(6)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ time         /), start=(/      record_number/)), '. [1] From write_extra_netcdf_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)

      ! Write the data of the variables:
      CASE(7)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_2D(i,:,:), start=(/1, 1, record_number/)), '. [2] From write_extra_netcdf_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE DEFAULT
       ! Do nothing: Ignore variables with another mode
      END SELECT
    END DO

    ! Synchronize the disk copy of a netcdf dataset: to minimize data loss in case of an abnormal termination:
    IF(C%synchronize_netcdf_writing) CALL handle_error(nf90_sync(nc%ncid)                                  , '. [3] From write_extra_netcdf_fields(): while synchronizing the file: '//TRIM(nc%file_name))
  END SUBROUTINE write_extra_netcdf_fields



  SUBROUTINE initialize_key_fields_file(file_name, nc) 
    USE configuration_module, ONLY: C
    USE netcdf_module, ONLY: netcdf_file_type
    USE netcdf, ONLY: nf90_int, nf90_double, nf90_unlimited
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*),       INTENT(IN)  :: file_name

    ! Output variables:
    TYPE(netcdf_file_type), INTENT(OUT) :: nc

    ! Local variables:
    INTEGER                             :: stat

    nc%NDIM = 4
    nc%NVAR = nc%NDIM + 8

    ! Setting the number of loops in which the dimensions + variables are written to the netcdf file:
    nc%N_loop = nc%NVAR

    ALLOCATE(nc%field_name       (nc%NVAR), &
             nc%field_unit       (nc%NVAR), &
             nc%field_longname   (nc%NVAR), &
             nc%id               (nc%NVAR), &
             nc%type             (nc%NVAR), &
             nc%case             (nc%NVAR), &
             nc%LEN_DIM          (nc%NDIM), &
             nc%grid_size        (nc%NDIM), &
             STAT=stat)
    IF(stat /= 0) STOP ' STOP message from: initialize_key_fields_file:  Could not allocate memory for the netcdf file struct.'

    ! The file name is stored for error messaging:
    nc%file_name = file_name

    nc%field_name(        1) = 'x          '
    nc%field_name(        2) = 'y          '
    nc%field_name(        3) = 'zeta       '
    nc%field_name(nc%NDIM  ) = 'time       '
    nc%field_name(nc%NDIM+1) = 'Hi         '
    nc%field_name(nc%NDIM+2) = 'Hb         '
    nc%field_name(nc%NDIM+3) = 'Hs         '
    nc%field_name(nc%NDIM+4) = 'bottom_melt'
    nc%field_name(nc%NDIM+5) = 'U          '
    nc%field_name(nc%NDIM+6) = 'V          '
    nc%field_name(nc%NDIM+7) = 'W          '
    nc%field_name(nc%NDIM+8) = 'Ti         '

    nc%field_unit(        1) = 'meter    ' ! x
    nc%field_unit(        2) = 'meter    ' ! y
    nc%field_unit(        3) = 'unitless ' ! zeta
    nc%field_unit(nc%NDIM  ) = 'year     ' ! time
    nc%field_unit(nc%NDIM+1) = 'm        ' ! Hi
    nc%field_unit(nc%NDIM+2) = 'm        ' ! Hb
    nc%field_unit(nc%NDIM+3) = 'm        ' ! Hs
    nc%field_unit(nc%NDIM+4) = 'mieq / y ' ! bottom_melt
    nc%field_unit(nc%NDIM+5) = 'm / y    ' ! U
    nc%field_unit(nc%NDIM+6) = 'm / y    ' ! V
    nc%field_unit(nc%NDIM+7) = 'm / y    ' ! W
    nc%field_unit(nc%NDIM+8) = 'K        ' ! Ti

    nc%field_longname(        1) = 'horizontal x-axis (meter)'
    nc%field_longname(        2) = 'horizontal y-axis (meter)'
    nc%field_longname(        3) = 'scaled vertical zeta-axis (unitless)'
    nc%field_longname(        6) = 'Time (year)'
    nc%field_longname(nc%NDIM+1) = 'Ice Thickness (meter)'
    nc%field_longname(nc%NDIM+2) = 'Height of Bedrock (meter) - relative to presentday sea level'
    nc%field_longname(nc%NDIM+3) = 'Surface Height (meter) - relative to presentday sea level'
    nc%field_longname(nc%NDIM+4) = 'Bottom melt (meter ice equivalent / year)'
    nc%field_longname(nc%NDIM+5) = 'Velocity in x-direction (meter / year)'
    nc%field_longname(nc%NDIM+6) = 'Velocity in y-direction (meter / year)'
    nc%field_longname(nc%NDIM+7) = 'Velocity in z-direction (meter / year)'
    nc%field_longname(nc%NDIM+8) = 'Ice Temperature (Kelvin)'

    nc%type(         1:nc%NVAR ) = nf90_double

    nc%case(                  1) =    1 ! The dimension: X
    nc%case(                  2) =    2 ! The dimension: Y
    nc%case(                  3) =    3 ! The dimension: zeta
    nc%case(nc%NDIM            ) =    6 ! The dimension: time
    nc%case(nc%NDIM+1:nc%NDIM+4) =    7 ! the NX, NY     , time case  [Default case]
    nc%case(nc%NDIM+5:nc%NDIM+8) =    8 ! the NX, NY, NZ , time case

    nc%LEN_DIM(1      )          = C%NX
    nc%LEN_DIM(2      )          = C%NY
    nc%LEN_DIM(3      )          = C%NZ
    nc%LEN_DIM(nc%NDIM)          = nf90_unlimited

    nc%grid_size(1      )        = C%dx
    nc%grid_size(2      )        = C%dy
   !nc%grid_size(3      )        =        ! non equidistant
   !nc%grid_size(nc%NDIM)        = C%dt   ! but the time step can be dynamical: see G%dt
  END SUBROUTINE initialize_key_fields_file



  SUBROUTINE write_key_fields(nc, record_number, Hi, Hb, Hs, bottom_melt, U, V, W, Ti, time)
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type, handle_error
    USE netcdf, ONLY: nf90_put_var, nf90_sync
    IMPLICIT NONE

    ! Input variables:
    TYPE(netcdf_file_type),                                         INTENT(IN)         :: nc
    INTEGER,                                                        INTENT(IN)         :: record_number
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hi
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hb
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hs
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: bottom_melt
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: U
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: V
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: W
    REAL(dp), DIMENSION(nc%LEN_DIM(3),nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Ti
    REAL(dp),                                                       INTENT(IN)         :: time

    ! Local variables:
    INTEGER                                                                            :: m, n        ! Grid point counters
    INTEGER                                                                            :: i           ! Counter over the dimensions or the variables
    REAL(dp), DIMENSION(nc%NDIM+1:nc%NDIM+4,nc%LEN_DIM(1),nc%LEN_DIM(2)              ) :: var_data_2D ! The 2D data of the variables in one array
    REAL(dp), DIMENSION(nc%NDIM+5:nc%NDIM+9,nc%LEN_DIM(1),nc%LEN_DIM(2),nc%LEN_DIM(3)) :: var_data_3D ! The 3D data of the variables in one array

    ! Transpose:
    DO m = 1, nc%LEN_DIM(1)
    DO n = 1, nc%LEN_DIM(2) 
      var_data_2D(nc%NDIM+1,m,n  ) = Hi         (  n,m)
      var_data_2D(nc%NDIM+2,m,n  ) = Hb         (  n,m)
      var_data_2D(nc%NDIM+3,m,n  ) = Hs         (  n,m)
      var_data_2D(nc%NDIM+4,m,n  ) = bottom_melt(  n,m)
      var_data_3D(nc%NDIM+5,m,n,:) = U          (:,n,m)
      var_data_3D(nc%NDIM+6,m,n,:) = V          (:,n,m)
      var_data_3D(nc%NDIM+7,m,n,:) = W          (:,n,m)
      var_data_3D(nc%NDIM+8,m,n,:) = Ti         (:,n,m)
    END DO           
    END DO

    ! Data writing mode:
    DO i = nc%NDIM, nc%N_loop
      SELECT CASE(nc%case(i))
      ! Write the data for the unlimited (time) dimension (the other dimension data is written by the subroutine create_netcdf_file):
      CASE(6)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ time /)          , start=(/         record_number/)), '. [1] From write_key_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)

      ! Write the data of the variables:
      CASE(7)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_2D(i,:,:  ), start=(/1, 1,    record_number/)), '. [2] From write_key_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(8)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_3D(i,:,:,:), start=(/1, 1, 1, record_number/)), '. [3] From write_key_fields(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE DEFAULT
       ! Do nothing: Ignore variables with another mode
      END SELECT
    END DO

    ! Synchronize the disk copy of a netcdf dataset: to minimize data loss in case of an abnormal termination:
    IF(C%synchronize_netcdf_writing) CALL handle_error(nf90_sync(nc%ncid)                                       , '. [4] From write_key_fields(): while synchronizing the file: '//TRIM(nc%file_name))
  END SUBROUTINE write_key_fields
  


  SUBROUTINE read_key_fields_file(file_name, record_number, Hi, Hb, Hs, bottom_melt, U, V, W, Ti, time)
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type, handle_error, open_netcdf_file, inquire_netcdf_file, count_netcdf_record_size, close_netcdf_file
    USE netcdf, ONLY: nf90_get_var
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*),                     INTENT(IN)            :: file_name
    INTEGER,                              INTENT(IN)            :: record_number

    ! Output variables:
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hi
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hb
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hs
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: bottom_melt
    REAL(dp), DIMENSION(C%NZ ,C%NY,C%NX), INTENT(OUT)           :: U
    REAL(dp), DIMENSION(C%NZ ,C%NY,C%NX), INTENT(OUT)           :: V
    REAL(dp), DIMENSION(C%NZ ,C%NY,C%NX), INTENT(OUT)           :: W
    REAL(dp), DIMENSION(C%NZ ,C%NY,C%NX), INTENT(OUT)           :: Ti
    REAL(dp),                             INTENT(OUT), OPTIONAL :: time

    ! Local variables:
    TYPE(netcdf_file_type)                                      :: nc      
    INTEGER                                                     :: m, n          ! Grid point counters
    INTEGER                                                     :: i             ! Counter over the dimensions or the variables
    INTEGER                                                     :: record_size
    INTEGER                                                     :: used_record ! The record number which is used for reading
    REAL(dp), DIMENSION(:,:   ,:         ), ALLOCATABLE         :: T_var_2D
    REAL(dp), DIMENSION(:,:   ,:   ,:    ), ALLOCATABLE         :: T_var_3D

    ! Output: netcdf 
    CALL initialize_key_fields_file(file_name, nc)

    ALLOCATE(T_var_2D(nc%NDIM+1:nc%NDIM+4,C%NX,C%NY     ))
    ALLOCATE(T_var_3D(nc%NDIM+5:nc%NDIM+8,C%NX,C%NY,C%NZ))

    ! In/Output: netcdf
    CALL open_netcdf_file(nc) 

    ! In/Output: netcdf
    CALL inquire_netcdf_file(nc)

    ! Output: record_size
    CALL count_netcdf_record_size(nc, record_size)

    IF(0 < record_number .AND. record_number <= record_size) THEN
     used_record = record_number
    ELSE IF(record_number == 0) THEN
     ! In this case the variables from the last record:
     used_record = record_size
    ELSE
     WRITE(UNIT=*,FMT='(A,I6,A)') 'ERROR: Record number ', record_number, ' does not exist in the file:  '//TRIM(nc%file_name)
     STOP
    END IF

    DO i = nc%NDIM, nc%NVAR
      SELECT CASE(nc%case(i))
       CASE(6)
       IF(PRESENT(time)) &
        CALL handle_error(nf90_get_var(nc%ncid, nc%id(i), time             , start=(/         used_record/) ), '. [1] From read_key_fields_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(7)
        CALL handle_error(nf90_get_var(nc%ncid, nc%id(i), T_var_2D(i,:,:  ), start=(/1, 1,    used_record/) ), '. [2] From read_key_fields_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(8)
        CALL handle_error(nf90_get_var(nc%ncid, nc%id(i), T_var_3D(i,:,:,:), start=(/1, 1, 1, used_record/) ), '. [3] From read_key_fields_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE DEFAULT
        !  Do nothing: Ignore variables with another mode
      END SELECT
    END DO

    ! Finish and closing writing output:
    CALL close_netcdf_file(nc)

    ! Transpose the read fields from {x,y,*} to {*,y,x}:
    DO m = 1, C%NX
    DO n = 1, C%NY 
      Hi         (  n,m) = T_var_2D(nc%NDIM+1,m,n  )
      Hb         (  n,m) = T_var_2D(nc%NDIM+2,m,n  )
      Hs         (  n,m) = T_var_2D(nc%NDIM+3,m,n  )
      bottom_melt(  n,m) = T_var_2D(nc%NDIM+4,m,n  )
      U          (:,n,m) = T_var_3D(nc%NDIM+5,m,n,:)
      V          (:,n,m) = T_var_3D(nc%NDIM+6,m,n,:)
      W          (:,n,m) = T_var_3D(nc%NDIM+7,m,n,:)
      Ti         (:,n,m) = T_var_3D(nc%NDIM+8,m,n,:)
    END DO
    END DO

    DEALLOCATE(T_var_2D)
    DEALLOCATE(T_var_3D)
  END SUBROUTINE read_key_fields_file



  SUBROUTINE initialize_restart_fields_siadyn(file_name, nc) 
    USE configuration_module, ONLY: C
    USE netcdf_module, ONLY: netcdf_file_type
    USE netcdf, ONLY: nf90_int, nf90_double, nf90_unlimited
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*),       INTENT(IN)  :: file_name

    ! Output variables:
    TYPE(netcdf_file_type), INTENT(OUT) :: nc

    ! Local variables:
    INTEGER                             :: stat

    nc%NDIM = 3
    nc%NVAR = nc%NDIM + 5

    ! Setting the number of loops in which the dimensions + variables are written to the netcdf file:
    nc%N_loop = nc%NVAR

    ALLOCATE(nc%field_name       (nc%NVAR), &
             nc%field_unit       (nc%NVAR), &
             nc%field_longname   (nc%NVAR), &
             nc%id               (nc%NVAR), &
             nc%type             (nc%NVAR), &
             nc%case             (nc%NVAR), &
             nc%LEN_DIM          (nc%NDIM), &
             nc%grid_size        (nc%NDIM), &
             STAT=stat)
    IF(stat /= 0) STOP ' STOP message from: initialize_restart_fields_siadyn:  Could not allocate memory for the netcdf file struct.'

    ! The file name is stored for error messaging:
    nc%file_name = file_name

    nc%field_name(        1) = 'x          '
    nc%field_name(        2) = 'y          '
    nc%field_name(        3) = 'zeta       '
    nc%field_name(nc%NDIM  ) = 'time       '
    nc%field_name(nc%NDIM+1) = 'Hi         '
    nc%field_name(nc%NDIM+2) = 'Hb         '
    nc%field_name(nc%NDIM+3) = 'Hs         '
    nc%field_name(nc%NDIM+4) = 'Us         '
    nc%field_name(nc%NDIM+5) = 'Vs         '

    nc%field_unit(        1) = 'meter    ' ! x          
    nc%field_unit(        2) = 'meter    ' ! y          
    nc%field_unit(        3) = 'unitless ' ! zeta       
    nc%field_unit(nc%NDIM  ) = 'year     ' ! time       
    nc%field_unit(nc%NDIM+1) = 'm        ' ! Hi         
    nc%field_unit(nc%NDIM+2) = 'm        ' ! Hb         
    nc%field_unit(nc%NDIM+3) = 'm        ' ! Hs         
    nc%field_unit(nc%NDIM+4) = 'm / y    ' ! Us         
    nc%field_unit(nc%NDIM+5) = 'm / y    ' ! Vs         

    nc%field_longname(        1) = 'horizontal x-axis (meter)'
    nc%field_longname(        2) = 'horizontal y-axis (meter)'
    nc%field_longname(        3) = 'scaled vertical zeta-axis (unitless)'
    nc%field_longname(nc%NDIM  ) = 'Time (year)'
    nc%field_longname(nc%NDIM+1) = 'Ice Thickness (meter)'
    nc%field_longname(nc%NDIM+2) = 'Height of Bedrock (meter) - relative to presentday sea level'
    nc%field_longname(nc%NDIM+3) = 'Surface Height (meter) - relative to presentday sea level'
    nc%field_longname(nc%NDIM+4) = 'Depth averaged velocity in x-direction (meter / year)'
    nc%field_longname(nc%NDIM+5) = 'Depth averaged velocity in y-direction (meter / year)'

    nc%type(         1:nc%NVAR )    = nf90_double

    nc%case(                  1)    =    1 ! The dimension: X
    nc%case(                  2)    =    2 ! The dimension: Y
    nc%case(nc%NDIM            )    =    6 ! The dimension: time
    nc%case(nc%NDIM+1:nc%NDIM+5)    =    7 ! the NX, NY     , time case  [Default case]

    nc%LEN_DIM(1      )             = C%NX
    nc%LEN_DIM(2      )             = C%NY
    nc%LEN_DIM(nc%NDIM)             = nf90_unlimited

    nc%grid_size(1      )           = C%dx
    nc%grid_size(2      )           = C%dy
   !nc%grid_size(nc%NDIM)           = C%dt   ! but the time step can be dynamical: see G%dt
  END SUBROUTINE initialize_restart_fields_siadyn



  SUBROUTINE write_restart_fields_siadyn(nc, record_number, time, Hi, Hb, Hs, Us, Vs)
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type, handle_error
    USE netcdf, ONLY: nf90_put_var, nf90_sync
    IMPLICIT NONE

    ! Input variables:
    TYPE(netcdf_file_type),                                         INTENT(IN)         :: nc
    INTEGER,                                                        INTENT(IN)         :: record_number
    REAL(dp),                                                       INTENT(IN)         :: time
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hi
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hb
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Hs
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Us
    REAL(dp), DIMENSION(              nc%LEN_DIM(2),nc%LEN_DIM(1)), INTENT(IN)         :: Vs

    ! Local variables:
    INTEGER                                                                            :: m, n        ! Grid point counters
    INTEGER                                                                            :: i           ! Counter over the dimensions or the variables
    REAL(dp), DIMENSION(nc%NDIM+1:nc%NDIM+5,nc%LEN_DIM(1),nc%LEN_DIM(2)              ) :: var_data_2D ! The 2D data of the variables in one array

    ! Transpose:
    DO m = 1, nc%LEN_DIM(1)
    DO n = 1, nc%LEN_DIM(2) 
      var_data_2D(nc%NDIM+1,m,n  ) = Hi         (  n,m)
      var_data_2D(nc%NDIM+2,m,n  ) = Hb         (  n,m)
      var_data_2D(nc%NDIM+3,m,n  ) = Hs         (  n,m)
      var_data_2D(nc%NDIM+4,m,n  ) = Us         (  n,m)
      var_data_2D(nc%NDIM+5,m,n  ) = Vs         (  n,m)
    END DO           
    END DO

    ! Data writing mode:
    DO i = nc%NDIM, nc%N_loop
      SELECT CASE(nc%case(i))
      ! Write the data for the unlimited (time) dimension (the other dimension data is written by the subroutine create_netcdf_file):
      CASE(6)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ time /)          , start=(/         record_number/)), '. [1] From write_restart_fields_siadyn(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)

      ! Write the data of the variables:
      CASE(7)
       CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), var_data_2D(i,:,:  ), start=(/1, 1,    record_number/)), '. [2] From write_restart_fields_siadyn(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE DEFAULT
       ! Do nothing: Ignore variables with another mode
      END SELECT
    END DO

    ! Synchronize the disk copy of a netcdf dataset: to minimize data loss in case of an abnormal termination:
    IF(C%synchronize_netcdf_writing) CALL handle_error(nf90_sync(nc%ncid)                                       , '. [4] From write_restart_fields_siadyn(): while synchronizing the file: '//TRIM(nc%file_name))
  END SUBROUTINE write_restart_fields_siadyn
  


  SUBROUTINE read_restart_file_siadyn(file_name, record_number, Hi, Hb, Hs, Us, Vs, time)
    USE configuration_module, ONLY: dp, C
    USE netcdf_module, ONLY: netcdf_file_type, handle_error, open_netcdf_file, inquire_netcdf_file, count_netcdf_record_size, close_netcdf_file
    USE netcdf, ONLY: nf90_get_var
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*),                     INTENT(IN)            :: file_name
    INTEGER,                              INTENT(IN)            :: record_number

    ! Output variables:
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hi
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hb
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Hs
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Us
    REAL(dp), DIMENSION(      C%NY,C%NX), INTENT(OUT)           :: Vs
    REAL(dp),                             INTENT(OUT), OPTIONAL :: time

    ! Local variables:
    TYPE(netcdf_file_type)                                      :: nc      
    INTEGER                                                     :: m, n          ! Grid point counters
    INTEGER                                                     :: i             ! Counter over the dimensions or the variables
    INTEGER                                                     :: record_size
    INTEGER                                                     :: used_record   ! The record number which is used for reading
    REAL(dp), DIMENSION(:,:   ,:         ), ALLOCATABLE         :: T_var_2D

    ! Output: netcdf
    CALL initialize_restart_fields_siadyn(file_name, nc)

    ALLOCATE(T_var_2D(nc%NDIM+1:nc%NDIM+5,C%NX,C%NY     ))

    ! In/Output: netcdf
    CALL open_netcdf_file(nc) 

    ! In/Output: netcdf 
    CALL inquire_netcdf_file(nc)

    ! Output: record_size
    CALL count_netcdf_record_size(nc, record_size)

    IF(0 < record_number .AND. record_number <= record_size) THEN
     used_record = record_number
    ELSE IF(record_number == 0) THEN
     ! In this case the variables from the last record:
     used_record = record_size
    ELSE
     WRITE(UNIT=*,FMT='(A,I6,A)') 'ERROR: Record number ', record_number, ' does not exist in the file:  '//TRIM(nc%file_name)
     STOP
    END IF

    DO i = nc%NDIM, nc%NVAR
      SELECT CASE(nc%case(i))
       CASE(6)
       IF(PRESENT(time)) &
        CALL handle_error(nf90_get_var(nc%ncid, nc%id(i), time             , start=(/         used_record/) ), '. [1] From read_restart_file_siadyn(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(7)
        CALL handle_error(nf90_get_var(nc%ncid, nc%id(i), T_var_2D(i,:,:  ), start=(/1, 1,    used_record/) ), '. [2] From read_restart_file_siadyn(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE DEFAULT
        !  Do nothing: Ignore variables with another mode
      END SELECT
    END DO

    ! Finish and closing writing output:
    CALL close_netcdf_file(nc)

    ! Transpose the read fields from {x,y,*} to {*,y,x}:
    DO m = 1, C%NX
    DO n = 1, C%NY 
      Hi         (  n,m) = T_var_2D(nc%NDIM+1,m,n  )
      Hb         (  n,m) = T_var_2D(nc%NDIM+2,m,n  )
      Hs         (  n,m) = T_var_2D(nc%NDIM+3,m,n  )
      Us         (  n,m) = T_var_2D(nc%NDIM+4,m,n  )
      Vs         (  n,m) = T_var_2D(nc%NDIM+5,m,n  )
    END DO
    END DO

    DEALLOCATE(T_var_2D)
  END SUBROUTINE read_restart_file_siadyn
  
  
  
  SUBROUTINE linear_vertical_profile(fake_label, Hi, Hb, Hs, var_zeta, var_z)
    ! Calculating per i,j point the linear vertical profile from the vertical non equidistant scaled profile. 
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    INTEGER,                    INTENT(IN)  :: fake_label ! This label is used as a index, to enable optimal color settings for bedrock, atmosphere and ocean  
    REAL(dp),                   INTENT(IN)  :: Hi
    REAL(dp),                   INTENT(IN)  :: Hb
    REAL(dp),                   INTENT(IN)  :: Hs
    REAL(dp), DIMENSION(C%NZ),  INTENT(IN)  :: var_zeta   ! Containing the values of the variable for each zeta coordinate
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NZL), INTENT(OUT) :: var_z      ! Containing the values of the variable for each vertical z coordinate
    
    ! Local variables:
    INTEGER                                 :: k_zeta                 ! The integer k which belongs to the zeta-coordinate
    INTEGER                                 :: k_z                    ! The integer k which belongs to the z   -coordinate
    REAL(dp), DIMENSION(C%NZ)               :: z_value_of_zeta_layer  ! The z-coordinate value at the layer zeta
    REAL(dp)                                :: zz                     ! The z-coordinate
    REAL(dp)                                :: aa                     ! Distance between z and the upper layer at zeta(k_zeta)
    REAL(dp)                                :: bb                     ! Distance between z and the lower layer at zeta(k_zeta+1)
   
    ! Calculating for each zeta layer its z value:
    DO k_zeta = 1, C%NZ
      z_value_of_zeta_layer(k_zeta) = Hs - C%zeta(k_zeta) * Hi
    END DO
   
   !var_z = 1E30_dp
    
    ! Calculating for each linear vertical z-coordinate the value of the variable at that z:
    DO k_z = 1, C%NZL
      zz = C%lowest_z_of_cross_section + (k_z - 1) * C%z_step

      IF(zz <= Hb) THEN
       ! Situation where z is below the bedrock level: 
       var_z(k_z) = C%fake_value_for_bedrock(fake_label)
      ELSE IF(zz >= Hs .OR. ((Hi == C%Hi_min) .AND. (zz >= Hs - C%Hi_min))) THEN 
       ! Situation where z is above the Hs level (or only a minimum default ice layer is present): 
       var_z(k_z) = C%fake_value_for_atmosphere(fake_label)
      ELSE IF((Hs - Hi > Hb) .AND. (zz <= Hs - Hi)) THEN
       ! Situation where z is an ocean point, possibly between Hb and the floating shelf above: 
       var_z(k_z) = C%fake_value_for_ocean(fake_label)          
      ELSE
       ! Find the closest zeta layers to z:
       DO k_zeta = 1, C%NZ - 1
        IF(zz == z_value_of_zeta_layer(k_zeta)) THEN
         var_z(k_z) = var_zeta(k_zeta)
         EXIT
        ELSE IF((zz < z_value_of_zeta_layer(k_zeta)) .AND. (zz > z_value_of_zeta_layer(k_zeta + 1))) THEN
         aa = z_value_of_zeta_layer(k_zeta) - zz
         bb = zz - z_value_of_zeta_layer(k_zeta + 1)
         var_z(k_z) = (1._dp - aa / (aa + bb)) * var_zeta(k_zeta) + (1._dp - bb / (aa + bb))  * var_zeta(k_zeta + 1)
         EXIT
        END IF
       END DO
      END IF
      ! A check if always an appropiate zeta level is found:
     !IF(var_z(k_z) == 1E30_dp) WRITE(UNIT=*, FMT='(A)')  ' WARNING from linear_vertical_profile: appropiate zeta level fails.'
    END DO
  END SUBROUTINE linear_vertical_profile
  
  

  SUBROUTINE cross_section(fake_label, k, l, Hi, Hb, Hs, var, var_linear_x_cross_sec, var_linear_y_cross_sec, var_linear)
    ! Making a cross section for a given x or y of the variable var, var must be a 3 dimensional quantity.
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    INTEGER,                              INTENT(IN)            :: fake_label        ! This label is used as a index, to enable optimal color settings for bedrock, atmosphere and ocean  
    INTEGER,                              INTENT(IN)            :: k, l
    REAL(dp), DIMENSION(C%NX,C%NY     ),  INTENT(IN)            :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY     ),  INTENT(IN)            :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY     ),  INTENT(IN)            :: Hs
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZ),  INTENT(IN)            :: var               ! This is the variable for every zeta coordinate
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NX,     C%NZL), INTENT(OUT), OPTIONAL :: var_linear_x_cross_sec
    REAL(dp), DIMENSION(     C%NY,C%NZL), INTENT(OUT), OPTIONAL :: var_linear_y_cross_sec
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZL), INTENT(OUT), OPTIONAL :: var_linear        ! This is the variable for every z coordinate
    
    ! Local variables:
    INTEGER                                                     :: i, j              ! Grid point counters
    REAL(dp), DIMENSION(C%NZ)                                   :: var_zeta          ! Containing the values of the variable for each zeta coordinate
    REAL(dp), DIMENSION(C%NZL)                                  :: var_z             ! Containing the values of the variable for each vertical z coordinate
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZL)                        :: var_linear_local  ! This is the variable for every z coordinate

    ! Calculation of the complete x,y,z field of the temperature:
    DO i = 1, C%NX
    DO j = 1, C%NY
      var_zeta = var(i,j,:)

      ! Output: var_z
      CALL linear_vertical_profile(fake_label, Hi(i,j), Hb(i,j), Hs(i,j), var_zeta, var_z)
      
      var_linear_local(i,j,:) = var_z
    END DO
    END DO
    
    ! Cross section paralel to the x-axis, so Y is fixed at one j = l value:
    IF(PRESENT(var_linear_x_cross_sec)) var_linear_x_cross_sec(:,:) = var_linear_local(:,l,:)

    ! Cross section paralel to the y-axis, so X is fixed at one i = k value:
    IF(PRESENT(var_linear_y_cross_sec)) var_linear_y_cross_sec(:,:) = var_linear_local(k,:,:)
    
    ! Copying the full 3D field on the x,y,z-coordinates to the optional Output variables:
    IF(PRESENT(var_linear)) var_linear = var_linear_local
  END SUBROUTINE cross_section              

END MODULE restart_file_module
