! File name: netcdf_module.f90
!
! Copyright (C) 2014 Thomas Reerink
!
! This file is part of the ICEDYN-package
!
! IMAU, Utrecht University, The Netherlands
!

MODULE netcdf_module
  USE configuration_module, ONLY: dp
  IMPLICIT NONE

  TYPE netcdf_file_type
    ! This TYPE contains the meta data and the data of the dimensions and the variables:
    INTEGER                                       :: ncid             ! The id of the netcdf file:  file_id
    INTEGER                                       :: number_of_fields ! The number of fields
    CHARACTER(LEN=256)                            :: file_name        ! The name of the netcdf file
    CHARACTER(LEN=128), DIMENSION(:), ALLOCATABLE :: field_name       ! The name of the field variable
    CHARACTER(LEN=128), DIMENSION(:), ALLOCATABLE :: field_unit       ! The units of the variable 
    CHARACTER(LEN=256), DIMENSION(:), ALLOCATABLE :: field_longname   ! The long name of the field variable (with units)
    INTEGER,            DIMENSION(:), ALLOCATABLE :: id               ! The ID number of the variable 
    INTEGER,            DIMENSION(:), ALLOCATABLE :: type             ! An integer determing the variable type: nf90_int or nf90_double
    INTEGER,            DIMENSION(:), ALLOCATABLE :: case             ! A number for each situation, for case selection

    INTEGER,            DIMENSION(:), ALLOCATABLE :: LEN_DIM          ! Contains the dimensional size of each dimension
    INTEGER                                       :: NDIM             ! The number of the amount of dimensions
    INTEGER                                       :: NVAR             ! The number of the amount of different variables
    INTEGER                                       :: N_loop           ! The number of loops in which the dimensions + variables are written to the netcdf file
    REAL(dp),           DIMENSION(:), ALLOCATABLE :: grid_size        ! The grid size for each dimension (Only if it is equidistant)
  END TYPE netcdf_file_type



CONTAINS
  SUBROUTINE handle_error(stat, message, message_counter)
    USE configuration_module, ONLY: C
    USE netcdf, ONLY: nf90_noerr, nf90_strerror
    IMPLICIT NONE

    ! Input variables:
    INTEGER,                    INTENT(IN) :: stat
    CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: message
    INTEGER,          OPTIONAL, INTENT(IN) :: message_counter

    IF(stat /= nf90_noerr) THEN
     IF(PRESENT(message) .AND. PRESENT(message_counter)) THEN
      WRITE(UNIT=*,FMT='(4A, I3/, A/)') C%ERROR, ' netcdf failed because: ', TRIM(nf90_strerror(stat)), message, message_counter, '        The used config file is: '//TRIM(C%config_filename)
     ELSE IF(PRESENT(message)) THEN
      WRITE(UNIT=*,FMT='(4A/    , A/)') C%ERROR, ' netcdf failed because: ', TRIM(nf90_strerror(stat)), message                 , '        The used config file is: '//TRIM(C%config_filename)
     ELSE
      WRITE(UNIT=*,FMT='(3A/    , A/)') C%ERROR, ' netcdf failed because: ', TRIM(nf90_strerror(stat))                          , '        The used config file is: '//TRIM(C%config_filename)
     END IF
     STOP
    END IF
  END SUBROUTINE handle_error
    


  SUBROUTINE check_file_existence(file_name)
    USE configuration_module, ONLY: C
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*), INTENT(IN) :: file_name

    ! Local variables:
    LOGICAL                      :: file_exists

    ! Continue or not if file exists:
    INQUIRE(EXIST = file_exists, FILE = file_name)
    IF(C%protect_file_overwriting .AND. file_exists) THEN
     WRITE(UNIT=*,FMT='(5A)') 'ERROR: The file "', TRIM(file_name), '" already exists! Remove it and run again!'
     STOP
    END IF
  END SUBROUTINE check_file_existence



  SUBROUTINE open_netcdf_file(nc) 
    USE netcdf, ONLY: nf90_open, nf90_write, nf90_share
    IMPLICIT NONE

    ! In/Output variables:
    TYPE(netcdf_file_type), INTENT(INOUT) :: nc
 
    ! Open netcdf file:
    CALL handle_error(nf90_open(nc%file_name, IOR(nf90_write,nf90_share), nc%ncid), '. The file '//TRIM(nc%file_name)//' is not found')
  END SUBROUTINE open_netcdf_file



  SUBROUTINE close_netcdf_file(nc)
    USE netcdf, ONLY: nf90_close
    IMPLICIT NONE
  
    ! In/Output variables:
    TYPE(netcdf_file_type), INTENT(INOUT) :: nc

    ! Close netcdf file:
    CALL handle_error(nf90_close(nc%ncid), '. From close_netcdf_file(): it concerns the file '//TRIM(nc%file_name))

    DEALLOCATE(nc%field_name    )
    DEALLOCATE(nc%field_unit    )
    DEALLOCATE(nc%field_longname)
    DEALLOCATE(nc%id            )
    DEALLOCATE(nc%type          )
    DEALLOCATE(nc%case          )
    DEALLOCATE(nc%LEN_DIM       )
    DEALLOCATE(nc%grid_size     )
  END SUBROUTINE close_netcdf_file



  SUBROUTINE inquire_dimension(i, nc)
    ! Inquire the id of a dimension and check that the length matches the length given by the user.
    USE netcdf, ONLY: nf90_inq_dimid, nf90_unlimited, nf90_inquire_dimension, nf90_INQUIRE
    IMPLICIT NONE
  
    ! Input variables:
    INTEGER,                INTENT(IN)    :: i

    ! In/Output variables:
    TYPE(netcdf_file_type), INTENT(INOUT) :: nc

    ! Local variables:
    INTEGER                               :: actual_length
    INTEGER                               :: id_of_unlimited_dimension

    CALL handle_error(nf90_inq_dimid(nc%ncid, nc%field_name(i), nc%id(i)), '. From inquire_dimension(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)

    IF(nc%LEN_DIM(i) /= nf90_unlimited) THEN
     CALL handle_error(nf90_inquire_dimension(nc%ncid, nc%id(i), len=actual_length))
     IF(nc%LEN_DIM(i) /= actual_length) THEN
      WRITE(UNIT=*,FMT='(3A,I5,3A,I5,A)') 'ERROR: Wrong dimensional size: ', TRIM(nc%field_name(i)), '(', actual_length, ') does not match the required ', TRIM(nc%field_name(i)), '(', nc%LEN_DIM(i), ')'
      STOP 'by inquire_dimension()'
     END IF
    ELSE
     CALL handle_error(nf90_INQUIRE(nc%ncid, unlimitedDimId=id_of_unlimited_dimension))
     IF(nc%id(i) /= id_of_unlimited_dimension) THEN
      WRITE(UNIT=*,FMT='(3A)') 'ERROR: The dimension of"', TRIM(nc%field_name(i)),'" is not unlimited as required.'
      STOP
     END IF
    END IF
  END SUBROUTINE inquire_dimension


  
  SUBROUTINE inquire_variable(i, id_dims, nc)
    ! Inquire the id of a variable and check that the dimensions of the variable match the dimensions given by the user and
    ! that the variable is of the correct type.
    USE netcdf, ONLY: nf90_inq_varid, nf90_inquire_variable, nf90_int, nf90_double
    IMPLICIT NONE
  
    ! Input variables:
    INTEGER,                INTENT(IN)    :: i
    INTEGER, DIMENSION(:),  INTENT(IN)    :: id_dims

    ! In/Output variables:
    TYPE(netcdf_file_type), INTENT(INOUT) :: nc

    ! Local variables:
    INTEGER                               :: actual_var_type
    INTEGER                               :: ndims

    CALL handle_error(nf90_inq_varid(nc%ncid, nc%field_name(i), nc%id(i)), '. From inquire_variable(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
    CALL handle_error(nf90_inquire_variable(nc%ncid, nc%id(i), xtype=actual_var_type, ndims=ndims))
    IF(nc%type(i) /= actual_var_type) THEN
      IF(nc%type(i) == nf90_int   ) WRITE(UNIT=*,FMT='(3A)') 'ERROR: The variable "', TRIM(nc%field_name(i)),'" should be of integer type.'
      IF(nc%type(i) == nf90_double) WRITE(UNIT=*,FMT='(3A)') 'ERROR: The variable "', TRIM(nc%field_name(i)),'" should be of double type.'
      STOP
    END IF
    IF(ndims /= SIZE(id_dims)) THEN
      WRITE(UNIT=*,FMT='(A,I5,3A,I5,2A)') 'ERROR: The actual number of the dimensions(',ndims, ') of variable "', TRIM(nc%field_name(i)), &
                                          '" does not match the required number of dimensions (',SIZE(id_dims),'), in the file: ', TRIM(nc%file_name)
      STOP
    END IF
  END SUBROUTINE inquire_variable



  SUBROUTINE count_netcdf_record_size(nc, record_size)
    ! Count the number of records in a netcdf file:
    USE netcdf, ONLY: nf90_inq_dimid, nf90_inquire_dimension
    IMPLICIT NONE
    
    ! Input variables:
    TYPE(netcdf_file_type), INTENT(IN)  :: nc

    ! Output variables:
    INTEGER,                INTENT(OUT) :: record_size

    ! Local variables:
    INTEGER                             :: dim_t
    
    CALL handle_error(nf90_inq_dimid(nc%ncid, nc%field_name(nc%NDIM), dim_t)            ,  '. From count_netcdf_record_size(): the field "'//TRIM(nc%field_name(nc%NDIM))//'" has an invalid ID.')
    CALL handle_error(nf90_inquire_dimension(ncid=nc%ncid, dimid=dim_t, len=record_size),  '. From count_netcdf_record_size(): the field "'//TRIM(nc%field_name(nc%NDIM))//'" has an invalid dimension')
  END SUBROUTINE count_netcdf_record_size



  SUBROUTINE create_netcdf_file(nc, optional_label_choice) 
    USE configuration_module, ONLY: C
    USE netcdf, ONLY: nf90_create, nf90_clobber, nf90_def_dim, nf90_def_var, nf90_put_att, nf90_enddef, nf90_put_var, nf90_sync
    IMPLICIT NONE

    ! In/Output variables:
    TYPE(netcdf_file_type), INTENT(INOUT)          :: nc

    ! Input variables:
    LOGICAL,                INTENT(IN),   OPTIONAL :: optional_label_choice

    ! Local variables:
    INTEGER                                        :: i                               ! Counter over the dimensions or the variables
    INTEGER                                        :: m                               ! Counter over the dimension data
    CHARACTER(LEN=256)                             :: field_longname                  ! The long name, including the field number
    LOGICAL                                        :: label_netcdf_fields_with_number

    CALL check_file_existence(nc%file_name)
    
    label_netcdf_fields_with_number = C%label_netcdf_fields_with_number
    IF(PRESENT(optional_label_choice)) label_netcdf_fields_with_number = optional_label_choice

    ! Create the netcdf file:
    CALL handle_error(nf90_create(nc%file_name, nf90_clobber, nc%ncid), '. [1] From create_netcdf_file(): it concerns the file: '//TRIM(nc%file_name))

    ! Definition mode:
    DO i = 1, nc%N_loop
      SELECT CASE(nc%case(i))
      ! Determine the dimension id's and the var id's of the dimensions:
      CASE(1:6)
       CALL handle_error(nf90_def_dim(nc%ncid, nc%field_name(i), nc%LEN_DIM(i),   nc%id(i)                                             ), '. [ 2] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CALL handle_error(nf90_def_var(nc%ncid, nc%field_name(i), nc%type(i)   , (/nc%id(i)                                 /), nc%id(i)), '. [ 3] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)

      ! Determine the var id's of the variables:
      CASE(7)
       CALL handle_error(nf90_def_var(nc%ncid, nc%field_name(i), nc%type(i), (/nc%id(1), nc%id(2),           nc%id(nc%NDIM)/), nc%id(i)), '. [ 4] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(8)
       CALL handle_error(nf90_def_var(nc%ncid, nc%field_name(i), nc%type(i), (/nc%id(1), nc%id(2), nc%id(3), nc%id(nc%NDIM)/), nc%id(i)), '. [ 5] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(10)
       CALL handle_error(nf90_def_var(nc%ncid, nc%field_name(i), nc%type(i), (/nc%id(1), nc%id(2), nc%id(4), nc%id(nc%NDIM)/), nc%id(i)), '. [ 6] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(11)
       CALL handle_error(nf90_def_var(nc%ncid, nc%field_name(i), nc%type(i), (/nc%id(1), nc%id(4),           nc%id(nc%NDIM)/), nc%id(i)), '. [ 7] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(12)
       CALL handle_error(nf90_def_var(nc%ncid, nc%field_name(i), nc%type(i), (/nc%id(2), nc%id(4),           nc%id(nc%NDIM)/), nc%id(i)), '. [ 8] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(13)
       CALL handle_error(nf90_def_var(nc%ncid, nc%field_name(i), nc%type(i), (/                              nc%id(nc%NDIM)/), nc%id(i)), '. [ 9] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE(20)
       CALL handle_error(nf90_def_var(nc%ncid, nc%field_name(i), nc%type(i), (/nc%id(1), nc%id(2)                          /), nc%id(i)), '. [10] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
      CASE DEFAULT
       ! Do nothing: Ignore variables with another mode
      END SELECT
      IF(nc%case(i) /= 1000) THEN
       ! In case the description equals still the initial dummy default, a neater one is generated:
       IF(nc%field_longname(i) == 'longname: ?') nc%field_longname(i) = TRIM(nc%field_name(i))//' ('//TRIM(nc%field_unit(i))//')'

       IF(label_netcdf_fields_with_number) THEN
        WRITE(field_longname, FMT='(A, I4, 2A)') 'Field No', i - nc%NDIM, ':  ', TRIM(nc%field_longname(i))
        CALL handle_error(nf90_put_att(nc%ncid, nc%id(i), 'long_name', TRIM(field_longname)      ), '. [11] From create_netcdf_file(): long_name definition, in the file '//TRIM(nc%file_name)//', i = ', i)
       ELSE
        CALL handle_error(nf90_put_att(nc%ncid, nc%id(i), 'long_name', TRIM(nc%field_longname(i))), '. [12] From create_netcdf_file(): long_name definition, in the file '//TRIM(nc%file_name)//', i = ', i)
       END IF
       CALL handle_error( nf90_put_att(nc%ncid, nc%id(i), 'units'    , TRIM(nc%field_unit(i))    ), '. [13] From create_netcdf_file(): units definition, in the file '//TRIM(nc%file_name)//', i = ', i)
      END IF
    END DO
    CALL handle_error(nf90_enddef(nc%ncid)                                                        , '. [14] From create_netcdf_file(): end definition mode, in the file '//TRIM(nc%file_name))

    ! Write data for limited dimensions:
    IF(C%use_grid_units_in_dimension) THEN
     ! The units of the dimensions are in grid units:
     DO i = 1, nc%NDIM-1
       SELECT CASE(nc%case(i))
       CASE(1)
        CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ (                                        m,m=1,nc%LEN_DIM(i)) /) ), '. [15] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(2)
        CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ (                                        m,m=1,nc%LEN_DIM(i)) /) ), '. [16] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(3)
        CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ (                                        m,m=1,nc%LEN_DIM(i)) /) ), '. [17] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(5)
        CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ (                                        m,m=1,nc%LEN_DIM(i)) /) ), '. [18] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE DEFAULT
        ! Do nothing: Ignore variables with another mode
      END SELECT
     END DO
    ELSE 
     ! The units of the dimensions are in physical units:
     DO i = 1, nc%NDIM-1
       SELECT CASE(nc%case(i))
       CASE(1)
        CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ (((m - (nc%LEN_DIM(i) + 1) / 2) * nc%grid_size(i)),m=1,nc%LEN_DIM(i)) /) ), '. [19] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(2)
        CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ (((m - (nc%LEN_DIM(i) + 1) / 2) * nc%grid_size(i)),m=1,nc%LEN_DIM(i)) /) ), '. [20] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(3)
        CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ C%zeta                                                                /) ), '. [21] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE(5)
        CALL handle_error(nf90_put_var(nc%ncid, nc%id(i), (/ (C%lowest_z_of_cross_section + m * nc%grid_size(i),m=1,nc%LEN_DIM(i)) /) ), '. [22] From create_netcdf_file(): it concerns the field "'//TRIM(nc%field_name(i))//'" in the file '//TRIM(nc%file_name)//', i = ', i)
       CASE DEFAULT
        ! Do nothing: Ignore variables with another mode
      END SELECT
     END DO
    END IF
    ! The unlimited time dimension  t  is written and not defined here.

    ! Synchronize the disk copy of a netcdf dataset: to minimize data loss in case of an abnormal termination:
    IF(C%synchronize_netcdf_writing) CALL handle_error(nf90_sync(nc%ncid), '. [23] From create_netcdf_file(): while synchronizing the file: '//TRIM(nc%file_name))
  END SUBROUTINE create_netcdf_file



  SUBROUTINE inquire_netcdf_file(nc) 
    ! Check if the right dimensions are present in the restart file, and if the variables have the right
    ! dimension: specie and length.
    IMPLICIT NONE
    
    ! In/Output variables:
    TYPE(netcdf_file_type), INTENT(INOUT) :: nc
 
    ! Local variables:
    INTEGER                               :: i           ! Counter over the dimensions or the variables

    DO i = nc%NDIM, nc%N_loop
      SELECT CASE(nc%case(i))
      ! Dimensions:
      CASE(1:6)
       CALL inquire_dimension(i, nc)
       CALL inquire_variable(i, (/ nc%id(i) /), nc)
      CASE(7)
       CALL inquire_variable(i, (/ nc%id(1), nc%id(2),           nc%id(nc%NDIM) /), nc)
      CASE(8)
       CALL inquire_variable(i, (/ nc%id(1), nc%id(2), nc%id(3), nc%id(nc%NDIM) /), nc)
      CASE(9)
       CALL inquire_variable(i, (/ nc%id(1), nc%id(2), nc%id(4), nc%id(nc%NDIM) /), nc)
      CASE(20)
       CALL inquire_variable(i, (/ nc%id(1), nc%id(2)                           /), nc)
      CASE DEFAULT
       ! Do nothing: Ignore variables with another mode
      END SELECT
    END DO
  END SUBROUTINE inquire_netcdf_file

END MODULE netcdf_module
