! File name: mask_module.f90
!
! Copyright (C) 2016 Thomas Reerink
!
! This file is part of the ICEDYN-package
!
! IMAU, Utrecht University, The Netherlands
!

MODULE mask_module

CONTAINS
  SUBROUTINE determine_mask(sealevel, Hi, Hb, mask, calve_mask)
    ! This routine calculates a new mask for ALL points:
    ! A shelf/ocean point and a sheet/rock point can NEVER be neighbours, since a
    ! grounded point that has a shelf/ocean point as neighbour is by definition a
    ! groundline point.
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)            :: sealevel
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)            :: Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)            :: Hb

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT)           :: mask
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: calve_mask

    ! Local variables:
    INTEGER                                               :: i, j

    ! Distinguish between shelf/ocean and sheet/rock points (by the combined
    ! ice floating condition and the sealevel-bedrock level comparison):
    WHERE((C%ice_density / C%seawater_density) * Hi <= (sealevel - Hb))
     mask = C%type_shelf
    ELSEWHERE
     mask = C%type_sheet
    END WHERE

    ! Mark sheet points at the domain boundary always as groundline:
    WHERE(mask(   1,   :) == C%type_sheet) mask(   1,   :) = C%type_groundline
    WHERE(mask(C%NY,   :) == C%type_sheet) mask(C%NY,   :) = C%type_groundline
    WHERE(mask(   :,   1) == C%type_sheet) mask(   :,   1) = C%type_groundline
    WHERE(mask(   :,C%NX) == C%type_sheet) mask(   :,C%NX) = C%type_groundline

    ! Marking the rest of the sheet points that border to shelf points
    ! as groundline points:
    DO i = 2, C%NX - 1
    DO j = 2, C%NY - 1
      IF(mask(j,i) == C%type_sheet) THEN
       IF(mask(j+1,i  ) == C%type_shelf .OR. mask(j-1,i  ) == C%type_shelf .OR. &
          mask(j+1,i+1) == C%type_shelf .OR. mask(j-1,i-1) == C%type_shelf .OR. &
          mask(j-1,i+1) == C%type_shelf .OR. mask(j+1,i-1) == C%type_shelf .OR. &
          mask(j  ,i+1) == C%type_shelf .OR. mask(j  ,i-1) == C%type_shelf) THEN
        mask(j,i) = C%type_groundline
       END IF
      END IF
    END DO
    END DO

    ! Shelf points with a thickness less then the calving thickness become ocean type, and a
    ! calving line is made between shelf and ocean points:
    IF(PRESENT(calve_mask)) THEN
     calve_mask = mask
     WHERE((calve_mask == C%type_shelf) .AND. (Hi < C%calve_thickness_shelf)) calve_mask = C%type_ocean
     DO i = 2, C%NX - 1
     DO j = 2, C%NY - 1
       IF(calve_mask(j,i) == C%type_shelf) THEN
        IF(calve_mask(j+1,i  ) == C%type_ocean .OR. calve_mask(j-1,i  ) == C%type_ocean .OR. &
           calve_mask(j+1,i+1) == C%type_ocean .OR. calve_mask(j-1,i-1) == C%type_ocean .OR. &
           calve_mask(j-1,i+1) == C%type_ocean .OR. calve_mask(j+1,i-1) == C%type_ocean .OR. &
           calve_mask(j  ,i+1) == C%type_ocean .OR. calve_mask(j  ,i-1) == C%type_ocean) THEN
         calve_mask(j,i) = C%type_calveline
        END IF
       END IF
     END DO
     END DO
    END IF
  END SUBROUTINE determine_mask



  SUBROUTINE determine_x_discretization_mask(mask, x_mask)
    ! If for example a derivative field has to be discretized in the x-direction it has to be known if the
    ! neighbours are available (near the domain border) and if it is desired to use them (at the 'wrong'
    ! side of the grounding line for example if there are no values for a certain field at the shelf).
    ! A bunch of situations occur: a central discretization, a one sided in the left or right direction,
    ! only two points, or even only a single groundline point, all of this kind of situations near the
    ! domain border. To distiguish between all this situations this subroutine creates a special mask.
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    LOGICAL,  DIMENSION(C%NY,C%NX), INTENT(IN)  :: mask    ! TRUE corresponds with active area, FALSE with inactive area

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: x_mask

    ! Local variables:
    INTEGER                                     :: i, j

    i = 1
    DO j = 1, C%NY
      IF(mask(j,i) .EQV. .FALSE.) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i+1) .EQV. .FALSE.) THEN
       x_mask(j,i) = C%mask_left_edge_single_point
      ELSE IF(mask(j,i+1) .AND. mask(j,i+2)) THEN
       x_mask(j,i) = C%mask_left_edge_point
      ELSE IF(mask(j,i+1) .AND. (mask(j,i+2) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 3I2)') 'Unkown situation in determine_x_discretization_mask() at i=', i, ': mask(j=', j, ', i=', i, ':', i+2, ') = ', mask(j, i:i+2)
      END IF
    END DO

    i = 2
    DO j = 1, C%NY
      IF(mask(j,i) .EQV. .FALSE.) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i-1) .AND. mask(j,i+1)) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) .AND. mask(j,i+1)) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF((mask(j,i-1) .EQV. .FALSE.) .AND. (mask(j,i+1) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_single_point
      ELSE IF((mask(j,i-1) .EQV. .FALSE.) .AND. mask(j,i+1) .AND. mask(j,i+2)) THEN
       x_mask(j,i) = C%mask_left_edge_point
      ELSE IF((mask(j,i-1) .EQV. .FALSE.) .AND. mask(j,i+1) .AND. (mask(j,i+2) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE IF((mask(j,i+1) .EQV. .FALSE.) .AND. mask(j,i-1)) THEN
       x_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 4I2)') 'Unkown situation in determine_x_discretization_mask() at i=', i, ': mask(j=', j, ', i=', i-1, ':', i+2, ') = ', mask(j, i-1:i+2)
      END IF
    END DO

    DO i = 3, C%NX - 2
    DO j = 1, C%NY
      IF(mask(j,i) .EQV. .FALSE.) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i-1) .AND. mask(j,i+1)) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) .AND. mask(j,i+1)) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF((mask(j,i-1) .EQV. .FALSE.) .AND. (mask(j,i+1) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_single_point
      ELSE IF((mask(j,i-1) .EQV. .FALSE.) .AND. mask(j,i+1) .AND. mask(j,i+2)) THEN
       x_mask(j,i) = C%mask_left_edge_point
      ELSE IF((mask(j,i-1) .EQV. .FALSE.) .AND. mask(j,i+1) .AND. (mask(j,i+2) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE IF((mask(j,i+1) .EQV. .FALSE.) .AND. mask(j,i-1) .AND. mask(j,i-2)) THEN
       x_mask(j,i) = C%mask_right_edge_point
      ELSE IF((mask(j,i+1) .EQV. .FALSE.) .AND. mask(j,i-1) .AND. (mask(j,i-2) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 5I2)') 'Unkown situation in determine_x_discretization_mask() at i=', i, ': mask(j=', j, ', i=', i-2, ':', i+2, ') = ', mask(j, i-2:i+2)
      END IF
    END DO
    END DO

    i = C%NX - 1
    DO j = 1, C%NY
      IF(mask(j,i) .EQV. .FALSE.) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i-1) .AND. mask(j,i+1)) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) .AND. mask(j,i+1)) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF((mask(j,i-1) .EQV. .FALSE.) .AND. (mask(j,i+1) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_single_point
      ELSE IF((mask(j,i+1) .EQV. .FALSE.) .AND. mask(j,i-1) .AND. mask(j,i-2)) THEN
       x_mask(j,i) = C%mask_right_edge_point
      ELSE IF((mask(j,i+1) .EQV. .FALSE.) .AND. mask(j,i-1) .AND. (mask(j,i-2) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE IF((mask(j,i-1) .EQV. .FALSE.) .AND. mask(j,i+1)) THEN
       x_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 4I2)') 'Unkown situation in determine_x_discretization_mask() at i=', i, ': mask(j=', j, ', i=', i-2, ':', i+1, ') = ', mask(j, i-2:i+1)
      END IF
    END DO

    i = C%NX
    DO j = 1, C%NY
      IF(mask(j,i) .EQV. .FALSE.) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i-1) .EQV. .FALSE.) THEN
       x_mask(j,i) = C%mask_right_edge_single_point
      ELSE IF(mask(j,i-1) .AND. mask(j,i-2)) THEN
       x_mask(j,i) = C%mask_right_edge_point
      ELSE IF(mask(j,i-1) .AND. (mask(j,i-2) .EQV. .FALSE.)) THEN
       x_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 3I2)') 'Unkown situation in determine_x_discretization_mask() at i=', i, ': mask(j=', j, ', i=', i-2, ':', i, ') = ', mask(j, i-2:i)
      END IF
    END DO
  END SUBROUTINE determine_x_discretization_mask



  SUBROUTINE determine_y_discretization_mask(mask, y_mask)
    ! If for example a derivative field has to be discretized in the y-direction it has to be known if the
    ! neighbours are available (near the domain border) and if it is desired to use them (at the 'wrong'
    ! side of the grounding line for example if there are no values for a certain field at the shelf).
    ! A bunch of situations occur: a central discretization, a one sided in the left or right direction,
    ! only two points, or even only a single groundline point, all of this kind of situations near the
    ! domain border. To distiguish between all this situations this subroutine creates a special mask.
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    LOGICAL,  DIMENSION(C%NY,C%NX), INTENT(IN)  :: mask    ! TRUE corresponds with active area, FALSE with inactive area

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: y_mask

    ! Local variables:
    INTEGER                                     :: i, j

    j = 1
    DO i = 1, C%NX
      IF(mask(j,i) .EQV. .FALSE.) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j+1,i) .EQV. .FALSE.) THEN
       y_mask(j,i) = C%mask_left_edge_single_point
      ELSE IF(mask(j+1,i)  .AND. mask(j+2,i) ) THEN
       y_mask(j,i) = C%mask_left_edge_point
      ELSE IF(mask(j+1,i)  .AND. (mask(j+2,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 3I2)') 'Unkown situation in determine_y_discretization_mask() at j=', j, ': mask(j=', j, ':', j+2, 'i=', i, ', ) = ', mask(j:j+2, i)
      END IF
    END DO

    j = 2
    DO i = 1, C%NX
      IF(mask(j,i) .EQV. .FALSE.) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j-1,i) .AND. mask(j+1,i)) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i)  .AND. mask(j+1,i) ) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF((mask(j-1,i) .EQV. .FALSE.) .AND. (mask(j+1,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_single_point
      ELSE IF((mask(j-1,i) .EQV. .FALSE.) .AND. mask(j+1,i)  .AND. mask(j+2,i) ) THEN
       y_mask(j,i) = C%mask_left_edge_point
      ELSE IF((mask(j-1,i) .EQV. .FALSE.) .AND. mask(j+1,i)  .AND. (mask(j+2,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_left_edge_only_two_points
        ELSE IF((mask(j+1,i) .EQV. .FALSE.) .AND. mask(j-1,i) ) THEN
         y_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 4I2)') 'Unkown situation in determine_y_discretization_mask() at j=', j, ': mask(j=', j-1, ':', j+2, 'i=', i, ', ) = ', mask(j-1:j+2, i)
      END IF
    END DO

    DO j = 3, C%NY - 2
    DO i = 1, C%NX
      IF(mask(j,i) .EQV. .FALSE.) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j-1,i) .AND. mask(j+1,i)) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i)  .AND. mask(j+1,i) ) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF((mask(j-1,i) .EQV. .FALSE.) .AND. (mask(j+1,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_single_point
      ELSE IF((mask(j-1,i) .EQV. .FALSE.) .AND. mask(j+1,i) .AND. mask(j+2,i) ) THEN
       y_mask(j,i) = C%mask_left_edge_point
      ELSE IF((mask(j-1,i) .EQV. .FALSE.) .AND. mask(j+1,i) .AND. (mask(j+2,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE IF((mask(j+1,i) .EQV. .FALSE.) .AND. mask(j-1,i) .AND. mask(j-2,i) ) THEN
       y_mask(j,i) = C%mask_right_edge_point
      ELSE IF((mask(j+1,i) .EQV. .FALSE.) .AND. mask(j-1,i) .AND. (mask(j-2,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 5I2)') 'Unkown situation in determine_y_discretization_mask() at j=', j, ': mask(j=', j-2, ':', j+2, 'i=', i, ', ) = ', mask(j-2:j+2, i)
      END IF
    END DO
    END DO

    j = C%NY - 1
    DO i = 1, C%NX
      IF(mask(j,i) .EQV. .FALSE.) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j-1,i) .AND. mask(j+1,i)) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i)  .AND. mask(j+1,i) ) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF((mask(j-1,i) .EQV. .FALSE.) .AND. (mask(j+1,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_single_point
      ELSE IF((mask(j+1,i) .EQV. .FALSE.) .AND. mask(j-1,i)  .AND. mask(j-2,i) ) THEN
       y_mask(j,i) = C%mask_right_edge_point
      ELSE IF((mask(j+1,i) .EQV. .FALSE.) .AND. mask(j-1,i)  .AND. (mask(j-2,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_right_edge_only_two_points
        ELSE IF((mask(j-1,i) .EQV. .FALSE.) .AND. mask(j+1,i) ) THEN
         y_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 4I2)') 'Unkown situation in determine_y_discretization_mask() at j=', j, ': mask(j=', j-2, ':', j+1, 'i=', i, ', ) = ', mask(j-2:j+1, i)
      END IF
    END DO

    j = C%NY
    DO i = 1, C%NX
      IF(mask(j,i) .EQV. .FALSE.) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j-1,i) .EQV. .FALSE.) THEN
       y_mask(j,i) = C%mask_right_edge_single_point
      ELSE IF(mask(j-1,i)  .AND. mask(j-2,i) ) THEN
       y_mask(j,i) = C%mask_right_edge_point
      ELSE IF(mask(j-1,i)  .AND. (mask(j-2,i) .EQV. .FALSE.)) THEN
       y_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 3I2)') 'Unkown situation in determine_y_discretization_mask() at j=', j, ': mask(j=', j-2, ':', j, 'i=', i, ', ) = ', mask(j-2:j, i)
      END IF
    END DO
  END SUBROUTINE determine_y_discretization_mask



  ! ALL ROUTINES BELOW ARE CURRENTLY ONLY USED WITH THE ICEDYN_PROGRAM



  ! Almost equal to determine_x_discretization_mask: C%type_shelf -> C%mask_inactive and C%type_sheet -> C%mask_central_point
  SUBROUTINE determine_x_discretization_mask_for_calving_edges(mask, x_mask)
    ! If for example a derivative field has to be discretized in the x-direction it has to be known if the
    ! neighbours are available (near the domain border) and if it is desired to use them (at the 'wrong'
    ! side of calving edge).
    ! A bunch of situations occur: a central discretization, a one sided in the left or right direction,
    ! only two points, or even only a single groundline point, all of this kind of situations near the
    ! domain border. To distiguish between all this situations this subroutine creates a special mask.
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(IN)  :: mask   ! the hi_x_mask

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: x_mask

    ! Local variables:
    INTEGER                                     :: i, j

    i = 1
    DO j = 1, C%NY
      IF(mask(j,i) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i+1) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_left_edge_single_point
      ELSE IF(mask(j,i+1) /= C%mask_inactive .AND. mask(j,i+2) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_left_edge_point
      ELSE IF(mask(j,i+1) /= C%mask_inactive .AND. mask(j,i+2) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 3I2)') 'Unkown situation in determine_x_discretization_mask_for_calving_edges() at i=', i, ': mask(j=', j, ', i=', i, ':', i+2, ') = ', mask(j, i:i+2)
      END IF
    END DO

    i = 2
    DO j = 1, C%NY
      IF(mask(j,i) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i  ) == C%mask_central_point) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) /= C%mask_inactive .AND. mask(j,i+1) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) == C%mask_inactive .AND. mask(j,i+1) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_single_point
      ELSE IF(mask(j,i-1) == C%mask_inactive .AND. mask(j,i+1) /= C%mask_inactive .AND. mask(j,i+2) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_left_edge_point
      ELSE IF(mask(j,i-1) == C%mask_inactive .AND. mask(j,i+1) /= C%mask_inactive .AND. mask(j,i+2) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE IF(mask(j,i+1) == C%mask_inactive .AND. mask(j,i-1) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 4I2)') 'Unkown situation in determine_x_discretization_mask_for_calving_edges() at i=', i, ': mask(j=', j, ', i=', i-1, ':', i+2, ') = ', mask(j, i-1:i+2)
      END IF
    END DO

    DO i = 3, C%NX - 2
    DO j = 1, C%NY
      IF(mask(j,i) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i  ) == C%mask_central_point) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) /= C%mask_inactive .AND. mask(j,i+1) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) == C%mask_inactive .AND. mask(j,i+1) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_single_point
      ELSE IF(mask(j,i-1) == C%mask_inactive .AND. mask(j,i+1) /= C%mask_inactive .AND. mask(j,i+2) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_left_edge_point
      ELSE IF(mask(j,i-1) == C%mask_inactive .AND. mask(j,i+1) /= C%mask_inactive .AND. mask(j,i+2) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE IF(mask(j,i+1) == C%mask_inactive .AND. mask(j,i-1) /= C%mask_inactive .AND. mask(j,i-2) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_right_edge_point
      ELSE IF(mask(j,i+1) == C%mask_inactive .AND. mask(j,i-1) /= C%mask_inactive .AND. mask(j,i-2) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 5I2)') 'Unkown situation in determine_x_discretization_mask_for_calving_edges() at i=', i, ': mask(j=', j, ', i=', i-2, ':', i+2, ') = ', mask(j, i-2:i+2)
      END IF
    END DO
    END DO

    i = C%NX - 1
    DO j = 1, C%NY
      IF(mask(j,i) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i  ) == C%mask_central_point) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) /= C%mask_inactive .AND. mask(j,i+1) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j,i-1) == C%mask_inactive .AND. mask(j,i+1) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_single_point
      ELSE IF(mask(j,i+1) == C%mask_inactive .AND. mask(j,i-1) /= C%mask_inactive .AND. mask(j,i-2) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_right_edge_point
      ELSE IF(mask(j,i+1) == C%mask_inactive .AND. mask(j,i-1) /= C%mask_inactive .AND. mask(j,i-2) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE IF(mask(j,i-1) == C%mask_inactive .AND. mask(j,i+1) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 4I2)') 'Unkown situation in determine_x_discretization_mask_for_calving_edges() at i=', i, ': mask(j=', j, ', i=', i-2, ':', i+1, ') = ', mask(j, i-2:i+1)
      END IF
    END DO

    i = C%NX
    DO j = 1, C%NY
      IF(mask(j,i) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i-1) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_right_edge_single_point
      ELSE IF(mask(j,i-1) /= C%mask_inactive .AND. mask(j,i-2) /= C%mask_inactive) THEN
       x_mask(j,i) = C%mask_right_edge_point
      ELSE IF(mask(j,i-1) /= C%mask_inactive .AND. mask(j,i-2) == C%mask_inactive) THEN
       x_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       x_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 3I2)') 'Unkown situation in determine_x_discretization_mask_for_calving_edges() at i=', i, ': mask(j=', j, ', i=', i-2, ':', i, ') = ', mask(j, i-2:i)
      END IF
    END DO
  END SUBROUTINE determine_x_discretization_mask_for_calving_edges



  ! Almost equal to determine_y_discretization_mask: C%type_shelf -> C%mask_inactive and C%type_sheet -> C%mask_central_point
  SUBROUTINE determine_y_discretization_mask_for_calving_edges(mask, y_mask)
    ! If for example a derivative field has to be discretized in the y-direction it has to be known if the
    ! neighbours are available (near the domain border) and if it is desired to use them (at the 'wrong'
    ! side of calving edge).
    ! A bunch of situations occur: a central discretization, a one sided in the left or right direction,
    ! only two points, or even only a single groundline point, all of this kind of situations near the
    ! domain border. To distiguish between all this situations this subroutine creates a special mask.
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(IN)  :: mask

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: y_mask

    ! Local variables:
    INTEGER                                     :: i, j

    j = 1
    DO i = 1, C%NX
      IF(mask(j,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j+1,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_left_edge_single_point
      ELSE IF(mask(j+1,i) /= C%mask_inactive .AND. mask(j+2,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_left_edge_point
      ELSE IF(mask(j+1,i) /= C%mask_inactive .AND. mask(j+2,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 3I2)') 'Unkown situation in determine_y_discretization_mask_for_calving_edges() at j=', j, ': mask(j=', j, ':', j+2, 'i=', i, ', ) = ', mask(j:j+2, i)
      END IF
    END DO

    j = 2
    DO i = 1, C%NX
      IF(mask(j,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i  ) == C%mask_central_point) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i) /= C%mask_inactive .AND. mask(j+1,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i) == C%mask_inactive .AND. mask(j+1,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_single_point
      ELSE IF(mask(j-1,i) == C%mask_inactive .AND. mask(j+1,i) /= C%mask_inactive .AND. mask(j+2,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_left_edge_point
      ELSE IF(mask(j-1,i) == C%mask_inactive .AND. mask(j+1,i) /= C%mask_inactive .AND. mask(j+2,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_left_edge_only_two_points
        ELSE IF(mask(j+1,i) == C%mask_inactive .AND. mask(j-1,i) /= C%mask_inactive) THEN
         y_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 4I2)') 'Unkown situation in determine_y_discretization_mask_for_calving_edges() at j=', j, ': mask(j=', j-1, ':', j+2, 'i=', i, ', ) = ', mask(j-1:j+2, i)
      END IF
    END DO

    DO j = 3, C%NY - 2
    DO i = 1, C%NX
      IF(mask(j,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i  ) == C%mask_central_point) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i) /= C%mask_inactive .AND. mask(j+1,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i) == C%mask_inactive .AND. mask(j+1,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_single_point
      ELSE IF(mask(j-1,i) == C%mask_inactive .AND. mask(j+1,i) /= C%mask_inactive .AND. mask(j+2,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_left_edge_point
      ELSE IF(mask(j-1,i) == C%mask_inactive .AND. mask(j+1,i) /= C%mask_inactive .AND. mask(j+2,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE IF(mask(j+1,i) == C%mask_inactive .AND. mask(j-1,i) /= C%mask_inactive .AND. mask(j-2,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_right_edge_point
      ELSE IF(mask(j+1,i) == C%mask_inactive .AND. mask(j-1,i) /= C%mask_inactive .AND. mask(j-2,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 5I2)') 'Unkown situation in determine_y_discretization_mask_for_calving_edges() at j=', j, ': mask(j=', j-2, ':', j+2, 'i=', i, ', ) = ', mask(j-2:j+2, i)
      END IF
    END DO
    END DO

    j = C%NY - 1
    DO i = 1, C%NX
      IF(mask(j,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j,i  ) == C%mask_central_point) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i) /= C%mask_inactive .AND. mask(j+1,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_central_point
      ELSE IF(mask(j-1,i) == C%mask_inactive .AND. mask(j+1,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_single_point
      ELSE IF(mask(j+1,i) == C%mask_inactive .AND. mask(j-1,i) /= C%mask_inactive .AND. mask(j-2,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_right_edge_point
      ELSE IF(mask(j+1,i) == C%mask_inactive .AND. mask(j-1,i) /= C%mask_inactive .AND. mask(j-2,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_right_edge_only_two_points
        ELSE IF(mask(j-1,i) == C%mask_inactive .AND. mask(j+1,i) /= C%mask_inactive) THEN
         y_mask(j,i) = C%mask_left_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 4I2)') 'Unkown situation in determine_y_discretization_mask_for_calving_edges() at j=', j, ': mask(j=', j-2, ':', j+1, 'i=', i, ', ) = ', mask(j-2:j+1, i)
      END IF
    END DO

    j = C%NY
    DO i = 1, C%NX
      IF(mask(j,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_inactive
      ELSE IF(mask(j-1,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_right_edge_single_point
      ELSE IF(mask(j-1,i) /= C%mask_inactive .AND. mask(j-2,i) /= C%mask_inactive) THEN
       y_mask(j,i) = C%mask_right_edge_point
      ELSE IF(mask(j-1,i) /= C%mask_inactive .AND. mask(j-2,i) == C%mask_inactive) THEN
       y_mask(j,i) = C%mask_right_edge_only_two_points
      ELSE
       y_mask(j,i) = C%mask_unknown
       WRITE(UNIT=*, FMT='(4(A, I4), A, 3I2)') 'Unkown situation in determine_y_discretization_mask_for_calving_edges() at j=', j, ': mask(j=', j-2, ':', j, 'i=', i, ', ) = ', mask(j-2:j, i)
      END IF
    END DO
  END SUBROUTINE determine_y_discretization_mask_for_calving_edges



! THE ROUTINES WHICH DO NOT DISTINGUISH BETWEEN A DIFFERENT SHEET AND SHELF CALVING THICKNESS:

  SUBROUTINE determine_hi_x_mask_for_a_straight_calving_front(Hi, calve_thickness, hi_x_mask)
    ! This routine determines a mask which represents ice cover and whether a covered point is a left or right side
    ! calving point. At each i value this routine takes a symmetric calving front by considering j = (C%NY + 1) / 2
    ! and extending that for all j.
    !
    ! Legend:
    !  hi_x_mask = C%mask_left_edge_point  = -1 means left side calving front  (the point left  of this point is ice-free)
    !  hi_x_mask = C%mask_central_point    =  0 means ice covered
    !  hi_x_mask = C%mask_right_edge_point =  1 means right side calving front (the point right of this point is ice-free)
    !  hi_x_mask = C%mask_inactive         = -5 means ice-free
    !
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: Hi
    REAL(dp),                       INTENT(IN)  :: calve_thickness

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: hi_x_mask

    ! Local variables:
    INTEGER                                     :: i, j

    ! Initializing all points to a ice-free state:
    hi_x_mask = C%mask_inactive

    j = (C%NY + 1) / 2

    i = 1
    ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the x-direction:
    IF((Hi(j,i) > calve_thickness) .AND. (Hi(j,i+1) > calve_thickness)) hi_x_mask(:,i) = C%mask_central_point

    DO i = 2, C%NX - 1
     ! Assign 'left side calving front' to points which are ice covered and have an ice-free left neighbour point in the x-direction:
     IF((Hi(j,i-1) <= calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j,i+1) >  calve_thickness)) &
      hi_x_mask(:,i) = C%mask_left_edge_point

     ! Assign 'ice covered' to points which are situated between two other ice covered points in the x-direction:
     IF((Hi(j,i-1) >  calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j,i+1) >  calve_thickness)) &
      hi_x_mask(:,i) = C%mask_central_point

     ! Assign 'right side calving front' to points which are ice covered and have an ice-free right neighbour point in the x-direction:
     IF((Hi(j,i-1) >  calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j,i+1) <= calve_thickness)) &
      hi_x_mask(:,i) = C%mask_right_edge_point
    END DO

    i = C%NX
    ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the x-direction:
    IF((Hi(j,i-1) > calve_thickness) .AND. (Hi(j,i) > calve_thickness)) hi_x_mask(:,i) = C%mask_central_point
  END SUBROUTINE determine_hi_x_mask_for_a_straight_calving_front



  SUBROUTINE determine_hi_y_mask_for_a_straight_calving_front(hi_x_mask, hi_y_mask)
    ! This routine determines the hi_y_mask which is consistent with the hi_x_mask if the
    ! latter is determined by determine_hi_x_mask_for_a_straight_calving_front()
    !
    ! Legend:
    !  hi_x_mask = C%mask_left_edge_point  = -1 means left side calving front  (the point left  of this point is ice-free)
    !  hi_x_mask = C%mask_central_point    =  0 means ice covered
    !  hi_x_mask = C%mask_right_edge_point =  1 means right side calving front (the point right of this point is ice-free)
    !  hi_x_mask = C%mask_inactive         = -5 means ice-free
    !
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(IN)  :: hi_x_mask

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: hi_y_mask

    ! Local variables:
    INTEGER                                     :: i, j

    DO i = 1, C%NX
    DO j = 1, C%NY
      IF(hi_x_mask(j,i) == C%mask_inactive) THEN
       hi_y_mask(j,i) = C%mask_inactive
      ELSE IF(j == 1) THEN
       hi_y_mask(j,i) = C%mask_left_edge_point
      ELSE IF(j == C%NY) THEN
       hi_y_mask(j,i) = C%mask_right_edge_point
      ELSE
       hi_y_mask(j,i) = C%mask_central_point
      END IF
    END DO
    END DO
  END SUBROUTINE determine_hi_y_mask_for_a_straight_calving_front



  ! Not in use yet: because the analogue routine causes a uniform calving front, which is preferred for now
  SUBROUTINE determine_hi_x_mask(Hi, calve_thickness, hi_x_mask)
    ! This routine determines a mask which represents ice cover and whether a covered point is a left or right side
    ! calving point.
    !
    ! Legend:
    !  hi_x_mask = C%mask_left_edge_point  = -1 means left side calving front  (the point left  of this point is ice-free)
    !  hi_x_mask = C%mask_central_point    =  0 means ice covered
    !  hi_x_mask = C%mask_right_edge_point =  1 means right side calving front (the point right of this point is ice-free)
    !  hi_x_mask = C%mask_inactive         = -5 means ice-free
    !
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: Hi
    REAL(dp),                       INTENT(IN)  :: calve_thickness

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: hi_x_mask

    ! Local variables:
    INTEGER                                     :: i, j

    ! Initializing all points to a ice-free state:
    hi_x_mask = C%mask_inactive

    i = 1
    DO j = 1, C%NY
     ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the x-direction:
     IF((Hi(j,i) > calve_thickness) .AND. (Hi(j,i+1) > calve_thickness)) hi_x_mask(j,i) = C%mask_central_point
    END DO

    DO i = 2, C%NX - 1
    DO j = 1, C%NY
      ! Assign 'left side calving front' to points which are ice covered and have an ice-free left neighbour point in the x-direction:
      IF((Hi(j,i-1) <= calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j,i+1) >  calve_thickness)) &
       hi_x_mask(j,i) = C%mask_left_edge_point

      ! Assign 'ice covered' to points which are situated between two other ice covered points in the x-direction:
      IF((Hi(j,i-1) >  calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j,i+1) >  calve_thickness)) &
       hi_x_mask(j,i) = C%mask_central_point

      ! Assign 'right side calving front' to points which are ice covered and have an ice-free right neighbour point in the x-direction:
      IF((Hi(j,i-1) >  calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j,i+1) <= calve_thickness)) &
       hi_x_mask(j,i) = C%mask_right_edge_point
    END DO
    END DO

    i = C%NX
    DO j = 1, C%NY
     ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the x-direction:
     IF((Hi(j,i-1) > calve_thickness) .AND. (Hi(j,i) > calve_thickness)) hi_x_mask(j,i) = C%mask_central_point
    END DO
  END SUBROUTINE determine_hi_x_mask



  ! Not in use yet: because the analogue routine causes a uniform calving front, which is preferred for now
  SUBROUTINE determine_hi_y_mask(Hi, calve_thickness, hi_y_mask)
    ! This routine determines a mask which represents ice cover and whether a covered point is a left or right side
    ! calving point.
    !
    ! Legend:
    !  hi_x_mask = C%mask_left_edge_point  = -1 means left side calving front  (the point left  of this point is ice-free)
    !  hi_x_mask = C%mask_central_point    =  0 means ice covered
    !  hi_x_mask = C%mask_right_edge_point =  1 means right side calving front (the point right of this point is ice-free)
    !  hi_x_mask = C%mask_inactive         = -5 means ice-free
    !
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: Hi
    REAL(dp),                       INTENT(IN)  :: calve_thickness

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: hi_y_mask

    ! Local variables:
    INTEGER                                     :: i, j

    ! Initializing all points to a ice-free state:
    hi_y_mask = C%mask_inactive

    j = 1
    DO i = 1, C%NX
     ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the y-direction:
     IF((Hi(j,i) > calve_thickness) .AND. (Hi(j+1,i) > calve_thickness)) hi_y_mask(j,i) = C%mask_central_point
    END DO

    DO j = 2, C%NY - 1
    DO i = 1, C%NX
     ! Assign 'left side calving front' to points which are ice covered and have an ice-free left neighbour point in the y-direction:
     IF((Hi(j-1,i) <= calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j+1,i) > calve_thickness)) &
      hi_y_mask(j,i) = C%mask_left_edge_point

     ! Assign 'ice covered' to points which are situated between two other ice covered points in the y-direction:
     IF((Hi(j-1,i) > calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j+1,i) > calve_thickness)) &
      hi_y_mask(j,i) = C%mask_central_point

     ! Assign 'right side calving front' to points which are ice covered and have an ice-free right neighbour point in the y-direction:
     IF((Hi(j-1,i) > calve_thickness) .AND. (Hi(j,i) > calve_thickness) .AND. (Hi(j+1,i) <= calve_thickness)) &
      hi_y_mask(j,i) = C%mask_right_edge_point
    END DO
    END DO

    j = C%NY
    DO i = 1, C%NX
     ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the y-direction:
     IF((Hi(j-1,i) > calve_thickness) .AND. (Hi(j,i) > calve_thickness)) hi_y_mask(j,i) = C%mask_central_point
    END DO
  END SUBROUTINE determine_hi_y_mask



! THE ROUTINES WHICH DISTINGUISH BETWEEN A DIFFERENT SHEET AND SHELF CALVING THICKNESS:

  SUBROUTINE determine_hi_x_mask_distinctive_for_a_straight_calving_front(Hi, mask, calve_thickness_sheet, calve_thickness_shelf, hi_x_mask)
    ! This routine determines a mask which represents ice cover and whether a covered point is a left or right side
    ! calving point. It distinguishes situations in which the calving thickness for the sheet and the shelf differ.
    ! At each i value this routine takes a symmetric calving front by considering j = (C%NY + 1) / 2 and extending
    ! that for all j.
    !
    ! Legend:
    !  hi_x_mask = C%mask_left_edge_point  = -1 means left side calving front  (the point left  of this point is ice-free)
    !  hi_x_mask = C%mask_central_point    =  0 means ice covered
    !  hi_x_mask = C%mask_right_edge_point =  1 means right side calving front (the point right of this point is ice-free)
    !  hi_x_mask = C%mask_inactive         = -5 means ice-free
    !
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: Hi
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(IN)  :: mask
    REAL(dp),                       INTENT(IN)  :: calve_thickness_sheet
    REAL(dp),                       INTENT(IN)  :: calve_thickness_shelf

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: hi_x_mask

    ! Local variables:
    INTEGER                                     :: i, j

    ! Initializing all points to a ice-free state:
    hi_x_mask = C%mask_inactive

    j = (C%NY + 1) / 2

    i = 1
    ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the x-direction:
    IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
       (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
     IF((mask(j,i+1) == C%type_shelf .AND. Hi(j,i+1) > calve_thickness_shelf) .OR. &
        (mask(j,i+1) /= C%type_shelf .AND. Hi(j,i+1) > calve_thickness_sheet)      ) hi_x_mask(:,i) = C%mask_central_point
    END IF

    DO i = 2, C%NX - 1
     ! Assign 'left side calving front' to points which are ice covered and have an ice-free left neighbour point in the x-direction:
     IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
        (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
      IF(((mask(j,i-1) == C%type_shelf .AND. Hi(j,i-1) <=  calve_thickness_shelf)  .OR.  &
          (mask(j,i-1) /= C%type_shelf .AND. Hi(j,i-1) <= calve_thickness_sheet)) .AND. &
         ((mask(j,i+1) == C%type_shelf .AND. Hi(j,i+1) >  calve_thickness_shelf)  .OR.  &
          (mask(j,i+1) /= C%type_shelf .AND. Hi(j,i+1) >  calve_thickness_sheet))) hi_x_mask(:,i) = C%mask_left_edge_point
     END IF

     ! Assign 'ice covered' to points which are situated between two other ice covered points in the x-direction:
     IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
        (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
      IF(((mask(j,i-1) == C%type_shelf .AND. Hi(j,i-1) >  calve_thickness_shelf)  .OR.  &
          (mask(j,i-1) /= C%type_shelf .AND. Hi(j,i-1) >  calve_thickness_sheet)) .AND. &
         ((mask(j,i+1) == C%type_shelf .AND. Hi(j,i+1) >  calve_thickness_shelf)  .OR.  &
          (mask(j,i+1) /= C%type_shelf .AND. Hi(j,i+1) >  calve_thickness_sheet))) hi_x_mask(:,i) = C%mask_central_point
     END IF

     ! Assign 'right side calving front' to points which are ice covered and have an ice-free right neighbour point in the x-direction:
     IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
        (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
      IF(((mask(j,i-1) == C%type_shelf .AND. Hi(j,i-1) >  calve_thickness_shelf)  .OR.  &
          (mask(j,i-1) /= C%type_shelf .AND. Hi(j,i-1) >  calve_thickness_sheet)) .AND. &
         ((mask(j,i+1) == C%type_shelf .AND. Hi(j,i+1) <=  calve_thickness_shelf)  .OR.  &
          (mask(j,i+1) /= C%type_shelf .AND. Hi(j,i+1) <= calve_thickness_sheet))) hi_x_mask(:,i) = C%mask_right_edge_point
     END IF
    END DO

    i = C%NX
    ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the x-direction:
    IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
       (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
     IF((mask(j,i-1) == C%type_shelf .AND. Hi(j,i-1) > calve_thickness_shelf) .OR. &
        (mask(j,i-1) /= C%type_shelf .AND. Hi(j,i-1) > calve_thickness_sheet)      ) hi_x_mask(:,i) = C%mask_central_point
    END IF
  END SUBROUTINE determine_hi_x_mask_distinctive_for_a_straight_calving_front



  ! Not in use yet: because the analogue routine causes a uniform calving front, which is preferred for now
  SUBROUTINE determine_hi_x_mask_distinctive(Hi, mask, calve_thickness_sheet, calve_thickness_shelf, hi_x_mask)
    ! This routine determines a mask which represents ice cover and whether a covered point is a left or right side
    ! calving point. It distinguishes situations in which the calving thickness for the sheet and the shelf differ.
    ! At each i value this routine takes a symmetric calving front by considering j = (C%NY + 1) / 2 and extending
    ! that for all j.
    !
    ! Legend:
    !  hi_x_mask = C%mask_left_edge_point  = -1 means left side calving front  (the point left  of this point is ice-free)
    !  hi_x_mask = C%mask_central_point    =  0 means ice covered
    !  hi_x_mask = C%mask_right_edge_point =  1 means right side calving front (the point right of this point is ice-free)
    !  hi_x_mask = C%mask_inactive         = -5 means ice-free
    !
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: Hi
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(IN)  :: mask
    REAL(dp),                       INTENT(IN)  :: calve_thickness_sheet
    REAL(dp),                       INTENT(IN)  :: calve_thickness_shelf

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: hi_x_mask

    ! Local variables:
    INTEGER                                     :: i, j

    ! Initializing all points to a ice-free state:
    hi_x_mask = C%mask_inactive

    i = 1
    DO j = 1, C%NY
     ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the x-direction:
     IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
        (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
      IF((mask(j,i+1) == C%type_shelf .AND. Hi(j,i+1) > calve_thickness_shelf) .OR. &
         (mask(j,i+1) /= C%type_shelf .AND. Hi(j,i+1) > calve_thickness_sheet)      ) hi_x_mask(j,i) = C%mask_central_point
     END IF
    END DO

    DO i = 2, C%NX - 1
    DO j = 1, C%NY
      ! Assign 'left side calving front' to points which are ice covered and have an ice-free left neighbour point in the x-direction:
      IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
         (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
       IF(((mask(j,i-1) == C%type_shelf .AND. Hi(j,i-1) <=  calve_thickness_shelf)  .OR.  &
           (mask(j,i-1) /= C%type_shelf .AND. Hi(j,i-1) <= calve_thickness_sheet)) .AND. &
          ((mask(j,i+1) == C%type_shelf .AND. Hi(j,i+1) >  calve_thickness_shelf)  .OR.  &
           (mask(j,i+1) /= C%type_shelf .AND. Hi(j,i+1) >  calve_thickness_sheet))) hi_x_mask(j,i) = C%mask_left_edge_point
      END IF

      ! Assign 'ice covered' to points which are situated between two other ice covered points in the x-direction:
      IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
         (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
       IF(((mask(j,i-1) == C%type_shelf .AND. Hi(j,i-1) >  calve_thickness_shelf)  .OR.  &
           (mask(j,i-1) /= C%type_shelf .AND. Hi(j,i-1) >  calve_thickness_sheet)) .AND. &
          ((mask(j,i+1) == C%type_shelf .AND. Hi(j,i+1) >  calve_thickness_shelf)  .OR.  &
           (mask(j,i+1) /= C%type_shelf .AND. Hi(j,i+1) >  calve_thickness_sheet))) hi_x_mask(j,i) = C%mask_central_point
      END IF

      ! Assign 'right side calving front' to points which are ice covered and have an ice-free right neighbour point in the x-direction:
      IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
         (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
       IF(((mask(j,i-1) == C%type_shelf .AND. Hi(j,i-1) >  calve_thickness_shelf)  .OR.  &
           (mask(j,i-1) /= C%type_shelf .AND. Hi(j,i-1) >  calve_thickness_sheet)) .AND. &
          ((mask(j,i+1) == C%type_shelf .AND. Hi(j,i+1) <=  calve_thickness_shelf)  .OR.  &
           (mask(j,i+1) /= C%type_shelf .AND. Hi(j,i+1) <= calve_thickness_sheet))) hi_x_mask(j,i) = C%mask_right_edge_point
      END IF
    END DO
    END DO

    i = C%NX
    DO j = 1, C%NY
     ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the x-direction:
     IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
        (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
      IF((mask(j,i-1) == C%type_shelf .AND. Hi(j,i-1) > calve_thickness_shelf) .OR. &
         (mask(j,i-1) /= C%type_shelf .AND. Hi(j,i-1) > calve_thickness_sheet)      ) hi_x_mask(j,i) = C%mask_central_point
     END IF
    END DO
  END SUBROUTINE determine_hi_x_mask_distinctive



  ! Not in use yet: because the analogue routine causes a uniform calving front, which is preferred for now
  SUBROUTINE determine_hi_y_mask_distinctive(Hi, mask, calve_thickness_sheet, calve_thickness_shelf, hi_y_mask)
    ! This routine determines a mask which represents ice cover and whether a covered point is a left or right side
    ! calving point. It distinguishes situations in which the calving thickness for the sheet and the shelf differ.
    !
    ! Legend:
    !  hi_x_mask = C%mask_left_edge_point  = -1 means left side calving front  (the point left  of this point is ice-free)
    !  hi_x_mask = C%mask_central_point    =  0 means ice covered
    !  hi_x_mask = C%mask_right_edge_point =  1 means right side calving front (the point right of this point is ice-free)
    !  hi_x_mask = C%mask_inactive         = -5 means ice-free
    !
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(IN)  :: Hi
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(IN)  :: mask
    REAL(dp),                       INTENT(IN)  :: calve_thickness_sheet
    REAL(dp),                       INTENT(IN)  :: calve_thickness_shelf

    ! Output variables:
    INTEGER,  DIMENSION(C%NY,C%NX), INTENT(OUT) :: hi_y_mask

    ! Local variables:
    INTEGER                                     :: i, j

    ! Initializing all points to a ice-free state:
    hi_y_mask = C%mask_inactive

    j = 1
    DO i = 1, C%NX
     ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the y-direction:
     IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
        (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
      IF((mask(j+1,i) == C%type_shelf .AND. Hi(j+1,i) > calve_thickness_shelf) .OR. &
         (mask(j+1,i) /= C%type_shelf .AND. Hi(j+1,i) > calve_thickness_sheet)      ) hi_y_mask(j,i) = C%mask_central_point
     END IF
    END DO

    DO j = 2, C%NY - 1
    DO i = 1, C%NX
      ! Assign 'left side calving front' to points which are ice covered and have an ice-free left neighbour point in the y-direction:
      IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
         (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
       IF(((mask(j-1,i) == C%type_shelf .AND. Hi(j-1,i) <=  calve_thickness_shelf)  .OR.  &
           (mask(j-1,i) /= C%type_shelf .AND. Hi(j-1,i) <= calve_thickness_sheet)) .AND. &
          ((mask(j+1,i) == C%type_shelf .AND. Hi(j+1,i) >  calve_thickness_shelf)  .OR.  &
           (mask(j+1,i) /= C%type_shelf .AND. Hi(j+1,i) >  calve_thickness_sheet))) hi_y_mask(j,i) = C%mask_left_edge_point
      END IF

      ! Assign 'ice covered' to points which are situated between two other ice covered points in the y-direction:
      IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
         (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
       IF(((mask(j-1,i) == C%type_shelf .AND. Hi(j-1,i) >  calve_thickness_shelf)  .OR.  &
           (mask(j-1,i) /= C%type_shelf .AND. Hi(j-1,i) >  calve_thickness_sheet)) .AND. &
          ((mask(j+1,i) == C%type_shelf .AND. Hi(j+1,i) >  calve_thickness_shelf)  .OR.  &
           (mask(j+1,i) /= C%type_shelf .AND. Hi(j+1,i) >  calve_thickness_sheet))) hi_y_mask(j,i) = C%mask_central_point
      END IF

      ! Assign 'right side calving front' to points which are ice covered and have an ice-free right neighbour point in the y-direction:
      IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
         (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
       IF(((mask(j-1,i) == C%type_shelf .AND. Hi(j-1,i) >  calve_thickness_shelf)  .OR.  &
           (mask(j-1,i) /= C%type_shelf .AND. Hi(j-1,i) >  calve_thickness_sheet)) .AND. &
          ((mask(j+1,i) == C%type_shelf .AND. Hi(j+1,i) <=  calve_thickness_shelf)  .OR.  &
           (mask(j+1,i) /= C%type_shelf .AND. Hi(j+1,i) <= calve_thickness_sheet))) hi_y_mask(j,i) = C%mask_right_edge_point
     END IF
    END DO
    END DO

    j = C%NY
    DO i = 1, C%NX
     ! Assign 'ice covered' to domain boundary points which can be considered as situated between two other ice covered points in the y-direction:
     IF((mask(j,i) == C%type_shelf .AND. Hi(j,i) > calve_thickness_shelf) .OR. &
        (mask(j,i) /= C%type_shelf .AND. Hi(j,i) > calve_thickness_sheet)      ) THEN
      IF((mask(j-1,i) == C%type_shelf .AND. Hi(j-1,i) > calve_thickness_shelf) .OR. &
         (mask(j-1,i) /= C%type_shelf .AND. Hi(j-1,i) > calve_thickness_sheet)      ) hi_y_mask(j,i) = C%mask_central_point
     END IF
    END DO
  END SUBROUTINE determine_hi_y_mask_distinctive

END MODULE mask_module
