! File name: initialization_module.f90
!
! Copyright (C) 2014 Thomas Reerink
!
! This file is part of the ICEDYN-package
!
! IMAU, Utrecht University, The Netherlands
!

MODULE initialization_module

CONTAINS
  SUBROUTINE initialize_antarctica(T_Hi, T_Hb, T_Us, T_Vs, T_Ti, smb)
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(OUT) :: T_Us
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(OUT) :: T_Vs
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX), INTENT(OUT) :: T_Ti
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(OUT) :: smb
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY     )              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY     )              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY     )              :: Us
    REAL(dp), DIMENSION(C%NX,C%NY     )              :: Vs
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZ)              :: Ti
    INTEGER                                          :: i, j, k, kdum
    REAL(dp)                                         :: fdum
    CHARACTER(LEN=14)                                :: tiformat

    CHARACTER(LEN=*), PARAMETER                      :: header_format = '(A80)'
    CHARACTER(LEN=80)                                :: header
    CHARACTER(LEN=80)                                :: input_format

    ! Initial value of ice thickness Hi: 
    OPEN(UNIT=2502,FILE='./data/initial/antarctica/ice_thickness_antarctica_pd_20x20km.dat')
    DO j=1,C%NY
    DO i=1,C%NX
      READ(UNIT=2502,FMT="(I3,X,I3,X,F10.2)") kdum, kdum, Hi(j,i)
    END DO
    END DO
    CLOSE(UNIT=2502)

    ! Initial value of bedrock height Hb:
    OPEN(UNIT=2503,FILE='./data/initial/antarctica/bedrock_antarctica_pd_20x20km.dat')
    DO j=1,C%NY
    DO i=1,C%NX
      READ(UNIT=2503,FMT="(I3,X,I3,X,F10.2)") kdum, kdum, Hb(j,i)
    END DO
    END DO
    CLOSE(UNIT=2503)

    ! Initial value of ice temperature Ti:
    OPEN(UNIT=2600,FILE='./data/initial/antarctica/modeled_pd_ice_temperature_antarctica_20x20km.dat')
    write (tiformat, '(a1,i3,a10)') '(', C%NZ, '(X,F10.2))' 
    ! Note that with C%NZ=15 we get: tiformat = '( 15(X,F10.2))'
    DO j=1,C%NY
    DO i=1,C%NX
      READ(UNIT=2600,FMT=tiformat) (Ti(j,i,k), k=1, C%NZ) 
    END DO
    END DO
    CLOSE(UNIT=2600)

    ! Initial value for shelf variables Us and Vs:
    OPEN(UNIT=2602,FILE='./data/initial/antarctica/modeled_pd_Us_Vs_on_shelf_antarctica_20x20km.dat')
    DO j=1,C%NY
    DO i=1,C%NX
      READ(UNIT=2602,FMT="(I3,X,I3,3F12.2,E12.2)") kdum, kdum, Us(j,i), Vs(j,i), fdum, fdum
    END DO
    END DO
    CLOSE(UNIT=2602)

    ! Read present day mass balance:
    OPEN(UNIT=1505,FILE='./data/reference/antarctica/accumulation.dat')
    READ(UNIT=1505, FMT=header_format) header
    READ(UNIT=1505, FMT=header_format) input_format
    DO j = 1, C%NY
      READ(UNIT=1505, FMT=input_format) kdum, (smb(j,i), i = 1, C%NX)
    END DO
    CLOSE(UNIT=1505)

    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Us, T_Us, Vs, T_Vs)
    CALL transpose_3D_fields(Ti, T_Ti)
  END SUBROUTINE initialize_antarctica
  
  
  
  SUBROUTINE initialize_greenland(T_Hi, T_Hb, T_Ts, T_MB)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_MB
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: MB
    REAL(dp), DIMENSION(C%NX,C%NY)              :: initial_sealevel
    INTEGER                                     :: i, j
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    REAL(dp)                                    :: dummy
    CHARACTER(LEN=256)                          :: MB_file_name
    CHARACTER(LEN=256)                          :: Hb_file_name

    initial_sealevel = 0._dp

    MB_file_name = './data/initial/greenland/bamber_data_20x20km/longitude_latitude_X_Y_Ts_pricipiation_greenland_pd_20x20km_at_bamber_grid.dat'
    Hb_file_name = './data/initial/greenland/bamber_data_20x20km/longitude_latitude_Hb_pd_20x20km.dat'

    CALL read_netcdf_file('./data/initial/greenland/bamber_data_20x20km/bamber_data_greenland_20x20km.nc', 'Hi', 'Hb', Hi, Hb)

    OPEN(UNIT=2605,FILE=MB_file_name)
    DO i = 1, C%NX
    DO j = 1, C%NY
      READ(UNIT=2605,FMT='(2F20.13, 2F13.1, 2E25.16)') dummy, dummy, dummy, dummy, dummy, MB(i,j)
    END DO
    END DO
    CLOSE(UNIT=2605)

    ! Rescale mm water equivalent per year to meter ice equivalent per year:
    MB = (MB / 1000._dp) * (C%fresh_water_density / C%ice_density) 
    
    ! Calculate the initial Hs from the initial Hb and Hi:
    CALL surface_height(initial_sealevel, MAX(Hi, C%Hi_min), Hb, Hs)
    
    ! Calculate the anual greenland present day surface temperature (here in degree Celcius):
    CALL greenlands_surface_temperature(Hb_file_name, Hs, Ts)
    ! Conversion from degree Celsius to Kelvin:
    Ts = Ts + C%triple_point_of_water

    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Ts, T_Ts, MB, T_MB)
  END SUBROUTINE initialize_greenland



  SUBROUTINE create_greenland_netcdf_bamber(grid_choice)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    USE oblimap_read_and_write_module, ONLY: oblimap_netcdf_file_type, oblimap_create_netcdf_file, oblimap_write_netcdf_fields, oblimap_close_netcdf_file
    IMPLICIT NONE

    ! Input variables:
    INTEGER,                       INTENT(IN) :: grid_choice

    ! Local variables:
    INTEGER                                   :: i, j
    INTEGER                                   :: field_counter
    REAL(dp), DIMENSION(C%NX,C%NY)            :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)            :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)            :: longitude
    REAL(dp), DIMENSION(C%NX,C%NY)            :: latitude
    REAL(dp), DIMENSION(C%NX,C%NY)            :: Ts
    REAL(dp), DIMENSION(C%NX,C%NY)            :: Hs
    REAL(dp)                                  :: dummy
    CHARACTER(LEN=256)                        :: Hb_file_name
    CHARACTER(LEN=256)                        :: Hi_file_name
    CHARACTER(LEN=256)                        :: created_file_name
    TYPE(oblimap_netcdf_file_type)            :: nc
    REAL(dp), DIMENSION(3,C%NX,C%NY)          :: fields
    REAL(dp), DIMENSION(C%NX,C%NY)            :: initial_sealevel

    initial_sealevel = 0._dp

    IF(grid_choice == 20) THEN
     Hb_file_name      = './data/initial/greenland/bamber_data_20x20km/longitude_latitude_Hb_pd_20x20km.dat'
     Hi_file_name      = './data/initial/greenland/bamber_data_20x20km/longitude_latitude_Hi_pd_20x20km.dat'
     created_file_name = 'bamber_data_greenland_20x20km.nc'
    ELSE IF(grid_choice == 10) THEN
     Hb_file_name      = './data/initial/greenland/bamber_data_10x10km/longitude_latitude_Hb_pd_10x10km.dat'
     Hi_file_name      = './data/initial/greenland/bamber_data_10x10km/longitude_latitude_Hi_pd_10x10km.dat'
     created_file_name = 'bamber_data_greenland_10x10km.nc'
    ELSE IF(grid_choice ==  5) THEN
     Hb_file_name      = './data/initial/greenland/bamber_data_05x05km/longitude_latitude_Hb_pd_05x05km.dat'
     Hi_file_name      = './data/initial/greenland/bamber_data_05x05km/longitude_latitude_Hi_pd_05x05km.dat'
     created_file_name = 'bamber_data_greenland_05x05km.nc'
    ELSE
     STOP 'Stop message from: create_greenland_netcdf_bamber() in the initialization_module: wrong grid_choice'
    END IF
    
    OPEN(UNIT=2603,FILE=Hb_file_name)
    OPEN(UNIT=2604,FILE=Hi_file_name)
    DO i = 1, C%NX
    DO j = 1, C%NY
      READ(UNIT=2603,FMT=*) longitude(i,j), latitude(i,j), Hb(i,j)
      READ(UNIT=2604,FMT=*) dummy,          dummy,         Hi(i,j)
    END DO
    END DO
    CLOSE(UNIT=2603)
    CLOSE(UNIT=2604)
    
    ! Reset lower-right corner to a different (unique) value:
    !!WHERE(Hb < -4500._dp) Hb = -2200._dp  ! Hb = -2250._dp
    
    ! Calculate the initial Hs from the initial Hb and Hi:
    CALL surface_height(initial_sealevel, MAX(Hi, C%Hi_min), Hb, Hs)
    
    ! Calculate the anual greenland present day surface temperature (here in degree Celcius):
    ! Overwiting: Ts
    CALL greenlands_surface_temperature(Hb_file_name, Hs, Ts)
    ! Conversion from degree Celsius to Kelvin:
    Ts = Ts + C%triple_point_of_water
    
    
    fields(1,:,:) = Hi
    fields(2,:,:) = Hb
    fields(3,:,:) = Hs

    C%NLON = C%NX
    C%NLAT = C%NY

    C%gcm_field_name(-3) = 'i_lon'
    C%gcm_field_name(-5) = 'j_lat'
    C%gcm_field_name(-2) = 'lon'
    C%gcm_field_name(-4) = 'lat'
    C%gcm_field_name( 1) = 'Hi'
    C%gcm_field_name( 2) = 'Hb'
    C%gcm_field_name( 3) = 'Hs'

    ! Output: nc
    CALL oblimap_create_netcdf_file(created_file_name, C%number_of_mapped_fields, .TRUE., (/ (2, field_counter = C%MND, C%number_of_mapped_fields) /), &
                                    C%gcm_field_name, C%gcm_field_unit, C%gcm_field_longname, C%NLON, C%NLAT, longitude, latitude, nc = nc)
    CALL oblimap_write_netcdf_fields(nc, 1, fields)
    CALL oblimap_close_netcdf_file(nc)
  END SUBROUTINE create_greenland_netcdf_bamber



  SUBROUTINE initialize_greenland_racmo(T_Hi, T_Hb, T_Ts, T_smb)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_smb
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: smb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    REAL(dp), DIMENSION(C%NX,C%NY)              :: initial_sealevel

    initial_sealevel = 0._dp

    CALL read_netcdf_file('./data/initial/greenland/mapped_bamber_greenland_at_10x10km/mapped_bamber_greenland_at_10x10km.nc', 'Hi', 'Hb', Hi, Hb)
    CALL read_netcdf_file('./data/racmo/greenland/im_Greenland_Ts_MB_surface_Hs_RG2.nc', 'Ts', 'MB_surface', Ts, smb)

    ! Calculate the initial Hs from the initial Hb and Hi:
    CALL surface_height(initial_sealevel, MAX(Hi, C%Hi_min), Hb, Hs)

    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Ts, T_Ts, smb, T_smb)
  END SUBROUTINE initialize_greenland_racmo



  SUBROUTINE initialize_greenland_GMS4(T_Hi, T_Hb, T_Ts, T_MB)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_MB
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: MB
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ice_mask
    INTEGER, DIMENSION(C%NY,C%NX)               :: mask_EI

    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    CHARACTER(LEN=256)                          :: MB_file_name
    CHARACTER(LEN=256)                          :: Hb_file_name
    
    Hb_file_name = './data/initial/greenland/mapped_bamber_greenland_at_20x20km_GMS4/mapped_bamber_greenland_at_20x20km_GMS4.nc'
    Mb_file_name = './data/racmo/greenland/CLRUN_1958-2007/im_racmo_greenland_RG_20x20km_GMS4.nc'

    CALL read_netcdf_file(Hb_file_name, 'Hi', 'Hb', Hi, Hb)
    CALL read_netcdf_file(Hb_file_name, 'Hs', 'Hb', Hs, Hb)
    CALL read_netcdf_file(Mb_file_name, 'ice_mask', 'MB_surface', Ice_mask, MB)
    CALL read_netcdf_file(Mb_file_name, 'MB_surface', 'Ts', MB, Ts)

    ! Adds ice on gridpoints that are within the RACMO ice mask, but contain no ice according the Bamber data on ice thickness:
    DO i = 1, C%NX
    DO j = 1, C%NY
      if (Ice_mask(i,j).gt.0.5_dp.and.Hi(i,j).le.0.100000001_dp) then
        Hi(i,j) = 10._dp
        print *, 'extra ice added: Hi=10 m', i, j
      endif
    END DO
    END DO
    
    !removes ice from Ellesmere Island and from grid points outside the racmo ice mask
    CALL mask_Ellesmere_Island_20x20km_GMS4(mask_EI)
    DO i = 1, C%NX
    DO j = 1, C%NY
!      if(ice_mask(i,j).lt.0.5_dp.or.mask_EI(j,i).eq.1) Hi(i,j) = 0.1_dp
      if(mask_EI(j,i).eq.1) Hi(i,j) = 0.1_dp
    END DO
    END DO

    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Ts, T_Ts, MB, T_MB)
  END SUBROUTINE initialize_greenland_GMS4



  SUBROUTINE mask_Ellesmere_Island_20x20km_GMS4(mask_EI)
    ! this subroutine recognizes grid point located on Ellesmere Island
    ! mask_EL = 1  Ellesmere Island
    ! mask_EL = 0  not Ellesmere Island
    ! only works when using the 20x20km_GMS4 grid
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! In/output:
    INTEGER, DIMENSION(C%NY,C%NX), INTENT(INOUT)  :: mask_EI

    ! Local variables:
    INTEGER                                       :: i, j

    mask_EI = 0
    DO j=1,C%NX
    DO i=1,C%NY
      if (j.eq. 1.and.i.gt.117) mask_EI(i,j) = 1
      if (j.eq. 2.and.i.gt.118) mask_EI(i,j) = 1
      if (j.eq. 3.and.i.gt.118) mask_EI(i,j) = 1
      if (j.eq. 4.and.i.gt.120) mask_EI(i,j) = 1
      if (j.eq. 5.and.i.gt.122) mask_EI(i,j) = 1
      if (j.eq. 6.and.i.gt.121) mask_EI(i,j) = 1
      if (j.eq. 7.and.i.gt.121) mask_EI(i,j) = 1
      if (j.eq. 8.and.i.gt.122) mask_EI(i,j) = 1
      if (j.eq. 9.and.i.gt.121) mask_EI(i,j) = 1
      if (j.eq.10.and.i.gt.122) mask_EI(i,j) = 1
      if (j.eq.11.and.i.gt.123) mask_EI(i,j) = 1
      if (j.eq.12.and.i.gt.123) mask_EI(i,j) = 1
      if (j.eq.13.and.i.gt.124) mask_EI(i,j) = 1
      if (j.eq.14.and.i.gt.125) mask_EI(i,j) = 1
      if (j.eq.15.and.i.gt.125) mask_EI(i,j) = 1
      if (j.eq.16.and.i.gt.125) mask_EI(i,j) = 1
      if (j.eq.17.and.i.gt.126) mask_EI(i,j) = 1
      if (j.eq.18.and.i.gt.126) mask_EI(i,j) = 1
      if (j.eq.19.and.i.gt.127) mask_EI(i,j) = 1
      if (j.eq.20.and.i.gt.129) mask_EI(i,j) = 1
      if (j.eq.21.and.i.gt.129) mask_EI(i,j) = 1
      if (j.eq.22.and.i.gt.129) mask_EI(i,j) = 1
      if (j.eq.23.and.i.gt.130) mask_EI(i,j) = 1
    END DO
    END DO
  END SUBROUTINE mask_Ellesmere_Island_20x20km_GMS4



  SUBROUTINE initialize_greenland_Bamber2003_Ettema2009_GMS5(T_Hi, T_Hb, T_Ts, T_MB)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height

    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_MB
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: MB

    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    CHARACTER(LEN=256)                          :: MB_file_name
    CHARACTER(LEN=256)                          :: Hb_file_name
    
    Hb_file_name = './input_fields/greenland/mapped_bamber_greenland_at_20x20km_1a.nc'
    Mb_file_name = './forcing_fields/greenland/forcing_racmo_greenland_20x20km_1a.nc'

    CALL read_netcdf_file(Hb_file_name, 'Ice Thickness', 'Bedrock Elevation', Hi, Hb)
    CALL read_netcdf_file(Hb_file_name, 'Surface Elevation', 'Bedrock Elevation', Hs, Hb)
    CALL read_netcdf_file(Mb_file_name, 'smb', 'ice_surface_temperature', MB, Ts)

    ! assigning an SMB value of -4 m i.e. y-1 to points outside the ice mask
    DO i = 1, C%NX
    DO j = 1, C%NY
      if (MB(i,j).eq.-9999) then 
        MB(i,j) = -4._dp
       else
        MB(i,j) = MB(i,j)/910._dp ! convert unit from kg m2 y-1 to m i.e. y-1
      endif
    END DO
    END DO
        
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Ts, T_Ts, MB, T_MB)
  END SUBROUTINE initialize_greenland_Bamber2003_Ettema2009_GMS5



  SUBROUTINE initialize_greenland_ice2sea_Ettema2009_GMS5(T_Hi, T_Hb, T_Ts, T_MB)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height

    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_MB
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: MB

    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    CHARACTER(LEN=256)                          :: MB_file_name
    CHARACTER(LEN=256)                          :: Hb_file_name
    
    Hb_file_name = './input_fields/greenland/initial_greenland_ice2sea_20x20km.nc'
    Mb_file_name = './forcing_fields/greenland/forcing_racmo_greenland_20x20km_1a.nc'

    CALL read_netcdf_file(Hb_file_name, 'Ice Thickness', 'Bedrock Elevation', Hi, Hb)
    CALL read_netcdf_file(Hb_file_name, 'Surface Elevation', 'Bedrock Elevation', Hs, Hb)
    CALL read_netcdf_file(Mb_file_name, 'smb', 'ice_surface_temperature', MB, Ts)

    ! assigning an SMB value of -4 m i.e. y-1 to points outside the ice mask
    DO i = 1, C%NX
    DO j = 1, C%NY
      if (MB(i,j).eq.-9999) then 
        MB(i,j) = -4._dp
       else
        MB(i,j) = MB(i,j)/910._dp ! convert unit from kg m2 y-1 to m i.e. y-1
      endif
    END DO
    END DO
        
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Ts, T_Ts, MB, T_MB)
  END SUBROUTINE initialize_greenland_ice2sea_Ettema2009_GMS5



  SUBROUTINE initialize_greenland_Bamber2003_racmo2_1960_2011_GMS5(T_Hi, T_Hb, T_Ts, T_MB)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height

    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_MB
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: MB

    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    CHARACTER(LEN=256)                          :: MB_file_name
    CHARACTER(LEN=256)                          :: Hb_file_name
    
    Hb_file_name = './input_fields/greenland/mapped_bamber_greenland_at_20x20km_1a.nc'
    Mb_file_name = './forcing_fields/greenland/forcing_fields_racmo2_greenland_20x20km_1960-2011-average.nc'

    CALL read_netcdf_file(Hb_file_name, 'Ice Thickness', 'Bedrock Elevation', Hi, Hb)
    CALL read_netcdf_file(Hb_file_name, 'Surface Elevation', 'Bedrock Elevation', Hs, Hb)
    CALL read_netcdf_file(Mb_file_name, 'smb', 'ice_surface_temperature', MB, Ts)

    ! assigning an SMB value of -4 m i.e. y-1 to points outside the ice mask
    DO i = 1, C%NX
    DO j = 1, C%NY
      if (MB(i,j).eq.-9999) then 
        MB(i,j) = -4._dp
       else
        MB(i,j) = MB(i,j)/910._dp ! convert unit from kg m2 y-1 to m i.e. y-1
      endif
    END DO
    END DO
        
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Ts, T_Ts, MB, T_MB)
  END SUBROUTINE initialize_greenland_Bamber2003_racmo2_1960_2011_GMS5



  SUBROUTINE initialize_greenland_ice2sea_racmo2_1960_2011_GMS5(T_Hi, T_Hb, T_Ts, T_MB)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height

    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_MB
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: MB

    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    CHARACTER(LEN=256)                          :: MB_file_name
    CHARACTER(LEN=256)                          :: Hb_file_name
    
    Hb_file_name = './input_fields/greenland/initial_greenland_ice2sea_20x20km.nc'
    Mb_file_name = './forcing_fields/greenland/forcing_fields_racmo2_greenland_20x20km_1960-2011-average.nc'

    CALL read_netcdf_file(Hb_file_name, 'Ice Thickness', 'Bedrock Elevation', Hi, Hb)
    CALL read_netcdf_file(Hb_file_name, 'Surface Elevation', 'Bedrock Elevation', Hs, Hb)
    CALL read_netcdf_file(Mb_file_name, 'smb', 'ice_surface_temperature', MB, Ts)

    ! assigning an SMB value of -4 m i.e. y-1 to points outside the ice mask
    DO i = 1, C%NX
    DO j = 1, C%NY
      if (MB(i,j).eq.-9999) then 
        MB(i,j) = -4._dp
       else
        MB(i,j) = MB(i,j)/910._dp ! convert unit from kg m2 y-1 to m i.e. y-1
      endif
    END DO
    END DO
        
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Ts, T_Ts, MB, T_MB)
  END SUBROUTINE initialize_greenland_ice2sea_racmo2_1960_2011_GMS5



  SUBROUTINE initialize_greenland_ice2sea_racmo2_1960_1989_GMS5(T_Hi, T_Hb, T_Ts, T_MB)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height

    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_MB
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: MB

    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    CHARACTER(LEN=256)                          :: MB_file_name
    CHARACTER(LEN=256)                          :: Hb_file_name
    
    Hb_file_name = './input_fields/greenland/initial_greenland_ice2sea_20x20km.nc'
    Mb_file_name = './forcing_fields/greenland/forcing_fields_racmo2_greenland_20x20km_1960-1989-average.nc'

    CALL read_netcdf_file(Hb_file_name, 'Ice Thickness', 'Bedrock Elevation', Hi, Hb)
    CALL read_netcdf_file(Hb_file_name, 'Surface Elevation', 'Bedrock Elevation', Hs, Hb)
    CALL read_netcdf_file(Mb_file_name, 'smb', 'ice_surface_temperature', MB, Ts)

    ! assigning an SMB value of -4 m i.e. y-1 to points outside the ice mask
    DO i = 1, C%NX
    DO j = 1, C%NY
      if (MB(i,j).eq.-9999) then 
        MB(i,j) = -4._dp
       else
        MB(i,j) = MB(i,j)/910._dp ! convert unit from kg m2 y-1 to m i.e. y-1
      endif
    END DO
    END DO
        
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Ts, T_Ts, MB, T_MB)
  END SUBROUTINE initialize_greenland_ice2sea_racmo2_1960_1989_GMS5



  SUBROUTINE read_netcdf_file(file_name, field_name_1, field_name_2, field_1, field_2)
    USE configuration_module, ONLY: dp, C
    USE oblimap_read_and_write_module, ONLY: oblimap_netcdf_file_type, oblimap_open_netcdf_file, oblimap_read_netcdf_fields, oblimap_close_netcdf_file
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=*),               INTENT(IN)  :: file_name
    CHARACTER(LEN=*),               INTENT(IN)  :: field_name_1
    CHARACTER(LEN=*),               INTENT(IN)  :: field_name_2

    ! Output variables:
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(OUT) :: field_1
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(OUT) :: field_2

    ! Local variables:
    TYPE(oblimap_netcdf_file_type)              :: nc
    REAL(dp), DIMENSION(2,C%NX,C%NY)            :: fields

    C%im_field_name(1)  = field_name_1
    C%im_field_name(2)  = field_name_2

    CALL oblimap_open_netcdf_file(file_name, 2, C%im_field_name, C%NX, C%NY, nc = nc)    
    CALL oblimap_read_netcdf_fields(nc, 1, fields)
    CALL oblimap_close_netcdf_file(nc)

    field_1 = fields(1,:,:)
    field_2 = fields(2,:,:)
  END SUBROUTINE read_netcdf_file


      
  SUBROUTINE greenlands_surface_temperature(file_name, Hs, Ts)
    ! --------------------------------------------------------------------------
    ! Determination of annual temperature at the corresponding surface altitude.
    ! --------------------------------------------------------------------------
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=256)            , INTENT(IN)  :: file_name ! The file name of the file which contains the lon and lat values
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN)  :: Hs        ! surface height [in meter relative to present day sea level]

    ! Output variables:
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(OUT) :: Ts        ! parameterized anual surface temperature of greenland [in degree Celcius]
    
    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp)                                    :: h_inv
    REAL(dp)                                    :: latitude
    REAL(dp)                                    :: dummy

    OPEN(UNIT=2607,FILE=file_name)
    DO i = 1, C%NX
    DO j = 1, C%NY
      READ(UNIT=2607,FMT=*) dummy, latitude, dummy
    
      h_inv = 300._dp * (latitude - 65._dp) / 15._dp                      ! Huybrechts 1999  
      IF(Hs(i,j) > h_inv) THEN                                                 
       Ts(i,j) = 49.13_dp - 0.007992_dp * Hs(i,j) - 0.7576_dp * latitude      
      ELSE                                                              
       Ts(i,j) = 49.13_dp - 0.007992_dp * h_inv   - 0.7576_dp * latitude      
      END IF
    END DO
    END DO
    CLOSE(UNIT=2607)
  END SUBROUTINE greenlands_surface_temperature



  SUBROUTINE initialize_eurasia_or_north_america(area_of_interest, T_Hi, T_Hb, T_MB, T_Ti)
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    CHARACTER(LEN=128),                  INTENT(IN)  :: area_of_interest

    ! Output variables:
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(     C%NY,C%NX), INTENT(OUT) :: T_MB
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX), INTENT(OUT) :: T_Ti
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY     )              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY     )              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY     )              :: MB
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZ)              :: Ti

    ! Local variables:
    INTEGER                                          :: i, j, k
    INTEGER                                          :: month_counter
    REAL(dp), DIMENSION(   C%NX,C%NY)                :: longitude
    REAL(dp), DIMENSION(   C%NX,C%NY)                :: latitude
    REAL(dp), DIMENSION(12,C%NX,C%NY)                :: Ts_month
    REAL(dp), DIMENSION(12,C%NX,C%NY)                :: accumulation_month
    
    ! Initial value of the ice thickness Hi: 
    Hi = 0._dp

    ! Initial value of the bedrock height Hb: 
    IF(area_of_interest == 'eurasia') THEN
     OPEN(UNIT=2610,FILE='./data/initial/eurasia/lon_lat_Hb_eurasia_40x40km.dat')
    ELSE IF(area_of_interest == 'north_america') THEN
     OPEN(UNIT=2610,FILE='./data/initial/north_america/lon_lat_Hb_north_america_40x40km.dat')
    ELSE
     WRITE(UNIT=*,FMT=*) 'Error, see subroutine: initialize_eurasia_or_north_america'
    END IF
    
    DO i = 1,C%NX
    DO j = 1,C%NY
      READ(UNIT=2610,FMT=*) longitude(i,j), latitude(i,j), Hb(i,j)
    END DO
    END DO
    CLOSE(UNIT=2610)
    
    ! Initial value of the bedrock height Hb: 
    IF(area_of_interest == 'eurasia') THEN
     OPEN(UNIT=2611,FILE='./data/initial/eurasia/monthly_average_surface_temperature_and_precipitation_eurasia_40x40km_pd_ncep_reanalysis.dat')
    ELSE IF(area_of_interest == 'north_america') THEN
     OPEN(UNIT=2611,FILE='./data/initial/north_america/monthly_average_surface_temperature_and_precipitation_north_america_40x40km_pd_ncep_reanalysis.dat')
    END IF
    
    DO i = 1,C%NX
    DO j = 1,C%NY
      READ(UNIT=2611,FMT=*)(Ts_month(month_counter,i,j), accumulation_month(month_counter,i,j), month_counter=1,12) 
    END DO
    END DO
    CLOSE(UNIT=2611)
    
    ! The annual average from the montly values for the surface temperature and the mass balance:
    MB = 0._dp
    Ti = 0._dp
    DO month_counter = 1, 12
     Ti(:,:,1) = Ti(:,:,1) + Ts_month(month_counter,:,:) / 12._dp
     MB(:,:)   = MB(:,:)   + accumulation_month(month_counter,:,:)          ! ice or water equivalent per month??
    END DO
    ! Because of zero ice situation, we initialize all ice layers at surface temperature:
    DO k = 2, C%NZ
     Ti(:,:,k) = Ti(:,:,1)
    END DO
    
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, MB, T_MB)
    CALL transpose_3D_fields(Ti, T_Ti)
  END SUBROUTINE initialize_eurasia_or_north_america



  SUBROUTINE initialize_hardangerjokulen(T_Hi, T_Hb, T_Hs, T_Ts, T_MB, T_glacier_mask)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hs
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_MB
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_glacier_mask
    
    ! Local variables:
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: MB
    REAL(dp), DIMENSION(C%NX,C%NY)              :: glacier_mask ! A mask of the present day glacier ice extend

    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi_input
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs_input
    INTEGER,  DIMENSION(C%NX,C%NY)              :: glacier_mask_input 

    OPEN(UNIT=2620,FILE='./data/initial/hardangerjokulen/surface_heigth_hardangerjokulen_pd_100x100m_140x132grid.dat')
    OPEN(UNIT=2621,FILE='./data/initial/hardangerjokulen/ice_thickness_hardangerjokulen_pd_100x100m_140x132grid.dat')
    OPEN(UNIT=2622,FILE='./data/initial/hardangerjokulen/mask_hardangerjokulen_pd_100x100m_140x132grid.dat')

    READ(UNIT=2620,FMT=*) Hs_input   
    READ(UNIT=2621,FMT=*) Hi_input   
    READ(UNIT=2622,FMT=*) glacier_mask_input
    glacier_mask_input = glacier_mask_input + 1 ! Ice covered points have value 2, other points have value 1.
    
    ! Due to non-standard grid conventions of the input data we do a mirror transformation in the j-direction:
    DO i = 1, C%NX
    DO j = 1, C%NY
     Hi(i,j)           = Hi_input(i,C%NY + 1 - j)
     Hs(i,j)           = Hs_input(i,C%NY + 1 - j)
     glacier_mask(i,j) = glacier_mask_input(i,C%NY + 1 - j)
    END DO
    END DO
    
    Hb = Hs - Hi
    
    ! Calculate the anual hardangerjokulen present day surface temperature:
    ! Output: Ts, MB
    CALL hardangerjokulen_surface_temperature_and_MB(Hs, Ts, MB)
    
    CLOSE(UNIT=2620)
    CLOSE(UNIT=2621)
    CLOSE(UNIT=2622)
    
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Hs, T_Hs, Ts, T_Ts, MB, T_MB, glacier_mask, T_glacier_mask)
  END SUBROUTINE initialize_hardangerjokulen



  SUBROUTINE hardangerjokulen_surface_temperature_and_MB(Hs, Ts, MB)
    ! --------------------------------------------------------------------------------------------
    ! Determination of annual temperature and a mass balance at the corresponding surface altitude
    ! --------------------------------------------------------------------------------------------
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN)  :: Hs  ! surface height [in meter relative to present day sea level]

    ! Output variables:
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(OUT) :: Ts  ! parameterized anual surface temperature of greenland [in degree Celcius]
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(OUT) :: MB  ! parameterized anual surface temperature of greenland [in degree Celcius]
    
    ! Local variables:
    INTEGER                                     :: i, j
    REAL(dp)                                    :: altifactor
    !REAL(dp)                                    :: ELA = 150._dp
    
    DO i = 1, C%NX
    DO j = 1, C%NY
      !MB(i,j) = 0.003_dp * (Hs(i,j) - ELA)
      !MB(i,j) = 0.005_dp * (Hs(i,j) - ELA)
      !IF(MB(i,j) > (0.1)) MB(i,j)=0.1
      
      altifactor = 1._dp
      IF(Hs(i,j) > 1675._dp) &
       altifactor = 1._dp - 0.0011_dp * (Hs(i,j) - 1675._dp) - 1.54e-5_dp * (Hs(i,j) - 1675._dp)**2
      MB(i,j) = (-15.86_dp + 0.00969_dp * (Hs(i,j))) * altifactor 
      !MB(i,j) = MB(i,j) -0.235_dp ! -0.5_dp
      ! Convert from meter water equivalent to meter ice equivalent per year:
      MB(i,j) = MB(i,j) * (C%fresh_water_density / C%ice_density) 

      ! Surface temperature in K, lapse rate 6.5K/km, Ts at AWS (1450m) is -1.35 deg Celcius: 
      Ts(i,j) = (-1.35_dp + C%triple_point_of_water) - 0.0065_dp * (Hs(i,j) - 1450._dp)
    END DO
    END DO
  END SUBROUTINE hardangerjokulen_surface_temperature_and_MB




  SUBROUTINE initialize_ITMIX(T_Hi, T_Hb, T_Hs, T_smb, T_icemask, T_velocity, T_relaxation)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hi         ! transpose of the ice thickness (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hb         ! transpose of the bed altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hs         ! transpose of the surface altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_smb        ! transpose of the surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_icemask    ! transpose of the ice mask 
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_velocity   ! transpose of the velocity field
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_relaxation ! transpose of the relaxation parameter field
   
    ! Local variables:
    INTEGER                                               :: ii,jj
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hi           ! ice thickness
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hb           ! bed elevation
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs           ! surface altitude
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: smb          ! surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: smb_input    ! surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask      ! ice mask glacier surface (2 for ice-covered, 1 for ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: velocity
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: velocity_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: bed
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: bed_input

    ! open files
    OPEN(UNIT=2630,FILE=C%filename_DEM)
    OPEN(UNIT=2631,FILE=C%filename_MASK)
    OPEN(UNIT=2632,FILE=C%filename_SMB)
    OPEN(UNIT=2633,FILE=C%filename_VEL)
    OPEN(UNIT=2634,FILE=C%filename_BED)
    OPEN(UNIT=2635,FILE=C%filename_K)

    ! read files
    READ(UNIT=2630,FMT=*) Hs_input
    READ(UNIT=2631,FMT=*) icemask_input
    READ(UNIT=2632,FMT=*) smb_input
    READ(UNIT=2633,FMT=*) velocity_input
    READ(UNIT=2634,FMT=*) bed_input
    READ(UNIT=2635,FMT=*) relaxation_input
	
    ! Due to non-standard (matlab) grid conventions of the input data we do a mirror transformation in the j-direction:
    DO ii = 1, C%NX
    DO jj = 1, C%NY
     Hs(ii,jj)         = Hs_input(ii, C%NY + 1 - jj)
     icemask(ii,jj)    = icemask_input(ii, C%NY + 1 - jj)
     smb(ii,jj)        = smb_input(ii, C%NY + 1 - jj)
     velocity(ii,jj)   = velocity_input(ii, C%NY + 1 - jj)
     bed(ii,jj)        = bed_input(ii, C%NY + 1 - jj)
     relaxation(ii,jj) = relaxation_input(ii, C%NY + 1 - jj)
    END DO
    END DO
    
    ! initialize bed topography       
    Hb = bed
    
    ! initialize ice thickness
    DO ii = 1, C%NX
    DO jj = 1, C%NY
     Hi(ii,jj) = Hs(ii,jj) - Hb(ii,jj)
    END DO
    END DO 
    
    CLOSE(UNIT=2630)
    CLOSE(UNIT=2631)
    CLOSE(UNIT=2632)
    CLOSE(UNIT=2633)
    CLOSE(UNIT=2634)
    CLOSE(UNIT=2635)
  
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Hs, T_Hs, smb, T_smb, icemask, T_icemask, velocity, T_velocity, relaxation, T_relaxation)
  END SUBROUTINE initialize_ITMIX
 

 

  SUBROUTINE initialize_svartisen(T_Hi, T_Hb, T_Hs, T_Ts, T_smb, T_icemask, T_relaxation)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hi         ! transpose of the ice thickness (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hb         ! transpose of the bed altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hs         ! transpose of the surface altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Ts         ! transpose of the surface temperature (K)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_smb        ! transpose of the surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_icemask    ! transpose of the ice mask in INTEGER, 1 = ice covered, 0 = ice-free grid points
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_relaxation ! transpose of the relaxation parameter field
    
    ! Local variables:
    INTEGER                                               :: ii,jj
    REAL(dp)                                              :: altifactor
    REAL(dp)                                              :: ELA
    REAL(dp)                                              :: smb_gradient
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hi           ! ice thickness
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hb           ! bed elevation
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs           ! surface altitude
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Ts           ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: smb          ! surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask      ! ice mask glacier surface( 1 = ice covered, 0 = ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask_input! ice mask glacier surface (2 for ice-covered, 1 for ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation   ! spatially differenciated relaxation parameter 
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation_input

    ! open files
    OPEN(UNIT=2630,FILE='./data/initial/svartisen_norway/statkart-svartisen-DEM-100x100m.txt')
    !OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/svartisen-glacier-mask-1988-100x100m.txt')
    OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/svartisenEandW-glacier-mask-1999-100x100m.txt')
    OPEN(UNIT=2635,FILE=C%filename_K)
    ! read files
    READ(UNIT=2630,FMT=*) Hs_input
    READ(UNIT=2631,FMT=*) icemask_input
    READ(UNIT=2635,FMT=*) relaxation_input

    ! Due to non-standard (matlab) grid conventions of the input data we do a mirror transformation in the j-direction:
    DO ii = 1, C%NX
    DO jj = 1, C%NY
     Hs(ii,jj)         = Hs_input(ii, C%NY + 1 - jj)
     icemask(ii,jj)    = icemask_input(ii, C%NY + 1 - jj)
     relaxation(ii,jj) = relaxation_input(ii, C%NY + 1 - jj)
    END DO
    END DO
    
    ! set ice mask 0/1 (no ice/ice) instead of 1/2
    icemask = icemask - 1.0_dp
    
    ! initialize with an uniform ice thickness of 10 m within the ice mask
    DO ii = 1, C%NX
    DO jj = 1, C%NY
        IF (icemask(ii,jj) > 0.5_dp) THEN
            Hi(ii,jj) = 10.0_dp
        ELSE
            Hi(ii,jj) = 0.0_dp    
        END IF
    END DO
    END DO 
    ! initialize bed topography such that surface topography is the same as the DEM topography       
    Hb = Hs - Hi
    
    ! Calculate the anual hardangerjokulen present day surface temperature:
    Ts = C%triple_point_of_water
    
    ! Calculate annual surface mass balance (smb) from simple parameterization with altitude
    ELA             = 1000._dp
    smb_gradient    = 0.007_dp
    DO ii = 1, C%NX
    DO jj = 1, C%NY
        altifactor      = 1._dp
        IF(Hs(ii,jj) > ELA) THEN
            altifactor = MAX(0._dp, 1._dp - 1e-4_dp * (Hs(ii,jj) - ELA) - 1e-6_dp * (Hs(ii,jj) - ELA + 200._dp)**2)
        END IF   
        smb(ii,jj) = altifactor * smb_gradient * (Hs(ii,jj) - ELA)
    END DO
    END DO
    
    CLOSE(UNIT=2630)
    CLOSE(UNIT=2631)
    CLOSE(UNIT=2635)
    
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Hs, T_Hs, Ts, T_Ts, smb, T_smb, icemask, T_icemask, relaxation, T_relaxation)
  END SUBROUTINE initialize_svartisen




  SUBROUTINE initialize_svartisen_bedmap(T_Hi, T_Hb, T_Hs, T_Ts, T_smb, T_icemask, T_relaxation)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hi         ! transpose of the ice thickness (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hb         ! transpose of the bed altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hs         ! transpose of the surface altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Ts         ! transpose of the surface temperature (K)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_smb        ! transpose of the surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_icemask    ! transpose of the ice mask in INTEGER, 1 = ice covered, 0 = ice-free grid points
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_relaxation ! transpose of the relaxation parameter field
    
    ! Local variables:
    INTEGER                                               :: ii,jj
    REAL(dp)                                              :: altifactor
    REAL(dp)                                              :: ELA
    REAL(dp)                                              :: smb_gradient
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hi           ! ice thickness
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hb           ! bed elevation
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs           ! surface altitude
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hb_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Ts           ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: smb          ! surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask      ! ice mask glacier surface (2 for ice-covered, 1 for ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation_input

    ! open files
    OPEN(UNIT=2630,FILE='./data/initial/svartisen_norway/statkart-svartisen-DEM-100x100m.txt')
    !OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/svartisen-glacier-mask-1988-100x100m.txt')
    OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/svartisenEandW-glacier-mask-1999-100x100m.txt')
    OPEN(UNIT=2632,FILE='./data/initial/svartisen_norway/svartisen-bedmap-DEM-NVEicethicknobs1992-100x100m.txt')
    OPEN(UNIT=2635,FILE=C%filename_K)

    ! read files
    READ(UNIT=2630,FMT=*) Hs_input
    READ(UNIT=2631,FMT=*) icemask_input
    READ(UNIT=2632,FMT=*) Hb_input
    READ(UNIT=2635,FMT=*) relaxation_input

    ! Due to non-standard (matlab) grid conventions of the input data we do a mirror transformation in the j-direction:
    DO ii = 1, C%NX
    DO jj = 1, C%NY
     Hs        (ii,jj) = Hs_input        (ii, C%NY + 1 - jj)
     icemask   (ii,jj) = icemask_input   (ii, C%NY + 1 - jj) - 1.0_dp
     Hb        (ii,jj) = Hb_input        (ii, C%NY + 1 - jj)
     relaxation(ii,jj) = relaxation_input(ii, C%NY + 1 - jj)
    END DO
    END DO
    
    ! initialize ice thickness by subtracting the bed elevation from the surface elevation 
    Hi = Hs - Hb
    
    ! Calculate the anual hardangerjokulen present day surface temperature:
    Ts = C%triple_point_of_water
    
    ! Calculate annual surface mass balance (smb) from simple parameterization with altitude
    ELA             = 1000._dp
    smb_gradient    = 0.007_dp
    DO ii = 1, C%NX
    DO jj = 1, C%NY
        altifactor      = 1._dp
        IF(Hs(ii,jj) > ELA) THEN
            altifactor = MAX(0._dp, 1._dp - 1e-4_dp * (Hs(ii,jj) - ELA) - 1e-6_dp * (Hs(ii,jj) - ELA + 200._dp)**2)
        END IF   
        smb(ii,jj) = altifactor * smb_gradient * (Hs(ii,jj) - ELA)
    END DO
    END DO
    
    CLOSE(UNIT=2630)
    CLOSE(UNIT=2631)
    
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Hs, T_Hs, Ts, T_Ts, smb, T_smb, icemask, T_icemask, relaxation, T_relaxation)   
  END SUBROUTINE initialize_svartisen_bedmap


  SUBROUTINE initialize_svartisen_250x250(T_Hi, T_Hb, T_Hs, T_Ts, T_smb, T_icemask, T_relaxation)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hi         ! transpose of the ice thickness (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hb         ! transpose of the bed altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hs         ! transpose of the surface altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Ts         ! transpose of the surface temperature (K)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_smb        ! transpose of the surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_icemask    ! transpose of the ice mask (1 = ice covered, 0 = ice-free grid points)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_relaxation ! transpose of the relaxation parameter field
    
    ! Local variables:
    INTEGER                                               :: ii,jj
    REAL(dp)                                              :: altifactor
    REAL(dp)                                              :: ELA
    REAL(dp)                                              :: smb_gradient
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hi           ! ice thickness
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hb           ! bed elevation
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs           ! surface altitude
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Ts           ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: smb          ! surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask      ! ice mask glacier surface (1 for ice-covered, 0 for ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask_input! ice mask glacier surface (2 for ice-covered, 1 for ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation_input

    ! open files
    OPEN(UNIT=2630,FILE='./data/initial/svartisen_norway/statkart-svartisen-DEM-250x250m.txt')
   !OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/svartisen-glacier-mask-1988-250x250m.txt')
    OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/svartisenEandW-glacier-mask-1999-250x250m.txt')
    OPEN(UNIT=2635,FILE=C%filename_K)
    ! read files
    READ(UNIT=2630,FMT=*) Hs_input
    READ(UNIT=2631,FMT=*) icemask_input
    READ(UNIT=2635,FMT=*) relaxation_input

    ! Due to non-standard (matlab) grid conventions of the input data we do a mirror transformation in the j-direction:
    DO ii = 1, C%NX
    DO jj = 1, C%NY
     Hs(ii,jj)         = Hs_input(ii, C%NY + 1 - jj)
     icemask(ii,jj)    = icemask_input(ii, C%NY + 1 - jj) - 1.0_dp
     relaxation(ii,jj) = relaxation_input(ii, C%NY + 1 - jj)
    END DO
    END DO
    
    ! initialize with an uniform ice thickness of 100 m within the ice mask
    DO ii = 1, C%NX
    DO jj = 1, C%NY
        IF (icemask(ii,jj) > 0.5_dp) THEN
            Hi(ii,jj) = 10.0_dp
        ELSE
            Hi(ii,jj) = 0.0_dp    
        END IF
    END DO
    END DO 
    ! initialize bed topography such that surface topography is the same as the DEM topography       
    Hb = Hs - Hi
    
    ! Calculate the anual hardangerjokulen present day surface temperature:
    Ts = C%triple_point_of_water
    
    ! Calculate annual surface mass balance (smb) from simple parameterization with altitude
    ELA             = 1000._dp
    smb_gradient    = 0.007_dp
    DO ii = 1, C%NX
    DO jj = 1, C%NY
        altifactor      = 1._dp
        IF(Hs(ii,jj) > ELA) THEN
            altifactor = MAX(0._dp, 1._dp - 1e-4_dp * (Hs(ii,jj) - ELA) - 1e-6_dp * (Hs(ii,jj) - ELA + 200._dp)**2)
        END IF   
        smb(ii,jj) = altifactor * smb_gradient * (Hs(ii,jj) - ELA)
    END DO
    END DO
    
    CLOSE(UNIT=2630)
    CLOSE(UNIT=2631)
    CLOSE(UNIT=2635)
    
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Hs, T_Hs, Ts, T_Ts, smb, T_smb, icemask, T_icemask, relaxation, T_relaxation)   
  END SUBROUTINE initialize_svartisen_250x250



  SUBROUTINE initialize_svartisen_250x250_bedmap(T_Hi, T_Hb, T_Hs, T_Ts, T_smb, T_icemask, T_relaxation)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hi         ! transpose of the ice thickness (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hb         ! transpose of the bed altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Hs         ! transpose of the surface altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_Ts         ! transpose of the surface temperature (K)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_smb        ! transpose of the surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_icemask    ! transpose of the ice mask INTEGER (1 = ice covered, 0 = ice-free grid points)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_relaxation ! transpose of the relaxation parameter field
    
    ! Local variables:
    INTEGER                                               :: ii,jj
    REAL(dp)                                              :: altifactor
    REAL(dp)                                              :: ELA
    REAL(dp)                                              :: smb_gradient
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hi           ! ice thickness
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hb           ! bed elevation
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs           ! surface altitude
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hs_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Hb_input
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: Ts           ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: smb          ! surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask      ! ice mask glacier surface (1 for ice-covered, 0 for ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: icemask_input! ice mask glacier surface (2 for ice-covered, 1 for ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation   ! relaxation parameter (K) spatially distributed 
    REAL(dp), DIMENSION(C%NX,C%NY)                        :: relaxation_input

    ! open files
    OPEN(UNIT=2630,FILE='./data/initial/svartisen_norway/statkart-svartisen-DEM-250x250m.txt')
    !OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/svartisen-glacier-mask-1988-100x100m.txt')
    OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/svartisenEandW-glacier-mask-1999-100x100m.txt')
    OPEN(UNIT=2632,FILE='./data/initial/svartisen_norway/svartisen-bedmap-DEM-NVEicethicknobs1992-250x250m.txt')
    OPEN(UNIT=2635,FILE=C%filename_K)

    ! read files
    READ(UNIT=2630,FMT=*) Hs_input
    READ(UNIT=2631,FMT=*) icemask_input
    READ(UNIT=2632,FMT=*) Hb_input
    READ(UNIT=2635,FMT=*) relaxation_input

    ! Due to non-standard (matlab) grid conventions of the input data we do a mirror transformation in the j-direction:
    DO ii = 1, C%NX
    DO jj = 1, C%NY
     Hs        (ii,jj) = Hs_input        (ii, C%NY + 1 - jj)
     icemask   (ii,jj) = icemask_input   (ii, C%NY + 1 - jj) - 1.0_dp
     Hb        (ii,jj) = Hb_input        (ii, C%NY + 1 - jj)
     relaxation(ii,jj) = relaxation_input(ii, C%NY + 1 - jj)
    END DO
    END DO
    
    ! initialize ice thickness by subtracting the bed elevation from the surface elevation 
    Hi = Hs - Hb
    
    ! Calculate the anual hardangerjokulen present day surface temperature:
    Ts = C%triple_point_of_water
    
    ! Calculate annual surface mass balance (smb) from simple parameterization with altitude
    ELA             = 1000._dp
    smb_gradient    = 0.007_dp
    DO ii = 1, C%NX
    DO jj = 1, C%NY
        altifactor      = 1._dp
        IF(Hs(ii,jj) > ELA) THEN
            altifactor = MAX(0._dp, 1._dp - 1e-4_dp * (Hs(ii,jj) - ELA) - 1e-6_dp * (Hs(ii,jj) - ELA + 200._dp)**2)
        END IF   
        smb(ii,jj) = altifactor * smb_gradient * (Hs(ii,jj) - ELA)
    END DO
    END DO
    
    CLOSE(UNIT=2630)
    CLOSE(UNIT=2631)
    CLOSE(UNIT=2632)
    CLOSE(UNIT=2635)
   
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Hs, T_Hs, Ts, T_Ts, smb, T_smb, icemask, T_icemask, relaxation, T_relaxation)   
  END SUBROUTINE initialize_svartisen_250x250_bedmap



  SUBROUTINE initialize_engabreen(T_Hi, T_Hb, T_Hs, T_Ts, T_smb, T_icemask)
    USE configuration_module, ONLY: dp, C
    USE surface_height_module, ONLY: surface_height
    IMPLICIT NONE

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hi         ! transpose of the ice thickness (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hb         ! transpose of the bed altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Hs         ! transpose of the surface altitude (m)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_Ts         ! transpose of the surface temperature (K)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_smb        ! transpose of the surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT) :: T_icemask    ! transpose of the ice mask in INTEGER, 1 = ice covered, 0 = ice-free grid points
    
    ! Local variables:
    INTEGER                                     :: ii,jj
    REAL(dp)                                    :: altifactor
    REAL(dp)                                    :: ELA
    REAL(dp)                                    :: smb_gradient
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hi           ! ice thickness
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hb           ! bed elevation
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs           ! surface altitude
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Hs_input
    REAL(dp), DIMENSION(C%NX,C%NY)              :: Ts           ! surface temperature [in degree Kelvin]
    REAL(dp), DIMENSION(C%NX,C%NY)              :: smb          ! surface mass balance (m w.e. a^-1)
    REAL(dp), DIMENSION(C%NX,C%NY)              :: icemask      ! ice mask glacier surface (1 for ice-covered, 0 for ice-free grid points)
    REAL(dp), DIMENSION(C%NX,C%NY)              :: icemask_input! ice mask glacier surface (2 for ice-covered, 1 for ice-free grid points)

    ! open files
    OPEN(UNIT=2630,FILE='./data/initial/svartisen_norway/statkart-engabreen-basin-DEM-100x100m.txt')
    OPEN(UNIT=2631,FILE='./data/initial/svartisen_norway/engabreen-basin-glacier-mask-1999-100x100m.txt')
    ! read files
    READ(UNIT=2630,FMT=*) Hs_input
    READ(UNIT=2631,FMT=*) icemask_input

    ! Due to non-standard (matlab) grid conventions of the input data we do a mirror transformation in the j-direction:
    DO ii = 1, C%NX
    DO jj = 1, C%NY
     Hs(ii,jj)         = Hs_input(ii, C%NY + 1 - jj)
     icemask(ii,jj)    = icemask_input(ii, C%NY + 1 - jj) - 1.0_dp
    END DO
    END DO
    
    ! initialize with an uniform ice thickness of 100 m within the ice mask
    DO ii = 1, C%NX
    DO jj = 1, C%NY
        IF (icemask(ii,jj) > 1.5_dp) THEN
            Hi(ii,jj) = 1.0_dp
        ELSE
            Hi(ii,jj) = 0.0_dp    
        END IF
    END DO
    END DO 
    ! initialize bed topography such that surface topography is the same as the DEM topography       
    Hb = Hs - Hi
    
    ! Calculate the anual hardangerjokulen present day surface temperature:
    Ts = C%triple_point_of_water
    
    ! Calculate annual surface mass balance (smb) from simple parameterization with altitude
    ELA             = 1000._dp
    smb_gradient    = 0.007_dp
    DO ii = 1, C%NX
    DO jj = 1, C%NY
        altifactor      = 1._dp
        IF(Hs(ii,jj) > ELA) THEN
            altifactor = MAX(0._dp, 1._dp - 1e-4_dp * (Hs(ii,jj) - ELA) - 1e-6_dp * (Hs(ii,jj) - ELA + 200._dp)**2)
        END IF   
        smb(ii,jj) = altifactor * smb_gradient * (Hs(ii,jj) - ELA)
    END DO
    END DO
    
    CLOSE(UNIT=2630)
    CLOSE(UNIT=2631)
    
    CALL transpose_2D_fields(Hi, T_Hi, Hb, T_Hb, Hs, T_Hs, Ts, T_Ts, smb, T_smb, icemask, T_icemask)
  END SUBROUTINE initialize_engabreen




  SUBROUTINE initialize_Ti_at_pressure_melting_point(Ts, Hi, delta_Ti_pmp_bottom, mask, Ti)
    ! Initialize ice temperature Ti in the lower layers (by a linear interpolation between the ice surface temperature
    ! and the pressure melting point temperature Ti_pmp at ice bottom (see equation (11.2)): 
    !  Ti_pmp(bottom) = C%triple_point_of_water - C%Claus_Clap_gradient * Hi
    ! Actually we can choose the bottom temperature a little bit below Ti_pmp: delta_Ti_pmp_bottom, if we want.
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(     C%NY,C%NX),  INTENT(IN)  :: Ts
    REAL(dp), DIMENSION(     C%NY,C%NX),  INTENT(IN)  :: Hi
    REAL(dp),                             INTENT(IN)  :: delta_Ti_pmp_bottom
    INTEGER,  DIMENSION(     C%NY,C%NX),  INTENT(IN)  :: mask

    ! Output variables:
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX),  INTENT(OUT) :: Ti
    
    ! Local variables:
    INTEGER                                           :: k

    Ti(1,:,:) = Ts
    DO k = 2, C%NZ
      Ti(k,:,:) = Ts + ((C%triple_point_of_water - C%Claus_Clap_gradient * Hi - delta_Ti_pmp_bottom) - Ts) * C%zeta(k)
      ! For bare rock the ice temperatures in the thin lower layers are taken equal to the surface temperature:
      WHERE((mask /= C%type_shelf) .AND. (Hi <= C%Hi_min)) Ti(k,:,:) = Ts
      ! The ice temperatures for ocean points:
      WHERE((mask == C%type_shelf) .AND. (Hi <= C%Hi_min)) Ti(k,:,:) = C%seawater_temperature
      ! The ice temperatures for shelf points:
      WHERE((mask == C%type_shelf) .AND. (Hi >  C%Hi_min)) Ti(k,:,:) = Ts + (C%seawater_temperature - Ts) * C%zeta(k)
    END DO
  END SUBROUTINE initialize_Ti_at_pressure_melting_point
  
  
  
  SUBROUTINE transpose_2D_fields(var_1, T_var_1, var_2, T_var_2, var_3, T_var_3, var_4, T_var_4, var_5, T_var_5, var_6, T_var_6, var_7, T_var_7)
    ! This subroutine transposes the initial fields
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN)            :: var_1
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN),  OPTIONAL :: var_2
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN),  OPTIONAL :: var_3
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN),  OPTIONAL :: var_4
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN),  OPTIONAL :: var_5
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN),  OPTIONAL :: var_6
    REAL(dp), DIMENSION(C%NX,C%NY), INTENT(IN),  OPTIONAL :: var_7

    ! Output variables:
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT)           :: T_var_1
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_2
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_3
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_4
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_5
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_6
    REAL(dp), DIMENSION(C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_7

    ! Local variables:
    INTEGER                                               :: m, n

    ! Transpose the read fields from {x,y} to {y,x}, or {m,n} to {n,m}:
    DO m = 1, C%NX 
    DO n = 1, C%NY
                           T_var_1(n,m) = var_1(m,n)
      IF(PRESENT(T_var_2)) T_var_2(n,m) = var_2(m,n)
      IF(PRESENT(T_var_3)) T_var_3(n,m) = var_3(m,n)
      IF(PRESENT(T_var_4)) T_var_4(n,m) = var_4(m,n)
      IF(PRESENT(T_var_5)) T_var_5(n,m) = var_5(m,n)
      IF(PRESENT(T_var_6)) T_var_6(n,m) = var_6(m,n)
      IF(PRESENT(T_var_7)) T_var_7(n,m) = var_7(m,n)
    END DO
    END DO
  END SUBROUTINE transpose_2D_fields



  SUBROUTINE transpose_3D_fields(var_1, T_var_1, var_2, T_var_2, var_3, T_var_3, var_4, T_var_4, var_5, T_var_5)
    ! This subroutine transposes the initial fields
    USE configuration_module, ONLY: dp, C
    IMPLICIT NONE

    ! Input variables:
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZ), INTENT(IN)            :: var_1
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZ), INTENT(IN) , OPTIONAL :: var_2
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZ), INTENT(IN) , OPTIONAL :: var_3
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZ), INTENT(IN) , OPTIONAL :: var_4
    REAL(dp), DIMENSION(C%NX,C%NY,C%NZ), INTENT(IN) , OPTIONAL :: var_5

    ! Output variables:
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX), INTENT(OUT)           :: T_var_1
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_2
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_3
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_4
    REAL(dp), DIMENSION(C%NZ,C%NY,C%NX), INTENT(OUT), OPTIONAL :: T_var_5

    ! Local variables:
    INTEGER                                                    :: m, n

    ! Transpose the read fields from {x,y,*} to {*,y,x}, or {m,n,:} to {:,n,m}:
    DO m = 1, C%NX 
    DO n = 1, C%NY
                           T_var_1(:,n,m) = var_1(m,n,:)
      IF(PRESENT(T_var_2)) T_var_2(:,n,m) = var_2(m,n,:)
      IF(PRESENT(T_var_3)) T_var_3(:,n,m) = var_3(m,n,:)
      IF(PRESENT(T_var_4)) T_var_4(:,n,m) = var_4(m,n,:)
      IF(PRESENT(T_var_5)) T_var_5(:,n,m) = var_5(m,n,:)
    END DO
    END DO
  END SUBROUTINE transpose_3D_fields

END MODULE initialization_module
