! File name: create_input_fields_program.f90
!
! Copyright (C) 2014 Thomas Reerink
!
! This file is part of the ICEDYN-package
!
! IMAU, Utrecht University, The Netherlands
!

PROGRAM create_input_fields_program
  ! This program creates the input netcdf file.
  !
  ! Run example: ./src/create_input_fields_program config-files/initialize_input_fields/config_initialize_input_fields_greenland_20x20km
  !
  ! Profiling:   gprof src/create_input_fields_program gmon.out > time_measurements.txt
  !
  USE configuration_module, ONLY: dp, C, initialize_config_variables, checking_specified_im_field_name, checking_specified_forcing_field_name
  USE oblimap_read_and_write_module, ONLY: oblimap_netcdf_file_type, oblimap_open_netcdf_file, oblimap_read_netcdf_fields, oblimap_close_netcdf_file
  USE netcdf_module, ONLY: netcdf_file_type, create_netcdf_file, close_netcdf_file
  USE restart_file_module, ONLY: initialize_restart_fields, write_restart_fields
  USE mask_module, ONLY: determine_mask
  USE robin_solution_module, ONLY: calculate_Ti_with_robin_solution
  USE smooth_module, ONLY: smooth

  USE initialization_module, ONLY: initialize_Ti_at_pressure_melting_point

  IMPLICIT NONE

  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE :: im_fields
  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE :: forcing_fields

  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: Hi
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: Hb
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: Hs
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: Us
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: Vs
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: bottom_melt
  INTEGER,  DIMENSION(  :,:), ALLOCATABLE :: mask
  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE :: Ti
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: Ts
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: smb 
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: Hi_smooth
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: Hb_smooth
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: adjusted_field
  
  TYPE(oblimap_netcdf_file_type)          :: nc
  TYPE(netcdf_file_type)                  :: restart_netcdf_file
  INTEGER                                 :: i, j, r
  REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: sealevel
  INTEGER                                 :: record_number = 1
  REAL(dp)                                :: time
  REAL(dp)                                :: integrated_smb
  INTEGER                                 :: counter

  ! Read the configuration file and initialization of the struckt C%:
  CALL initialize_config_variables()

  ! Allocate all arrays for which the dimension sizes are given in the config file:
  ALLOCATE(im_fields     (C%number_of_mapped_fields ,C%NX,C%NY))
  ALLOCATE(forcing_fields(C%number_of_forcing_fields,C%NX,C%NY))

  ALLOCATE(Hi            (     C%NY,C%NX)) 
  ALLOCATE(Hb            (     C%NY,C%NX)) 
  ALLOCATE(Hs            (     C%NY,C%NX)) 
  ALLOCATE(Us            (     C%NY,C%NX)) 
  ALLOCATE(Vs            (     C%NY,C%NX)) 
  ALLOCATE(bottom_melt   (     C%NY,C%NX)) 
  ALLOCATE(mask          (     C%NY,C%NX)) 
  ALLOCATE(Ti            (C%NZ,C%NY,C%NX)) 
  ALLOCATE(Ts            (     C%NY,C%NX)) 
  ALLOCATE(smb           (     C%NY,C%NX)) 
  ALLOCATE(Hi_smooth     (     C%NY,C%NX)) 
  ALLOCATE(Hb_smooth     (     C%NY,C%NX)) 
  ALLOCATE(adjusted_field(     C%NY,C%NX))
  ALLOCATE(sealevel      (     C%NY,C%NX))
  
  sealevel = 0._dp

  ! A fixed order of the im_fields is required, this order in the config file is checked here:
  CALL checking_specified_im_field_name(1, 'Ice Thickness'    )
  CALL checking_specified_im_field_name(2, 'Bedrock Elevation')
  CALL checking_specified_im_field_name(3, 'Surface Elevation')

  ! A fixed order of the forcing_fields is required, this order in the config file is checked here:
  CALL checking_specified_forcing_field_name( 1, 'smb'                    )
  CALL checking_specified_forcing_field_name( 2, 'ice_surface_temperature')


  ! Read the geometrical input data:

  ! Output: nc
  CALL oblimap_open_netcdf_file(C%im_input_filename, C%number_of_mapped_fields, C%im_field_name, C%NX, C%NY, nc = nc)

  ! Output: im_fields, time
  CALL oblimap_read_netcdf_fields(nc, 1, im_fields, time)

  ! Output: -
  CALL oblimap_close_netcdf_file(nc)

  ! Transpose the read fields from {x,y} to {y,x}:
  DO i = 1, C%NX 
  DO j = 1, C%NY
    Hi(j,i) = im_fields(1,i,j)
    Hb(j,i) = im_fields(2,i,j)
    Hs(j,i) = im_fields(3,i,j)
  END DO
  END DO


  ! Read the forcing data:

  ! Output: nc
  CALL oblimap_open_netcdf_file(C%forcing_fields_filename, C%number_of_forcing_fields, C%forcing_field_name, C%NX, C%NY, nc = nc)

  ! Output: forcing_fields, time
  CALL oblimap_read_netcdf_fields(nc, C%forcing_record, forcing_fields, time)

  ! Output: -
  CALL oblimap_close_netcdf_file(nc)


  ! Transpose the read fields from {x,y} to {y,x}:
  DO i = 1, C%NX 
  DO j = 1, C%NY
    smb(j,i) = forcing_fields(1,i,j)
    Ts (j,i) = forcing_fields(2,i,j)
  END DO
  END DO

  integrated_smb = 0._dp
  DO i = 1, C%NX 
  DO j = 1, C%NY
    IF(Hi(j,i) > C%Hi_min) integrated_smb = integrated_smb + smb(j,i) * C%dx * C%dy
  END DO
  END DO  
  print *, 'ice sheet integrated_smb [Gt yr-1] = ', integrated_smb / 1.E12_dp

  smb = (smb / 1000._dp) * (C%fresh_water_density / C%ice_density)  ! Conversion of mmweq to mieq

  ! Optional smoothing of bedrock:
  WRITE(UNIT=*,FMT='(A)')
  IF(C%smooth_radius > 0._dp) THEN
   CALL smooth(C%NX, C%NY, C%smooth_radius, C%shepard_exponent, Hi, Hi_smooth)
   CALL smooth(C%NX, C%NY, C%smooth_radius, C%shepard_exponent, Hb, Hb_smooth)
   Hi = Hi_smooth
   Hb = Hb_smooth
   WRITE(UNIT=*,FMT='(A, I4, A, F12.4)') ' create_input_fields_program: smoothing with a smooth radius = ', C%smooth_radius, ' and with a shepard exponent = ', C%shepard_exponent
  ELSE
   WRITE(UNIT=*,FMT='(A)') ' create_input_fields_program: No smoothing'
  END IF
  WRITE(UNIT=*,FMT='(A)')


  IF(C%replace_invalid_values_by_smooting) THEN
   ! Replacing the invalid values of the right lower corner in Hb by a reasonable estimate:
   CALL smooth(C%NX, C%NY, C%replace_invalid_values_smooth_radius, 2._dp, Hb, Hb_smooth, Hb /= C%invalid_input_value(1))
   WHERE(Hb == C%invalid_input_value(1)) Hb = Hb_smooth
  END IF


  IF(C%adjust_single_gap_points) THEN
   DO r = 1, 100
    adjusted_field = 0._dp
    counter = 0
    DO i = 2, C%NX-1
    DO j = 2, C%NY-1
      IF(Hb(j  ,i-1) - Hb(j,i) > C%single_gap_threshold .AND. Hb(j  ,i+1) - Hb(j,i) > C%single_gap_threshold .AND. Hb(j-1,i  ) - Hb(j,i) > C%single_gap_threshold .AND. Hb(j+1,i  ) - Hb(j,i) > C%single_gap_threshold) THEN
       adjusted_field(j,i) = ( Hb(j,i-1) + Hb(j,i+1) + Hb(j-1,i) + Hb(j+1,i) ) / 4._dp
       counter = counter + 1
      END IF
    END DO
    END DO
    WHERE(adjusted_field /= 0._dp) Hb = adjusted_field
    WHERE(adjusted_field /= 0._dp .AND. Hi > C%Hi_min) Hi = Hs - Hb
    WRITE(UNIT=*, FMT='(A, I10, A)') ' Adjustment of ', counter, ' points'
    IF(counter == 0) EXIT
   END DO
  END IF


  ! Creating a consistent set of Hi-Hb-Hs-mask and the flotation criterion:
  counter = 0
  DO i = 1, C%NX
  DO j = 1, C%NY
   IF(C%ice_density / C%seawater_density * Hi(j,i) < sealevel(j,i) - Hb(j,i)) THEN
    ! Floating situations:
    IF(Hi(j,i) <= C%Hi_min) THEN
     Hi(j,i) = C%Hi_min ! Setting a minimum ice thickness
    ELSE
     ! Keep Hi(j,i)
    !WRITE(UNIT=*, FMT='(A, 2I4, 3F12.4)') '  Floating point detected: ', i, j, Hi(j,i), Hb(j,i), Hs(j,i)
     counter = counter + 1
    END IF
    ! Keep Hb(j,i)
    Hs(j,i) = Hi(j,i) + sealevel(j,i) - C%ice_density / C%seawater_density * Hi(j,i)  ! For shelf points the ice thickness is leading, and the surface heigth is derived by combining ice thickness with the floatation criterion
    IF(C%ice_density / C%seawater_density * Hi(j,i) > sealevel(j,i) - Hb(j,i)) THEN
    !WRITE(UNIT=*, FMT='(A, 2I4)') ' An Hi adjustment leads to a shelf to sheet swap for point ', i, j
     Hs(j,i) = Hb(j,i) + Hi(j,i)  ! Hs for a grounded point
    END IF
   ELSE
    ! Grounded situations:
    IF(Hi(j,i) <= C%Hi_min) THEN
     ! Keep Hb(j,i)
     Hi(j,i) = C%Hi_min           ! Setting a minimum ice thickness
     Hs(j,i) = Hb(j,i) + C%Hi_min ! For ice free sheet points the bedrock level is leading, and the surface heigth is derived by taking the bedrock level plus the minimum ice thickness
    ELSE
     ! Keep Hi(j,i)
     ! Keep Hs(j,i)
     Hb(j,i) = Hs(j,i) - Hi(j,i)  ! For ice covered sheet points the surface heigth is leading, and the bedrock level is derived by taking the surface heigth minus the ice thickness
     IF(C%ice_density / C%seawater_density * Hi(j,i) < sealevel(j,i) - Hb(j,i)) THEN
     !WRITE(UNIT=*, FMT='(A, 2I4)') ' An Hb adjustment leads to a sheet to shelf swap for point ', i, j
      Hs(j,i) = Hi(j,i) + sealevel(j,i) - C%ice_density / C%seawater_density * Hi(j,i) ! Hs for a floating point
     END IF
    END IF
   END IF
  END DO
  END DO
  WRITE(UNIT=*, FMT='(A, I10, A)') ' There are ', counter, ' floating points detected'


  ! Output: mask
  CALL determine_mask(sealevel, Hi, Hb, mask)


  ! Finally check the floatation - mask - Hb - Hi - Hs consistency:
  counter = 0
  DO i = 1, C%NX
  DO j = 1, C%NY
   IF(C%ice_density / C%seawater_density * Hi(j,i) < sealevel(j,i) - Hb(j,i)) THEN
    IF(mask(j,i) /= C%type_shelf) WRITE(UNIT=*, FMT='(A)') ' Inconsistent mask'
    IF(Hs(j,i) /= Hi(j,i) + sealevel(j,i) - C%ice_density / C%seawater_density * Hi(j,i)) THEN
     WRITE(UNIT=*, FMT=*) ' Inconsistency in:  Hs = sealevel - (1 - C%ice_density / C%seawater_density) Hi at the shelf'
     counter = counter + 1
    END IF
   ELSE
    IF(mask(j,i) == C%type_shelf) WRITE(UNIT=*, FMT='(A)') ' Inconsistent mask'
    IF(Hs(j,i) /= Hb(j,i) + Hi(j,i)) THEN
     WRITE(UNIT=*, FMT=*) ' Inconsistency in:  Hs = Hb + Hi  at the sheet', i, j, mask(j,i), Hs(j,i), Hb(j,i) + Hi(j,i), Hb(j,i), Hi(j,i)
     counter = counter + 1
    END IF
   END IF
  END DO
  END DO
  IF(counter == 1) THEN
   WRITE(UNIT=*, FMT='(A, I10, A)') ' There is ',  counter, ' inconsistent point found'
  ELSE
   WRITE(UNIT=*, FMT='(A, I10, A)') ' There are ', counter, ' inconsistent points found'
  END IF


  ! Initialize the 3D ice temperature field using the Robin solution:
  ! Output: Ti
  !CALL calculate_Ti_with_robin_solution(Ts, Hi, smb, mask, Ti)

  CALL initialize_Ti_at_pressure_melting_point(Ts, Hi, 1._dp, mask, Ti)


  ! Set the initial dHi_dt and dHb_dt to zero:
  Us          = 0._dp
  Vs          = 0._dp
  bottom_melt = 0._dp


  ! Output: restart_netcdf_file
  CALL initialize_restart_fields(C%initial_restart_filename, restart_netcdf_file)

  ! In/Output: restart_netcdf_file
  CALL create_netcdf_file(restart_netcdf_file, .FALSE.)

  ! Output: -
  CALL write_restart_fields(restart_netcdf_file, record_number, time, Hi, Hb, Hs, Us, Vs, bottom_melt, Ti)

  ! Finish writing initial file:
  CALL close_netcdf_file(restart_netcdf_file)


  ! Finishing message:
  WRITE(UNIT=*, FMT='(/3A/2A/)') ' Finished! The file  ', TRIM(C%initial_restart_filename), '  is created. Which can be viewed by:', '  ncview ', TRIM(C%initial_restart_filename)

  DEALLOCATE(im_fields     )
  DEALLOCATE(forcing_fields)

  DEALLOCATE(Hi            )
  DEALLOCATE(Hb            )
  DEALLOCATE(Hs            )
  DEALLOCATE(Us            )
  DEALLOCATE(Vs            )
  DEALLOCATE(bottom_melt   )
  DEALLOCATE(mask          )
  DEALLOCATE(Ti            )
  DEALLOCATE(Ts            )
  DEALLOCATE(smb           )
  DEALLOCATE(Hi_smooth     )
  DEALLOCATE(Hb_smooth     )
  DEALLOCATE(adjusted_field)

END PROGRAM create_input_fields_program
