MODULE zeta_derivatives_ant_module
  USE configuration_main_module, ONLY: dp, C
  IMPLICIT NONE

 ! This TYPE contains the derivatives of zeta, which are used in the coordinate 
 ! transformations (see table 2). The components of this type are allocated in the main program
 TYPE zeta_derivatives_ant                
   REAL(dp), DIMENSION(:,:,:), ALLOCATABLE :: t  ! zeta_t   or  dzeta_dt
   REAL(dp), DIMENSION(:,:,:), ALLOCATABLE :: x  ! zeta_x   or  dzeta_dx 
   REAL(dp), DIMENSION(:,:,:), ALLOCATABLE :: y  ! zeta_y   or  dzeta_dy 
   REAL(dp), DIMENSION(  :,:), ALLOCATABLE :: z  ! zeta_z   or  dzeta_dz        
 END TYPE zeta_derivatives_ant

 ! Declaring a global struct which contains all derivatives of the scaled
 ! coordinate zeta. This global struct is used in the thermodynamical and
 ! uvw_mumps routine and updated each time step in anice.
 TYPE(zeta_derivatives_ant), SAVE :: dzeta                                    ! Global variable !

CONTAINS
  SUBROUTINE calculate_zeta_derivatives(Hi,             &
                                        dHi_dt,         &
                                        dHi_dx, dHi_dy, &
                                        dHs_dt,         &
                                        dHs_dx, dHs_dy  )
    ! This subroutine calculates the global struct which contains the derivatives of 
    ! zeta, which are used in the transformation to the t,x,y,zeta coordinates. 
    ! This zeta derivatives are the Jacobians. See table 3.
    
    ! Input variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: dHi_dt
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: dHi_dx, dHi_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: dHs_dt                     ! dHs_dt = dHi_dt + dHb_dt
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: dHs_dx, dHs_dy

    ! Local variables:
    INTEGER                                    :: i, j, k
    REAL(dp)                                   :: inverse_Hi                 ! Contains the inverse of Hi
    REAL(dp)                                   :: inverse_Hi2                ! Contains the squared inverse of Hi
    
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      inverse_Hi  = 1._dp / Hi(i,j)
      inverse_Hi2 = inverse_Hi * inverse_Hi

      dzeta%z(i,j)  = -inverse_Hi                                                      
      DO k = 1, C%NZ
        dzeta%t(i,j,k)  =  inverse_Hi * (dHs_dt(i,j)  - C%zeta(k) * dHi_dt(i,j))                                   
        dzeta%x(i,j,k)  =  inverse_Hi * (dHs_dx(i,j)  - C%zeta(k) * dHi_dx(i,j))                                   
        dzeta%y(i,j,k)  =  inverse_Hi * (dHs_dy(i,j)  - C%zeta(k) * dHi_dy(i,j))                                   
      END DO
    END DO
    END DO
  END SUBROUTINE calculate_zeta_derivatives
END MODULE zeta_derivatives_ant_module
