! In this module ascii output is written in two files:
! 1 - standard output of the ice sheet, means over the entire ice covered surface
! 2 - mass balance variables
! New module added BdB: 02/2013

MODULE write_ascii_output_ant_module
  USE configuration_main_module, ONLY: dp, C
  USE configuration_ant_module, ONLY: T_ANT
  USE insolation_forcing_module, ONLY: insolation_data
  IMPLICIT NONE

CONTAINS

  SUBROUTINE write_means_and_mb_icesheet(sealevel, global_temperature, mean_isoice, mean_temp, slev_contr, calve_mask, &
                                         Hi, dHi_dt, MB_surface, MB_basal, Accum, Abla, Refr, fluxes_per_dt, icedis_field)
    ! determine the applied mass changes for the separate mass balance components 
    ! and calculate the flux towards to ocean, setting Hi to Hi_min for ocean grid points with Hi < calve_thickness ( = 200 m default)


    REAL(dp),                                   INTENT(IN) :: sealevel
    REAL(dp),                                   INTENT(IN) :: global_temperature
    REAL(dp),                                   INTENT(IN) :: mean_isoice
    REAL(dp),                                   INTENT(IN) :: mean_temp
    REAL(dp),                                   INTENT(IN) :: slev_contr

    INTEGER,  DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN) :: calve_mask
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN) :: dHi_dt
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN) :: Hi
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN) :: MB_surface
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN) :: MB_basal
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN) :: Accum
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN) :: Abla
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN) :: Refr
    ! all fluxes of ice_thickness change
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, 6), INTENT(IN) :: fluxes_per_dt
    
    ! ice discharge as a map
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(OUT) :: icedis_field

    ! local variables
    INTEGER                                                :: i,j
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)                 :: MB_surf_appl, MB_bas_appl, MB_appl_per_yr
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)                 :: Abla_appl, Acc_appl, Rfr_appl

    REAL(dp)                                               :: volume_land, volume_shelf
    REAL(dp)                                               :: area_land, area_shelf

    REAL(dp)                                               :: total_Hi_change, total_MB_change, total_icedis
    REAL(dp)                                               :: total_SMB_change, total_BMB_change
    REAL(dp)                                               :: total_Acc_change, total_Abl_change, total_Rfr_change
    REAL(dp)                                               :: dHi_posneg  ! positive or negative mass change indicator (+1 or -1)
    REAL(dp)                                               :: weight      ! weighting of oceanic melt

    REAL(dp), PARAMETER                                    :: m3_to_Gt_ice = 9.1E-10_dp ! to convert from cubic meters ice to Gt (ice density divided by 10E12)
    
! Added BdB (11/2009): Calculate volumes and areas of ice sheet
!
! Ice volume is now calculated as total ice over land and the ice over ocean, above the calve_thickness.
! The sealevel contribution only takes into account ice volume over land. Although a correction for ice below sea level 
! could be applied, if one assumes that when all ice is removed, the land below sea level will be filled with ocean water, but 
! then again also GIA should be taken into account.... For now this is ok. 
!
! applied mass balance is the corrected MB from the mass continuity equation (MB applied is per time step!), to conserve mass

    
    ! total grounded volume, omitting soil and shelf/ocean points
    volume_land   = C%dx_ant * C%dy_ant * SUM(Hi, MASK = (calve_mask <= C%type_groundline .AND. Hi > C%Hi_min))
    ! same as above, but the area
    area_land     = C%dx_ant * C%dy_ant * COUNT(calve_mask <= C%type_groundline .AND. Hi >  C%Hi_min)
    ! total shelf volume, omitting ocean points with Hi < calve_thickness (= 200 m)
    volume_shelf  = C%dx_ant * C%dy_ant * SUM(Hi, MASK = (calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline))
    ! same as above, but the area
    area_shelf    = C%dx_ant * C%dy_ant * COUNT(calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline)

    ! The fluxes per time step ar defined as:
    ! 1 = ice flux towards i+1
    ! 2 = ice flux towards i-1
    ! 3 = ice flux towards j+1
    ! 4 = ice flux towards j-1
    ! 5 = total advective flux (1+2+3+4)
    ! 6 = total MB applied
    
    ! Initialize all variables from the original data
    MB_appl_per_yr = fluxes_per_dt(:,:,6) / T_ANT%dt
    MB_surf_appl = MB_surface
    MB_bas_appl  = MB_basal
    Abla_appl    = Abla
    Acc_appl     = Accum
    Rfr_appl     = Refr

    ! When mb_applied is different from mb, adapt the smallest MB term (so the negative one, MB_basal or MB_surface), 
    ! if this is MB_surface, change ablation. Save MB_surface, MB_basal and ablation in local arrays, so they can be adjusted 
    ! for calculating the total mass balance over the ice sheet
    WHERE ( MB_appl_per_yr /= 0._dp .AND. (MB_appl_per_yr - MB_surface - MB_basal) > 0._dp  ) 
      WHERE (Abla > (MB_appl_per_yr - MB_surface - MB_basal))
        MB_surf_appl = MB_appl_per_yr - MB_basal
        Abla_appl    = Abla + MB_surface + MB_basal - MB_appl_per_yr
      ELSE WHERE
        MB_bas_appl  = MB_appl_per_yr - MB_surface
      END WHERE
    END WHERE
    
    ! Set MB terms to zero where MB applied is zero (mostly where Hi = C%Hi_min, so no ice)
    WHERE (MB_appl_per_yr == 0._dp) 
      MB_bas_appl  = 0._dp
      MB_surf_appl = 0._dp
      Acc_appl     = 0._dp
      Abla_appl    = 0._dp
      Rfr_appl     = 0._dp
    END WHERE
    
    total_icedis = 0._dp
    icedis_field = 0._dp


    ! Determine the ice flux towards the ocean
    DO j = 2, C%NY_ant - 1
    DO i = 2, C%NX_ant - 1
      ! NET calving from ice shelves
      IF (calve_mask(i,j) == C%type_calveline) THEN
        IF (calve_mask(i+1,j  ) == C%type_ocean) total_icedis = total_icedis - (fluxes_per_dt(i,j,1) - fluxes_per_dt(i+1,j  ,2)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i-1,j  ) == C%type_ocean) total_icedis = total_icedis - (fluxes_per_dt(i,j,2) - fluxes_per_dt(i-1,j  ,1)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j+1) == C%type_ocean) total_icedis = total_icedis - (fluxes_per_dt(i,j,3) - fluxes_per_dt(i  ,j+1,4)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j-1) == C%type_ocean) total_icedis = total_icedis - (fluxes_per_dt(i,j,4) - fluxes_per_dt(i  ,j-1,3)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        ! for the field:
        IF (calve_mask(i+1,j  ) == C%type_ocean) icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,1) - fluxes_per_dt(i+1,j  ,2)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i-1,j  ) == C%type_ocean) icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,2) - fluxes_per_dt(i-1,j  ,1)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j+1) == C%type_ocean) icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,3) - fluxes_per_dt(i  ,j+1,4)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j-1) == C%type_ocean) icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,4) - fluxes_per_dt(i  ,j-1,3)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
      END IF
      ! NET calving from the grounding line
      IF (calve_mask(i,j) == C%type_groundline) THEN
        IF (calve_mask(i+1,j  ) == C%type_ocean) total_icedis = total_icedis - (fluxes_per_dt(i,j,1) - fluxes_per_dt(i+1,j  ,2)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i-1,j  ) == C%type_ocean) total_icedis = total_icedis - (fluxes_per_dt(i,j,2) - fluxes_per_dt(i-1,j  ,1)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j+1) == C%type_ocean) total_icedis = total_icedis - (fluxes_per_dt(i,j,3) - fluxes_per_dt(i  ,j+1,4)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j-1) == C%type_ocean) total_icedis = total_icedis - (fluxes_per_dt(i,j,4) - fluxes_per_dt(i  ,j-1,3)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        ! for the field:
        IF (calve_mask(i+1,j  ) == C%type_ocean) icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,1) - fluxes_per_dt(i+1,j  ,2)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i-1,j  ) == C%type_ocean) icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,2) - fluxes_per_dt(i-1,j  ,1)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j+1) == C%type_ocean) icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,3) - fluxes_per_dt(i  ,j+1,4)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j-1) == C%type_ocean) icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,4) - fluxes_per_dt(i  ,j-1,3)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
      END IF
      ! NET calving from grounded ice to land points
      IF (calve_mask(i,j) <= C%type_groundline) THEN
        IF (calve_mask(i+1,j  ) <= C%type_groundline .AND. Hi(i+1,j  ) == C%Hi_min) & 
                                                 total_icedis = total_icedis - (fluxes_per_dt(i,j,1) - fluxes_per_dt(i+1,j  ,2)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i-1,j  ) <= C%type_groundline .AND. Hi(i-1,j  ) == C%Hi_min) & 
                                                 total_icedis = total_icedis - (fluxes_per_dt(i,j,2) - fluxes_per_dt(i-1,j  ,1)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j+1) <= C%type_groundline .AND. Hi(i  ,j+1) == C%Hi_min) & 
                                                 total_icedis = total_icedis - (fluxes_per_dt(i,j,3) - fluxes_per_dt(i  ,j+1,4)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j-1) <= C%type_groundline .AND. Hi(i  ,j-1) == C%Hi_min) & 
                                                 total_icedis = total_icedis - (fluxes_per_dt(i,j,4) - fluxes_per_dt(i  ,j-1,3)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        ! for the field:
        IF (calve_mask(i+1,j  ) <= C%type_groundline .AND. Hi(i+1,j  ) == C%Hi_min) & 
                                                 icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,1) - fluxes_per_dt(i+1,j  ,2)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i-1,j  ) <= C%type_groundline .AND. Hi(i-1,j  ) == C%Hi_min) & 
                                                 icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,2) - fluxes_per_dt(i-1,j  ,1)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j+1) <= C%type_groundline .AND. Hi(i  ,j+1) == C%Hi_min) & 
                                                 icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,3) - fluxes_per_dt(i  ,j+1,4)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
        IF (calve_mask(i  ,j-1) <= C%type_groundline .AND. Hi(i  ,j-1) == C%Hi_min) & 
                                                 icedis_field(i,j) = icedis_field(i,j) - (fluxes_per_dt(i,j,4) - fluxes_per_dt(i  ,j-1,3)) * C%dx_ant * C%dy_ant * m3_to_Gt_ice / T_ANT%dt
     END IF
    END DO
    END DO


    ! total changes in ice mass balance and fluxes in m^3 / yr ( * 910 kg / m3 = kg / yr ==> 1 Gt / yr = 10^12 kg / yr)
    ! only for Hi > Hi_min over land and Hi > calve_thickness over ocean:
    total_Hi_change    = C%dx_ant * C%dy_ant * SUM(dHi_dt, MASK = ((calve_mask < C%type_shelf .AND. Hi >  C%Hi_min)                  &
                                             .OR. calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline)) * m3_to_Gt_ice
    total_MB_change    = C%dx_ant * C%dy_ant * SUM(MB_appl_per_yr, MASK = ((calve_mask < C%type_shelf .AND. Hi >  C%Hi_min)          &
                                             .OR. calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline)) * m3_to_Gt_ice
    total_SMB_change   = C%dx_ant * C%dy_ant * SUM(MB_surf_appl, MASK = ((calve_mask < C%type_shelf .AND. Hi >  C%Hi_min)            &
                                             .OR. calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline)) * m3_to_Gt_ice
    total_BMB_change   = C%dx_ant * C%dy_ant * SUM(MB_bas_appl, MASK = ((calve_mask < C%type_shelf .AND. Hi >  C%Hi_min)             &
                                             .OR. calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline)) * m3_to_Gt_ice

    ! Surface mass balance contributions over all grid points
    total_Acc_change   = C%dx_ant * C%dy_ant * SUM(Acc_appl, MASK = ((calve_mask < C%type_shelf .AND. Hi >  C%Hi_min)                &
                                             .OR. calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline)) * m3_to_Gt_ice
    total_Abl_change   =  - C%dx_ant * C%dy_ant * SUM(Abla_appl, MASK = ((calve_mask < C%type_shelf .AND. Hi >  C%Hi_min)            &
                                             .OR. calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline)) * m3_to_Gt_ice
    total_Rfr_change   = C%dx_ant * C%dy_ant * SUM(Rfr_appl, MASK = ((calve_mask < C%type_shelf .AND. Hi >  C%Hi_min)                &
                                             .OR. calve_mask == C%type_shelf .OR. calve_mask == C%type_calveline)) * m3_to_Gt_ice

      dHi_posneg =  0._dp
    IF (total_Hi_change > 0._dp) THEN
      dHi_posneg =  1._dp
    ELSE IF (total_Hi_change < 0._dp) THEN
      dHi_posneg = -1._dp
    END IF
!
! --- WRITE STANDARD OUTPUT OF THE ICE SHEET
! 
    WRITE(4001, 103) T_ANT%time / 1000._dp, sealevel, global_temperature, mean_temp, mean_isoice, slev_contr,   &
                     volume_land+volume_shelf, volume_land, volume_shelf, area_land, area_shelf

    CALL FLUSH(4001)
!
! --- WRITE MASS BALANCE VARIABLES OF THE ICE SHEET
!
    ! wright weighting factor of oceanic melt in file
    weight = MAX(0._dp, MIN(2._dp, 1._dp + global_temperature/12._dp + MAX(0._dp, (insolation_data%Jan80S - 532.19_dp)/40._dp)))
   
    WRITE(4011, 105) T_ANT%time / 1000._dp, total_Hi_change, total_MB_change, total_icedis, total_BMB_change,  &
                     total_SMB_change, total_Acc_change, total_Abl_change, total_Rfr_change, dHi_posneg,       &
                     weight, global_temperature/12._dp, MAX(0._dp, (insolation_data%Jan80S - 532.19_dp)/40._dp)

    CALL FLUSH(4011)
!
! --- Formats
!

103 format (2x,6F12.3,5E18.6)
105 format (9F14.3,F8.1,3F10.3)

  END SUBROUTINE write_means_and_mb_icesheet

END MODULE write_ascii_output_ant_module
