MODULE velocity_ant_module
  USE configuration_main_module, ONLY: dp, C
  USE configuration_ant_module, ONLY: C_ANT, T_ANT
  IMPLICIT NONE
  
CONTAINS
  SUBROUTINE vertical_integrate(f, int_f)
    ! This subroutine calcualtes the integral int_f:
    !  int_f(k) = INTEGRAL_bottom[C%zeta(k=C%NZ)=1]^zeta[C%zeta(k)] f(zeta) dzeta
    ! In case the integrant f is positive (our cases) our result must be negative because we
    ! integrate from C%zeta(k=C%NZ) = 1 opposite to the zeta-axis up to C%zeta(k). (Our dzeta's are 
    ! negative).
    ! This subroutine returns the integral for each layer k from bottom to this layer, so
    ! inf_f is an array with length C%NZ: int_f(k=1:C%NZ)
    ! The value of the integrant f at some integration step k is the average of f(k+1) and f(k).
    !  int_f(k) = int_f(k+1) + 0.5*(f(k+1) + f(k))*(-dzeta)
    ! So for f > 0  int_f < 0.
    
    ! Input variable:
    REAL(dp), DIMENSION(C%NZ), INTENT(IN) :: f

    ! Output variable:
    REAL(dp), DIMENSION(C%NZ), INTENT(OUT):: int_f  
    
    ! Local variable:
    INTEGER                               :: k
    
    int_f(C%NZ) = 0._dp

    DO k = C%NZ-1, 1, -1
      int_f(k) = int_f(k+1) - 0.5_dp * (f(k+1) + f(k)) * (C%zeta(k+1) - C%zeta(k))
    END DO
  END SUBROUTINE vertical_integrate


  
  SUBROUTINE calculate_D_uv_3D(Hi, dHs_dx, dHs_dy, Hb, sealevel, Ti_bottom, Ti_pmp_bottom, A_flow, mask_check, D_uv_3D)
    ! This subroutine calculates the three-dimensional (3D) common part in the uv calculation
    ! for grounded points following the integral-SIA approach. For the ground line there is an 
    ! additional second term D_gl_3D (see (7.63)).
    USE parameters_main_module, ONLY: n_flow, m_flow, ice_density, seawater_density, grav, A_sliding
    
    ! Input variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHs_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHs_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Hb
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: sealevel    
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Ti_bottom
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Ti_pmp_bottom
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: A_flow
    LOGICAL,  DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: mask_check  ! Selecting the sheet or groundline points or both.
    
    ! Output variable:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(OUT) :: D_uv_3D

    ! Local variables:
    INTEGER                                          :: i, j, k
    REAL(dp)                                         :: f_buoyancy_correction
    REAL(dp)                                         :: D_0, D_0_slid
    REAL(dp), DIMENSION(C%NZ)                        :: D_deformation
    REAL(dp)                                         :: D_sliding
    
    REAL(dp), PARAMETER                              :: cutoff = 1._dp            ! To be determined....

    ! Initializing each time step all to zero (no old values at points which switched to shelf):
    D_uv_3D = 0._dp

!DoE 02/06/2017
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      ! Considering only grounded points:
      IF(mask_check(i,j)) THEN
       
       ! Sliding or no sliding:
       IF(((C_ANT%choice_sliding_method == 1) .AND. (Ti_bottom(i,j) >= Ti_pmp_bottom(i,j) - C%melt_range)) .OR. C%always_sliding) THEN
        ! Sliding calculation in case Ti is near or above the pressure melting point Ti_pmp
        
        ! Calculation of the buoyancy correction factor for different situations:
        IF(Hb(i,j) >= sealevel(i,j)) THEN                                                                                    
         ! In this case we are above sealevel, no sealevel-buoyancy effect at all
         ! Just devision by Hi:                                                  
         ! See equation ():
         f_buoyancy_correction = 1._dp / Hi(i,j)
        ELSE                                                                                                            
         ! Floating points are already deselected in the mask_check procedure
         ! so if the mask is always updatted, no devision by zero will be made: 
         ! See equation ():
         IF((Hb(i,j) - sealevel(i,j)) > cutoff) THEN
          f_buoyancy_correction = 1._dp / (Hi(i,j) + (seawater_density / ice_density) * (Hb(i,j) - sealevel(i,j)))
         ELSE
         ! Near the floating criterium  (Hb - sealevel)  decreases to very low values, 
         ! so here we require that v_sliding has a maximum corresponding with:
         !  Hb(i,j) - sealevel(i,j) = cutoff:
          f_buoyancy_correction = 1._dp / (Hi(i,j) + (seawater_density / ice_density) * cutoff)
          !WRITE(333, FMT='(2(a, e12.3), a, 2i4)') 'Hb - sealevel', Hb(i,j) - sealevel(i,j), 'is set to', cutoff, ' at i, j= ', i, j
         END IF
        END IF
        
         ! See equation (7.38):
         D_sliding = - f_buoyancy_correction * A_sliding

       ELSE
        ! No sliding:
        D_sliding = 0._dp
       END IF ! End: Sliding or no sliding

       ! See equation (7.31):
       D_0      = (ice_density * grav * Hi(i,j))**n_flow * ((dHs_dx(i,j)**2 + dHs_dy(i,j)**2))**((n_flow - 1._dp) / 2._dp)

       ! if m_flow is different D_0 for sliding changes...
       D_0_slid = (ice_density * grav * Hi(i,j))**m_flow * ((dHs_dx(i,j)**2 + dHs_dy(i,j)**2))**((m_flow - 1._dp) / 2._dp)
       
       ! See equation (7.37), integration from bottom up to zeta:
       CALL vertical_integrate(A_flow(i,j,:) * C%zeta(:)**n_flow, D_deformation)
       D_deformation = 2._dp * Hi(i,j) * D_deformation
       
       ! See equation (7.36):
       D_uv_3D(i,j,:) = D_0 * D_deformation(:) + D_0_slid * D_sliding

      END IF ! End: Considering only grounded points
      
      ! Check for very large D_uv_3D's, causing very large velocities RESET --DIRTY
      DO k = 1, C%NZ
       IF(D_uv_3D(i,j,k) < C%d_uv_3d_cutoff) THEN
!        IF(k==1) WRITE(UNIT=C%stdlog, FMT='(A, I3, A, I3, A, I2, A, E12.3, A, E12.3, A, F14.2)') &
!         'ANT D_uv_3D(i=', i, ',j=', j, ',k=', k, ') = ', D_uv_3D(i,j,k), ' is resetted to ', C%d_uv_3d_cutoff, ', at time ', T_ANT%time
        D_uv_3D(i,j,k) = C%d_uv_3d_cutoff
       END IF
      END DO

    END DO
    END DO


  END SUBROUTINE calculate_D_uv_3D


  SUBROUTINE sheet_velocity_2D(Hi, dHs_dx, dHs_dy, D_uv_3D, mask_check, Us, Vs, D_2D)
    ! Implementation of the formula for the velocity, which can be used on groundline and sheet.
    USE parameters_main_module, ONLY: vertical_average
    
    ! Input variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHs_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHs_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: D_uv_3D
    LOGICAL,  DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: mask_check

    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(OUT) :: Us
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(OUT) :: Vs
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(OUT) :: D_2D

    ! Local variables:
    INTEGER                                                  :: i, j
    REAL(dp)                                                 :: D_uv_2D

    ! Initializing each time step all to zero (zero for shelf and ocean)
    D_2D = 0._dp
    Us   = 0._dp
    Vs   = 0._dp
                                                    
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      IF(mask_check(i,j)) THEN
       D_uv_2D = vertical_average(D_uv_3D(i,j,:))
       
       D_2D(i,j)  = Hi(i,j) * D_uv_2D     ! See equation (8.5)
       Us(i,j) = D_uv_2D * dHs_dx(i,j)    ! See equation (7.39)
       Vs(i,j) = D_uv_2D * dHs_dy(i,j)    ! See equation (7.40)
      END IF
      
    END DO
    END DO
  END SUBROUTINE sheet_velocity_2D
  

  SUBROUTINE grounded_area_velocity_3d(Hi, dHb_dt, dHi_dx, dHi_dy, dHs_dx, dHs_dy, D_uv_3D, U_slid, V_slid, mask, U_stable, V_stable, W_stable)
    ! The 3d arrays U, V and W are calculated on mask == C%type_sheet points and on mask == C%type_groundline points
    
    ! Input variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHb_dt
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHi_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHi_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHs_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHs_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: D_uv_3D
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: mask

    ! The sliding velocities, calculated with the SSA
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: U_slid
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: V_slid
                                                     
    ! Output variables:  
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(OUT) :: U_stable
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(OUT) :: V_stable
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(OUT) :: W_stable
      
    ! Local variables:    
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ)              :: U
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ)              :: V
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     )              :: dHb_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     )              :: dHb_dy
    INTEGER                                                   :: i, j, k
    REAL(dp)                                                 :: w1, w2, w3, w4
                                                     
    ! Setting the shelf values to zero:
    U_stable = 0._dp
    V_stable = 0._dp
    W_stable = 0._dp
    U        = 0._dp
    V        = 0._dp

    ! Direct caluculation of U and V for the ice sheet:
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      IF(mask(i,j) /= C%type_shelf) THEN 
       dHb_dx(i,j) = dHs_dx(i,j) - dHi_dx(i,j)
       dHb_dy(i,j) = dHs_dy(i,j) - dHi_dy(i,j)
       
       ! To be completely consistent, only add the SSA basal velocities when calculated:
       IF (C_ANT%choice_sliding_method == 2) THEN
        U(i,j,:)    = D_uv_3D(i,j,:) * dHs_dx(i,j) + U_slid(i,j)
        V(i,j,:)    = D_uv_3D(i,j,:) * dHs_dy(i,j) + V_slid(i,j)
       ! Sliding is calculated with Weertman type sliding law (included in D_uv) or no sliding is used:       
       ELSE
        U(i,j,:)    = D_uv_3D(i,j,:) * dHs_dx(i,j)
        V(i,j,:)    = D_uv_3D(i,j,:) * dHs_dy(i,j)
       END IF  
      END IF
    END DO
    END DO

    IF(.TRUE.) THEN
     ! Caluculation of a staggered U and V for grounded points:
     DO j = 2, C%NY_ant-1
     DO i = 2, C%NX_ant-1
       IF(mask(i,j) == C%type_sheet) THEN
        U_stable(i,j,:) = 0.25_dp * U(i-1,j  ,:) + 0.5_dp * U(i,j,:) + 0.25_dp * U(i+1,j  ,:)
        V_stable(i,j,:) = 0.25_dp * V(i  ,j-1,:) + 0.5_dp * V(i,j,:) + 0.25_dp * V(i  ,j+1,:)
       ELSE !  IF groundline or shelf:
        U_stable(i,j,:) = U(i,j,:)
        V_stable(i,j,:) = V(i,j,:)
       END IF
     END DO
     END DO
     CALL horizontal_domain_edges_set_by_neumann(U_stable)
     CALL horizontal_domain_edges_set_by_neumann(V_stable)
    ELSE
     ! This option performs the non-staggered approach:
     U_stable = U
     V_stable = V
    END IF

    DO j = 2, C%NY_ant-1
    DO i = 2, C%NX_ant-1
      IF(mask(i,j) /= C%type_shelf) THEN
       W_stable(i,j,C%NZ) = dHb_dt(i,j) + U_stable(i,j,C%NZ) * dHb_dx(i,j) + V_stable(i,j,C%NZ) * dHb_dy(i,j)                      
       ! The integrant is calculated half way the layer of integration at k+1/2. This integrant is multiplied with the layer thickness and added to the integral
       ! of all layers below, giving the integral up to and including this layer:
       DO k = C%NZ - 1, 1, -1
         IF(mask(i,j) == C%type_sheet) THEN
          w1             = (0.5_dp * (U_stable(i+1,j  ,k+1) + U_stable(i+1,j  ,k)) -    &
                            0.5_dp * (U_stable(i-1,j  ,k+1) + U_stable(i-1,j  ,k))) / (2._dp * C%dx_ant)
          w2             = (0.5_dp * (V_stable(i  ,j+1,k+1) + V_stable(i  ,j+1,k)) -    &
                            0.5_dp * (V_stable(i  ,j-1,k+1) + V_stable(i  ,j-1,k))) / (2._dp * C%dy_ant)
                            
         ELSE !  IF groundline:
          ! First order gradients (only single groundline points use a zero-shelf value):
          IF(mask(i-1,j) /= C%type_shelf) THEN
           w1            = (0.5_dp * (U_stable(i  ,j  ,k+1) + U_stable(i  ,j  ,k))  -   &
                            0.5_dp * (U_stable(i-1,j  ,k+1) + U_stable(i-1,j  ,k)) ) / (C%dx_ant)
          ELSE
           w1            = (0.5_dp * (U_stable(i+1,j  ,k+1) + U_stable(i+1,j  ,k))  -   &
                            0.5_dp * (U_stable(i  ,j  ,k+1) + U_stable(i  ,j  ,k)) ) / (C%dx_ant)
          END IF
          IF(mask(i,j-1) /= C%type_shelf) THEN
           w2            = (0.5_dp * (V_stable(i  ,j  ,k+1) + V_stable(i  ,j  ,k))  -   &
                            0.5_dp * (V_stable(i  ,j-1,k+1) + V_stable(i  ,j-1,k)) ) / (C%dy_ant)
          ELSE
           w2            = (0.5_dp * (V_stable(i  ,j+1,k+1) + V_stable(i  ,j+1,k))  -   &
                            0.5_dp * (V_stable(i  ,j  ,k+1) + V_stable(i  ,j  ,k)) ) / (C%dy_ant)
          END IF
         END IF
         w3              = ((dHs_dx(i,j) - 0.5_dp * (C%zeta(k+1) + C%zeta(k)) * dHi_dx(i,j)) / Hi(i,j)) *   &
                           ((U_stable(i,j,k+1) - U_stable(i,j,k)) / (C%zeta(k+1) - C%zeta(k)))
         w4              = ((dHs_dy(i,j) - 0.5_dp * (C%zeta(k+1) + C%zeta(k)) * dHi_dy(i,j)) / Hi(i,j)) *   &
                           ((V_stable(i,j,k+1) - V_stable(i,j,k)) / (C%zeta(k+1) - C%zeta(k)))

         W_stable(i,j,k) = W_stable(i,j,k+1) - Hi(i,j) * (w1 + w2 + w3 + w4) * (C%zeta(k+1) - C%zeta(k))
       END DO
      END IF
    END DO
    END DO
    CALL horizontal_domain_edges_set_by_neumann(W_stable)

  END SUBROUTINE grounded_area_velocity_3d



  SUBROUTINE horizontal_domain_edges_set_by_neumann(field_var)
    ! In this subroutine the horizontal edges of a 3D field variable are set by a neumann condition
    ! i.e. in this case the closest inward neighbour values are copied at each edge point.

    ! In/Output variable:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(INOUT) :: field_var
    
    field_var(2:C%NX_ant, C%NY_ant, :) = field_var(2:C%NX_ant, C%NY_ant-1, :)
    field_var(2:C%NX_ant,        1, :) = field_var(2:C%NX_ant,          2, :)
    field_var(C%NX_ant  ,        :, :) = field_var(C%NX_ant-1,          :, :)
    field_var(         1,        :, :) = field_var(         2,          :, :)
  END SUBROUTINE horizontal_domain_edges_set_by_neumann
END MODULE velocity_ant_module
