MODULE time_recording_ant_module
  USE configuration_main_module, ONLY: dp, C
  USE configuration_ant_module, ONLY: T_ANT
  IMPLICIT NONE
            
  ! Names of dimensions:
  CHARACTER(LEN=*), PARAMETER :: name_t                               = 'time                           '

  ! Names of variables:
  CHARACTER(LEN=*), PARAMETER :: name_critical_time_step              = 'critical_time_step             '
  CHARACTER(LEN=*), PARAMETER :: name_total_ice_volume                = 'total_ice_volume               '
  CHARACTER(LEN=*), PARAMETER :: name_total_ice_volume_grounded       = 'total_ice_volume_grounded      '
  CHARACTER(LEN=*), PARAMETER :: name_total_ice_volume_shelf          = 'total_ice_volume_shelf         '

  CHARACTER(LEN=*), PARAMETER :: name_total_ice_area                  = 'total_ice_area                 '
  CHARACTER(LEN=*), PARAMETER :: name_total_ice_area_grounded         = 'total_ice_area_grounded        '
  CHARACTER(LEN=*), PARAMETER :: name_total_ice_area_shelf            = 'total_ice_area_shelf           '

  CHARACTER(LEN=*), PARAMETER :: name_total_MB                        = 'total_MB                       '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_surface                = 'total_MB_surface               '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_surface_negative       = 'total_MB_surface_negative      '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_surface_positive       = 'total_MB_surface_positive      '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_basal                  = 'total_MB_basal                 '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_basal_negative         = 'total_MB_basal_negative        '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_basal_positive         = 'total_MB_basal_positive        '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_applied                = 'total_MB_applied               '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_applied_negative       = 'total_MB_applied_negative      '
  CHARACTER(LEN=*), PARAMETER :: name_total_MB_applied_positive       = 'total_MB_applied_positive      '

  CHARACTER(LEN=*), PARAMETER :: name_mass_conservation               = 'mass_conservation              '
  CHARACTER(LEN=*), PARAMETER :: name_domain_leaving_flux             = 'domain_leaving_flux            '

  CHARACTER(LEN=*), PARAMETER :: name_sea_level                       = 'sea_level                      '
  CHARACTER(LEN=*), PARAMETER :: name_global_temperature              = 'global_temperature             '
  CHARACTER(LEN=*), PARAMETER :: name_time_step                       = 'time_step                      '

  TYPE recording_file_type_ant
    PRIVATE

    ! id for netCDF file: 
    INTEGER ::  ncid
    ! id's for variables:
    INTEGER :: var_t

    INTEGER :: var_critical_time_step                       
    INTEGER :: var_total_ice_volume               
    INTEGER :: var_total_ice_volume_grounded      
    INTEGER :: var_total_ice_volume_shelf         

    INTEGER :: var_total_ice_area                 
    INTEGER :: var_total_ice_area_grounded        
    INTEGER :: var_total_ice_area_shelf           

    INTEGER :: var_total_MB          
    INTEGER :: var_total_MB_surface          
    INTEGER :: var_total_MB_surface_negative 
    INTEGER :: var_total_MB_surface_positive 
    INTEGER :: var_total_MB_basal            
    INTEGER :: var_total_MB_basal_negative   
    INTEGER :: var_total_MB_basal_positive   
    INTEGER :: var_total_MB_applied          
    INTEGER :: var_total_MB_applied_negative 
    INTEGER :: var_total_MB_applied_positive 

    INTEGER :: var_mass_conservation   
    INTEGER :: var_domain_leaving_flux 

    INTEGER :: var_sea_level
    INTEGER :: var_global_temperature           
    INTEGER :: var_time_step
        
  END TYPE recording_file_type_ant

  ! Save struct recording_fields to save id's of variables 
  TYPE(recording_file_type_ant), SAVE :: recording_fields


CONTAINS
  SUBROUTINE create_recording_fields_file(file_name) 
    USE netcdf, ONLY: nf90_create, nf90_clobber, nf90_share, nf90_unlimited , nf90_enddef, nf90_put_var, nf90_sync
    IMPLICIT NONE
    
    ! Input variables:
    CHARACTER(LEN=*),              INTENT(IN)  :: file_name

    ! Local variables:
    INTEGER                                :: dim_t

    ! Create netCDF file
    CALL handle_error(nf90_create(file_name,IOR(nf90_clobber,nf90_share),recording_fields%ncid), 'create_recording_fields_file')
    

    ! Define dimensions:
    CALL create_dim(recording_fields, name_t, nf90_unlimited, dim_t)

    ! Define the variables (The order of the CALL statements for the different variables determines their order of appearence in the netcdf file.):

    ! time:
    CALL create_double_var(recording_fields,  name_t                              ,(/ dim_t /), recording_fields%var_t                              , long_name='time                           ', units='years')
  
    ! Critical time step:
    CALL create_double_var(recording_fields,  name_critical_time_step             ,(/ dim_t /), recording_fields%var_critical_time_step             , long_name='critical time step             ', units='%')
    
    ! Total Ice Volume:
    IF(C%which_recording_fields >=  3) &
     CALL create_double_var(recording_fields, name_total_ice_volume               ,(/ dim_t /), recording_fields%var_total_ice_volume               , long_name='total ice volume               ', units='m^3')

    ! Total Ice Volume grounded:
    IF(C%which_recording_fields >=  4) &
     CALL create_double_var(recording_fields, name_total_ice_volume_grounded      ,(/ dim_t /), recording_fields%var_total_ice_volume_grounded      , long_name='total ice volume grounded      ', units='m^3')

    ! Total Ice Volume shelf:
    IF(C%which_recording_fields >=  5) &
     CALL create_double_var(recording_fields, name_total_ice_volume_shelf         ,(/ dim_t /), recording_fields%var_total_ice_volume_shelf         , long_name='total ice volume shelf         ', units='m^3')

    ! Total Ice Area:
    IF(C%which_recording_fields >=  6) &
     CALL create_double_var(recording_fields, name_total_ice_area                 ,(/ dim_t /), recording_fields%var_total_ice_area                 , long_name='total ice area                 ', units='m^2')

    ! Total Ice Area grounded:
    IF(C%which_recording_fields >=  7) &
     CALL create_double_var(recording_fields, name_total_ice_area_grounded        ,(/ dim_t /), recording_fields%var_total_ice_area_grounded        , long_name='total ice area grounded        ', units='m^2')

    ! Total Ice Area shelf:
    IF(C%which_recording_fields >=  8) &
     CALL create_double_var(recording_fields, name_total_ice_area_shelf           ,(/ dim_t /), recording_fields%var_total_ice_area_shelf           , long_name='total ice area shelf           ', units='m^2')

    ! Total MB:
    IF(C%which_recording_fields >=  9) &
     CALL create_double_var(recording_fields, name_total_MB                       ,(/ dim_t /), recording_fields%var_total_MB                       , long_name='total MB                       ', units='mieq')

    ! Total MB surface:
    IF(C%which_recording_fields >= 10) &
     CALL create_double_var(recording_fields, name_total_MB_surface               ,(/ dim_t /), recording_fields%var_total_MB_surface               , long_name='total MB surface               ', units='mieq')

    ! Total MB surface negative:
    IF(C%which_recording_fields >= 11) &
     CALL create_double_var(recording_fields, name_total_MB_surface_negative      ,(/ dim_t /), recording_fields%var_total_MB_surface_negative      , long_name='total MB surface negative      ', units='mieq')

    ! Total MB surface positive:
    IF(C%which_recording_fields >= 12) &
     CALL create_double_var(recording_fields, name_total_MB_surface_positive      ,(/ dim_t /), recording_fields%var_total_MB_surface_positive      , long_name='total MB surface positive      ', units='mieq')

    ! Total MB basal:
    IF(C%which_recording_fields >= 13) &
     CALL create_double_var(recording_fields, name_total_MB_basal                 ,(/ dim_t /), recording_fields%var_total_MB_basal                 , long_name='total MB basal                 ', units='mieq')

    ! Total MB basal negative:
    IF(C%which_recording_fields >= 14) &
     CALL create_double_var(recording_fields, name_total_MB_basal_negative        ,(/ dim_t /), recording_fields%var_total_MB_basal_negative        , long_name='total MB basal negative        ', units='mieq')

    ! Total MB basal positive:
    IF(C%which_recording_fields >= 15) &
     CALL create_double_var(recording_fields, name_total_MB_basal_positive        ,(/ dim_t /), recording_fields%var_total_MB_basal_positive        , long_name='total MB basal positive        ', units='mieq')

    ! Total MB applied:
    IF(C%which_recording_fields >= 16) &
     CALL create_double_var(recording_fields, name_total_MB_applied               ,(/ dim_t /), recording_fields%var_total_MB_applied               , long_name='total MB applied               ', units='mieq')

    ! Total MB applied negative:
    IF(C%which_recording_fields >= 17) &
     CALL create_double_var(recording_fields, name_total_MB_applied_negative      ,(/ dim_t /), recording_fields%var_total_MB_applied_negative      , long_name='total MB applied negative      ', units='mieq')

    ! Total MB applied positive:
    IF(C%which_recording_fields >= 18) &
     CALL create_double_var(recording_fields, name_total_MB_applied_positive      ,(/ dim_t /), recording_fields%var_total_MB_applied_positive      , long_name='total MB applied positive      ', units='mieq')

    ! Mass Conservation:
     CALL create_double_var(recording_fields, name_mass_conservation              ,(/ dim_t /), recording_fields%var_mass_conservation              , long_name='mass conservation              ', units='mieq')

    ! Domain Leaving Flux:
     CALL create_double_var(recording_fields, name_domain_leaving_flux            ,(/ dim_t /), recording_fields%var_domain_leaving_flux            , long_name='domain leaving_flux            ', units='mieq')

    ! Sea Level:
     CALL create_double_var(recording_fields, name_sea_level                      ,(/ dim_t /), recording_fields%var_sea_level                      , long_name='sea level                      ', units='meter')

    ! Global Temperature Signal:
     CALL create_double_var(recording_fields, name_global_temperature             ,(/ dim_t /), recording_fields%var_global_temperature             , long_name='global temperature             ', units='Kelvin')

    ! Time Step:
     CALL create_double_var(recording_fields, name_time_step                      ,(/ dim_t /), recording_fields%var_time_step                      , long_name='time step                      ', units='year')

    ! Leave definition mode:
    CALL handle_error(nf90_enddef(recording_fields%ncid))

    ! Synchronize with disk (otherwise it doesn't seem to work on a MAC)
    CALL handle_error(nf90_sync(recording_fields%ncid))
  END SUBROUTINE create_recording_fields_file


  SUBROUTINE create_dim(netcdf_file, dim_name, length, id_dim)
    ! Subroutine for creating netCDF dimensions more convenient:
    USE netcdf, ONLY: nf90_def_dim
    IMPLICIT NONE
  
    ! Input variables:
    TYPE(recording_file_type_ant), INTENT(IN) :: netcdf_file
    CHARACTER(LEN=*),              INTENT(IN) :: dim_name
    INTEGER,                       INTENT(IN) :: length
    
    ! Output variables:
    INTEGER, INTENT(OUT)                  :: id_dim

    CALL handle_error(nf90_def_dim(netcdf_file%ncid,dim_name,length,id_dim), 'create_dim '//TRIM(dim_name))
  END SUBROUTINE create_dim



  SUBROUTINE create_double_var(netcdf_file, var_name, id_dims, id_var, long_name, units, valid_min, valid_max, missing_value)
    ! Subroutine for creating netCDF variables of type nf90_DOUBLE more convenient:
    USE netcdf, ONLY: nf90_def_var, nf90_double, nf90_put_att
  
    ! Input variables:
    TYPE(recording_file_type_ant), INTENT(IN)  :: netcdf_file
    CHARACTER(LEN=*),              INTENT(IN)  :: var_name
    INTEGER, DIMENSION(:),         INTENT(IN)  :: id_dims
    CHARACTER(LEN=*),    OPTIONAL, INTENT(IN)  :: long_name
    CHARACTER(LEN=*),    OPTIONAL, INTENT(IN)  :: units
    REAL(dp),            OPTIONAL, INTENT(IN)  :: valid_min
    REAL(dp),            OPTIONAL, INTENT(IN)  :: valid_max
    REAL(dp),            OPTIONAL, INTENT(IN)  :: missing_value

    ! Output variables:
    INTEGER,                      INTENT(OUT) :: id_var

    CALL handle_error(nf90_def_var(netcdf_file%ncid,var_name,nf90_double,id_dims,id_var))
    IF(PRESENT(long_name))     CALL handle_error(nf90_put_att(netcdf_file%ncid,id_var,'long_name',long_name))
    IF(PRESENT(units))         CALL handle_error(nf90_put_att(netcdf_file%ncid,id_var,'units',units))
    IF(PRESENT(valid_min))     CALL handle_error(nf90_put_att(netcdf_file%ncid,id_var,'valid_min',valid_min))
    IF(PRESENT(valid_max))     CALL handle_error(nf90_put_att(netcdf_file%ncid,id_var,'valid_max',valid_max))
    IF(PRESENT(missing_value)) CALL handle_error(nf90_put_att(netcdf_file%ncid,id_var,'missing_value',missing_value))
  END SUBROUTINE create_double_var


  SUBROUTINE write_recording_fields_file(recno              , &
                                         time               , &
                                         critical_time_step , &
                                         MB_surface         , &
                                         MB_basal           , &
                                         MB_applied         , &
                                         Hi                 , &
                                         domain_leaving_flux, &
                                         mass_conservation  , &
                                         mask               , &
                                         sealevel           , &
                                         global_temperature)

    USE netcdf, ONLY: nf90_put_var, nf90_sync
  
    ! Input variables:
    INTEGER,                                INTENT(IN) :: recno
    REAL(dp),                               INTENT(IN) :: time
    REAL(dp),                               INTENT(IN) :: critical_time_step
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: MB_surface
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: MB_basal
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: MB_applied
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: Hi
    REAL(dp),                               INTENT(IN) :: domain_leaving_flux
    REAL(dp),                               INTENT(IN) :: mass_conservation
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: mask
    REAL(dp),                               INTENT(IN) :: sealevel
    REAL(dp),                               INTENT(IN) :: global_temperature

    ! Local variables:
    REAL(dp)                                           :: volume_land, volume_shelf
    REAL(dp)                                           :: area_land, area_shelf

    ! Added BdB (11/2009): Calculate variables:
    volume_land   = C%dx_ant * C%dy_ant * SUM(Hi, MASK = (mask /= C%type_shelf .AND. Hi > C%Hi_min))                  ! total grounded volume, omitting soil and shelf/ocean points
    volume_shelf  = C%dx_ant * C%dy_ant * SUM(Hi, MASK = (mask == C%type_shelf .AND. Hi > C%calve_thickness))         ! total shelf volume, omitting ocean points with Hi < calve_thickness (= 200 m)
    area_land     = C%dx_ant * C%dy_ant * COUNT(mask /= C%type_shelf .AND. Hi >  C%Hi_min)                            ! same as above, but the area
    area_shelf    = C%dx_ant * C%dy_ant * COUNT(mask == C%type_shelf .AND. Hi >= C%calve_thickness)                   ! same as above, but the area

    ! Write variables to file: 
    CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_t                             , (/ time /)                                      , start=(/ recno /)))
    CALL handle_error(nf90_put_var(recording_fields%ncid,  recording_fields%var_critical_time_step             , 100._dp * T_ANT%dt / critical_time_step                   , start=(/ recno /)))

    IF(C%which_recording_fields >=  3) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_ice_volume               , volume_land + volume_shelf                                , start=(/ recno /)))
    IF(C%which_recording_fields >=  4) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_ice_volume_grounded      , volume_land                                               , start=(/ recno /)))
    IF(C%which_recording_fields >=  5) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_ice_volume_shelf         , volume_shelf                                              , start=(/ recno /)))
    IF(C%which_recording_fields >=  6) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_ice_area                 , area_land + area_shelf                                    , start=(/ recno /)))
    IF(C%which_recording_fields >=  7) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_ice_area_grounded        , area_land                                                 , start=(/ recno /)))
    IF(C%which_recording_fields >=  8) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_ice_area_shelf           , area_shelf                                                , start=(/ recno /)))
    IF(C%which_recording_fields >=  9) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB                       , C%dx_ant * C%dy_ant * SUM(MB_surface + MB_basal)          , start=(/ recno /)))
    IF(C%which_recording_fields >=  10) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_surface               , C%dx_ant * C%dy_ant * SUM(MB_surface)                     , start=(/ recno /)))
    IF(C%which_recording_fields >=  11) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_surface_negative      , C%dx_ant * C%dy_ant * SUM(MIN(0._dp,MB_surface))          , start=(/ recno /)))
    IF(C%which_recording_fields >=  12) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_surface_positive      , C%dx_ant * C%dy_ant * SUM(MAX(0._dp,MB_surface))          , start=(/ recno /)))
    IF(C%which_recording_fields >= 13) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_basal                 , C%dx_ant * C%dy_ant * SUM(MB_basal)                       , start=(/ recno /)))
    IF(C%which_recording_fields >= 14) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_basal_negative        , C%dx_ant * C%dy_ant * SUM(MIN(0._dp,MB_basal))            , start=(/ recno /)))
    IF(C%which_recording_fields >= 15) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_basal_positive        , C%dx_ant * C%dy_ant * SUM(MAX(0._dp,MB_basal))            , start=(/ recno /)))
    IF(C%which_recording_fields >= 16) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_applied               , C%dx_ant * C%dy_ant * SUM(MB_applied)                     , start=(/ recno /)))
    IF(C%which_recording_fields >= 17) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_applied_negative      , C%dx_ant * C%dy_ant * SUM(MIN(0._dp,MB_applied))          , start=(/ recno /)))
    IF(C%which_recording_fields >= 18) &
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_total_MB_applied_positive      , C%dx_ant * C%dy_ant * SUM(MAX(0._dp,MB_applied))          , start=(/ recno /)))

     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_mass_conservation              , C%dx_ant * C%dy_ant * mass_conservation * 9.1E-10_dp      , start=(/ recno /)))
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_domain_leaving_flux            , C%dx_ant * C%dy_ant * domain_leaving_flux / T_ANT%dt      , start=(/ recno /)))
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_sea_level                      , sealevel                                                  , start=(/ recno /)))
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_global_temperature             , global_temperature                                        , start=(/ recno /)))
     CALL handle_error(nf90_put_var(recording_fields%ncid, recording_fields%var_time_step                      , T_ANT%dt                                                  , start=(/ recno /)))

    ! Synchronize to make sure that data is really on disk (in case the program crashes later on):
    CALL handle_error(nf90_sync(recording_fields%ncid))
    
  END SUBROUTINE write_recording_fields_file


  SUBROUTINE close_recording_file()
    USE netcdf, ONLY: nf90_close
    IMPLICIT NONE
  
    ! Close netCDF file:
    CALL handle_error(nf90_close(recording_fields%ncid))
    
  END SUBROUTINE close_recording_file


  SUBROUTINE handle_error(stat, message)
    USE netcdf, ONLY: nf90_noerr, nf90_strerror
    IMPLICIT NONE
  
    ! Input variables:
    INTEGER,                    INTENT(IN) :: stat
    CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: message

    IF(stat /= nf90_noerr) THEN
     IF(PRESENT(message)) THEN
      WRITE(*,'(A,A,A,A)') 'ERROR: netCDF failed (in time_recording_module) because: ', TRIM(nf90_strerror(stat)), ' concerning: ', message
     ELSE
      WRITE(*,'(A,A)')     'ERROR: netCDF failed (in time_recording_module) because: ', TRIM(nf90_strerror(stat))
     END IF
     STOP
    END IF
  END SUBROUTINE handle_error
END MODULE time_recording_ant_module
