MODULE thermodynamics_ant_module
  USE configuration_main_module, ONLY : dp, C
  USE configuration_ant_module, ONLY: C_ANT, T_ANT
  IMPLICIT NONE
  
CONTAINS
  SUBROUTINE bottom_frictional_heating(Hi, dHs_dx, dHs_dy, tau_yield, U_sliding, V_sliding, Ti_bottom, Ti_pmp_bottom, mask, frictional_heating)
    ! Calculation of the frictional heating at the bottom due to sliding at the sheet/Gl - bedrock interface. See section 8.5.
    USE parameters_main_module, ONLY: ice_density, grav, q_plastic, u_threshold, delta_v
    
    ! Input variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: dHs_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: dHs_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: U_sliding
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: V_sliding
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Ti_bottom
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Ti_pmp_bottom     ! The melting temperature, depending on pressure [K]
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: mask

    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: frictional_heating

    ! added variables to calculate the basal shear stress, from the water pressure
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: tau_yield        ! basal yield stress
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: beta_base        ! basal shear stress coefficient

    ! calculate beta_base the coefficient for the basal stress (tau_base = beta * V)
    beta_base = tau_yield * ( (delta_v**2 + U_sliding**2 + V_sliding**2)**(0.5_dp * (q_plastic-1._dp)) ) / (u_threshold**q_plastic) 

    frictional_heating = 0._dp

    ! Use Weertman type sliding velocity, depends on basal ice temperature
    IF (C_ANT%choice_sliding_method == 1) THEN    
     ! See equation (8.51 and 8.53):
     WHERE((mask /= C%type_shelf) .AND. ((Ti_bottom >= Ti_pmp_bottom - C%melt_range) .OR. C%always_sliding)) & 
      frictional_heating = ice_density * grav * Hi * (ABS(U_sliding * dHs_dx) + ABS(V_sliding * dHs_dy))

    ! Use SSA as sliding, computed with basal yield stress everywhere
    ELSE IF (C_ANT%choice_sliding_method == 2) THEN
     ! Do not use the driving stress but the yield stress (function of velocity..)
     WHERE(mask /= C%type_shelf) & 
      frictional_heating = beta_base * (U_sliding**2 + V_sliding**2)  
    END IF

  END SUBROUTINE bottom_frictional_heating
   

  SUBROUTINE temperature(Hi, dHs_dx, dHs_dy, Ti, Ti_pmp, Cpi, Ki, U, V, W, frictional_heating, mask, Fr, Ti_new)
    ! The equation numbers without an extra reference refer to the temperature document in trunk/src/Doc/temperature_v12.pdf.
    ! This subroutine numerically solves the temperature equation (8.9).
    USE parameters_main_module, ONLY: ice_density, grav, SMT, L_fusion
    USE zeta_derivatives_ant_module, ONLY : dzeta
    USE tridiagonal_solve_ant_module, ONLY : tridiagonal_solve

    ! Input variables:
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: U
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: V
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: W
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHs_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: dHs_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: frictional_heating
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: Ti        ! The ice temperature at the previous time step t, Ti(i-2:i+2,j-2:j+2,:) are used.
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: Ti_pmp    ! The pressure melting point temperature for each depth and for all grid points.
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: Cpi       ! The specific heat capacity of ice at each x,y,zeta point [J kg^-1 K^-1]
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: Ki        ! The conductivity of ice at each x,y,zeta point [J m^-1 K^-1 yr^-1]
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Fr        ! The heat flux from rock to ice.

    ! Output variable:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(OUT) :: Ti_new    ! The new ice temperature at time t + dt. The surface temperature T(1,:,:) is 
                                                                          ! calculated with ice_surface_temperature()
    ! Local variables:
    INTEGER                                          :: i, j, k, kk
    REAL(dp)                                         :: dTi_dx
    REAL(dp)                                         :: dTi_dy
    REAL(dp)                                         :: f1, f2, f3
    REAL(dp), DIMENSION(2:C%NZ)                      :: alpha
    REAL(dp), DIMENSION(C%NZ)                        :: beta
    REAL(dp), DIMENSION(C%NZ-1)                      :: gamma
    REAL(dp), DIMENSION(C%NZ)                        :: delta
    REAL(dp)                                         :: addapted_number_Ti_pmp
    
    ! Loop over the whole x, y -domain, but not the two outer boundaries:

    DO j = 3, C%NY_ant-2
    DO i = 3, C%NX_ant-2
      IF(mask(i,j) /= C%type_shelf) THEN
        ! Ice surface boundary condition, see equations (8.28 - 8.30):
        beta(1)  = 1._dp
        gamma(1) = 0._dp
        delta(1) = Ti(i,j,1)
    
        ! Loop over the whole vertical domain but not the surface (k=1) and the bottom (k=NZ):
        DO k = 2, C%NZ-1
         IF(.TRUE.) THEN
          ! Upwind temperature gradient scheme (see equation (14.40)):                  
          IF(U(i,j,k) > 0._dp) THEN
           dTi_dx = ( 2._dp * Ti(i+1,j,k) + 3._dp * Ti(i,j,k)   - 6._dp * Ti(i-1,j,k) + 1._dp * Ti(i-2,j,k)) / (6._dp * C%dx_ant)
          ELSE
           dTi_dx = (-1._dp * Ti(i+2,j,k) + 6._dp * Ti(i+1,j,k) - 3._dp * Ti(i,j,k)   - 2._dp * Ti(i-1,j,k)) / (6._dp * C%dx_ant)
          END IF
          IF(V(i,j,k) > 0._dp) THEN 
           dTi_dy = ( 2._dp * Ti(i,j+1,k) + 3._dp * Ti(i,j,k)   - 6._dp * Ti(i,j-1,k) + 1._dp * Ti(i,j-2,k)) / (6._dp * C%dy_ant)
          ELSE
           dTi_dy = (-1._dp * Ti(i,j+2,k) + 6._dp * Ti(i,j+1,k) - 3._dp * Ti(i,j,k)   - 2._dp * Ti(i,j-1,k)) / (6._dp * C%dy_ant)
          END IF
         ELSE
          ! Upwind temperature gradient, one sided schemes:                  
          IF(U(i,j,k) > 0._dp) THEN
           dTi_dx = C_ANT%z_x_minus * Ti(i-2,j,k) + C_ANT%a_x_minus * Ti(i-1,j,k) + C_ANT%b_x_minus * Ti(i,j,k)
          ELSE
           dTi_dx = C_ANT%b_x_plus  * Ti(i,j,k)   + C_ANT%c_x_plus  * Ti(i+1,j,k) + C_ANT%d_x_plus  * Ti(i+2,j,k)
          END IF
          IF(V(i,j,k) > 0._dp) THEN 
           dTi_dy = C_ANT%z_y_minus * Ti(i-2,j,k) + C_ANT%a_y_minus * Ti(i-1,j,k) + C_ANT%b_y_minus * Ti(i,j,k) 
          ELSE
           dTi_dy = C_ANT%b_y_plus  * Ti(i,j,k)   + C_ANT%c_y_plus  * Ti(i+1,j,k) + C_ANT%d_y_plus  * Ti(i+2,j,k)
          END IF
         END IF
  
          ! See equations (8.17 - 8.19):
          f1 = (Ki(i,j,k) * dzeta%z(i,j)**2) / (ice_density * Cpi(i,j,k))

          f2 = dzeta%t(i,j,k) + dzeta%x(i,j,k) * U(i,j,k) + dzeta%y(i,j,k) * V(i,j,k) + dzeta%z(i,j) * W(i,j,k)

          f3 = ((- grav * C%zeta(k)) / Cpi(i,j,k)) * ( &
                 (C_ANT%a_zeta(k) * U(i,j,k-1) + C_ANT%b_zeta(k) * U(i,j,k) + C_ANT%c_zeta(k) * U(i,j,k+1)) * dHs_dx(i,j) + &
                 (C_ANT%a_zeta(k) * V(i,j,k-1) + C_ANT%b_zeta(k) * V(i,j,k) + C_ANT%c_zeta(k) * V(i,j,k+1)) * dHs_dy(i,j) ) &
               + (U(i,j,k) * dTi_dx + V(i,j,k) * dTi_dy) + C_ANT%b_t * Ti(i,j,k)

          ! See equations (8.23 - 8.26):
          alpha(k) = f1 * C_ANT%a_zetazeta(k) - f2 * C_ANT%a_zeta(k)
          beta (k) = f1 * C_ANT%b_zetazeta(k) - f2 * C_ANT%b_zeta(k) - C_ANT%s_t
          gamma(k) = f1 * C_ANT%c_zetazeta(k) - f2 * C_ANT%c_zeta(k)
          delta(k) = f3
        END DO
   
        ! Variable boundary condition (see equation (8.45)): 
        IF((Ti(i,j,C%NZ-1) + C_ANT%a_N * Hi(i,j) * (Fr(i,j) + frictional_heating(i,j)) / Ki(i,j,C%NZ)) > Ti_pmp(i,j,C%NZ)) THEN
         ! In case the bottom ice layer has reached pressure melting point (the ice temperature
         ! in the bottom layer can not exceed pressure melting point):
         ! See equations 8.46 - 8.48:
         alpha(C%NZ) = 0._dp
         beta(C%NZ)  = 1._dp
         delta(C%NZ) = Ti_pmp(i,j,C%NZ)
        ELSE
         ! Usual ice/bedrock boundary condition (2nd order gradient).See equations (8.39 - 8.41):
         alpha(C%NZ)  = C_ANT%z_zeta_minus(C%NZ) * beta(C%NZ-1)  - C_ANT%a_zeta_minus(C%NZ) * alpha(C%NZ-1)
         beta(C%NZ)   = C_ANT%z_zeta_minus(C%NZ) * gamma(C%NZ-1) - C_ANT%b_zeta_minus(C%NZ) * alpha(C%NZ-1) 
         delta(C%NZ)  = C_ANT%z_zeta_minus(C%NZ) * delta(C%NZ-1) - Hi(i,j) * ((Fr(i,j) + frictional_heating(i,j)) / Ki(i,j,C%NZ)) * alpha(C%NZ-1) 
        END IF

        Ti_new(i,j,:) = tridiagonal_solve(alpha, beta, gamma, delta, 'thermodynamics_ant_module [temperature]')
    
      ! At the shelf/ocean (setting of 3 dim Ti field for if a shelf point changes into a grounded point):
      ELSE
       Ti_new(i,j,1:C%NZ) = Ti(i,j,1) + C%zeta(1:C%NZ) * (SMT - Ti(i,j,1))  
      END IF
    END DO
    END DO



    ! Checking that ice doesn't get a temperature above the melting point (see equation 8.42):
    addapted_number_Ti_pmp = COUNT(Ti_new > Ti_pmp)

!    WRITE(UNIT=C%stdlog, FMT='(f7.3, a, f10.3, a)') addapted_number_Ti_pmp * 100.0_dp / SIZE(Ti_new), &
!      '% of the 3d grounded cases is setted to melting point temperature, with an average difference of ', &      
!      SUM(ABS(Ti_new - Ti_pmp), Ti_new > Ti_pmp) / addapted_number_Ti_pmp, ' Kelvin'    

    ! See equations (8.42 - 8.43):
    WHERE(Ti_new > Ti_pmp) Ti_new = Ti_pmp

    ! Neumann at domain edges (for two grid-size thick boundaries):
    Ti_new(:,1,:)      = Ti_new(:,3,:) 
    Ti_new(:,2,:)      = Ti_new(:,3,:) 
    Ti_new(:,C%NY_ant,:)   = Ti_new(:,C%NY_ant-2,:)
    Ti_new(:,C%NY_ant-1,:) = Ti_new(:,C%NY_ant-2,:)
    Ti_new(1,:,:)      = Ti_new(3,:,:) 
    Ti_new(2,:,:)      = Ti_new(3,:,:) 
    Ti_new(C%NX_ant,:,:)   = Ti_new(C%NX_ant-2,:,:)
    Ti_new(C%NX_ant-1,:,:) = Ti_new(C%NX_ant-2,:,:)

    DO k = 1,C%NZ
      DO j = 1, C%NY_ant
      DO i = 1, C%NX_ant
        !if (i == 53 .and. j == 50 .and. k == 1) write (111,'(3f12.3)') Ti(i,j,k),Ti_new(i,j,k),W(i,j,k)
        
        IF ( Ti_new(i,j,k) .NE. Ti_new(i,j,k) ) WRITE(111,'(A10,4I5,7F14.5,F12.3)') 'ANT Ti = ',i,j,k, mask(i,j), & 
                                                Ti_new(i,j,k), Ti(i,j,k), U(i,j,C%NZ), V(i,j,C%NZ), U(i,j,k), V(i,j,k), W(i,j,k), T_ANT%time
        IF ( Ti_new(i,j,k) < 0._dp ) THEN 
          WRITE (111,'(A10,4I5,7F14.5,F12.3)') ' ANT Ti = ',i,j,k, mask(i,j), &
                            Ti_new(i,j,k), Ti(i,j,k), U(i,j,C%NZ), V(i,j,C%NZ), U(i,j,k), V(i,j,k), W(i,j,k), T_ANT%time
          WRITE (111,'(A10,15F14.5)') 'Ti all lrs: ',(Ti_new(i,j,kk),kk=1,C%NZ)
        END IF                             
        IF (Ti_new(i,j,k) .NE. Ti_new(i,j,k)) STOP
        IF (Ti_new(i,j,k) < 0._dp) STOP
      END DO
      END DO
    END DO

  END SUBROUTINE temperature
END MODULE thermodynamics_ant_module
