MODULE shelf_ant_module
  USE configuration_main_module, ONLY : dp, C
  USE configuration_ant_module, ONLY: C_ANT, T_ANT
  IMPLICIT NONE

CONTAINS
  SUBROUTINE ssa_velocity(calve_mask, dHs_dx, dHs_dy, A_flow_mean, & 
                            tau_yield, Us_old, Vs_old, Us, Vs)
    ! Solve the system of elliptic eqautions:
    !   2 Us_xx + 1/2 Us_yy + 3/2 Vs_xy = rhs_x = ice_density g dHs_dx / C_uv
    !   2 Vs_yy + 1/2 Vs_xx + 3/2 Us_xy = rhs_y = ice_density g dHs_dy / C_uv
    ! with:
    !                2         2                                      2               (1-n)/2n
    ! C_uv = ( dUs_dx  + dVs_dy  + dUs_dx dVs_dy + 1/4 (dUs_dx+dVs_dy) + epsilon_sq_0)        (see Huybrechts 4.28)
    !
    ! Derivatives are numerically calculated at internal shelf/ocean points.
    ! At shelf/ocean points along the domain boundary appropriate boundary conditions are imposed.
    ! Herefore the "boundary condition"-array was used, now only Neumann condition.
    ! At all points other than shelf ocean the solution is fixed to the initial value.
    ! C_uv is not updated during the iteration (might be better?)
    ! At entrance an initial guess for the velocity fields is used. An iterative method 
    ! is used (Red/Black Gauss-Seidel + over relaxation).
    ! At exit it is the calculated solution. At sheet points the values of Us and Vs are not changed.

    USE parameters_main_module, ONLY : n_flow, q_plastic, u_threshold, ice_density, grav, epsilon_sq_0, delta_v

    ! Input variables:
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: calve_mask         ! The shelf-sheet-ocean mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: dHs_dx             ! The right hand side of the differential equations
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: dHs_dy             ! The right hand side of the differential equations
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: A_flow_mean        ! Vertical mean of the flow parameter A
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Us_old
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Vs_old

    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: Us                 ! The x velocity fields
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: Vs                 ! The y velocity fields

    ! Local variables controling the PDE solving:
    LOGICAL,  PARAMETER                  :: debug = .FALSE.                   ! Turn debug mode on or off
    INTEGER                              :: explicit_update_iteration
    INTEGER                              :: iter_gauss_seidel

    ! Local variables
    INTEGER                                             :: i, j
    INTEGER                                             :: sweep
    INTEGER                                             :: start_odd_or_even
    REAL(dp)                                            :: largest_rest_x
    REAL(dp)                                            :: largest_rest_y
    REAL(dp)                                            :: smallest_eu_ij
    REAL(dp)                                            :: smallest_ev_ij
    REAL(dp)                                            :: eu_ij, ev_ij       ! The coefficient of U_ij and V_ij
    REAL(dp)                                            :: Us_xy, Vs_xy
    REAL(dp)                                            :: Rx, Ry             ! The rest terms
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: rhs_x              ! right hand side x
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: rhs_y              ! right hand side y
    REAL(dp)                                            :: dUs_dx
    REAL(dp)                                            :: dUs_dy
    REAL(dp)                                            :: dVs_dx
    REAL(dp)                                            :: dVs_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: C_uv
    LOGICAL,  DIMENSION(C%NX_ant,C%NY_ant)              :: mask_check         ! grid points that should be included in the iteration

    ! added variables to calculate the basal shear stress, from the water pressure
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: tau_yield          ! basal yield stress (Mohr-Coulomb model) (eqn (10) Martin,11)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: beta_base          ! basal shear stress coefficient (plastic till model) (eqn (27) B&B,09)

    ! Because of splitting INOUT in IN and OUT
    Us = Us_old
    Vs = Vs_old

    mask_check = .FALSE.
    C_uv       = 1._dp
    rhs_x      = 0._dp
    rhs_y      = 0._dp
    beta_base  = 0._dp
   
    ! In case of grounded ice points, include in case of sliding  
    WHERE (calve_mask <= C%type_groundline .AND. C_ANT%choice_sliding_method == 2 )
       mask_check = .TRUE.
    END WHERE

    ! In case of floating ice, include for shelves, do not include ocean grid points
    WHERE (calve_mask >= C%type_shelf .AND. (.NOT. C_ANT%no_shelf) )
      mask_check = .TRUE.
    END WHERE


!!-- In this loop the explicit Us, Vs fields are updated, and than (again) the kernel iteration is done:
    DO explicit_update_iteration = 1, 3

      ! Calculate the 'fixed' non-linear term C_uv for the current solution {Us, Vs} at interior points,
      ! and the right hand sides: rhs_x and rhs_y:
      DO j = 2, C%NY_ant - 1
      DO i = 2, C%NX_ant - 1

        IF( mask_check(i,j)) THEN

         dUs_dx = (Us(i+1,  j) - Us(i-1,j  )) / (2._dp * C%dx_ant)  ! d Us / C%dx_ant
         dUs_dy = (Us(i  ,j+1) - Us(i  ,j-1)) / (2._dp * C%dy_ant)  ! d Us / C%dy_ant
         dVs_dx = (Vs(i+1,  j) - Vs(i-1,j  )) / (2._dp * C%dx_ant)  ! d Vs / C%dx_ant
         dVs_dy = (Vs(i  ,j+1) - Vs(i  ,j-1)) / (2._dp * C%dy_ant)  ! d Vs / C%dy_ant
         C_uv(i,j) = A_flow_mean(i,j)**(-1._dp / n_flow) * (dUs_dx**2 + dVs_dy**2 + dUs_dx * dVs_dy &
                         + 0.25_dp * (dUs_dy + dVs_dx)**2 + epsilon_sq_0)**((1._dp - n_flow) / (2._dp * n_flow))

         ! Determine beta_base, for basal stress (beta in eqn (27) B&B, 2009), here equivalent to the sliding parameter (A**-1/m)
         beta_base(i,j) = tau_yield(i,j) * ( (delta_v**2 + Us(i,j)**2 + Vs(i,j)**2)**(0.5_dp * (q_plastic-1._dp)) ) & 
                          / (u_threshold**q_plastic) 

         ! The right hand side of the PDE, the gravitational driving stress and static stress at the shelf edge
         rhs_x(i,j) = (ice_density * grav * dHs_dx(i,j)) / C_uv(i,j)
         rhs_y(i,j) = (ice_density * grav * dHs_dy(i,j)) / C_uv(i,j)
         
         IF (C_uv(i,j) .NE. C_uv(i,j)) THEN
           WRITE(111,'(A,3I5,2E15.3,I5)') 'C_uv = ',i,j,calve_mask(i,j),C_uv(i,j),A_flow_mean(i,j),explicit_update_iteration
           WRITE(111,'(A15,4F12.4)') 'U all 8 = ',Us(i+1,j),Us(i,j-1),Us(i-1,j),Us(i,j+1)
           WRITE(111,'(A15,4F12.4)') 'V all 8 = ',Vs(i+1,j),Vs(i,j-1),Vs(i-1,j),Vs(i,j+1)
           WRITE(111,'(A15,4I12)') 'msk all 8 = ',calve_mask(i+1,j),calve_mask(i,j-1),calve_mask(i-1,j),calve_mask(i,j+1)
           WRITE(111,'(A15,2E15.3)') ' rhs: ', rhs_x(i,j), rhs_y(i,j)
           STOP
         END IF

        END IF
      END DO
      END DO



      ITERATE: DO  iter_gauss_seidel = 1, C%max_iter_gauss_seidel

        ! Update {Us,Vs} at restricted interior points:
        ! First sweep: i,j starts at (2,2) and is even; second sweep: i,j starts at (3,3) and is odd
        largest_rest_x = 0._dp
        largest_rest_y = 0._dp
        
        smallest_eu_ij = 1.0E10_dp
        smallest_ev_ij = 1.0E10_dp

        ! Red black scheme:
        DO sweep = 0, 1 
          DO j = 2, C%NY_ant - 1
            start_odd_or_even = 2 + MOD(j+sweep,2)
            DO i = start_odd_or_even, C%NX_ant - 1, 2
              IF( mask_check(i,j) ) THEN

               Us_xy = ((Us(i+1,j+1)- Us(i-1,j+1)) - (Us(i+1,j-1)-Us(i-1,j-1))) / (4._dp * C%dx_ant * C%dy_ant) ! Taken into account with Gauss-Seidel way
               Vs_xy = ((Vs(i+1,j+1)- Vs(i-1,j+1)) - (Vs(i+1,j-1)-Vs(i-1,j-1))) / (4._dp * C%dx_ant * C%dy_ant) ! Taken into account with Gauss-Seidel way

               ! coefficients for Us(i,j) and Vs(i,j) when using SSA for sliding
               eu_ij = -(8._dp / (C%dx_ant**2) + 2._dp / (C%dy_ant**2)) - 2._dp * beta_base(i,j) / C_uv(i,j)
               ev_ij = -(2._dp / (C%dx_ant**2) + 8._dp / (C%dy_ant**2)) - 2._dp * beta_base(i,j) / C_uv(i,j)

               ! Rest of x-velocity equation:  4 Us_xx + Us_yy + 3 Vs_xy = 2 rhs_x  (See Numerical recepies eq (19.5.28))
               ! The first four terms of Rx and Ry use 'old' values at the first sweep and 'new' values at the second sweep 
               !(the red-black effect):
               Rx = 4._dp * (Us(i+1,j) + Us(i-1,j)) / (C%dx_ant**2) + (Us(i,j+1) + Us(i,j-1)) / (C%dy_ant**2) + eu_ij * Us(i,j) + &
                    3._dp * Vs_xy - 2._dp * rhs_x(i,j)

               ! Rest of y-velocity equation:  4 Vs_yy + Vs_xx + 3 Us_xy = 2 rhs_y  (See Numerical recepies eq (19.5.28))
               Ry = 4._dp * (Vs(i,j+1) + Vs(i,j-1)) / (C%dy_ant**2) + (Vs(i+1,j) + Vs(i-1,j)) / (C%dx_ant**2) + ev_ij * Vs(i,j) + &
                    3._dp * Us_xy - 2._dp * rhs_y(i,j)

               ! Update by calculating Us and Vs with the successive over relaxation (SOR) scheme (See Numerical recepies eq (19.5.29)):
               IF (eu_ij == 0._dp) THEN
                 Us(i,j) = Us(i,j)
               ELSE 
                 Us(i,j) = Us(i,j) - C%gauss_seidel_relaxation * Rx / eu_ij
               END IF  
               IF (ev_ij == 0._dp) THEN
                 Vs(i,j) = Vs(i,j)
               ELSE 
                 Vs(i,j) = Vs(i,j) - C%gauss_seidel_relaxation * Ry / ev_ij               
               END IF
               
               
               IF (Us(i,j) .NE. Us(i,j)) THEN
                WRITE(111,'(A,3I5,2F12.3,2I5)') 'Us = ',i,j,calve_mask(i,j),Us(i,j),Vs(i,j),iter_gauss_seidel,explicit_update_iteration
                WRITE(111,'(A15,8F12.4)') 'U all 8 = ',Us(i+1,j),Us(i+1,j-1),Us(i,j-1),Us(i-1,j-1),Us(i-1,j),Us(i-1,j+1),Us(i,j+1),Us(i+1,j+1)
                WRITE(111,'(A15,8F12.4)') 'V all 8 = ',Vs(i+1,j),Vs(i+1,j-1),Vs(i,j-1),Vs(i-1,j-1),Vs(i-1,j),Vs(i-1,j+1),Vs(i,j+1),Vs(i+1,j+1)
                WRITE(111,'(A15,8I12)') 'msk all 8 = ',calve_mask(i+1,j),calve_mask(i+1,j-1),calve_mask(i,j-1),calve_mask(i-1,j-1),  &
                                                       calve_mask(i-1,j),calve_mask(i-1,j+1),calve_mask(i,j+1),calve_mask(i+1,j+1)
                WRITE(111,'(A15,3E15.3)') ' rhs and C_uv: ', rhs_x(i,j), rhs_y(i,j), C_uv(i,j)
                STOP
               END IF

               ! Keeping the largest rests for iteration-stop-check:
               largest_rest_x = MAX(largest_rest_x,ABS(Rx)) ! largest_rest_x + ABS(Rx) !
               largest_rest_y = MAX(largest_rest_y,ABS(Ry)) ! largest_rest_y + ABS(Ry) ! 
               
               ! determine smallest value of the eu_ij, ev_ij terms for the iteration stop
               smallest_eu_ij = MIN(smallest_eu_ij,ABS(eu_ij)) ! smallest_eu_ij + ABS(eu_ij) ! 
               smallest_ev_ij = MIN(smallest_ev_ij,ABS(ev_ij)) ! smallest_ev_ij + ABS(ev_ij) !

              END IF
            END DO
          END DO
        END DO ! sweep
        
        ! Step 2b: Update {Us,Vs} along some edges:
        ! Conditions for the velocity component that is orthogonal to the boundary,
        ! i.e. specify Us at RIGHT and LEFT boundary, and Vs at TOP and BOTTOM boundary.

        DO j = 2, C%NY_ant - 1
          Us(1       ,j) = Us(2         ,j)
          Us(C%NX_ant,j) = Us(C%NX_ant-1,j)
        END DO

        DO i = 2, C%NX_ant - 1
          Vs(i,       1) = Vs(i,         2)
          Vs(i,C%NY_ant) = Vs(i,C%NY_ant-1)
        END DO

        ! Step 2c: Update {Us,Vs} along the other edges.
        ! Conditions for the velocity component that is to the boundary,
        ! i.e. specify Us at TOP and BOTTOM boundary, and Vs at RIGHT and LEFT boundary.
        ! Impose dUs_dy + dVs_dx = 0 along the boundary:

        DO i = 2, C%NX_ant - 1
          Us(i,       1) = Us(i,2         )
          Us(i,C%NY_ant) = Us(i,C%NY_ant-1)
        END DO

        DO j = 2, C%NY_ant - 1
          Vs(1       ,j) = Vs(2         ,j)
          Vs(C%NX_ant,j) = Vs(C%NX_ant-1,j)
        END DO


        ! Step 2d: Update {Us,Vs} at the corner.
        ! Set Us and Vs at the 4 corner-points:
        Us(       1,       1) = Us(         2,         1)
        Vs(       1,       1) = Vs(         1,         2)
        Us(C%NX_ant,       1) = Us(C%NX_ant-1,         1)
        Vs(C%NX_ant,       1) = Vs(C%NX_ant  ,         2)
        Us(       1,C%NY_ant) = Us(         2,C%NY_ant  )
        Vs(       1,C%NY_ant) = Vs(         1,C%NY_ant-1)
        Us(C%NX_ant,C%NY_ant) = Us(C%NX_ant-1,C%NY_ant  )
        Vs(C%NX_ant,C%NY_ant) = Vs(C%NX_ant  ,C%NY_ant-1)

        IF(debug) THEN
          WRITE(C%stdlog,FMT='(A,I4,A,F13.1,2(A,2E15.6))') 'solve_UsVs: at iteration: ',              &
                iter_gauss_seidel, ', at time: ', T_ANT%time, ' yr, largest_rest: ',      &
                largest_rest_x, largest_rest_y, ' smallest eu: ', smallest_eu_ij, smallest_ev_ij
        END IF

        IF( MAX((largest_rest_x / smallest_eu_ij), (largest_rest_y / smallest_ev_ij)) < C%residual_epsilon) THEN
!         WRITE (111,FMT='(A,I4,A,E15.6,2(A,2E15.6))') 'solve_UsVs: Iteration ', iter_gauss_seidel, ': convergence!, residual: ', &
!               MAX((largest_rest_x / smallest_eu_ij), (largest_rest_y / smallest_ev_ij)), ' largest_rest: ',      &
!                largest_rest_x, largest_rest_y, ' smallest eu: ', smallest_eu_ij, smallest_ev_ij
         
         EXIT ITERATE

        ELSE IF(iter_gauss_seidel == C%max_iter_gauss_seidel) THEN
!         WRITE(111,FMT='(A,I4,A,F13.1,A,E15.6)') 'solve_UsVs: No convergence after iteration: ', &
!               C%max_iter_gauss_seidel, ', at time: ', T_ANT%time, ' yr, residual: ',        &
!               MAX((largest_rest_x / smallest_eu_ij), (largest_rest_y / smallest_ev_ij))

         ! When iteratation does not converge set Us and Vs to zero and redo the iteration
         Us = 0._dp
         Vs = 0._dp
         
        END IF

      END DO ITERATE

    END DO ! End: explicit_update_iteration

  END SUBROUTINE ssa_velocity


  SUBROUTINE basal_yield_stress(Hb, mask, sealevel, tau_yield, lambda_p)
    ! Calculate the basal yield stress, which is used to determine the basal stress, used 
    ! for sliding. Using the parameterisations given by Martin et al. (TCD: 2010) 
    ! As used in the PISM-PIK model
    ! Added by BdB 08/2011
    USE parameters_main_module, ONLY : ice_density, grav
  
    ! input variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hb                ! the bedrock height [m]
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: mask              ! mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: sealevel          ! sea level relative to PD [m]
    !REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: bottom_heat_ratio ! heating ratio at the bottom of the ice sheet, generating water


    ! output variable
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: tau_yield         ! the basal yield stress [Pa]
    
    ! local variables
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: lambda_p          ! scaling of the pore water pressure
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: P_wat             ! pore water pressure [Pa]
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: phi_fric          ! the friction angle (degrees)

    REAL(dp), PARAMETER :: pf1 = -1000._dp  ! scaling of phi_frac - original = -1000
    REAL(dp), PARAMETER :: pf2 =     0._dp  ! scaling of phi_frac - original = 0 
    REAL(dp), PARAMETER :: p_min =   5._dp  ! minimum value of phi_fric
    REAL(dp), PARAMETER :: p_max =  20._dp  ! maximum value of phi_fric
  
    
    ! the pore water pressure is scaled with a bedrock height dependend parameterisation
    ! Equation (13) in Martin et al. (2011)
    WHERE ((Hb - sealevel) <= C%wat_level_min)
      lambda_p = 1._dp
    ELSE WHERE ((Hb - sealevel) >= C%wat_level_max)
      lambda_p = 0._dp
    ELSE WHERE ! between 0 and 1000
      lambda_p = 1._dp - (C%wat_level_max - Hb + sealevel) / (C%wat_level_max - C%wat_level_min)
    END WHERE
    
    ! adapt lambda_p according to the bottom heating ratio
    !lambda_p = MAX(0._dp,MIN(1._dp,(lambda_p + bottom_heat_ratio)))

    ! The pore water pressure, equation (12) in Martin et al. (2011)
    P_wat = 0.96_dp * ice_density * grav * lambda_p

    ! The friction angle, used for the yield stress, equation (11) in Martin et al. (2011)   
    WHERE (Hb <= pf1)
      phi_fric = p_min
    ELSE WHERE (Hb >= pf2)
      phi_fric = p_max
    ELSE WHERE ! between pf2 and pf1
      phi_fric = p_min + (p_max - p_min) * (1._dp + (Hb - pf2) / (pf2 - pf1))
    END WHERE 
     
    ! set the yield stress to zero for ocean or shelf grid points
    WHERE (mask == C%type_shelf)
      tau_yield = 0._dp
    ! The yield stress, equation (10) in Martin et al. (2011)
    ELSE WHERE
      tau_yield = TAN(C%deg2rad * phi_fric) * (ice_density * grav - P_wat)
    END WHERE
   
  END SUBROUTINE basal_yield_stress
END MODULE shelf_ant_module
