MODULE restart_file_ant_module
  ! This module defines methods to read and write the current state of the model from or to a file.
  ! To open an existing data file use OPEN_RESTART_FILE while CREATE_RESTART_FILE overwrites an existing data
  ! file or creates a new one.
  ! Then use READ_RESTART_FILE and WRITE_RESTART_FILE to read or write the state variables to the file.
  ! Finally use CLOSE_RESTART_FILE when you are finished using the file.

  ! Adjusted BdB 10/2010 - The structure 'restart' is saved in the module to save the 
  ! id's of the variables written in the restart and help_field files
  ! furthermore, the writing of the help field is (drastically) changes, since a lot of fields 
  ! are (according to me - BdB) not necessary, which saves time and space since the help_field is quite large

  USE netcdf, ONLY: nf90_max_var_dims, nf90_create, nf90_close, nf90_clobber, nf90_share, nf90_unlimited , &
                    nf90_enddef, nf90_put_var, nf90_sync, nf90_def_var, nf90_int, nf90_put_att, nf90_def_dim, &
                    nf90_open, nf90_write, nf90_inq_dimid, nf90_inquire_dimension, nf90_inquire, nf90_double, &
                    nf90_inq_varid, nf90_inquire_variable, nf90_get_var, nf90_noerr, nf90_strerror

  USE configuration_main_module, ONLY: dp, C
  USE configuration_ant_module, ONLY: T_ANT
  IMPLICIT NONE

  PRIVATE
  PUBLIC :: netcdf_file_type_ant, create_restart_file, open_netcdf_file, inquire_restart_file,   &
            read_restart_file, write_restart_file, count_netcdf_record_size, close_netcdf_file,  &
            write_help_fields_file, create_help_fields_file, close_restart_file,                 &
            close_help_fields_file, read_monthly_file, inquire_monthly_file,                     &
            read_extra_file, inquire_extra_file, read_ocean_temp_file, inquire_ocean_temp_file
            
  TYPE netcdf_file_type_ant
    PRIVATE

    ! id for netCDF file: 
    INTEGER ::  ncid
    ! id's for variables:
    INTEGER :: var_x
    INTEGER :: var_y
    INTEGER :: var_zeta
    INTEGER :: var_t
    INTEGER :: var_Hi
    INTEGER :: var_Hs
    INTEGER :: var_Hb
    INTEGER :: var_Ti
    INTEGER :: var_Tr
    INTEGER :: var_Us
    INTEGER :: var_Vs
    INTEGER :: var_bm
    INTEGER :: var_MB_surface
    INTEGER :: var_MB_basal
    INTEGER :: var_mask
    INTEGER :: var_calve_mask
    INTEGER :: var_dHs_dx
    INTEGER :: var_dHs_dy
    INTEGER :: var_D_2D
    INTEGER :: var_A_flow_mean
    INTEGER :: var_dummy1_2D
    INTEGER :: var_dummy2_2D
    INTEGER :: var_dummy3_2D
    INTEGER :: var_dummy4_2D
    INTEGER :: var_dummy5_2D
    INTEGER :: var_dummy6_2D
    INTEGER :: var_D_uv_3D
    INTEGER :: var_A_flow
    INTEGER :: var_U
    INTEGER :: var_V
    INTEGER :: var_W
    INTEGER :: var_Cpi
    INTEGER :: var_Ki
    INTEGER :: var_dummy1_3D
    INTEGER :: var_dummy2_3D
    INTEGER :: var_dummy3_3D
    INTEGER :: var_dummy4_3D
    INTEGER :: var_dummy5_3D
    INTEGER :: var_dummy6_3D
    INTEGER :: var_Ts

!!-- Added BdB (11/2009)
    INTEGER :: var_Tmon
    INTEGER :: var_Pmon 
    INTEGER :: var_Smon 

!!-- Added BdB (01/2010): for reading files with 5 variables (Long_Lat_Uwind_Vwind_Hflux)
    INTEGER :: var_Lon
    INTEGER :: var_lat
    INTEGER :: var_Hflux
!!-- Added BdB (05/2016): for reading ocean temperatures from a climate model
    INTEGER :: var_Tocean
    INTEGER :: var_d

  END TYPE netcdf_file_type_ant

  ! Save structure restart and help_fields to save the id's of the variables
  TYPE(netcdf_file_type_ant), SAVE :: restart
  TYPE(netcdf_file_type_ant), SAVE :: help_fields  


  ! Names of dimensions:
  CHARACTER(LEN=*), PARAMETER :: name_x                     = 'x                    '
  CHARACTER(LEN=*), PARAMETER :: name_y                     = 'y                    '
  CHARACTER(LEN=*), PARAMETER :: name_zeta                  = 'zeta                 '
  CHARACTER(LEN=*), PARAMETER :: name_t                     = 'time                 '
  CHARACTER(LEN=*), PARAMETER :: name_z                     = 'z                    '

  ! Names of variables:
  CHARACTER(LEN=*), PARAMETER :: name_Hi                    = 'Hi                   '
  CHARACTER(LEN=*), PARAMETER :: name_Hb                    = 'Hb                   '
  CHARACTER(LEN=*), PARAMETER :: name_Hs                    = 'Hs                   '
  CHARACTER(LEN=*), PARAMETER :: name_Ti                    = 'Ti                   '
  CHARACTER(LEN=*), PARAMETER :: name_Tr                    = 'Tr                   '
  CHARACTER(LEN=*), PARAMETER :: name_Us                    = 'Us                   '
  CHARACTER(LEN=*), PARAMETER :: name_Vs                    = 'Vs                   '
  CHARACTER(LEN=*), PARAMETER :: name_bm                    = 'bottom_melt          '
  CHARACTER(LEN=*), PARAMETER :: name_MB_surface            = 'MB_surface           '
  CHARACTER(LEN=*), PARAMETER :: name_MB_basal              = 'MB_basal             '
  CHARACTER(LEN=*), PARAMETER :: name_mask                  = 'mask                 '
  CHARACTER(LEN=*), PARAMETER :: name_calve_mask            = 'calve_mask           '
  CHARACTER(LEN=*), PARAMETER :: name_dHs_dx                = 'dHs_dx               '
  CHARACTER(LEN=*), PARAMETER :: name_dHs_dy                = 'dHs_dy               '
  CHARACTER(LEN=*), PARAMETER :: name_D_2D                  = 'D_2D                 '
  CHARACTER(LEN=*), PARAMETER :: name_A_flow_mean           = 'A_flow_mean          '
  CHARACTER(LEN=*), PARAMETER :: name_dummy1_2D             = 'dummy1_2D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy2_2D             = 'dummy2_2D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy3_2D             = 'dummy3_2D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy4_2D             = 'dummy4_2D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy5_2D             = 'dummy5_2D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy6_2D             = 'dummy6_2D            '
  CHARACTER(LEN=*), PARAMETER :: name_D_uv_3D               = 'D_uv_3D              '
  CHARACTER(LEN=*), PARAMETER :: name_A_flow                = 'A_flow               '
  CHARACTER(LEN=*), PARAMETER :: name_U                     = 'U                    '
  CHARACTER(LEN=*), PARAMETER :: name_V                     = 'V                    '
  CHARACTER(LEN=*), PARAMETER :: name_W                     = 'W                    '
  CHARACTER(LEN=*), PARAMETER :: name_Cpi                   = 'Cpi                  '
  CHARACTER(LEN=*), PARAMETER :: name_Ki                    = 'Ki                   '
  CHARACTER(LEN=*), PARAMETER :: name_dummy1_3D             = 'dummy1_3D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy2_3D             = 'dummy2_3D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy3_3D             = 'dummy3_3D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy4_3D             = 'dummy4_3D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy5_3D             = 'dummy5_3D            '
  CHARACTER(LEN=*), PARAMETER :: name_dummy6_3D             = 'dummy6_3D            '
  CHARACTER(LEN=*), PARAMETER :: name_Ts                    = 'Ts                   '

  CHARACTER(LEN=*), PARAMETER :: name_Tmon                  = 'T2m                  '
  CHARACTER(LEN=*), PARAMETER :: name_Pmon                  = 'Precip               '
  CHARACTER(LEN=*), PARAMETER :: name_Smon                  = 'SWdown               '
  CHARACTER(LEN=*), PARAMETER :: name_Lon                   = 'Lon                  '
  CHARACTER(LEN=*), PARAMETER :: name_Lat                   = 'Lat                  '
  CHARACTER(LEN=*), PARAMETER :: name_Hflux                 = 'Hflux                '
  CHARACTER(LEN=*), PARAMETER :: name_Tocean                = 'Tocean               '
  CHARACTER(LEN=*), PARAMETER :: name_d                     = 'd                    '

  CHARACTER(LEN=*), PARAMETER :: name_x2                    = 'NX                   '
  CHARACTER(LEN=*), PARAMETER :: name_y2                    = 'NY                   '
  CHARACTER(LEN=*), PARAMETER :: name_t2                    = 'NTIME                '

CONTAINS
  SUBROUTINE create_restart_file(NX, NY, dx, dy, file_name, write_differences) 
    IMPLICIT NONE
    
    ! Input variables:
    INTEGER,                INTENT(IN)  :: NX
    INTEGER,                INTENT(IN)  :: NY
    REAL(dp),               INTENT(IN)  :: dx
    REAL(dp),               INTENT(IN)  :: dy
    CHARACTER(LEN=*),       INTENT(IN)  :: file_name
    ! This option is used in case the differences of two restart files are written:
    INTEGER, OPTIONAL,      INTENT(IN)  :: write_differences 

    ! Local variables:
    INTEGER                             :: m
    INTEGER                             :: dim_x, dim_y, dim_zeta, dim_t
    LOGICAL                             :: file_exists

    ! Create a new restart file if none exists and, to prevent loss of data, 
    ! stop with an error message if one already exists (not when differences are considered):
    INQUIRE(EXIST=file_exists, FILE = file_name)
    IF(file_exists .AND. (PRESENT(write_differences) .eqv. .FALSE.)) THEN
     WRITE(0,'(5A)') 'ERROR: ', TRIM(file_name), ' already exists! Remove the ', &
                                         TRIM(file_name), ' and run again!'
     STOP
    END IF
    
    ! Create netCDF file
    CALL handle_error(nf90_create(file_name,IOR(nf90_clobber,nf90_share),restart%ncid))
    
!    write (0,*) 'create restart file; ncid = ', restart%ncid

    ! Define dimensions:
    CALL create_dim(restart, name_x, NX, dim_x)
    CALL create_dim(restart, name_y, NY, dim_y)
    CALL create_dim(restart, name_zeta, C%NZ, dim_zeta)
    CALL create_dim(restart, name_t, nf90_unlimited, dim_t)

    ! Define variables:
    ! The order of the CALL statements for the different variables determines their
    ! order of appearence in the netcdf file.

    ! x:
    CALL create_double_var(restart, name_x, (/ dim_x /), restart%var_x, units='meters')
    ! y:
    CALL create_double_var(restart, name_y, (/ dim_y /), restart%var_y, units='meters')
    
    ! zeta:
    CALL create_double_var(restart, name_zeta, (/ dim_zeta /), restart%var_zeta, &
                           long_name='scaled vertical zeta-axis (unitless)',units='unitless')

    ! time:
    CALL create_double_var(restart, name_t, (/ dim_t /), restart%var_t, long_name='Time', units='years')

    ! ice thickness:
    CALL create_double_var(restart, name_Hi, (/ dim_x, dim_y, dim_t /), restart%var_Hi, &
                            long_name='Ice Thickness', units='meter')
                            
    ! bedrock height:
    CALL create_double_var(restart, name_Hb, (/ dim_x, dim_y, dim_t /), restart%var_Hb, &
                           long_name='Bedrock Height',units='meter')

    ! surface height:
    CALL create_double_var(restart, name_Hs, (/ dim_x, dim_y, dim_t /), restart%var_Hs, &
                           long_name='Surface Height', units='meter')

    ! z-averaged horizontal velocity in x-direction:
    CALL create_double_var(restart, name_Us, (/ dim_x, dim_y, dim_t /), restart%var_Us, &
                           long_name='Vertically Averaged Ice Velocity in x-direction', &
                           units='meter per year')

    ! z-averaged horizontal velocity in y-direction:
    CALL create_double_var(restart, name_Vs, (/ dim_x, dim_y, dim_t /), restart%var_Vs, &
                           long_name='Vertically Averaged Ice Velocity in y-direction', &
                           units='meter per year')

    ! bottom melting underneath the ice sheet
    CALL create_double_var(restart, name_bm, (/ dim_x, dim_y, dim_t /), restart%var_bm, &
                           long_name='bottom melt', units='meter per year')

    ! ice temperature: 
    CALL create_double_var(restart, name_Ti, (/ dim_x, dim_y, dim_zeta, dim_t /), restart%var_Ti, &
                            long_name='Ice Temperature',units='Kelvin')

    ! Leave definition mode:
    CALL handle_error(nf90_enddef(restart%ncid))

    ! Write data for limited dimensions:
    
    ! x and y units are in meters:
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_x, (/ (((m-(NX+1)/2)*dx),m=1,NX) /)))
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_y, (/ (((m-(NY+1)/2)*dy),m=1,NY) /)))

    CALL handle_error(nf90_put_var(restart%ncid, restart%var_zeta, C%zeta))

    ! The unlimited time dimension  t  is written and not defined here.

    ! Synchronize with disk (otherwise it doesn't seem to work on a MAC)
    CALL handle_error(nf90_sync(restart%ncid))
  END SUBROUTINE create_restart_file



  SUBROUTINE create_help_fields_file(NX, NY, dx, dy, file_name) 
    
    ! Input variables:
    INTEGER,                INTENT(IN)  :: NX
    INTEGER,                INTENT(IN)  :: NY
    REAL(dp),               INTENT(IN)  :: dx
    REAL(dp),               INTENT(IN)  :: dy
    CHARACTER(LEN=*),       INTENT(IN)  :: file_name

    ! Local variables:
    INTEGER                             :: i
    INTEGER                             :: dim_x, dim_y, dim_zeta, dim_t
    LOGICAL                             :: file_exists
    
    ! Create a new help_fields file if none exists and, to prevent loss of data, 
    ! stop with an error message if one already exists (not when differences are considered):
    INQUIRE(EXIST=file_exists, FILE = file_name)
    IF(file_exists) THEN
     WRITE(0,'(5A)') 'ERROR: ', TRIM(file_name), ' already exists! Remove the ', &
                                         TRIM(file_name), ' and run again!'
     STOP
    END IF

    ! Create netCDF file
    CALL handle_error(nf90_create(file_name,IOR(nf90_clobber,nf90_share),help_fields%ncid))
    
!      write (0,*) 'create help_fields file; ncid = ', help_fields%ncid

    ! Define dimensions:
    CALL create_dim(help_fields, name_x, NX, dim_x)
    CALL create_dim(help_fields, name_y, NY, dim_y)
    CALL create_dim(help_fields, name_zeta, C%NZ, dim_zeta)
    CALL create_dim(help_fields, name_t, nf90_unlimited, dim_t)

    ! Define variables:
    ! The order of the CALL statements for the different variables determines their
    ! order of appearence in the netcdf file.
    
    ! x and y units are in meters:
    CALL create_double_var(help_fields, name_x, (/ dim_x /), help_fields%var_x, units='meters')
    CALL create_double_var(help_fields, name_y, (/ dim_y /), help_fields%var_y, units='meters')
    
    ! zeta:
    IF(C%which_help_fields >= 40) &
     CALL create_double_var(help_fields, name_zeta, (/ dim_zeta /), help_fields%var_zeta, &
                            long_name='Normalized Vertical Coordinate')

    ! time:
    CALL create_double_var(help_fields, name_t, (/ dim_t /), help_fields%var_t, long_name='Time', units='years')
 
    ! Surface Mass Balance:
    CALL create_double_var(help_fields, name_MB_surface, (/ dim_x, dim_y, dim_t /), help_fields%var_MB_surface, &
                            long_name='Surface Mass Balance', units='mieq')

    ! Basal Mass Balance:
    CALL create_double_var(help_fields, name_MB_basal, (/ dim_x, dim_y, dim_t /), help_fields%var_MB_basal, &
                            long_name='Basal Mass Balance', units='mieq')

    ! mask:
    IF(C%which_help_fields >=  3) &
     CALL create_double_var(help_fields, name_mask, (/ dim_x, dim_y, dim_t /), help_fields%var_mask, &
                          long_name='Type of Land')

    ! calve_mask:
    IF(C%which_help_fields >=  4) &
     CALL create_double_var(help_fields, name_calve_mask, (/ dim_x, dim_y, dim_t /), help_fields%var_calve_mask, &
                          long_name='Type of Land')

    ! dHs_dx:
    IF(C%which_help_fields >=  6) &
     CALL create_double_var(help_fields, name_dHs_dx, (/ dim_x, dim_y, dim_t /), help_fields%var_dHs_dx, &
                             long_name='dHs_dx', units='m/m')

    ! dHs_dy:
    IF(C%which_help_fields >=  6) &
     CALL create_double_var(help_fields, name_dHs_dy, (/ dim_x, dim_y, dim_t /), help_fields%var_dHs_dy, &
                             long_name='dHs_dy', units='m/m')

    ! D:
    IF(C%which_help_fields >=  7) &
     CALL create_double_var(help_fields, name_D_2D, (/ dim_x, dim_y, dim_t /), help_fields%var_D_2D, &
                             long_name='D_2D', units='none')

    ! A_flow_mean:
    IF(C%which_help_fields >=  8) &
     CALL create_double_var(help_fields, name_A_flow_mean, (/ dim_x, dim_y, dim_t /), help_fields%var_A_flow_mean, &
                             long_name='A_flow_mean', units='Pa^-3 y^-1')

    ! dummy1_2D:
    IF(C%which_help_fields >=  9) &
     CALL create_double_var(help_fields, name_dummy1_2D, (/ dim_x, dim_y, dim_t /), help_fields%var_dummy1_2D, &
                             long_name='dummy1_2D', units='?')

    ! dummy2_2D:
    IF(C%which_help_fields >= 10) &
     CALL create_double_var(help_fields, name_dummy2_2D, (/ dim_x, dim_y, dim_t /), help_fields%var_dummy2_2D, &
                             long_name='dummy2_2D', units='?')

    ! dummy3_2D:
    IF(C%which_help_fields >= 11) &
     CALL create_double_var(help_fields, name_dummy3_2D, (/ dim_x, dim_y, dim_t /), help_fields%var_dummy3_2D, &
                             long_name='dummy3_2D', units='?')

    ! dummy4_2D:
    IF(C%which_help_fields >= 12) &
     CALL create_double_var(help_fields, name_dummy4_2D, (/ dim_x, dim_y, dim_t /), help_fields%var_dummy4_2D, &
                             long_name='dummy4_2D', units='?')

    ! dummy5_2D:
    IF(C%which_help_fields >= 13) &
     CALL create_double_var(help_fields, name_dummy5_2D, (/ dim_x, dim_y, dim_t /), help_fields%var_dummy5_2D, &
                             long_name='dummy5_2D', units='?')

    ! dummy6_2D:
    IF(C%which_help_fields >= 14) &
     CALL create_double_var(help_fields, name_dummy6_2D, (/ dim_x, dim_y, dim_t /), help_fields%var_dummy6_2D, &
                             long_name='dummy6_2D', units='?')

    ! D_uv_3D: 
    IF(C%which_help_fields >= 40) &
     CALL create_double_var(help_fields, name_D_uv_3D, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_D_uv_3D, &
                             long_name='D_uv_3D',units='none')
     
    ! A_flow: 
    IF(C%which_help_fields >= 41) &
     CALL create_double_var(help_fields, name_A_flow, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_A_flow, &
                             long_name='A_flow',units='Pa^-3 y^-1')
     
    ! U: 
    IF(C%which_help_fields >= 45) &
     CALL create_double_var(help_fields, name_U, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_U, &
                             long_name='U',units='meter per year')
     
    ! V: 
    IF(C%which_help_fields >= 46) &
     CALL create_double_var(help_fields, name_V, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_V, &
                             long_name='V',units='meter per year')
     
    ! W: 
    IF(C%which_help_fields >= 47) &
     CALL create_double_var(help_fields, name_W, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_W, &
                             long_name='W',units='meter per year')
     
    ! Cpi: 
    IF(C%which_help_fields >= 51) &
     CALL create_double_var(help_fields, name_Cpi, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_Cpi, &
                             long_name='Cpi',units='J kg^-1 K^-1')
     
    ! Ki: 
    IF(C%which_help_fields >= 52) &
     CALL create_double_var(help_fields, name_Ki, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_Ki, &
                             long_name='Ki',units='J m^-1 K^-1 yr^-1')
     
    ! dummy1_3D: 
    IF(C%which_help_fields >= 61) &
     CALL create_double_var(help_fields, name_dummy1_3D, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_dummy1_3D, &
                             long_name='dummy1_3D',units='?')
     
    ! dummy2_3D: 
    IF(C%which_help_fields >= 62) &
     CALL create_double_var(help_fields, name_dummy2_3D, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_dummy2_3D, &
                             long_name='dummy2_3D',units='?')

    ! dummy3_3D: 
    IF(C%which_help_fields >= 63) &
     CALL create_double_var(help_fields, name_dummy3_3D, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_dummy3_3D, &
                             long_name='dummy3_3D',units='?')
     
    ! dummy4_3D: 
    IF(C%which_help_fields >= 64) &
     CALL create_double_var(help_fields, name_dummy4_3D, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_dummy4_3D, &
                             long_name='dummy4_3D',units='?')

    ! dummy5_3D: 
    IF(C%which_help_fields >= 65) &
     CALL create_double_var(help_fields, name_dummy5_3D, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_dummy5_3D, &
                             long_name='dummy5_3D',units='?')
     
    ! dummy6_3D: 
    IF(C%which_help_fields >= 66) &
     CALL create_double_var(help_fields, name_dummy6_3D, (/dim_x, dim_y, dim_zeta, dim_t /), help_fields%var_dummy6_3D, &
                             long_name='dummy6_3D',units='?')

    ! Leave definition mode:
    CALL handle_error(nf90_enddef(help_fields%ncid))

    ! Write data for limited dimensions:
    
    ! The units of the dimensions are in physical units:
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_x, (/ (((i-(NX+1)/2)*dx),i=1,NX) /)))
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_y, (/ (((i-(NY+1)/2)*dy),i=1,NY) /)))
     IF(C%which_help_fields >= 40) &
      CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_zeta, C%zeta))

    ! Synchronize with disk (otherwise it doesn't seem to work on a MAC)
    CALL handle_error(nf90_sync(help_fields%ncid))
  END SUBROUTINE create_help_fields_file


  SUBROUTINE create_dim(netcdf_file, dim_name, length, id_dim)
    ! Subroutine for creating netCDF dimensions more convenient:
    IMPLICIT NONE
  
    ! Input variables:
    TYPE(netcdf_file_type_ant), INTENT(IN) :: netcdf_file
    CHARACTER(LEN=*),           INTENT(IN) :: dim_name
    INTEGER,                    INTENT(IN) :: length
    
    ! Output variables:
    INTEGER, INTENT(OUT)               :: id_dim

    CALL handle_error(nf90_def_dim(netcdf_file%ncid,dim_name,length,id_dim))
  END SUBROUTINE create_dim



  SUBROUTINE create_double_var(netcdf_file, var_name, id_dims, id_var, long_name, units, missing_value)
    ! Subroutine for creating netCDF variables of type nf90_DOUBLE more convenient:
  
    ! Input variables:
    TYPE(netcdf_file_type_ant),   INTENT(IN)  :: netcdf_file
    CHARACTER(LEN=*),             INTENT(IN)  :: var_name
    INTEGER, DIMENSION(:),        INTENT(IN)  :: id_dims
    CHARACTER(LEN=*),   OPTIONAL, INTENT(IN)  :: long_name
    CHARACTER(LEN=*),   OPTIONAL, INTENT(IN)  :: units
    REAL(dp),           OPTIONAL, INTENT(IN)  :: missing_value

    ! Output variables:
    INTEGER,                      INTENT(OUT) :: id_var

    CALL handle_error(nf90_def_var(netcdf_file%ncid,var_name,nf90_double,id_dims,id_var))
    IF(PRESENT(long_name))     CALL handle_error(nf90_put_att(netcdf_file%ncid,id_var,'long_name',long_name))
    IF(PRESENT(units))         CALL handle_error(nf90_put_att(netcdf_file%ncid,id_var,'units',units))
    IF(PRESENT(missing_value)) CALL handle_error(nf90_put_att(netcdf_file%ncid,id_var,'missing_value',missing_value))
  END SUBROUTINE create_double_var



  SUBROUTINE open_netcdf_file(filename, netcdf) 
    IMPLICIT NONE
    
    ! Input variable:
    CHARACTER(LEN=*), INTENT(IN)        :: filename

    ! Output variable:
    TYPE(netcdf_file_type_ant), INTENT(OUT) :: netcdf
    
    ! Open netCDF file:
    CALL handle_error(nf90_open(filename, IOR(nf90_write,nf90_share), netcdf%ncid))

  END SUBROUTINE open_netcdf_file



  SUBROUTINE inquire_restart_file(netcdf_file) 
    ! Check if the right dimensions are present in the restart file, and if the variables have the right
    ! dimension: specie and length.
    
    ! Input/output variable:
    TYPE(netcdf_file_type_ant), INTENT(INOUT) :: netcdf_file
 
    ! Local variables:
    INTEGER                               :: dim_x
    INTEGER                               :: dim_y
    INTEGER                               :: dim_t
    INTEGER                               :: dim_zeta
    
    ! Inquire dimensions id's. Check that all required dimensions exist and their lengts are correct:
    CALL inquire_dim(netcdf_file, name_x,      C%NX_ant,             dim_x)
    CALL inquire_dim(netcdf_file, name_y,      C%NY_ant,             dim_y)
    CALL inquire_dim(netcdf_file, name_zeta,   C%NZ,              dim_zeta)
    CALL inquire_dim(netcdf_file, name_t,      nf90_unlimited,       dim_t)

    ! Inquire variable id's. Make sure that each variable has the correct dimensions:
    CALL inquire_double_var(netcdf_file, name_t,      (/ dim_t /),                           netcdf_file%var_t)
    CALL inquire_double_var(netcdf_file, name_Hi,     (/ dim_x, dim_y, dim_t /),             netcdf_file%var_Hi)
    CALL inquire_double_var(netcdf_file, name_Hb,     (/ dim_x, dim_y, dim_t /),             netcdf_file%var_Hb)
    CALL inquire_double_var(netcdf_file, name_Hs,     (/ dim_x, dim_y, dim_t /),             netcdf_file%var_Hs)
    CALL inquire_double_var(netcdf_file, name_Us,     (/ dim_x, dim_y, dim_t /),             netcdf_file%var_Us)
    CALL inquire_double_var(netcdf_file, name_Vs,     (/ dim_x, dim_y, dim_t /),             netcdf_file%var_Vs)
    CALL inquire_double_var(netcdf_file, name_bm,     (/ dim_x, dim_y, dim_t /),             netcdf_file%var_bm)
    CALL inquire_double_var(netcdf_file, name_Ti,     (/ dim_x, dim_y, dim_zeta, dim_t /),   netcdf_file%var_Ti)
  END SUBROUTINE inquire_restart_file


!!-- Added BdB (11/2009): for reading monthly T2m (Ts) and Precip (w.e. per month: total is per year)

  SUBROUTINE inquire_monthly_file(reference) 
    ! Check if the right dimensions are present in the monthly file, and if the variables 
    ! have the right dimension
    
    ! Input/output variable:
    TYPE(netcdf_file_type_ant), INTENT(INOUT) :: reference
 
    ! Local variables:
    INTEGER                               :: dim_x
    INTEGER                               :: dim_y
    INTEGER                               :: dim_t
    
    ! Inquire dimensions id's. Check that all required dimensions exist and their lengts are correct:
    CALL inquire_dim(reference, name_x2, C%NX_ant,           dim_x)
    CALL inquire_dim(reference, name_y2, C%NY_ant,           dim_y)
    CALL inquire_dim(reference, name_t2, nf90_unlimited,     dim_t)

    ! Inquire variable id's. Make sure that each variable has the correct dimensions:
    CALL inquire_double_var(reference, name_Tmon,       (/ dim_x, dim_y, dim_t /), reference%var_Tmon)
    CALL inquire_double_var(reference, name_Pmon,       (/ dim_x, dim_y, dim_t /), reference%var_Pmon)
    IF(C%use_sw_down_flux_at_surface) CALL inquire_double_var(reference, name_Smon, (/ dim_x, dim_y, dim_t /), reference%var_Smon)

  END SUBROUTINE inquire_monthly_file

  
  SUBROUTINE inquire_extra_file(reference,doing_a_restart) 

    ! Check if the right dimensions are present in the file, and if the variables 
    ! have the right dimension

    ! Input/output variable:
    TYPE(netcdf_file_type_ant), INTENT(INOUT) :: reference
    LOGICAL,                    INTENT(IN)    :: doing_a_restart
 
    ! Local variables:
    INTEGER                               :: dim_x
    INTEGER                               :: dim_y
    
    ! Inquire dimensions id's. Check that all required dimensions exist and their lengts are correct:
    CALL inquire_dim(reference, name_x2, C%NX_ant,       dim_x)
    CALL inquire_dim(reference, name_y2, C%NY_ant,       dim_y)

    ! Inquire variable id's. Make sure that each variable has the correct dimensions:
    CALL inquire_double_var(reference, name_Lon,       (/ dim_x, dim_y /), reference%var_Lon)
    CALL inquire_double_var(reference, name_Lat,       (/ dim_x, dim_y /), reference%var_Lat)
    CALL inquire_double_var(reference, name_Hflux,     (/ dim_x, dim_y /), reference%var_Hflux)
    CALL inquire_double_var(reference, name_Hs,        (/ dim_x, dim_y /), reference%var_Hs)

    IF(doing_a_restart) CALL inquire_double_var(reference, name_Hi, (/ dim_x, dim_y /), reference%var_Hi)
    IF(doing_a_restart) CALL inquire_double_var(reference, name_Hb, (/ dim_x, dim_y /), reference%var_Hb)
  END SUBROUTINE inquire_extra_file



!!-- Added BdB (05/2016): for reading 3D ocean temperatures
  SUBROUTINE inquire_ocean_temp_file(reference) 
    ! Check if the right dimensions are present in the ocean temp file, and if the variables 
    ! have the right dimension
    
    ! Input/output variable:
    TYPE(netcdf_file_type_ant), INTENT(INOUT) :: reference
 
    ! Local variables:
    INTEGER                               :: dim_x
    INTEGER                               :: dim_y
    INTEGER                               :: dim_d
    
    ! Inquire dimensions id's. Check that all required dimensions exist and their lengts are correct:
    CALL inquire_dim(reference, name_x2, C%NX_ant,                 dim_x)
    CALL inquire_dim(reference, name_y2, C%NY_ant,                 dim_y)
    CALL inquire_dim(reference, name_t2, C%number_of_ocean_layers, dim_d)

    ! Inquire variable id's. Make sure that each variable has the correct dimensions:
    CALL inquire_double_var(reference, name_Tocean,     (/ dim_x, dim_y, dim_d /), reference%var_Tocean)
    CALL inquire_double_var(reference, name_d,          (/ dim_d /),               reference%var_d)
  END SUBROUTINE inquire_ocean_temp_file



  SUBROUTINE inquire_dim(netcdf, dim_name, length, id_dim)
    ! Inquire the id of a dimension and check that the length matches the length given by the user.
    IMPLICIT NONE
  
    ! Input variables:
    TYPE(netcdf_file_type_ant), INTENT(IN)  :: netcdf
    CHARACTER(LEN=*),           INTENT(IN)  :: dim_name
    INTEGER,                    INTENT(IN)  :: length

    ! Output variables:
    INTEGER,                    INTENT(OUT) :: id_dim

    ! Local variables:
    INTEGER                             :: actual_length
    INTEGER                             :: dim_unlimited

    CALL handle_error(nf90_inq_dimid(netcdf%ncid,dim_name,id_dim))
    IF(length /= nf90_unlimited) THEN
     CALL handle_error(nf90_inquire_dimension(netcdf%ncid, id_dim, len=actual_length))
     IF(length /= actual_length) THEN
      WRITE(0,'(A,I5,3A,I5,A)') 'ERROR: Actual length (',actual_length, &
             ' of dimension "',dim_name,'" does not match required length (',length,').'
      STOP
     END IF
    ELSE
     CALL handle_error(nf90_INQUIRE(netcdf%ncid, unlimitedDimId=dim_unlimited))
     IF(dim_unlimited /= id_dim) THEN
      WRITE(0,'(3A)') 'ERROR: Dimension "',dim_name,'" is not unlimited as required.'
      STOP
     END IF
    END IF
  END SUBROUTINE inquire_dim



  SUBROUTINE inquire_double_var(netcdf, var_name, id_dims, id_var)
    ! Inquire the id of a variable and check that the dimensions of the variable match the dimensions given by the user and
    ! that the variable is of type nf90_DOUBLE.
    IMPLICIT NONE
  
    ! Input variables:
    TYPE(netcdf_file_type_ant), INTENT(IN)    :: netcdf
    CHARACTER(LEN=*),           INTENT(IN)    :: var_name
    INTEGER, DIMENSION(:),      INTENT(IN)    :: id_dims

    ! Output variables:
    INTEGER,                INTENT(OUT)   :: id_var

    ! Local variables:
    INTEGER                               :: xtype, ndims
    INTEGER, DIMENSION(nf90_max_var_dims) :: actual_id_dims

    CALL handle_error(nf90_inq_varid(netcdf%ncid, var_name, id_var))
    CALL handle_error(nf90_inquire_variable(netcdf%ncid, id_var, xtype=xtype,ndims=ndims,dimids=actual_id_dims))
    IF(xtype /= nf90_double) THEN
     WRITE(0,'(3A)') 'ERROR: Actual type of variable "',var_name,'" is not nf90_DOUBLE.'
     STOP
    END IF
    IF(ndims /= SIZE(id_dims)) THEN
     WRITE(0,'(A,I5,3A,I5,A)') 'ERROR: Actual number of dimensions(', &
            ndims,') of variable "',var_name,'": does not match required number of dimensions (',SIZE(id_dims),').'
     STOP
    END IF
    IF(ANY(actual_id_dims(1:ndims) /= id_dims)) THEN
     WRITE(0,'(3A)') 'ERROR: Actual dimensions of variable "',var_name,'" does not match required dimensions.'
     STOP
    END IF
  END SUBROUTINE inquire_double_var


  
  SUBROUTINE read_restart_file(netcdf_file, recno, Hi, Hb, Hs, Us, Vs, bottom_melt, Ti, t)
   
    ! Input variables:
    INTEGER, INTENT(IN)                               :: recno   ! The record number in the file from which to read the variables.
    TYPE(netcdf_file_type_ant),           INTENT(IN)  :: netcdf_file ! The file from which to read.
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(OUT) :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(OUT) :: Hb
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(OUT) :: Hs
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(OUT) :: Us
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(OUT) :: Vs
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(OUT) :: bottom_melt
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(OUT) :: Ti
    REAL(dp),                                     INTENT(OUT) :: t

    ! Local variables:
    INTEGER                                           :: record_size
    
!    write (6,*) 'read restart file: ncid = ', netcdf_file%ncid
    
    ! Output: record_size
    CALL count_netcdf_record_size(netcdf_file, record_size)

    IF((recno > record_size) .OR. (recno < 0)) THEN
      WRITE(0,'(A,I6,A)') 'read_restart_file ERROR: Record number ', recno, ' does not exist!'
      STOP
    END IF
    
    IF(recno == 0) THEN
     ! Read the variables from the last record:
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_t,      t,           start=(/         record_size/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Hi,     Hi,          start=(/1, 1,    record_size/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Hb,     Hb,          start=(/1, 1,    record_size/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Hs,     Hs,          start=(/1, 1,    record_size/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Us,     Us,          start=(/1, 1,    record_size/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Vs,     Vs,          start=(/1, 1,    record_size/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_bm,     bottom_melt, start=(/1, 1,    record_size/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Ti,     Ti,          start=(/1, 1, 1, record_size/)))
    ELSE
    
     ! Read the variables:
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_t,      t,           start=(/         recno/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Hi,     Hi,          start=(/1, 1,    recno/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Hb,     Hb,          start=(/1, 1,    recno/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Hs,     Hs,          start=(/1, 1,    recno/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Us,     Us,          start=(/1, 1,    recno/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Vs,     Vs,          start=(/1, 1,    recno/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_bm,     bottom_melt, start=(/1, 1,    recno/)))
     CALL handle_error(nf90_get_var(netcdf_file%ncid, netcdf_file%var_Ti,     Ti,          start=(/1, 1, 1, recno/)))
    END IF
    
  END SUBROUTINE read_restart_file                                                            


  SUBROUTINE read_monthly_file(reference, recno, T2m, Precip, SWdown)
  
    ! Input variables:
    INTEGER,                        INTENT(IN)  :: recno     ! The record number in the file from which to read the variables.
    TYPE(netcdf_file_type_ant),     INTENT(IN)  :: reference ! The file from which to read.
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: T2m, Precip
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), OPTIONAL, INTENT(OUT) :: SWdown

    ! Local variables:
    INTEGER                                     :: record_size

    ! Output: record_size
    CALL count_netcdf_record_size2(reference, record_size)
    
    IF((recno > record_size) .OR. (recno < 1)) THEN
      WRITE(0,'(A,I6,A)') 'read_monthly_file ERROR: Record number ', recno, ' does not exist!'
      STOP
    END IF
    
    ! Read the variables:
    CALL handle_error(nf90_get_var(reference%ncid, reference%var_Tmon,       T2m,        start=(/1, 1, recno/)))
    CALL handle_error(nf90_get_var(reference%ncid, reference%var_Pmon,       Precip,     start=(/1, 1, recno/)))
    IF(PRESENT(SWdown)) CALL handle_error(nf90_get_var(reference%ncid, reference%var_Smon, SWdown, start=(/1, 1, recno/)))
    
  END SUBROUTINE read_monthly_file



  SUBROUTINE read_extra_file(reference, Lon, Lat, Hflux, Hs, Hi, Hb)
  
    ! Input variables:
    TYPE(netcdf_file_type_ant),             INTENT(IN)  :: reference ! The file from which to read.
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT)           :: Lon, Lat, Hflux, Hs
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT), OPTIONAL :: Hi, Hb

    ! Read the variables:
    CALL handle_error(nf90_get_var(reference%ncid, reference%var_Lon,   Lon,       start=(/1, 1/)))
    CALL handle_error(nf90_get_var(reference%ncid, reference%var_Lat,   Lat,       start=(/1, 1/)))
    CALL handle_error(nf90_get_var(reference%ncid, reference%var_Hflux, Hflux,     start=(/1, 1/)))
    CALL handle_error(nf90_get_var(reference%ncid, reference%var_Hs,    Hs,        start=(/1, 1/)))

    ! When a restart is performed, these variables are also read from a file
    IF(PRESENT(Hi)) CALL handle_error(nf90_get_var(reference%ncid, reference%var_Hi, Hi, start=(/1, 1/)))
    IF(PRESENT(Hb)) CALL handle_error(nf90_get_var(reference%ncid, reference%var_Hb, Hb, start=(/1, 1/)))

  END SUBROUTINE read_extra_file



!!-- Added BdB (05/2016): for reading 3D ocean temperature
  SUBROUTINE read_ocean_temp_file(reference, recno, Tocean, d)
  
    ! Input variables:
    INTEGER,                                INTENT(IN)  :: recno     ! The record number in the file from which to read the variables.
    TYPE(netcdf_file_type_ant),             INTENT(IN)  :: reference ! The file from which to read.
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: Tocean
    REAL(dp),                               INTENT(OUT) :: d

    ! Read the variables:
    CALL handle_error(nf90_get_var(reference%ncid, reference%var_d,          d,          start=(/      recno/)))
    CALL handle_error(nf90_get_var(reference%ncid, reference%var_Tocean,     Tocean,     start=(/1, 1, recno/)))
  END SUBROUTINE read_ocean_temp_file



  SUBROUTINE write_restart_file(NX, NY, recno, Hi, Hb, Hs, Us, Vs, bm, Ti)

    ! Input variables:
    INTEGER,                          INTENT(IN) :: NX
    INTEGER,                          INTENT(IN) :: NY
    INTEGER,                          INTENT(IN) :: recno
    REAL(dp), DIMENSION(NX,NY),       INTENT(IN) :: Hi, Hs, Hb, Us, Vs, bm
    REAL(dp), DIMENSION(NX,NY,C%NZ),  INTENT(IN) :: Ti

   ! Units in years:
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_t, (/ T_ANT%time /), start=(/ recno /))) 
    
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_Hi, Hi, start=(/1, 1,    recno/)))
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_Hb, Hb, start=(/1, 1,    recno /)))
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_Hs, Hs, start=(/1, 1,    recno /)))
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_Us, Us, start=(/1, 1,    recno /)))
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_Vs, Vs, start=(/1, 1,    recno /)))
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_bm, bm, start=(/1, 1,    recno /)))
    CALL handle_error(nf90_put_var(restart%ncid, restart%var_Ti, Ti, start=(/1, 1, 1, recno /)))

    ! Synchronize to make sure that data is really on disk (in case the program crashes later on):
    CALL handle_error(nf90_sync(restart%ncid))
  END SUBROUTINE write_restart_file



  SUBROUTINE write_help_fields_file( recno                , &
                                     MB_surface           , &
                                     MB_basal             , &
                                     mask                 , &
                                     calve_mask           , &
                                     dHs_dx               , &
                                     dHs_dy               , &
                                     D_2D                 , &
                                     A_flow_mean          , &
                                     dummy1_2D            , &
                                     dummy2_2D            , &
                                     dummy3_2D            , &
                                     dummy4_2D            , &
                                     dummy5_2D            , &
                                     dummy6_2D            , &
                                     D_uv_3D              , &
                                     A_flow               , &
                                     U                    , &
                                     V                    , &
                                     W                    , &
                                     Cpi                  , &
                                     Ki                   , &
                                     dummy1_3D            , &
                                     dummy2_3D            , &
                                     dummy3_3D            , &
                                     dummy4_3D            , &
                                     dummy5_3D            , &
                                     dummy6_3D)

  
    ! Input variables:
    INTEGER,                              INTENT(IN) :: recno

    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: MB_surface
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: MB_basal
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: mask
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: calve_mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: dHs_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: dHs_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: D_2D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: A_flow_mean
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: dummy1_2D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: dummy2_2D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: dummy3_2D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: dummy4_2D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: dummy5_2D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant),       INTENT(IN) :: dummy6_2D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: D_uv_3D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: A_flow
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: U
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: V
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: W
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: Cpi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: Ki
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: dummy1_3D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: dummy2_3D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: dummy3_3D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: dummy4_3D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: dummy5_3D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ),  INTENT(IN) :: dummy6_3D

!     write (0,*) 'write help_fields file; ncid = ', help_fields%ncid    

    ! Write variables to file: 
    
    ! Units in years:
    CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_t, (/ T_ANT%time /), start=(/ recno /)))

    CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_MB_surface, MB_surface, start=(/1,1,recno/)))
    IF(C%which_help_fields >=  2) &
    CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_MB_basal, MB_basal, start=(/1,1,recno/)))
    IF(C%which_help_fields >=  3) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_mask, mask, start=(/1,1,recno/)))
    IF(C%which_help_fields >=  4) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_calve_mask, calve_mask, start=(/1,1,recno/)))
    IF(C%which_help_fields >=  6) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dHs_dx, dHs_dx, start=(/1,1,recno/)))
    IF(C%which_help_fields >=  6) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dHs_dy, dHs_dy, start=(/1,1,recno/)))
    IF(C%which_help_fields >=  7) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_D_2D, D_2D, start=(/1, 1, recno /)))
    IF(C%which_help_fields >=  8) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_A_flow_mean, A_flow_mean, start=(/1, 1, recno /)))
    IF(C%which_help_fields >=  9) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy1_2D, dummy1_2D, start=(/1,1,recno/)))
    IF(C%which_help_fields >= 10) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy2_2D, dummy2_2D, start=(/1,1,recno/)))
    IF(C%which_help_fields >= 11) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy3_2D, dummy3_2D, start=(/1,1,recno/)))
    IF(C%which_help_fields >= 12) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy4_2D, dummy4_2D, start=(/1,1,recno/)))
    IF(C%which_help_fields >= 13) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy5_2D, dummy5_2D, start=(/1,1,recno/)))
    IF(C%which_help_fields >= 14) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy6_2D, dummy6_2D, start=(/1,1,recno/)))
    IF(C%which_help_fields >= 40) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_D_uv_3D, D_uv_3D, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 41) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_A_flow, A_flow, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 45) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_U, U, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 46) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_V, V, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 47) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_W, W, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 51) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_Cpi, Cpi, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 52) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_Ki, Ki, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 61) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy1_3D, dummy1_3D, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 62) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy2_3D, dummy2_3D, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 63) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy3_3D, dummy3_3D, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 64) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy4_3D, dummy4_3D, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 65) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy5_3D, dummy5_3D, start=(/1, 1, 1, recno /)))
    IF(C%which_help_fields >= 66) &
     CALL handle_error(nf90_put_var(help_fields%ncid, help_fields%var_dummy6_3D, dummy6_3D, start=(/1, 1, 1, recno /)))

    ! Synchronize to make sure that data is really on disk (in case the program crashes later on):
    CALL handle_error(nf90_sync(help_fields%ncid))

  END SUBROUTINE write_help_fields_file

  ! Count the number of records in a restart file (start counting from 1 instead of 0):
  SUBROUTINE count_netcdf_record_size(netcdf, record_size)
    IMPLICIT NONE
    
    ! Input variables:
    TYPE(netcdf_file_type_ant), INTENT(IN)  :: netcdf

    ! Output variable:
    INTEGER,                    INTENT(OUT) :: record_size

    ! Local variables:
    INTEGER                             :: dim_t
    
    CALL handle_error(nf90_inq_dimid(netcdf%ncid, name_T, dim_t))
    CALL handle_error(nf90_inquire_dimension(netcdf%ncid, dim_t, len=record_size))
  END SUBROUTINE count_netcdf_record_size



  ! Count the number of records in the monthly and extra fields files
  SUBROUTINE count_netcdf_record_size2(netcdf, record_size)
    IMPLICIT NONE
    
    ! Input variables:
    TYPE(netcdf_file_type_ant), INTENT(IN)  :: netcdf

    ! Output variable:
    INTEGER,                    INTENT(OUT) :: record_size

    ! Local variables:
    INTEGER                             :: dim_t
    
    CALL handle_error(nf90_inq_dimid(netcdf%ncid, name_t2, dim_t))
    CALL handle_error(nf90_inquire_dimension(netcdf%ncid, dim_t, len=record_size))
  END SUBROUTINE count_netcdf_record_size2



  ! Close a netcdf file, used for the initial file
  SUBROUTINE close_netcdf_file(netcdf_file)
    IMPLICIT NONE
  
    ! Input/Output variables:
    TYPE(netcdf_file_type_ant), INTENT(INOUT) :: netcdf_file

    ! Close netCDF file:
    CALL handle_error(nf90_close(netcdf_file%ncid))
   
  END SUBROUTINE close_netcdf_file

  ! Close the restart file, created and writing within this module
  SUBROUTINE close_restart_file()
    IMPLICIT NONE


    ! Close restart file:
    CALL handle_error(nf90_close(restart%ncid))
   
  END SUBROUTINE close_restart_file


  ! Close the help_fields file, created and writing within this module
  SUBROUTINE close_help_fields_file()
    IMPLICIT NONE


    ! Close help_fields file:
    CALL handle_error(nf90_close(help_fields%ncid))
   
  END SUBROUTINE close_help_fields_file

  SUBROUTINE handle_error(stat, message)
    USE netcdf, ONLY: nf90_noerr, nf90_strerror
    IMPLICIT NONE
    
    ! Input variables:
    INTEGER,                    INTENT(IN) :: stat
    CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: message

    IF(stat /= nf90_noerr) THEN
     IF(PRESENT(message)) THEN
      WRITE(0,'(A,A,A,A)') 'ERROR: netCDF failed (in restart_file_ant_module) because: ', TRIM(nf90_strerror(stat)), ' concerning: ', message
     ELSE
      WRITE(0,'(A,A)')     'ERROR: netCDF failed (in restart_file_ant_module) because: ', TRIM(nf90_strerror(stat))
     END IF
     STOP
    END IF
    
  END SUBROUTINE handle_error
END MODULE restart_file_ant_module
