MODULE reference_fields_ant_module
  USE configuration_main_module, ONLY: dp, C
  IMPLICIT NONE
 
  REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: Hi_ref
  REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: Hb_ref
  REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: Hs_ref
  REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: Ts_ref
  REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: Iso_ref

  ! Allocate and save initial monthly fields of Temperature and Precipitation
  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: Tmon_ref      ! Monthly temperature
  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: Pmon_ref      ! Monthly precipitation

  ! Allocate and save latitude and longitude fields (Added BdB 12/2009)
  REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: Latitude      ! Latitude  (degrees N)
  REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: Longitude     ! Longitude (degrees E)

  ! Allocate and save geothermal heatflux field 
  REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: Heatflux      ! Geothermal heatflux (W m-2)

  REAL(dp),                                SAVE :: vaf_ref       ! total reference value of volume above flotation
  REAL(dp),                                SAVE :: slev_ref      ! total reference value of sea level contribution

  ! Allocate and save monthly anaomlies fields of Temperature and Precipitation
  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: Tmon_clim     ! Monthly temperature time series
  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: Pmon_clim     ! Monthly precipitation time series

  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: ocean_temp    ! 3D ocean temperatures (Celcius)
  REAL(dp), DIMENSION(    :), ALLOCATABLE, SAVE :: ocean_depth   ! ocean depth (m)

  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: SWdown_ref    ! Monthly SW flux downward at the surface

CONTAINS
  SUBROUTINE read_reference_fields(Hi, Hb)
    ! This subroutine reads the reference data (e.g. present day data) from a netcdf file

    USE restart_file_ant_module, ONLY: netcdf_file_type_ant, open_netcdf_file,  &
          close_netcdf_file, inquire_monthly_file, read_monthly_file,           &
          inquire_extra_file, read_extra_file, read_ocean_temp_file, inquire_ocean_temp_file

    USE parameters_main_module, ONLY: ice_density, seawater_density

    ! input fields from the initial fields
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hi   ! Ice thickness (m)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hb   ! Bedrock elevation (m)
        
    TYPE(netcdf_file_type_ant)                          :: monthly_file     ! monthly file necessary to start a run
    TYPE(netcdf_file_type_ant)                          :: extra_file       ! additional fields
    TYPE(netcdf_file_type_ant)                          :: climate_file     ! monthly climate fields
    TYPE(netcdf_file_type_ant)                          :: ocean_temp_file  ! ocean_temp file for sub-shelf oceanic melt

    ! for reading monthly data:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: T_one_month     ! 2-meter air Temperature (Kelvin)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: P_one_month     ! Precipitation (m / month)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: T_one_depth     ! Ocean temperature at one depth (degrees C)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: S_one_month     ! SW flux down for one month (W m-2)

    INTEGER                                             :: m,i,j,d
    REAL(dp)                                            :: depth

    ! Allocate the global variables:
    ALLOCATE(Hi_ref      (C%NX_ant,C%NY_ant)   )
    ALLOCATE(Hb_ref      (C%NX_ant,C%NY_ant)   )
    ALLOCATE(Hs_ref      (C%NX_ant,C%NY_ant)   )
    ALLOCATE(Ts_ref      (C%NX_ant,C%NY_ant)   )
    ALLOCATE(Iso_ref     (C%NX_ant,C%NY_ant)   )

    ! Allocate monthly variables
    ALLOCATE(Tmon_ref    (C%NX_ant,C%NY_ant,12))
    ALLOCATE(Pmon_ref    (C%NX_ant,C%NY_ant,12))
        
    ! Allocate longitude, latitude and heatlfux fields (Added BdB 12/2009)
    ALLOCATE(Latitude    (C%NX_ant,C%NY_ant)   )
    ALLOCATE(Longitude   (C%NX_ant,C%NY_ant)   )
    ALLOCATE(Heatflux    (C%NX_ant,C%NY_ant)   )

    ! Allocate monthly climate fields
    ALLOCATE(Tmon_clim   (C%NX_ant,C%NY_ant,C%climate_fields_length))
    ALLOCATE(Pmon_clim   (C%NX_ant,C%NY_ant,C%climate_fields_length))

    ALLOCATE(SWdown_ref  (C%NX_ant,C%NY_ant,12))

    ! Allocate 3D ocean temperatures
    ALLOCATE(ocean_temp  (C%NX_ant,C%NY_ant,C%number_of_ocean_layers))
    ALLOCATE(ocean_depth (                  C%number_of_ocean_layers))
    
!    IF (.NOT. C%restart_model .AND. C%choice_forcing /= C%climate_model_forcing) THEN
    IF (.NOT. C%restart_model) THEN
      WRITE (0,'(A)') 'set reference Hi and Hb from the initial file'
      Hi_ref = Hi
      Hb_ref = Hb
    END IF

!!---- Added BdB 11/2009: read in monthy reference fields for Bintanja Mass balance routine


      ! For the coupling with a climate model (now FAMOUS), read the 120 time steps of temperature and precipitation 
      ! These cover monthly data over 10 time steps, each every 100 years (so 1000 years).
!      IF(C%choice_forcing == C%climate_model_forcing) THEN
!
!        ! Output: climate fields
!        CALL open_netcdf_file(C%climate_fields_ant_filename, climate_file)
!
!        ! In/Output: monthly climate time series
!        CALL inquire_monthly_file(climate_file)
!
!        DO m=1,C%climate_fields_length
!          IF(C%use_sw_down_flux_at_surface) THEN
!            ! Output: Tmon_ref, Pmon_ref, Smon_ref
!            CALL read_monthly_file(climate_file, m, T_one_month, P_one_month, S_one_month)
!            ! save SW downflux at surface
!            SWdown_ref(:,:,m) = S_one_month
!          ELSE
!            ! Output: Tmon_clim, Pmon_clim
!            CALL read_monthly_file(climate_file, m, T_one_month, P_one_month)
!          END IF
!          ! Save the two fields    
!          Tmon_clim(:,:,m) = T_one_month
!          Pmon_clim(:,:,m) = P_one_month
!        END DO
!
!        ! In/Output: monthly_file
!        CALL close_netcdf_file(climate_file)
!
!        ! Mean temperature over the year (take first 12 months)
!        DO j = 1, C%NY_ant
!        DO i = 1, C%NX_ant
!          Ts_ref(i,j) = SUM(Tmon_clim(i,j,1:12)) / 12._dp
!        END DO
!        END DO
!
!      ELSE

        ! Output: monthly_file 
        CALL open_netcdf_file(C%monthly_ant_filename, monthly_file)

        ! In/Output: monthly_file 
        CALL inquire_monthly_file(monthly_file)

        DO m=1,12
          IF(C%use_sw_down_flux_at_surface) THEN
            ! Output: Tmon_ref, Pmon_ref, Smon_ref
            CALL read_monthly_file(monthly_file, m, T_one_month, P_one_month, S_one_month)
            ! save SW downflux at surface
            SWdown_ref(:,:,m) = S_one_month
          ELSE
            ! Output: Tmon_ref, Pmon_ref
            CALL read_monthly_file(monthly_file, m, T_one_month, P_one_month)
          END IF
          ! Save the two fields for temperature and precipitation
          Tmon_ref(:,:,m) = T_one_month
          Pmon_ref(:,:,m) = P_one_month
        END DO

        ! In/Output: monthly_file
        CALL close_netcdf_file(monthly_file)

        ! Mean temperature over the year
        DO j = 1, C%NY_ant
        DO i = 1, C%NX_ant
          Ts_ref(i,j) = SUM(Tmon_ref(i,j,1:12)) / 12._dp
        END DO
        END DO

!      END IF

      ! Calculate reference field of d18O of precipitation (Zwally and Giovinetto, 1997):
      Iso_ref = 0.852_dp * Ts_ref - 239.507_dp

!!---- Added BdB 12/2009: read in latitude/longitude fields for insolation

      ! Output: extra_file 
      CALL open_netcdf_file(C%extra_fields_ant_filename, extra_file)

!      IF(C%restart_model .OR. C%choice_forcing == C%climate_model_forcing) THEN
      IF(C%restart_model) THEN
        WRITE (0,'(A)') 'set reference Hi and Hb from the reference file'
        ! In/Output: extra_file 
        CALL inquire_extra_file(extra_file, .TRUE.)
        ! Output: Lon, Lat, hflux, Hs, Hi, Hb
        CALL read_extra_file(extra_file, Longitude, Latitude, Heatflux, Hs_ref, Hi_ref, Hb_ref)
      ELSE
        ! In/Output: extra_file 
        CALL inquire_extra_file(extra_file, .FALSE.)
        ! Output: Lon, Lat, hflux, Hi
        CALL read_extra_file(extra_file, Longitude, Latitude, Heatflux, Hs_ref)
      END IF
      
       ! In/Output: extra_file
      CALL close_netcdf_file(extra_file)
      
      ! For reference calculate the initial contribution to sea level, just ice on land
      slev_ref = (ice_density / seawater_density) * SUM(Hi_ref, MASK = ((ice_density/seawater_density) * Hi_ref > -Hb_ref))
      
      ! For reference calculate the initial contribution to sea level, ice thickness above flotation, corrected for bedrock change
      vaf_ref = SUM((ice_density / seawater_density) * Hi_ref, MASK = ((ice_density/seawater_density) * Hi_ref > -Hb_ref)) & 
              + SUM(min(0._dp,Hb_ref),                         MASK = ((ice_density/seawater_density) * Hi_ref > -Hb_ref)) &
              - SUM(min(0._dp,Hb_ref))


!!---- Added BdB 05/2016: read in 3D ocean temperatures for shelf melting
      
    IF(C%use_3D_ocean_temperature) THEN

      ! Output: ocean_temp_file 
      CALL open_netcdf_file(C%ocean_temp_ant_filename, ocean_temp_file)

      ! In/Output: ocean_temp_file 
      CALL inquire_ocean_temp_file(ocean_temp_file)

      ! Output: Tocean (3D)
      DO d = 1,C%number_of_ocean_layers
        CALL read_ocean_temp_file(ocean_temp_file, d, T_one_depth, depth)
        ! save the fields
        ocean_depth(    d) = depth
        ocean_temp (:,:,d) = T_one_depth
        !write (0,'(i5,f12.3)') d, ocean_depth(d)
      END DO
      
      ! In/Output: ocean_temp_file
      CALL close_netcdf_file(ocean_temp_file)
      
    END IF

  END SUBROUTINE read_reference_fields


  ! Initialize new reference fields when starting to use SELEN, after a spin up of 1 glacial cycle from 410-330 kyr ago
  SUBROUTINE change_reference_fields_for_selen(Hi, Hb, Hs, Ts, Tmon)

    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hb
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hs
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Ts
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,12), INTENT(IN)  :: Tmon
    
    ! Define new reference fields to be used by SELEN
    Hi_ref   = Hi
    Hb_ref   = Hb
    Hs_ref   = Hs
    Iso_ref  = 0.852_dp * Ts - 239.507_dp
    Tmon_ref = Tmon
    Ts_ref   = Ts

  END SUBROUTINE change_reference_fields_for_selen

END MODULE reference_fields_ant_module
