MODULE oceanmelt_ant_module
  USE configuration_main_module, ONLY: dp, C
  USE insolation_forcing_module, ONLY: insolation_data
  IMPLICIT NONE

CONTAINS
  SUBROUTINE iceshelf_ocean_melt(Hi, Hb, mask, sealevel, gT_offset, MB_basal)
  ! Added BdB 03/2011: parametrisation of sub-ice-shelf ocean melt as described by Martin et al. (TCD, 2010),
  ! as function of temperature difference between shelf and ocean
  ! Use a weighting function (from Pollard & DeConto, 2009) to scale the ocena temperature beneath the shelves

  USE parameters_main_module, ONLY: T0, L_fusion, ice_density, seawater_density
  USE reference_fields_ant_module, ONLY: ocean_temp, ocean_depth
  
    ! Input variables: 
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: mask                        ! mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hi                          ! ice thickness (m)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hb                          ! Height of bedrock (m)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: sealevel                    ! sea level relative to PD
    REAL(dp),                               INTENT(IN)  :: gT_offset                   ! surface-air temperature relative to PD
    
    ! local variables:
    REAL(dp)                                            :: weight                      ! weighting index for changes in ocean melt
    REAL(dp)                                            :: T_ocean                     ! temperature beneath the  shelves [Celcius]
    REAL(dp)                                            :: M_deep                      ! melt for deep-ocean areas [m/year]
    REAL(dp)                                            :: M_expo                      ! melt for exposed shelf [m/year]
    REAL(dp)                                            :: z_deep                      ! deep-ocean weighting
    REAL(dp)                                            :: z_expo                      ! exposed-shelf weighting
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: sub_angle                   ! substended angle to open ocean (degrees)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)              :: dist_open                   ! distance to closest open-ocean point (m)

    INTEGER                                             :: i,j,k

    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: MB_basal                    ! Basal melt at ice-shelf

    ! PISM basal melting:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)      :: T_freeze                    ! Freezing temperature at the base of the shelf (Celcius)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)      :: S_flux                      ! melt flux from ocean to shelf
    REAL(dp), PARAMETER                         :: cp0        = 3974._dp       ! specific heat capacity of the ocean mixed layer (J kg-1 K-1) 
    REAL(dp), PARAMETER                         :: gamma_T    = 1.0E-04_dp     ! Thermal exchange velocity (m s-1)
    
    ! See Excel spreadsheet Melt_weighting_d18O_Q80S.xlsx for test, parameters are still subject to change.. (BdB 22-03-2011)
    REAL(dp), PARAMETER                         :: T_ocean_CD = -5._dp         ! cold period temperature of the ocean beneath the shelves [Celcius]
    REAL(dp), PARAMETER                         :: T_ocean_PD = -1.7_dp        ! present day temperature of the ocean beneath the shelves [Celcius]
    REAL(dp), PARAMETER                         :: T_ocean_WM =  2._dp         ! warm period temperature of the ocean beneath the shelves [Celcius]
            
    REAL(dp), PARAMETER                         :: M_deep_CD =  2._dp          ! cold period value for deep-ocean areas [m/year]
    REAL(dp), PARAMETER                         :: M_deep_PD =  5._dp          ! present day value for deep-ocean areas [m/year]
    REAL(dp), PARAMETER                         :: M_deep_WM = 10._dp          ! warm period value for deep-ocean areas [m/year]    
 
    REAL(dp), PARAMETER                         :: M_expo_CD =  0._dp          ! cold period value for exposed areas [m/year]
    REAL(dp), PARAMETER                         :: M_expo_PD =  3._dp          ! present day value for exposed areas [m/year]
    REAL(dp), PARAMETER                         :: M_expo_WM =  6._dp          ! warm period value for exposed areas [m/year]

    REAL(dp), DIMENSION(4)                      :: exposed                     ! exposed area of groundling line points
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)      :: F_melt_grline               ! percentage of the grounded ice which is exposed to oceanic melting

    ! Optional for AO-GCM: use a 2-D ocean field (from 3D temperatures: Maris, 2013/2014)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant)      :: T_ocean_2D                 ! 2D ocean temperatures
    REAL(dp)                                    :: shelf_depth                ! depth of shelf below 
    REAL(dp)                                    :: d_rest

!!---- Calculate the minimum distance and subtended angle to the open ocean
    ! initialize values:
    dist_open = 0._dp
    sub_angle = 360._dp

!DoE 02/06/2017
    DO j = 3, C%NY_ant-2
    DO i = 3, C%NX_ant-2
      IF (mask(i,j) > C%type_groundline .AND. Hi(i,j) > C%Hi_min) THEN 
        ! Determine the closest ocean grid point for all directions. 
        dist_open(i,j) = distance_open_ocean(i, j, mask, Hi)
      
        ! Determine the subtanded andist_opengle, i.e. the angle to open ocean grid points
        sub_angle(i,j) = subtanded_angle(i, j, mask, Hi)
      END IF  
    END DO
    END DO


    ! Determine the weighting index for changes in ocean melt as function of time;
    ! - two variables vary through time: January 80S insolation (relative to the PD value of 532.19 W m-2)
    ! - and sea level (is already relative to PD)
    weight = MAX(0._dp, MIN(2._dp, 1._dp + gT_offset/12._dp + MAX(0._dp, (insolation_data%Jan80S - 532.19_dp)/40._dp)))
    
    IF (weight >= 0._dp .AND. weight < 1._dp) THEN
      T_ocean = weight * T_ocean_PD + (1._dp - weight) * T_ocean_CD
      M_deep  = weight * M_deep_PD  + (1._dp - weight) * M_deep_CD
      M_expo  = weight * M_expo_PD  + (1._dp - weight) * M_expo_CD
    ELSE IF (weight >= 1._dp .AND. weight <= 2._dp) THEN
      T_ocean = (2._dp - weight) * T_ocean_PD + (weight - 1._dp) * T_ocean_WM
      M_deep  = (2._dp - weight) * M_deep_PD  + (weight - 1._dp) * M_deep_WM
      M_expo  = (2._dp - weight) * M_expo_PD  + (weight - 1._dp) * M_expo_WM
    ELSE 
      T_ocean = T_ocean_PD
      M_deep  = M_deep_PD
      M_expo  = M_expo_PD
    END IF

    ! Calculate the 2D temperatures from the 3D ocean temperature fields
    IF(C%use_3D_ocean_temperature) THEN
      T_ocean_2D = 0._dp
      ! Maximum depth of input file
      DO j = 1, C%NY_ant
      DO i = 1, C%NX_ant
        ! For all shelf points, interpolate the GCM 3D ocean temperatures to a 2D field
        IF ((mask(i,j) > C%type_groundline .AND. Hi(i,j) > C%Hi_min) ) THEN
          shelf_depth = Hi(i,j) * ice_density / seawater_density
          DO k = 1, C%number_of_ocean_layers-1
            ! Interpolate the ocean temperatures (in Celcius)
            IF (shelf_depth >= ocean_depth(k) .AND. shelf_depth < ocean_depth(k+1)) THEN
              d_rest = (shelf_depth - ocean_depth(k)) / (ocean_depth(k+1) - ocean_depth(k))
              T_ocean_2D(i,j) = (1._dp-d_rest) * ocean_temp(i,j,k) + d_rest * ocean_temp(i,j,k+1)
             !WRITE (111,'(3I5,5F12.3)') i,j,k,T_ocean_2D(i,j),shelf_depth,d_rest,ocean_temp(i,j,k),ocean_temp(i,j,k+1)
           END IF
          END DO
        ELSE IF ((mask(i,j) > C%type_groundline .AND. Hi(i,j) == C%Hi_min) ) THEN
          T_ocean_2D(i,j) = ocean_temp(i,j,1)
        END IF
      END DO
      END DO
    ELSE 
      T_ocean_2D = T_ocean
    END IF



    ! Calculate the two weighting factors for the deep-ocean and exposed shelves
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant    
      ! initialize
      F_melt_grline(i,j) = 0._dp ! when calculated this should first be set to zero for all land points..
      exposed = 0._dp

      ! Find area of grounding line points to determine fraction of oceanic melting
      IF (mask(i,j) == C%type_groundline .and. Hb(i,j) < sealevel(i,j) .and. Hi(i,j) > C%Hi_min) THEN
        IF (mask(i+1,j) == C%type_shelf .AND. Hi(i+1,j) > C%Hi_min) THEN
          IF (Hb(i+1,j) <= Hb(i,j)) exposed(1) = MAX(0._dp,           - Hb(i,j) - (ice_density/seawater_density) * Hi(i+1,j)) + sealevel(i,j)
          IF (Hb(i+1,j) >  Hb(i,j)) exposed(1) = MAX(0._dp, Hb(i+1,j) - Hb(i,j) - (ice_density/seawater_density) * Hi(i+1,j)) + sealevel(i,j)
        ELSE IF (mask(i-1,j) == C%type_shelf .AND. Hi(i-1,j) > C%Hi_min) THEN
          IF (Hb(i-1,j) <= Hb(i,j)) exposed(2) = MAX(0._dp,           - Hb(i,j) - (ice_density/seawater_density) * Hi(i-1,j)) + sealevel(i,j)
          IF (Hb(i-1,j) >  Hb(i,j)) exposed(2) = MAX(0._dp, Hb(i-1,j) - Hb(i,j) - (ice_density/seawater_density) * Hi(i-1,j)) + sealevel(i,j)
        ELSE IF (mask(i,j+1) == C%type_shelf .AND. Hi(i,j+1) > C%Hi_min) THEN
          IF (Hb(i,j+1) <= Hb(i,j)) exposed(3) = MAX(0._dp,           - Hb(i,j) - (ice_density/seawater_density) * Hi(i,j+1)) + sealevel(i,j)
          IF (Hb(i,j+1) >  Hb(i,j)) exposed(3) = MAX(0._dp, Hb(i,j+1) - Hb(i,j) - (ice_density/seawater_density) * Hi(i,j+1)) + sealevel(i,j)
        ELSE IF (mask(i,j-1) == C%type_shelf .AND. Hi(i,j-1) > C%Hi_min) THEN
          IF (Hb(i,j-1) <= Hb(i,j)) exposed(4) = MAX(0._dp,           - Hb(i,j) - (ice_density/seawater_density) * Hi(i,j-1)) + sealevel(i,j)
          IF (Hb(i,j-1) >  Hb(i,j)) exposed(4) = MAX(0._dp, Hb(i,j-1) - Hb(i,j) - (ice_density/seawater_density) * Hi(i,j-1)) + sealevel(i,j)
        END IF

        F_melt_grline(i,j) = SUM(exposed(1:4)) / C%dx_ant ! ratio relative to the surface area of 1 grid box (= dx * dx)

      ! For ocean grid points set the factor to 1
      ELSE IF (mask(i,j) == C%type_shelf) THEN
        F_melt_grline(i,j) = 1._dp
      END IF

      IF (mask(i,j) > C%type_groundline .AND. Hi(i,j) > C%Hi_min) THEN
        ! freezing temperature at the bottom of the ice shelves       ! This is ice shelf thickness below sea level
        T_freeze(i,j) = 0.0939_dp - 0.057_dp * 35._dp - 7.64E-04_dp * Hi(i,j) * ice_density / seawater_density

        ! basal melting, mass flux, from shelf to ocean (Martin, TC, 2011) - melt values, when T_ocean > T_freeze.
        S_flux(i,j)   = seawater_density * cp0 * C%sec_per_year * gamma_T * C%subshelf_melt_factor * &
                        (T_ocean_2D(i,j) - T_freeze(i,j)) / (L_fusion * ice_density)  !!!* F_melt_grline(i,j)

      ELSE
        S_flux(i,j) = 0._dp
      END IF

      IF (mask(i,j) > C%type_groundline) THEN
        z_deep = MAX(0._dp, MIN(1._dp, (sealevel(i,j) - Hb(i,j) - 1800._dp)/200._dp))   ! changed to 1800 meters
        z_expo = MAX(0._dp, MIN(1._dp, (sub_angle(i,j) - 80._dp)/30._dp)) * EXP(-dist_open(i,j)*C%dx_ant/100000._dp)
        
        MB_basal(i,j) = (z_deep - 1._dp) * ( (1._dp - z_expo) * S_flux(i,j) + z_expo * M_expo) - z_deep * M_deep
        
      ELSE  
        MB_basal(i,j) = 0._dp
      END IF

    END DO
    END DO



  END SUBROUTINE iceshelf_ocean_melt


  FUNCTION distance_open_ocean(i_shelf, j_shelf, mask, Hi) RESULT(open_distance)
    ! Determine the distance to the open ocean. A  solution is found for all 16 directions
    ! also when encountering land points, keep searching to find a grid point. 
    ! The minimum value is the solution

    ! Input
    INTEGER,                                INTENT(IN)  :: j_shelf, i_shelf     ! only shelf grid points are used
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: mask                 ! mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hi                   ! ice thickness (m)
    
    ! local variables
    REAL(dp), DIMENSION(16)                             :: distance             ! list of all distances for 16 directions (m)
    INTEGER                                             :: iloop,imax           ! loop integer and maximum value (max of NX/NY)
    INTEGER                                             :: ii,jj
    
    ! output, result
    REAL(dp)                                            :: open_distance
    
    imax = MAX(C%NX_ant,C%NY_ant)

!      for 16 directions (sub-loops):
!
!              12 13 14
!             11 ---- 15
!            10 --  -- 16
!           09 -- ij -- 01
!            08 --  -- 02 
!             07 ---- 03
!              06 05 04 
!

   
      ! direction 01: (should work fine with imax, no possibility that jj or ii will exceed nx or ny)
      ! but what about when jj or ii are lower than 1..



      DO iloop = 1,imax
        ii = i_shelf
        jj = j_shelf + iloop
         
          ! first check if the boundaries of the grid are reached
          IF (jj > C%NY_ant) THEN
            distance(1) = 100._dp
            EXIT
          END IF  
          
          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(1) = real(jj) - real(j_shelf)
            EXIT
          END IF    
        END DO

        ! direction 02:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf + 2*iloop
          IF (jj > C%NY_ant .OR. ii < 1) THEN
            distance(2) = 100._dp
            EXIT
          END IF  
          
          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(2) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 03:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf + iloop
          IF (jj > C%NY_ant .OR. ii < 1) THEN
            distance(3) = 100._dp
            EXIT
          END IF  

          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(3) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 04:
        DO iloop = 1,imax   
          ii = i_shelf - 2*iloop
          jj = j_shelf + iloop
          IF (jj > C%NY_ant .OR. ii < 1) THEN
            distance(4) = 100._dp
            EXIT
          END IF  
          
          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(4) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 05:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf
          IF ( ii < 1) THEN
            distance(5) = 100._dp
            EXIT
          END IF  

          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(5) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 06:
        DO iloop = 1,imax   
          ii = i_shelf - 2*iloop
          jj = j_shelf - iloop
          IF (jj < 1 .OR. ii < 1) THEN
            distance(6) = 100._dp
            EXIT
          END IF
          
          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(6) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 07:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf - iloop
          IF (jj < 1 .OR. ii < 1) THEN
            distance(7) = 100._dp
            EXIT
          END IF 
                   
          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(7) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 08:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf - 2*iloop
          IF (jj < 1 .OR. ii < 1) THEN
            distance(8) = 100._dp
            EXIT
          END IF
                    
          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(8) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO
        ! direction 09:
        DO iloop = 1,imax   
          ii = i_shelf 
          jj = j_shelf - iloop
          IF (jj < 1) THEN
            distance(9) = 100._dp
            EXIT
          END IF
                    
          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(9) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 10:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf - 2*iloop
          IF (jj < 1 .OR. ii > C%NX_ant) THEN
            distance(10) = 100._dp
            EXIT
          END IF
                    
          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(10) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 11:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf - iloop
          IF (jj < 1 .OR. ii > C%NX_ant) THEN
            distance(11) = 100._dp
            EXIT
          END IF

          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(11) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 12:
        DO iloop = 1,imax   
          ii = i_shelf + 2*iloop 
          jj = j_shelf - iloop
          IF (jj < 1 .OR. ii > C%NX_ant) THEN
            distance(12) = 100._dp
            EXIT
          END IF

          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(12) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 13:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf
          IF (ii > C%NX_ant) THEN
            distance(13) = 100._dp
            EXIT
          END IF

          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(13) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 14:
        DO iloop = 1,imax   
          ii = i_shelf + 2*iloop 
          jj = j_shelf + iloop
          IF (jj > C%NY_ant .OR. ii > C%NX_ant) THEN
            distance(14) = 100._dp
            EXIT
          END IF

          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(14) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 15:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf + iloop
          IF (jj > C%NY_ant .OR. ii > C%NX_ant) THEN
            distance(15) = 100._dp
            EXIT
          END IF

          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(15) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! direction 16:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf + 2*iloop
          IF (jj > C%NY_ant .OR. ii > C%NX_ant) THEN
            distance(16) = 100._dp
            EXIT
          END IF

          IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            distance(16) = SQRT( (real(jj) - real(j_shelf))**2 + (real(ii) - real(i_shelf))**2 )
            EXIT
          END IF    
        END DO

        ! Calculate minimum distance
        open_distance = MINVAL(distance(1:16))

  END FUNCTION distance_open_ocean



  FUNCTION subtanded_angle(i_shelf, j_shelf, mask, Hi) RESULT(angle_sub)
    ! Determine the subtanded angle to the open ocean. When encountering land points
    ! the angle is zet to zero (i.e. a loss of 1/16th of the fully possible 360). 
    ! The minimum value is the solution
    
    ! Input
    INTEGER,                                INTENT(IN)  :: j_shelf, i_shelf     ! only shelf grid points are used
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: mask                 ! mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hi                   ! ice thickness (m)
    
    ! local variables
    REAL(dp), DIMENSION(16)                     :: angle                !  list of open-ocean angles for 16 directions (degrees)
    INTEGER                                     :: iloop,imax           ! loop integer and maximum value (max of NX/NY)
    INTEGER                                     :: ii,jj
    
    ! output, result
    REAL(dp)                                    :: angle_sub

    
    imax = MAX(C%NX_ant,C%NY_ant)
    
    ! replace calve_thickness with Hi_min??
    
        ! direction 01: (should work fine with imax, no possibility that jj or ii will exceed nx or ny)
        ! but what about when jj or ii are lower than 1..



        DO iloop = 1,imax   
          ii = i_shelf
          jj = j_shelf + iloop
          
          ! first check if the boundaries of the grid are reached
          IF (jj > C%NY_ant) THEN
            angle(1) = 0._dp
            EXIT
          END IF  
          
          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(1) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(1) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 02:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf + 2*iloop
          IF (jj > C%NY_ant .OR. ii < 1) THEN
            angle(2) = 0._dp
            EXIT
          END IF  
          
          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(2) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(2) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 03:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf + iloop
          IF (jj > C%NY_ant .OR. ii < 1) THEN
            angle(3) = 0._dp
            EXIT
          END IF  

          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(3) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(3) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 04:
        DO iloop = 1,imax   
          ii = i_shelf - 2*iloop
          jj = j_shelf + iloop
          IF (jj > C%NY_ant .OR. ii < 1) THEN
            angle(4) = 0._dp
            EXIT
          END IF  
          
          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(4) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(4) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 05:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf
          IF ( ii < 1) THEN
            angle(5) = 0._dp
            EXIT
          END IF  

          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(5) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(5) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 06:
        DO iloop = 1,imax   
          ii = i_shelf - 2*iloop
          jj = j_shelf - iloop
          IF (jj < 1 .OR. ii < 1) THEN
            angle(6) = 0._dp
            EXIT
          END IF
          
          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(6) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(6) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 07:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf - iloop
          IF (jj < 1 .OR. ii < 1) THEN
            angle(7) = 0._dp
            EXIT
          END IF 
                   
          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(7) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(7) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 08:
        DO iloop = 1,imax   
          ii = i_shelf - iloop
          jj = j_shelf - 2*iloop
          IF (jj < 1 .OR. ii < 1) THEN
            angle(8) = 0._dp
            EXIT
          END IF
                    
          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(8) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(8) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 09:
        DO iloop = 1,imax   
          ii = i_shelf 
          jj = j_shelf - iloop
          IF (jj < 1) THEN
            angle(9) = 0._dp
            EXIT
          END IF
                    
          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(9) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(9) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 10:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf - 2*iloop
          IF (jj < 1 .OR. ii > C%NX_ant) THEN
            angle(10) = 0._dp
            EXIT
          END IF
                    
          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(10) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(10) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 11:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf - iloop
          IF (jj < 1 .OR. ii > C%NX_ant) THEN
            angle(11) = 0._dp
            EXIT
          END IF

          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(11) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(11) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 12:
        DO iloop = 1,imax   
          ii = i_shelf + 2*iloop 
          jj = j_shelf - iloop
          IF (jj < 1 .OR. ii > C%NX_ant) THEN
            angle(12) = 0._dp
            EXIT
          END IF

          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(12) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(12) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 13:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf
          IF (ii > C%NX_ant) THEN
            angle(13) = 0._dp
            EXIT
          END IF

          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(13) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(13) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 14:
        DO iloop = 1,imax   
          ii = i_shelf + 2*iloop 
          jj = j_shelf + iloop
          IF (jj > C%NY_ant .OR. ii > C%NX_ant) THEN
            angle(14) = 0._dp
            EXIT
          END IF

          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(14) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(14) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 15:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf + iloop
          IF (jj > C%NY_ant .OR. ii > C%NX_ant) THEN
            angle(15) = 0._dp
            EXIT
          END IF

          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(15) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(15) = 1._dp
            EXIT
          END IF    
        END DO

        ! direction 16:
        DO iloop = 1,imax   
          ii = i_shelf + iloop 
          jj = j_shelf + 2*iloop
          IF (jj > C%NY_ant .OR. ii > C%NX_ant) THEN
            angle(16) = 0._dp
            EXIT
          END IF

          IF (mask(ii,jj) /= C%type_shelf) THEN
            angle(16) = 0._dp
            EXIT
          ELSE IF (mask(ii,jj) == C%type_shelf .AND. Hi(ii,jj) <= C%Hi_min) THEN
            angle(16) = 1._dp
            EXIT
          END IF    
        END DO

        ! Calculate total subtanded angle
        angle_sub = SUM(angle(1:16)) * 360._dp / 16._dp

  END FUNCTION subtanded_angle
END MODULE oceanmelt_ant_module
