MODULE mb_surface_bintanja_ant_module
  USE configuration_main_module, ONLY: C, dp, Insolation_Adjusted, Model_Time
  USE configuration_ant_module, ONLY: T_ANT
  IMPLICIT NONE
  
CONTAINS
  SUBROUTINE surface_massbalance_Bintanja(T2m_mon, Hi, mask, MB_surface, accumulation, ablation, refreezing, Mean_Tsurf, transmis)

    ! Added by Bas de Boer (October/November 2009)
    ! Mass balance parametrisation according to Bintanja and Van de Wal (2005/2008)
    !  - uses an precipitation-field adjusted with the C-C relation (as function of free-atmospheric temperature)
    !  - reads in insolation for all latitude to calculate Q at TOA (W/m2)
    !  - and an albedo/temperature depended ablation parametrisation
    USE reference_fields_ant_module, ONLY: Tmon_ref, Pmon_ref, Latitude, Tmon_clim, Pmon_clim, SWdown_ref
    USE parameters_main_module, ONLY: T0, ice_density, L_fusion
    USE insolation_forcing_module, ONLY: insolation_data
     
    IMPLICIT NONE

    ! Input variables: 
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,12), INTENT(IN)  :: T2m_mon        ! monthly 2-meter surface air temperature
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hi             ! Ice thickness (m)
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: mask           ! mask

    ! Output variables:         
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: MB_surface     ! Surface Mass Balance [m ice equivalent per year]
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: accumulation
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: ablation
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: refreezing
    REAL(dp),                                  INTENT(OUT) :: Mean_Tsurf     ! Continental mean surface air temperature (Kelvin)

    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: transmis       ! transmissivity of the atmosphere
  
    ! Local variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   )              :: T2m           ! mean of T2m_mon: for writing in help field
    INTEGER                                                :: i,j,j2,m, step
    INTEGER                                                :: Nmean
    
    ! Tinv: inversion Temperature (at 1000m above surface), where the condensation takes place  
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,12)              :: Tinv_ref,Tinv

    ! Ablation/Accumulation variables
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,12)             :: Precip         ! Precipitation in meters per year
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   )             :: Accum,Abla     ! Accumulation and Ablation in mieq per year
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   )             :: Snow           ! Depth of snow layer (from mass balance)
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   )             :: Refr, Rain     ! Refreezing and rain in m s.e. per year
    REAL(dp)                                              :: snowfrac       ! Snow fraction of rain

    ! Monthly and yearly values of ablation, accumulation, snow depth and liquid water (for refreezing)
    REAL(dp), DIMENSION(12)                               :: abla_month, acc_month, lwat_month, snow_month
    REAL(dp)                                              :: abla_year,  acc_year, snow_year

    REAL(dp)                                              :: sup_imp_wat    ! Super imposed water, maximum amount of refreezing
    REAL(dp), DIMENSION(12)                               :: refr_month     ! Effective refreezing = MIN(SI,LW) < Pmon
    REAL(dp)                                              :: refr_year, rain_year
    
    ! Allocatable and saved data:
    REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE         :: Snow_prev      ! Snow layer of previous year (to use at the start of a new one)
    REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE         :: Abla_prev      ! Ablation of previous year (to use at the start of a new one)
    REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE         :: Q_at_TOA       ! Monthly insolation (W/m2) over icegrid at Top Of the Atmosphere

    ! Local parameters of the albedo
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   )            :: Albedo          ! The parametrized albedo
    REAL(dp)                                             :: alsurf          ! Albedo of the surface (one of the 4 below)

    REAL(dp), PARAMETER :: alwater = 0.10_dp                      ! Albedo of ocean water
    REAL(dp), PARAMETER :: alsoil  = 0.20_dp                      ! Albedo of bare rock
    REAL(dp), PARAMETER :: alice   = 0.50_dp                      ! Albedo of (glacier) ice (changed from 0.45 to 0.50) (55?)
    REAL(dp), PARAMETER :: alsnow  = 0.85_dp                      ! Albedo of snow (changed from 0.80 to 0.85)
  
!!---- Adjust monthly temperature field and calculate monthly free atmospheric temperature
 
    ! Use the precipitation fields from the climate model
!    IF(C%choice_forcing == C%climate_model_forcing) THEN
!      ! changing of steps depends on length of run, current time step and length of the climate fields
!      step = 12 * INT(C%climate_fields_length/12 * (T_ANT%time - C%start_time_of_run + 0.01_dp) / (C%end_time_of_run - C%start_time_of_run))
!
!      do m = 1,12
!        ! Method of Jouzel and Merlivat (1984), see equation (4.82) in Huybrechts (1992):
!        Tinv_ref(:,:,m) = 88.9_dp + 0.67_dp * Tmon_clim(:,:,m+step)
!        Tinv(:,:,m)     = 88.9_dp + 0.67_dp * T2m_mon(:,:,m)
!
!        ! set precipitation
!        Precip(:,:,m) =  Pmon_clim(:,:,m+step) * (Tinv_ref(:,:,m) / Tinv(:,:,m))**2 * &
!                          EXP(22.47_dp * (T0 / Tinv_ref(:,:,m) - T0 / Tinv(:,:,m)))
!      end do
!    ELSE
      ! Method of Jouzel and Merlivat (1984), see equation (4.82) in Huybrechts (1992): 
      Tinv_ref = 88.9_dp + 0.67_dp * Tmon_ref
      Tinv     = 88.9_dp + 0.67_dp * T2m_mon

!---- Adjust precipation as function of Free atmospheric temperature (Tinv)
!      Precip = Pmon_ref * 1.04**(Tinv - Tinv_ref)
      Precip = Pmon_ref * (Tinv_ref / Tinv)**2 * &
                          EXP(22.47_dp * (T0 / Tinv_ref - T0 / Tinv))
                          ! As with Lorius/Jouzel method (also Huybrechts, 2002)
!    END IF

    ! Calculate the insolation at TOA over the ice model grid
    ! This is only needed at the very first time step and at every first time point when the insolation
    ! is adjusted from the main routine (Insolation_Adjusted = .TRUE.). 
    IF (T_ANT%time == C%start_time_of_run .OR. (T_ANT%time == Model_Time .AND. Insolation_Adjusted) ) THEN
!     WRITE (111,'(A,2F12.3)') 'Insolation ANT adjusted at time: ',T_ANT%time, Model_Time

     ! The very first time:
     IF (T_ANT%time == C%start_time_of_run) THEN
       ALLOCATE(Q_at_TOA(C%NX_ant,C%NY_ant,12))
     ! all other moments that Q_at_TOA is adjusted
     ELSE
       DEALLOCATE(Q_at_TOA)
       ALLOCATE(Q_at_TOA(C%NX_ant,C%NY_ant,12))
     END IF

     DO j = 1, C%NY_ant
     DO i = 1, C%NX_ant    
       DO j2=2,181
         IF ( Latitude(i,j) >= insolation_data%Phi(j2-1) .AND. & 
              Latitude(i,j) <= insolation_data%Phi(j2  ) ) THEN
           DO m=1,12
             Q_at_TOA(i,j,m)  = insolation_data%Qins(m,j2-1) * ABS(Latitude(i,j) - insolation_data%Phi(j2  )) + &
                                insolation_data%Qins(m,j2  ) * ABS(Latitude(i,j) - insolation_data%Phi(j2-1))
           END DO
           ! in ISM_3D there is a goto statement here to exit the j2-loop, instead we use exit (should work)
           EXIT 
         END IF
       END DO
     END DO
     END DO
    END IF

    ! initiate Snow depth from reference field of precipitation
    IF (T_ANT%time == C%start_time_of_run ) THEN
      ALLOCATE(Snow_prev(C%NX_ant,C%NY_ant))
      !WHERE (Hi <= C%Hi_min) Snow_prev =  0._dp
      !WHERE (Hi >  C%Hi_min) Snow_prev =  10._dp   
      Snow_prev = 0._dp
      ALLOCATE(Abla_prev(C%NX_ant,C%NY_ant))
      Abla_prev = 0._dp
    END IF

    transmis = 0._dp

    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      abla_year = 0._dp
      acc_year  = 0._dp
      refr_year = 0._dp
      rain_year = 0._dp
      snow_year = Snow_prev(i,j)

      ! background albedo
      IF ( mask(i,j) /= C%type_shelf ) Albedo(i,j) = alsoil
      IF ( mask(i,j) == C%type_shelf ) Albedo(i,j) = alwater
      IF ( Hi(i,j)   >  C%Hi_min     ) Albedo(i,j) = alice
       
      ! ONLY FOR ICE GRID POINTS
      !IF (Hi(i,j) >= C%Hi_min) THEN

       DO m = 1,12          ! Month loop
        ! Compute daily (shape function) of insolation and temperature 
        ! and normalize to be used for all time points

!!----  Determine the albedo for all surface types, for calculating the ablation
        IF ( mask(i,j) /= C%type_shelf ) alsurf = alsoil
        IF ( mask(i,j) == C%type_shelf ) alsurf = alwater
        IF ( Hi(i,j)   >  C%Hi_min     ) alsurf = alice
          
        Albedo(i,j) = MIN(alsnow, MAX(alsurf,alsnow - (alsnow-alsurf) *                 &
                      EXP(-15._dp * snow_year) - 0.015_dp * Abla_prev(i,j)))

!!---- Determine albation as function af surface temperature and albedo/insolation
!      according to Bintanja et al. (2002)   

         ! When using SW downward flux at the surface from the climate input
         IF(C%use_sw_down_flux_at_surface) THEN
          abla_month(m) = ( C%C_abl_Ts * (T2m_mon(i,j,m) - T0) +                        &
                            C%C_abl_Q  * SWdown_ref(i,j,m) * (1.0_dp - Albedo(i,j)) -     &
                            C%C_abl_constant) * C%sec_per_year / (L_fusion * 1000._dp * 12._dp)

          ! check transmissivity between TOA insolation and at the surface, only from Nov-Feb
          if (m==1 .or. m==2 .or. m==11 .or. m==12) then
          if (SWdown_ref(i,j,m) /= 0._dp .AND. Q_at_TOA(i,j,m) /= 0._dp) then 
           transmis(i,j) = transmis(i,j) + SWdown_ref(i,j,m) / (4._dp * Q_at_TOA(i,j,m))
          end if
          end if

        ELSE
          abla_month(m) = ( C%C_abl_Ts * (T2m_mon(i,j,m) - T0) +                        &
                            C%C_abl_Q  * Q_at_TOA(i,j,m) * (1.0_dp - Albedo(i,j)) -     &
                            C%C_abl_constant) * C%sec_per_year / (L_fusion * 1000._dp * 12._dp)
        END IF
        ! Here, surface air temperature is used (T2m_mon) which is not adjusted to T0
        IF (abla_month(m) < 0._dp) abla_month(m) = 0._dp

!!---- Determine accumulation with snow/rain fraction from Ohmura et al. (1999),
!      liquid water content (rain and melt water) and snowdepth

        snowfrac = 0.5_dp * (1._dp - ATAN((T2m_mon(i,j,m) - T0)/3.5_dp)/1.25664_dp)
        IF (snowfrac > 1._dp ) snowfrac = 1._dp
        IF (snowfrac < 0._dp ) snowfrac = 0._dp
        
        acc_month(m)  = Precip(i,j,m) * snowfrac
        lwat_month(m) = Precip(i,j,m) * (1._dp - snowfrac) + abla_month(m)

!!---- Refreezing, according to Janssens & Huybrechts, 2000)
!      The refreezing (=effective retention) is the minimum value of the amount of super imposed 
!      water and the available liquid water, with a maximum value of the total precipitation.
!      (see also Huybrechts & de Wolde, 1999)

        sup_imp_wat   = 0.012_dp * MAX(T0 - T2m_mon(i,j,m), 0._dp)
        refr_month(m) = MIN(MIN(sup_imp_wat, lwat_month(m)), Precip(i,j,m))

        ! Monthly snowfall, adjusted according to the current snowdepth
        snow_month(m) = acc_month(m) - abla_month(m)
        
        ! Sum up monthly values to a yearly value which is saved in the total field
        snow_year = MIN(10._dp, MAX(0._dp, snow_year + snow_month(m)))
        abla_year = abla_year + abla_month(m)
        acc_year  = acc_year  + acc_month(m)
        refr_year = refr_year + refr_month(m)
        rain_year = rain_year + Precip(i,j,m) * (1._dp - snowfrac)
                
        ! set values to zero for ocean gridpoints (use calve_mask??)
        IF (mask(i,j) == C%type_shelf .AND. Hi(i,j) <= C%Hi_min) THEN
          snow_year = 0._dp
          refr_year = 0._dp
          rain_year = 0._dp
        END IF

       END DO ! end of m-loop for the months        

      !END IF ! ice points    
      
      Abla(i,j)     = abla_year
      Accum(i,j)    = acc_year
      Snow(i,j)     = snow_year
      Refr(i,j)     = refr_year
      Rain(i,j)     = rain_year

      ! calculate T 2 meter for writing to help field
      T2m(i,j) = SUM(T2m_mon(i,j,1:12)) / 12._dp
    END DO    ! end of j-loop
    END DO    ! end of i-loop

      ! Save mass balance terms for output in m w.e. per year
      accumulation = Accum * 1000._dp / ice_density
      ablation     = Abla  * 1000._dp / ice_density
      refreezing   = Refr  * 1000._dp / ice_density

      ! Calculate total surface mass balance
      MB_surface = (Accum - Abla + Refr) * 1000._dp / ice_density
      
      ! First, deallocate previous list of snow_prev and abla_prev
      DEALLOCATE(Snow_prev, Abla_prev)

      ! Then, allocate new list
      ALLOCATE(Snow_prev(C%NX_ant,C%NY_ant), Abla_prev(C%NX_ant,C%NY_ant))

      Snow_prev = Snow
      Abla_prev = Abla
      
      ! Calculate mean surface temperature, for writing in recording_fields
      Nmean      = COUNT(mask /= C%type_shelf)
      Mean_Tsurf = SUM(T2m, MASK = (mask /= C%type_shelf)) / dble(Nmean) - T0

  END SUBROUTINE surface_massbalance_Bintanja

END MODULE mb_surface_bintanja_ant_module
