MODULE massbalance_ant_module
  USE configuration_main_module, ONLY: dp, C
  USE configuration_ant_module, ONLY: C_ANT
  IMPLICIT NONE
  
CONTAINS
  SUBROUTINE surface_massbalance(T2m_mon, Hi, mask, MB_surface, Accumulation, Ablation, Refreezing, Mean_Tsurf, transmis)
                                    
    ! This routine takes one of the options to calculate the (air-ice) surface mass balance (MB)
    ! BdB (10/2009): Added surface mass balance parametrisation according to Bintanja/Van de Wal (2005/2008)
    ! subroutine is called from mb_surface_bintantja_module.f90
    USE parameters_main_module, ONLY: T0                                    ! T0 = triple point of water
    USE mb_surface_bintanja_ant_module, ONLY: surface_massbalance_Bintanja  ! surface MB routine from ISM_3D
    
    ! Input variables: 
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: mask       ! mask                                           
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,12), INTENT(IN)  :: T2m_mon    ! 2-meter air temperature [K]  
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hi         ! ice thickness

    ! Output variables:         
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: MB_surface ! Surface Mass Balance [m ice equivalent per year]
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: Accumulation
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: Ablation
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: Refreezing
    REAL(dp),                                  INTENT(OUT) :: Mean_Tsurf ! Continental mean surface air temperature (Kelvin)

    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: transmis       ! transmissivity of the atmosphere
    
    SELECT CASE(C%choice_MB_surface_model)
      CASE(10)
       ! Output: MB_surface, and three MB_terms (default: Accumulation, Ablation, Snow depth)
       CALL surface_massbalance_Bintanja(T2m_mon, Hi, mask, MB_surface, Accumulation, Ablation, Refreezing, Mean_Tsurf, transmis)
      CASE DEFAULT
       STOP ' Stop message from: surface_massbalance:  A non valid choice for C%choice_MB_surface_model'
    END SELECT
   
   END SUBROUTINE surface_massbalance     


  SUBROUTINE basal_massbalance(mask, Hi, Hb, Hs, sealevel, gT_offset, MB_basal)
    ! This routine calculates the basal mass balance (MB_basal). This basal mass balance
    ! exists in principle of two contributions at the shelf: refreezing of ocean water and
    ! basal melt due to the ocean water contact. 
    ! At the sheet there can be basal melt as well due to the pressure, frictional heating and 
    ! the rock contact (the geothermal heat flux), this is (can be) done in the thermodynamics_module
    USE oceanmelt_ant_module, ONLY: iceshelf_ocean_melt
    
    ! Input variables: 
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: mask       ! mask                                           
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hs         ! Height of surface                              
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hi         ! Ice thickness                          
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hb         ! Height of bedrock                           
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: sealevel   ! sea level relative to PD
    REAL(dp),                               INTENT(IN)  :: gT_offset  ! Temperature relative to PD    
    
    ! Output variable:         
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: MB_basal   ! Mass Balance [m ice equivalent]

    SELECT CASE(C_ANT%choice_MB_basal_model)
    CASE(1)
     ! The basal-shelf-melt is substracted at shelf points: a constant:
     WHERE(mask == C%type_shelf) 
               MB_basal = - C_ANT%basal_shelf_melt
     ELSEWHERE
               MB_basal = 0._dp
     END WHERE                              
    CASE(2)
     ! The basal-shelf-melt is substracted at shelf points: linear and cutoff:
     WHERE(mask == C%type_shelf) 
               MB_basal = - MIN(C_ANT%basal_shelf_melt *  (Hs - Hi - Hb) / 1000._dp    , C_ANT%basal_shelf_melt)             
     ELSEWHERE
               MB_basal = 0._dp
     END WHERE                              
    CASE(3)
     ! The basal-shelf-melt is substracted at shelf points: squared and cutoff:
     WHERE(mask == C%type_shelf) 
               MB_basal = - MIN(C_ANT%basal_shelf_melt * ((Hs - Hi - Hb) / 2000._dp)**2, C_ANT%basal_shelf_melt)        
     ELSEWHERE
               MB_basal = 0._dp
     END WHERE
     
    CASE(4)
     ! Added BdB (11/2009): parameterization of sub-ice-shelf ocean melt (Pollard and DeConto, 2009)
     CALL iceshelf_ocean_melt(Hi, Hb, mask, sealevel, gT_offset, MB_basal)

    CASE DEFAULT
     STOP ' Stop message from basal_massbalance:  A non valid choice for C_ANT%choice_MB_basal_model'
    END SELECT
   END SUBROUTINE basal_massbalance 

      
END MODULE massbalance_ant_module
