MODULE isotopes_ant_module
  USE configuration_main_module, ONLY: C, dp
  USE configuration_ant_module, ONLY: T_ANT
  IMPLICIT NONE

CONTAINS
  SUBROUTINE oxygen_isotope_icesheet(Ts,Hi_old,Hi,Hs,calve_mask,IsoIce_adv,MeanIsoHi,VolIso,fluxes_per_dt)
    ! Added by Bas de Boer (November 2010).
    !
    ! using the implicit ice fluxes (vertical averaged) and applied mass balance fields from the 
    ! ice thickness subroutine to calculate the advected isotope flux from all 4 directions.
    !
    ! Because of the dynamic shelf, we can also calculate IsoIce over the shelf area and threat 
    ! all ice covered gridpoints the same since we are using vertically averaged velocities
    !
    ! IsoIce_adv should be zero for no ice (or at least have a small value, just like Hi)
    ! As in all subroutines, arrays are used as i,j (x,y)
    
    USE reference_fields_ant_module, ONLY: Ts_ref, Hs_ref, Iso_ref
    USE parameters_main_module, ONLY:  Atmos_lapserate_low, Atmos_lapserate_high
        
    IMPLICIT NONE

    ! Input variables: 
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Ts               ! Surface temperature of ice [K]
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Hi               ! Ice thickness [m]
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Hi_old           ! Ice thickness at previous time step [m]
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Hs               ! Height of the surface (m) 
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, 6), INTENT(IN)  :: fluxes_per_dt

    INTEGER,  DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: calve_mask       ! mask with specific values for ice sheet and shelf

    ! Output variables:         
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   )              :: IsoSurf          ! Surface Isotopic depletion from precipitation
    REAL(dp),                                   INTENT(OUT) :: MeanIsoHi        ! Weighted mean of IsoIce_adv * Hi
    REAL(dp),                                   INTENT(OUT) :: VolIso           ! Total ice volume used to calculate MeanIsoHi
    
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(OUT) :: IsoIce_adv       ! d18O of ice (time-derivative equation)
    REAL(dp)                                                :: MeanIso          ! Mean of IsoSurf
    REAL(dp)                                                :: MeanIso2         ! Mean of IsoIce_adv
        
    ! Local variables:
    INTEGER                                                 :: i,j
    INTEGER                                                 :: nmean            ! counters for averaging
    INTEGER                                                 :: nmean2,nmean3

    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   )              :: MB_iso           ! IsoIce weighted by the surface MB
    
    ! Allocatable and saved data:
    REAL(dp), DIMENSION(:,:), ALLOCATABLE, SAVE             :: IsoIce           ! d18O of ice (time-derivative equation)    

    REAL(dp)                                                :: IsoMin,IsoMax    ! minimum and maximum value of IsoIce
    REAL(dp)                                                :: HiIso,HiTot      ! for weighted average of IsoIce
    REAL(dp)                                                :: MeanIso_Sheet    ! Mean d18O of ice-sheet (including grouding line)
    REAL(dp)                                                :: MeanIso_Shelf    ! Mean d18O of ice-shelf

    REAL(dp)                                                :: Alr, h15         ! Atmospheric lapse rate and 1500 m threshold height
    REAL(dp)                                                :: hmin, hmax       ! used to calculate Alr
  
    h15 = 1500._dp             
    
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      ! Determine the lapse rate, should be the same as used in the ice_surface_temperature routine,
      ! Although thermodynamics are calculated each 4 years...
      IF (Hi(i,j) > C%Hi_min) THEN
        hmin = MIN(Hs(i,j), Hs_ref(i,j))
        hmax = MAX(Hs(i,j), Hs_ref(i,j))
        IF (hmax <= h15) THEN
          Alr = Atmos_lapserate_low
        ELSE IF (hmin >= h15) THEN
          Alr = Atmos_lapserate_high
        ELSE ! hmin < h15 < hmax
          Alr = ((h15-hmin)* Atmos_lapserate_low + (hmax-h15) * Atmos_lapserate_high) / (hmax - hmin)
        END IF
        
        IsoSurf(i,j) = Iso_ref(i,j)                                     &
                       + 0.6_dp    * (Ts(i,j) - Ts_ref(i,j)             &
                       + Alr       * (Hs(i,j) - Hs_ref(i,j)))           &
                       - 0.0112_dp * (Hs(i,j) - Hs_ref(i,j))            ! from Clarke et al., 2005  
      ELSE
        IsoSurf(i,j) = 0._dp ! = No ice
      END IF           

    END DO
    END DO
    
    IsoMax = Maxval(IsoSurf)
    IsoMin = Minval(IsoSurf)

!!-- Initialize the mean d18O of the ice at the first time step
      IF (T_ANT%time == C%start_time_of_run) THEN
        ALLOCATE(IsoIce(C%NX_ant,C%NY_ant))
      
        WHERE ( Hi > C%Hi_min ) 
          IsoIce = IsoSurf
        ELSE WHERE
          IsoIce = 0._dp
        END WHERE
      END IF

!!-- Calculate the mass gain/loss of d18O
    WHERE ( fluxes_per_dt(:,:,6) > 0._dp ) 
      MB_iso = fluxes_per_dt(:,:,6) * IsoSurf    ! (applied MB) mass gain, so d18O from precipitation
    ELSE WHERE
      MB_iso = fluxes_per_dt(:,:,6) * IsoIce     ! (applied MB) mass loss, so d18O from ice
    END WHERE

!!---- Calculate the new d18O_ice from the ice fluxes and applied mass balance

    ! Initialize IsoIce_adv with IsoSurf
    IsoIce_adv = IsoSurf

    DO j = 2, C%NY_ant - 1
    DO i = 2, C%NX_ant - 1
      IF ( Hi(i,j) > C%Hi_min ) THEN  ! for all points with ice
      
        IsoIce_adv(i,j) = (Hi_old(i,j)*IsoIce(i,j) + MB_iso(i,j)                                            &
                                                   + fluxes_per_dt(i-1,j  ,1) * IsoIce(i-1,j  )             &   ! from the left
                                                   + fluxes_per_dt(i+1,j  ,2) * IsoIce(i+1,j  )             &   ! from the right
                                                   + fluxes_per_dt(i  ,j-1,3) * IsoIce(i  ,j-1)             &   ! from the bottom
                                                   + fluxes_per_dt(i  ,j+1,4) * IsoIce(i  ,j+1)             &   ! from the top
                                                   - fluxes_per_dt(i  ,j  ,5) * IsoIce(i  ,j  ) ) / Hi(i,j)     ! leaving this gp

        ! Check for outliers (can happen due to low Hi, and changes from small to high d18O values)
        IF (IsoIce_adv(i,j) > IsoMax) IsoIce_adv(i,j) = IsoMax
        IF (IsoIce_adv(i,j) < IsoMin) IsoIce_adv(i,j) = IsoMin

      END IF
    
    END DO
    END DO


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!  Calculate Mean ice-sheet Isotopic content (also separate for shelf/sheet)   !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    MeanIso  = 0._dp
    MeanIso2 = 0._dp    
    HiIso    = 0._dp
    HiTot    = 0._dp
    nmean    = 0
    
    MeanIso_Sheet  = 0._dp
    MeanIso_Shelf  = 0._dp
    nmean2 = 0
    nmean3 = 0

    DO j = 1, C%NY_ant    
    DO i = 1, C%NX_ant      
      ! For all points with ice
      IF ( Hi(i,j) > C%Hi_min .AND. calve_mask(i,j) <= C%type_calveline ) THEN 
        MeanIso  = MeanIso  + IsoSurf(i,j)
        MeanIso2 = MeanIso2 + IsoIce_adv(i,j)
        nmean    = nmean    + 1
        HiIso    = HiIso    + Hi(i,j) * IsoIce_adv(i,j)
        HiTot    = HiTot    + Hi(i,j)
      END IF

      ! For all ice-sheet gridpoints 
      IF ( Hi(i,j) > C%Hi_min .AND. calve_mask(i,j) <= C%type_groundline ) THEN
        MeanIso_Sheet = MeanIso_Sheet + IsoSurf(i,j)
        nmean2 = nmean2 + 1
      END IF

      ! For all ice-shelf gridpoints
      IF ( calve_mask(i,j) == C%type_shelf .OR. calve_mask(i,j) == C%type_calveline ) THEN
        MeanIso_Shelf = MeanIso_Shelf + IsoSurf(i,j)
        nmean3 = nmean3 + 1
      END IF    

    END DO
    END DO
    
    ! Average of IsoSurf
    IF (nmean /= 0) THEN
      MeanIso  = MeanIso  / dble(nmean)
      MeanIso2 = MeanIso2 / dble(nmean)
    ELSE
      MeanIso  = 0._dp
      MeanIso2 = 0._dp
    END IF
    
    ! Weighted average of IsoIce_adv with Hi
    IF (HiTot /= 0._dp) THEN
      MeanIsoHi = HiIso / HiTot
      VolIso    = HiTot * C%dx_ant * C%dy_ant
    ELSE
      MeanIsoHi = 0._dp
      VolIso    = 0._dp
    END IF

    ! Average of IsoSurf for ice-sheet
    IF (nmean2 /= 0) THEN
      MeanIso_Sheet = MeanIso_Sheet / dble(nmean2)
    ELSE
      MeanIso_Sheet = 0._dp
    END IF

    ! Average of IsoSurf for ice-shelf
    IF (nmean3 /= 0) THEN
      MeanIso_Shelf = MeanIso_Shelf / dble(nmean3)
    ELSE
      MeanIso_Shelf = 0._dp
    END IF
    
    DEALLOCATE(IsoIce)
    
    ALLOCATE(IsoIce(C%NX_ant,C%NY_ant))
    
    IsoIce = IsoIce_adv

    ! Output of IsoSurf (zero for no ice):
    WHERE (Hi <= C%Hi_min) IsoSurf = 0._dp
 
  END SUBROUTINE oxygen_isotope_icesheet

END MODULE isotopes_ant_module
